# -------------------------------------------------------------------------
# MODULE:      HierarchyEditor
#
# DESCRIPTION: 
#     Contains the HierarchyEditor class.
# 
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>
#

import vp, Xmd

from TreeNode         import TreeNode
from HierarchyBrowser import HierarchyBrowser
from WidgetButton     import PushButton, ArrowButton
from Box              import Box, Glue
from Frame            import Frame
from TextControls     import TextField
from Menu             import MenuBar, PopupMenu

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         HierarchyEditor
#
# INHERITS FROM: HierarchyBrowser
#
# DESCRIPTION: 
#     An editor for browsing and editing tree structures. The nodes
#     should be of type TreeNode.  
#
#     Args can be the following: 
#        - 'hierarchy'  : <oid>, default=None
#		 - 'show_path'  : [TRUE=default | FALSE]
#        - 'path_label' : <string>, default='     Path:'
#        - 'show_leaf'  : [TRUE=default | FALSE]
#        - 'leaf_label' : <string>, default='Leaf node:'
#
#     Callbacks:
#         - 'curr_node' ,    cb-data = <the current node>
#         - 'node_deleted' , cb-data = <the node to be deleted>
#

NODE_ADDED   = 1
NODE_DELETED = 2

class HierarchyEditor( HierarchyBrowser ):

	# ------------------------------------------------------------------
	# Initialization methods

	def __init__( self, argdict = {} ):
		self.new_node = None
		self.change   = 0
		HierarchyBrowser.__init__( self, argdict )


	def CreateGraphicObjects( self, argdict = {} ): 
		alignment = vp.VLEFT
		children = HierarchyBrowser.CreateGraphicObjects(self, argdict)
		if argdict.has_key( 'alignment' ): 
			alignment = argdict['alignment']

		browser_box = Box({ 
			  'alignment' : vp.VLEFT,
			  'child_list': children
			  })
		if alignment in [vp.VLEFT, vp.VCENTER, vp.VRIGHT]: 
			al = vp.HTOP
		else:
			al = vp.VLEFT
		edit_box = Box({ 
			  'alignment'     : al,
			  'stretchability': (vp.ELASTIC, vp.FIXED),
			  'child_list'    : [
				  self._CreateCurrNodeBox(),
				  self._CreateNewNodeBox()]
			  })

		return [browser_box, Glue(10, vp.FIXED), edit_box] 


	def _CreateCurrNodeBox( self ):
		self.curr_node_field = field = TextField({ 'editable': vp.FALSE })
		self.curr_node_menu  = PopupMenu({
			  'title'  : 'Actions',
			  'enabled': vp.FALSE,
			  'items'  : [('Delete', self.DeleteCurrentCB)]
			  })
		abut = ArrowButton({ 
			  'popup'         : self.curr_node_menu, 
			  'stretchability': (vp.FIXED, vp.FIXED),
			  'xres'          : {'arrowDirection': Xmd.ARROW_DOWN}
			  })
		return Frame({
			  'label': 'Current node',
			  'child': Box({ 'child_list': [field, abut] }) 
			  })


	def _CreateNewNodeBox( self ):
		self.new_node_field = field = TextField({ 'editable': vp.FALSE })
		self.new_node_menu  = PopupMenu({
			  'title'  : 'Actions',
			  'enabled': vp.FALSE,
			  'items'  : [('Add to current',     self.AddToCurrentCB),
			              ('Add after current',  self.AddAfterCurrentCB),
						  ('Add before current', self.AddBeforeCurrentCB)]
			  })
		abut = ArrowButton({ 
			  'popup'         : self.new_node_menu, 
			  'stretchability': (vp.FIXED, vp.FIXED),
			  'xres'          : {'arrowDirection': Xmd.ARROW_DOWN}
			  })
		return Frame({
			  'label': 'New node',
			  'child': Box({ 'child_list': [field, abut] }) 
			  })		
		

	# ------------------------------------------------------------------
	# Misc public methods

	def SetNewNode( self, tnode ):
		if not tnode == None:
			if not tnode.IsA( TreeNode ):
				print 'HierarchyEditor.SetNewNode: new node is not a TreeNode!'
				return

			if self.hierarchy == None:
				self.SetHierarchyRoot( tnode )
			else:
				self.new_node_field.SetText( tnode.GetName() )
				self.new_node = tnode
				if self.curr_node != None: self.EnableNewNodeMenuItems()
		else:
			self.new_node_field.SetText( '' )
			self.new_node = None
			self.new_node_menu.Disable()
			

	# ------------------------------------------------------------------
	# Callbacks for user input

	def DeleteCurrentCB( self, button ):
		if self.curr_node != None:
			pn = self.curr_node.parent
			if self.curr_node == self.hierarchy: self.hierarchy = None
			self.curr_node.Finalize()
			self.ExecuteCallback( 'node_deleted', self.curr_node )
			self.change = NODE_DELETED
			self.DoUpdate( pn )


	def AddToCurrentCB( self, button ):
		if self.curr_node != None and self.new_node != None:
			self.change = NODE_ADDED
			self.curr_node.AddChild( self.new_node )
			self.DoUpdate( self.new_node )
			self.SetNewNode( None )


	def AddAfterCurrentCB( self, button ):
		if self.curr_node != None and self.new_node != None:
			self.change = NODE_ADDED
			self.curr_node.parent.AddChildAfter(self.new_node, self.curr_node)
			self.DoUpdate( self.new_node )
			self.SetNewNode( None )


	def AddBeforeCurrentCB( self, button ):
		if self.curr_node != None and self.new_node != None:
			self.change = NODE_ADDED
			self.curr_node.parent.AddChildBefore(self.new_node, self.curr_node)
			self.DoUpdate( self.new_node )
			self.SetNewNode( None )
		

	# ------------------------------------------------------------------
	# Misc private methods

	def SetCurrentNode( self, node ):
		HierarchyBrowser.SetCurrentNode( self, node )
		if node != None:
			self.curr_node_field.SetText( node.GetName() )
			self.curr_node_menu.Enable()
		else:
			self.curr_node_field.SetText( '' )
			self.curr_node_menu.Disable()


	def DoUpdate( self, node ):
		if self.change in [NODE_ADDED,NODE_DELETED]:
			if node == None:
				self.SetListItems( 1, None )
				self.SetListItems( 2, None )
				self.curr_node_menu.Disable()
				self.new_node_menu.Disable()
				
			elif node.IsRoot():
				self.SetListItems( 1, node )
				self.SetListItems( 2, None )

			elif node.parent.GetName() == self.label1.GetText():
				self.SetListItems( 1, node.parent )
				if node.HasChildren():
					self.SetListItems( 2, node )
				else:
					self.SetListItems( 2, None )

			elif node.parent.parent.GetName() == self.label1.GetText():  
				self.SetListItems( 1, node.parent.parent )
				self.SetListItems( 2, node.parent )

			elif node.parent.GetName() == self.label2.GetText():
				self.SetListItems( 2, node.parent )

			self.change = 0
			self.SetCurrentNode( node )
			self.SetPath( node )
		else:
			HierarchyBrowser.DoUpdate( self, node )


	def EnableNewNodeMenuItems( self ):
		if not self.curr_node.IsRoot():
			self.new_node_menu.items['Add after current'].Enable()
			self.new_node_menu.items['Add before current'].Enable()
		else:
			self.new_node_menu.items['Add after current'].Disable()
			self.new_node_menu.items['Add before current'].Disable()

		if self.curr_node.IsLeafOnly():
			self.new_node_menu.items['Add to current'].Disable()
		else:
			self.new_node_menu.items['Add to current'].Enable()

