# -------------------------------------------------------------------------
# MODULE:      HierarchyBrowser
#
# DESCRIPTION: 
#     Contains the HierarchyBrowser class.
# 
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>
#

import vp, X, Xmd

from TreeNode     import TreeNode
from Box          import Box, Glue
from TextControls import TextList, TextField, LabeledTextField
from WidgetButton import ArrowButton

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         HierarchyBrowser
#
# INHERITS FROM: Box
#
# DESCRIPTION: 
#     A browser for browsing tree structures. The nodes should be 
#     of type TreeNode. 
#
#     Args can be the following: 
#        - 'hierarchy'  : <oid>, default=None
#		 - 'show_path'  : [TRUE=default | FALSE]
#        - 'path_label' : <string>, default='     Path:'
#        - 'show_leaf'  : [TRUE=default | FALSE]
#        - 'leaf_label' : <string>, default='Leaf node:'
#
#     Callbacks:
#        - 'curr_node' , cb-data = <the current node>
#

class HierarchyBrowser( Box ):

	# ------------------------------------------------------------------
	# Initialization methods

	def __init__( self, argdict = {} ):
		self.show_path      = vp.TRUE
		self.path_label     = '     Path:'
		self.show_leaf      = vp.TRUE
		self.leaf_label     = 'Leaf node:'
		self.hierarchy      = None      

		self.path           = []
		self.cursor         = 0
		self.curr_leaf_node = None

		children = self.CreateGraphicObjects( argdict )

		argdict = self.MergeDefaults( argdict, {
			  'alignment' : vp.VLEFT,
			  'child_list': children
			  })
		Box.__init__( self, argdict )

		self.SetHierarchyRoot( self.hierarchy )


	def CreateGraphicObjects( self, argdict = {} ):
		for key in ['show_path', 'path_label', 'show_leaf', 'leaf_label']:
			if argdict.has_key(key):
				setattr(self, key, argdict[key])
				del argdict[key]

		if self.show_path:
			self.path_field = LabeledTextField({
				  'label'         : self.path_label,
				  'editable'      : vp.FALSE,
				  'stretchability': (vp.ELASTIC, vp.FIXED)
				  })
		if self.show_leaf:
			self.leaf_field = LabeledTextField({
				  'label'         : self.leaf_label,
				  'editable'      : vp.FALSE,
				  'stretchability': (vp.ELASTIC, vp.FIXED)
				  })
		al_box = Box({ 'child_list': [
			  self._CreateArrowBox(),
			  self._CreateList1Box(),
			  self._CreateList2Box()]
			  })
		children = [al_box]
		if self.show_path or self.show_leaf:
			children.append( Glue(5, vp.FIXED) )
			if self.show_path: children.append( self.path_field )
			if self.show_leaf: children.append( self.leaf_field )

		return children


	def _CreateList1Box( self ):
		self.label1 = TextField({
			  'editable'      : vp.FALSE,
			  'stretchability': (vp.ELASTIC, vp.FIXED)
			  })
		self.label1.Subscribe( vp.MOUSE_DOWN, self.LabelMouseDownEH, None )
		self.list1 = TextList({
			  'name'    : 'List1',
			  'size'    : (100,100),
			  'order'   : vp.UNSORTED,
			  'callback': self.SelectionCB 
			  })
		return Box({ 
			  'alignment'     : vp.VCENTER, 
			  'child_list'    : [self.label1, self.list1] })		


	def _CreateList2Box( self ):
		self.label2 = TextField({
			  'editable'      : vp.FALSE,
			  'stretchability': (vp.ELASTIC, vp.FIXED)
			  })
		self.label2.Subscribe( vp.MOUSE_DOWN, self.LabelMouseDownEH, None )
		self.list2 = TextList({
			  'name'    : 'List2',
			  'size'    : (100,100),
			  'order'   : vp.UNSORTED,
			  'callback': self.SelectionCB 
			  })
		return Box({ 
			  'alignment' : vp.VCENTER, 
			  'child_list': [self.label2, self.list2] })		


	def _CreateArrowBox( self ):
		al = ArrowButton({ 
			  'callback': self.MoveLeftCB,
			  'xres'    : {'arrowDirection': Xmd.ARROW_LEFT} 
			  })
		ar = ArrowButton({  
			  'callback': self.MoveRightCB,
			  'xres'    : {'arrowDirection': Xmd.ARROW_RIGHT} 
			  })
		return Box({ 
			  'alignment'     : vp.VCENTER, 
			  'stretchability': (vp.FIXED, vp.ELASTIC),
			  'child_list'    : [al,ar] })


	# ------------------------------------------------------------------
	# Public methods

	def SetHierarchyRoot( self, hr ):
		self.hierarchy = hr

		if hr != None:
			if hr.IsA( TreeNode ):
				self.SetPath( hr )
				self.SetListItems( 1, hr )
				self.SetListItems( 2, None )
				self.SetCurrentNode( hr )
			else:
				print 'HierarchyBrowser: the nodes should be of type TreeNode!'
		else:
			self.SetPath( None )
			self.SetListItems( 1, None )
			self.SetListItems( 2, None )
			self.SetCurrentNode( None )


	# ------------------------------------------------------------------
	# Callbacks and event-handlers for user input

	def LabelMouseDownEH( self, label, client_data, xevent, e ):
		name = label.GetText()
		if name not in [None, '']: self.DoUpdate( self.GetNode( name ) )


	def SelectionCB( self, list ):
		name = list.GetSelection()
		if name != None: self.DoUpdate( self.GetNode( name ) )


	def MoveLeftCB( self, button ):
		if self.cursor > 0:
			self.cursor = self.cursor - 1
			self.SetListItems( 1, self.path[self.cursor] )
			self.SetListItems( 2, self.path[self.cursor+1] )


	def MoveRightCB( self, button ):
		if self.cursor < len(self.path) - 2:
			self.cursor = self.cursor + 1
			self.SetListItems( 1, self.path[self.cursor] )
			self.SetListItems( 2, self.path[self.cursor+1] )


	# ------------------------------------------------------------------
	# Private methods

	def GetNode( self, name ):
		if name[len(name)-1] == '>': name = name[0:len(name)-2]

		for node in self.path:
			if name == node.GetName():
				return node
			for node in node.children:
				if name == node.GetName():
					return node


	def DoUpdate( self, node ):
		if debug: print 'DoUpdate: node =', node

		if node.IsRoot():
			self.SetListItems( 1, node )
			self.SetListItems( 2, None )

		elif node.parent.GetName() == self.label1.GetText():  # list1
			self.SetListItems( 1, node.parent )
			if node.HasChildren():
				self.SetListItems( 2, node )
			else:
				self.SetListItems( 2, None )

		elif node.parent.GetName() == self.label2.GetText():  # list2
			if node.HasChildren():
				self.SetListItems( 1, node.parent )
				self.SetListItems( 2, node )
			else:
				self.SetListItems( 1, node.parent.parent )
				self.SetListItems( 2, node.parent )
			
		self.SetCurrentNode( node )
		self.SetPath( node )
			

	def SetListItems( self, list_no, tnode ):
		label = getattr( self, 'label' + `list_no` )
		list  = getattr( self, 'list' + `list_no` )

		if tnode != None:
			label.SetText( tnode.GetName() )
			list.SetList( self.GetListItems( tnode ) )
		else:
			label.SetText( '' )
			list.SetList( [] )
		

	def SetPath( self, node ):
		if node == None:
			self.path   = []
			self.cursor = 0
		elif node.IsRoot():
			self.path   = [node]
			self.cursor = 0
		elif node.HasChildren():
			self.path   = self.GetPath( node )
			self.cursor = self.path.index( node ) - 1
		else:
			self.path   = self.GetPath( node.parent )
			self.cursor = self.path.index( node.parent ) 

		if self.show_path: self.path_field.SetText( self._GetPathString() )


	def SetCurrentNode( self, node ):
		self.curr_node = node
		self.ExecuteCallback( 'curr_node', node )

		if debug: print 'Current node =', node.GetName()

		if self.show_leaf:
			if node != None and node.IsLeaf():
				self.leaf_field.SetText( node.GetName() )
			else:
				self.leaf_field.SetText( '' )


	def GetPath( self, node ):
		path = []
		self._GetPath( path, node )
		if len(path) != 1: path.reverse()
		return path


	def _GetPath( self, path, node ):
		path.append( node )
		if node.parent != None: self._GetPath( path, node.parent )
		

	def _GetPathString( self ):
		str = ''
		for node in self.path:
			if str != '':
				str = str + '/' + node.GetName()
			else:
				str = node.GetName()
		return str
 
		
	def GetListItems( self, node ):
		items = []

		if node != None:
			if node.HasChildren():
				for child in node.children:
					if child.HasChildren():
						items.append( child.GetName() + ' >' )
					else:
						items.append( child.GetName() )
		else:
			if self.hierarchy.HasChildren():
				items.append( self.hierarchy.GetName() + ' >' )
			else:
				items.append( self.hierarchy.GetName() )

		return items

