# -------------------------------------------------------------------------
# MODULE:      GraphicLink
#
# DESCRIPTION: 
#     Contains the Link, ArrowLink and LineLink classes. These classes can
#     be used to link graphical objects. The links will automatically install
#     the needed graphical constraints in order to keep the links attached to
#     the graphical objects.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl
#

import vp

from vp           import TRUE, FALSE
from Object       import Object
from Mixin        import Mixin
from Graphic      import Graphic
from WidgetObject import WidgetObject
from Circle       import Circle
from Lines        import Arrow, Line
from Datum        import Datum, Tuple, Add, Divide
from math         import sin, cos, atan2

debug       = FALSE
debug_final = FALSE

CIRCLE    = 0
RECTANGLE = 1

# -------------------------------------------------------------------------
# CLASS:         Link
#
# INHERITS FROM: Mixin
#
# DESCRIPTION: 
#     Mixin-class for ArrowLink and LineLink.
#

class Link( Mixin ):

	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		#
		# Args can be:
		# - 'begin_graphic'  : <object-id> (required)
		# - 'end_graphic'    : <object-id> (required)
		# - 'name'           : <string> 
		# - 'head_length'    : <int>
		# - 'head_width'     : <int>
		#
		self.wc_mode = TRUE
		apply( self.GetSuperAttr( Link, '__init__' ), (self, argdict ))


	def Realize( self ):
		self.GetSuperAttr( Link, 'Realize' )( self )
		self.InitLink()


	def Finalize( self ):
		if debug_final: print 'Link.Finalize called'

		if self.begin_graphic != None: self.UnLink( self.begin_graphic )
		if self.end_graphic != None: self.UnLink( self.end_graphic )
		self.GetSuperAttr( Link, 'Finalize' )( self )


	def InitLink( self ):
		if not hasattr( self,'begin_graphic' ) and \
			  not hasattr( self, 'end_graphic' ):
			print 'Link.InitLink: link requires a begin_graphic', \
				  'and an end_graphic!'
			return

		if self.begin_graphic == None:
			print 'Link.InitLink: begin_graphic == None'
			self.Hide()
			return

		if self.end_graphic == None:
			print 'Link.InitLink: end_graphic == None'
			self.Hide()
			return

		if debug:
			print 'Link.InitLink: ', \
				  'begin_graphic =', self.begin_graphic.GetName(), \
				  'end_graphic =', self.end_graphic.GetName()

		if not hasattr( self, '_update' ):
			self.AddFunctionDatum( '_update', self.UpdateCB )

		# the link should be notified if any of its graphic objects are deleted

		self.begin_graphic.AddCallback( 'deleted', self.GraphicDeletedCB )
		self.end_graphic.AddCallback( 'deleted', self.GraphicDeletedCB )

		bgt = self.GetGeometryType( self.begin_graphic)
		egt = self.GetGeometryType( self.end_graphic )

		if bgt == CIRCLE and egt == CIRCLE:
			self._Update = self._CC_Update
			self.begin_graphic.wc_x.LinkTo( self._update )
			self.begin_graphic.wc_y.LinkTo( self._update )
			self.end_graphic.wc_x.LinkTo( self._update )
			self.end_graphic.wc_y.LinkTo( self._update )

		elif  bgt == CIRCLE and egt == RECTANGLE:
			self._Update = self._CR_Update
			self.begin_graphic.wc_x.LinkTo( self._update )
			self.begin_graphic.wc_y.LinkTo( self._update )
			self.GetEndHook( self.end_graphic ).LinkTo( self.end )

		elif  bgt == RECTANGLE and egt == CIRCLE:
			self._Update = self._RC_Update
			self.GetBeginHook( self.begin_graphic ).LinkTo( self.begin )
			self.end_graphic.wc_x.LinkTo( self._update )
			self.end_graphic.wc_y.LinkTo( self._update )

		else:
			self.GetBeginHook( self.begin_graphic ).LinkTo( self.begin )
			self.GetEndHook( self.end_graphic ).LinkTo( self.end )

		# The link must only be visible when both the begin- and end-graphic
		# are visible.

		if not(self.begin_graphic.IsVisible() and \
			  self.end_graphic.IsVisible()):
			self.Hide()
		else:
			self.Show()

		self.begin_graphic.AddCallback( 'visible', self.GraphicVisibleCB )
		self.begin_graphic.AddCallback( 'invisible', self.GraphicInVisibleCB )
		self.end_graphic.AddCallback( 'visible', self.GraphicVisibleCB )
		self.end_graphic.AddCallback( 'invisible', self.GraphicInVisibleCB )

	# ------------------------------------------------------------------
	# Callback methods

	def GraphicDeletedCB( self, graphic ):
		if graphic == self.begin_graphic:
			self.UnLink( self.begin_graphic )
		elif graphic == self.end_graphic:
			self.UnLink( self.end_graphic )


	def GraphicVisibleCB( self, graphic ):
		if debug: print 'Link.GraphicVisibleCB called'

		if self.begin_graphic.IsVisible() and self.end_graphic.IsVisible():
			self.Show()


	def GraphicInVisibleCB( self, graphic ): 
		if debug: print 'Link.GraphicInVisibleCB called'

		self.Hide()
		

	# ------------------------------------------------------------------
	# Misc public methods

	def SetBeginGraphic( self, graphic ):
		if self.IsRealized(): 
			self.UnLink( self.begin_graphic )
			self.begin_graphic = graphic
			self.Link( graphic )
		else:
			self.begin_graphic = graphic
		

	def SetEndGraphic( self, graphic ):
		if self.IsRealized(): 
			self.UnLink( self.end_graphic )
			self.end_graphic = graphic
			self.Link( graphic )
		else:
			self.end_graphic = graphic


	# ------------------------------------------------------------------
	# Methods which might be overridden in subclasses

	def GetEndHook( self, graphic ):
		if not hasattr(graphic, 'lhook'):
			if graphic.IsA( WidgetObject ):
				x = graphic.x
				y = graphic.y
			else:
				x = graphic.wc_x
				y = graphic.wc_y
			w = graphic.width
			h = graphic.height
			graphic.AddDatum('lhook', Tuple(x, Add(y, Divide(h, 2) )))

		return graphic.lhook

		
	def GetBeginHook( self, graphic ):
		if not hasattr(graphic, 'rhook'):
			if graphic.IsA( WidgetObject ):
				x = graphic.x
				y = graphic.y
			else:
				x = graphic.wc_x
				y = graphic.wc_y
			w = graphic.width
			h = graphic.height
			graphic.AddDatum('rhook', Tuple( Add(x, w), Add(y, Divide(h, 2)))) 

		return graphic.rhook


	# ------------------------------------------------------------------
	# Misc private methods

	def UnLink( self, graphic ):
		#
		# Unlink the datums 
		#
		gt = self.GetGeometryType( graphic )
		graphic.RemoveCallback( 'visible', self.GraphicVisibleCB )
		graphic.RemoveCallback( 'invisible', self.GraphicInVisibleCB )
		
		if graphic == self.begin_graphic:
			if gt == CIRCLE:
				graphic.wc_x.Unlink( self._update )
				graphic.wc_y.Unlink( self._update )
			else:
				self.GetBeginHook( graphic ).Unlink( self.begin )
			self.begin_graphic = None
		else:
			if gt == CIRCLE:
				graphic.wc_x.Unlink( self._update )
				graphic.wc_y.Unlink( self._update )
			else:
				self.GetEndHook( graphic ).Unlink( self.begin )
			self.end_graphic = None
			

	def Link( self, graphic ):
		if self.begin_graphic == None: 
			self.begin_graphic = graphic
		else: 
			self.end_graphic = graphic
		self.InitLink()
			
		
	def GetGeometryType( self, graphic ):
		if graphic.IsA( Circle ):
			return CIRCLE
		else:
			return RECTANGLE


	# ------------------------------------------------------------------
	# Link constraint methods:
	# ------------------------
	# 
	# The 'begin' and 'end' points of the graphic are set in window coordinates
	# in order to make links that go acrosss 'pseudo-windows' (for example
	# NestableComposites).
	#

	def UpdateCB( self, void ):
		self.parent.RedrawOff()
		self._Update()
		self.parent.RedrawOn()


	def _CC_Update( self ):
		r   = self.begin_graphic.GetWidth()/2
		bbb = self.begin_graphic._GetWcBBox()
		ebb = self.end_graphic._GetWcBBox()
		ob  = (bbb[0]+r, bbb[1]+r)
		oe  = (ebb[0]+r, ebb[1]+r)
		try:
			angle = atan2( oe[1] - ob[1], oe[0] - ob[0] )
			dx    = int(cos( angle ) * r)
			dy    = int(sin( angle ) * r)

			self.begin.Set( (ob[0]+dx, ob[1]+dy) )
			self.end.Set(   (oe[0]-dx, oe[1]-dy) )
		except:
			pass


	def _CR_Update( self ):
		r   = self.begin_graphic.GetWidth()/2
		bbb = self.begin_graphic._GetWcBBox()
		ob  = (bbb[0]+r, bbb[1]+r)

		oe  = self.end.Get()
		try:
			angle = atan2( oe[1] - ob[1], oe[0] - ob[0] )
			dx    = int(cos( angle ) * r)
			dy    = int(sin( angle ) * r)

			self.begin.Set( (ob[0]+dx, ob[1]+dy) )
		except:
			pass


	def _RC_Update( self ):
		ob  = self.begin.Get()

		r   = self.end_graphic.GetWidth()/2
		ebb = self.end_graphic._GetWcBBox()
		oe  = (ebb[0]+r, ebb[1]+r)
		try:
			angle = atan2( oe[1] - ob[1], oe[0] - ob[0] )
			dx    = int(cos( angle ) * r)
			dy    = int(sin( angle ) * r)

			self.end.Set(   (oe[0]-dx, oe[1]-dy) )
		except:
			pass



# -------------------------------------------------------------------------
# CLASS:         ArrowLink
#
# INHERITS FROM: Arrow : Line : Primitive : Graphic : ..
#
# DESCRIPTION: 
#     The ArrowLink is drawn in 'wc_mode' meaning that its begin and end
#     points are given in window (widget) coordinates instead of relative
#     to the (primitive) parent. This is done so that a link can cross 
#     'primitive' borders.
#

class ArrowLink( Link, Arrow ):

	IsPicked   = Graphic.IsPicked


# -------------------------------------------------------------------------
# CLASS:         LineLink
#
# INHERITS FROM: Line : Line : Primitive : Graphic : ..
#
# DESCRIPTION: 
#

class LineLink( Link, Line ):

	IsPicked   = Graphic.IsPicked




