# -------------------------------------------------------------------------
# MODULE:      Graphic
#
# DESCRIPTION: 
#     Contains the Graphic class.
#
# USAGE:      
#     > from Graphic import Graphic
# 
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>

import X
import vp, EventNotifier

from vp        import TRUE, FALSE
from Responder import Responder
from TreeNode  import TreeNode
from Datum     import *

debug       = FALSE
debug_final = FALSE

# -------------------------------------------------------------------------
# CLASS:         Graphic
#
# INHERITS FROM: (TreeNode,Responder,DatumDict) : Object
#
# DESCRIPTION: 
#     Base-class for all graphical objects
#

class Graphic( TreeNode, Responder, DatumDict ):

	# ------------------------------------------------------------------
	# Initialisation.

	def __init__( self, argdict = {} ):
		if hasattr( self, 'eh_dict' ):
			print 'Graphic: init already called for', self.GetName()
			return  

		Responder.__init__( self )
		TreeNode.__init__( self )   # will init DatumDict aswell

		# attributes settable via argdict

		self.draggable       = FALSE 
		self.stretchability  = (vp.ELASTIC, vp.ELASTIC) # w,h
		self.editmode        = FALSE
		self.visible         = TRUE

		# private attributes

		self.realized        = FALSE
		self.show_cntrl_pnts = FALSE
		self._cp             = 0
		self.natural_size    = (0,0)
		self.mouse_pos       = (0,0)
		self.AddDatums()                 # 'x', 'y', 'width', and 'height'
		self.AddFunctionDatum('popup', self.AddPopupCB)
		self.xresources     = {}         # normal X resources
		self.hilight        = FALSE

		self.ProcessArgs( argdict )
			

	def AddDatums( self ):
		#
		# Dummy method which must be overriden in the Widget and Primitive
		# classes. The datums defined here should specify the positon (upper
		# left corner) and the total size, including border, of the graphical
		# object relative to its parent.
		#
		self.AddDatum('wc_x',   Datum(0))
		self.AddDatum('wc_y',   Datum(0))
		self.AddDatum('x',      Datum(0))
		self.AddDatum('y',      Datum(0))
		self.AddDatum('width',  Datum(0))
		self.AddDatum('height', Datum(0))


	def ProcessArgs( self, argdict ):
		if debug: print 'Graphic.ProcessArgs: argdict =', argdict

		for key in argdict.keys():
			if key == 'xres':
				self.SetXResources( argdict[key] ) 

			elif key == 'size':
				(w,h) = argdict[key]
				self.SetSize( w, h )

			elif key == 'pos':
				(x,y) = argdict[key]
				self.SetPosition( x, y )

			elif key == 'draggable':
				self.SetDraggable( argdict[key] )

			elif key == 'editmode':
				self.SetEditMode( argdict[key] )

			elif key in ('name', 'label'):
				self.SetName( argdict[key] )

			elif self.datums.has_key( key ):
				self.datums[ key ].Set( argdict[key] )

			else:
				setattr( self, key, argdict[key] )


	# ------------------------------------------------------------------
	# Public methods for setting and accessing attributes (both datum, 
	# xresources and others). These methods (__setitem__, __getitem__, and 
	# GetAttrKeys) make is possible to set attributes with the dictionary 
	# syntax.

	def __setitem__( self, key, value ):
		if key in self.datums.keys():
			self.datums[key] = value

		elif key == 'size':
			(w,h) = value
			self.SetSize( w, h )
			
		elif key == 'pos': 
			(x,y) = value
			self.SetPosition( x, y )
			
		elif key == 'draggable': 
			self.SetDraggable( value )

		elif key == 'editmode':
			self.SetEditMode( value )

		elif key == 'stretchability':
			(hor,ver) = value
			self.SetStretchability( hor, ver )

		elif key in ('name', 'label'): 
			self.SetName( value )

		elif key == 'xres':
			self.SetXResources( {key: value} )

		else:
			setattr( self, key, value )


	def __getitem__( self, key ):
		if key in self.datums.keys():
			return self.datums[keys].value

		elif key == 'size':
			return self.GetSize()
			
		elif key == 'pos': 
			self.GetPosition()
			
		elif key in ('draggable','editmode','stretchability'):
			return getattr( self, key )

		elif key in ('name', 'label'): 
			self.GetName()

		elif hasattr( self, 'w' ):
			return getattr( self.w, key )

		elif self.xresources.has_key(key):
			return self.xresources[key]

		else:
			return None


	def GetAttrKeys( self ): 
		return ['x','y','width','height','name',
				'size','pos','draggable','editmode']


##	def GetWindow( self ):
##		#
##		# Traverse the graphic hierarchy to find the window
##		#
##		from Window import Window
##
##		if self.IsA( Window ):
##			return self
##		elif self.parent != None:
##			return self.parent.GetWindow()
##		else:
##			return None
##
##


	# ------------------------------------------------------------------
	# Destruction.

	def Finalize( self ):
		if debug_final: print 'Graphic.Finalize called for',self.GetClassName()

		self.Hide()

		# the TreeNode class will Finalize the children

		TreeNode.Finalize( self )      

		# now clean up the rest

		if self in vp.theSelection.slist: vp.theSelection.Remove( self )
		DatumDict.Finalize( self )
		Responder.Finalize( self )

		self.xresources = None
		if hasattr( self, '_popup' ): self._popup = None

		
	# ------------------------------------------------------------------
	# Child management

	def InitAddedNode( self, node ):
		#
		# This will cause the widget or primitive to be realized
		#
		TreeNode.InitAddedNode(self, node)
		if self.IsRealized() and not node.IsRealized():
			self.RedrawOff()
			node.Realize()
			self.RedrawOn()


	def AddPopupCB( self, popup ):
		#
		# Called by the 'popup' datum 
		#
		self._popup = popup
		if popup.parent == None:
			#
			# The following works on Motif 1.1 but not on 1.2. The problem
			# is that the widget of theApplication is a *shell* widget. Adding
			# a popup-menu to a shell widget does not work in Motif 1.2. I
			# have tried the following to make it work on Motif 1.2:
			# 
			# - adding the popup as a child of 'self': this works as long as 
			#   the parent does not have a popup as well
			#
			# - adding the popup as a child of the window: this seems to 
			#   give problems always
			#
			vp.theApplication.AddChild( popup )
##			self.AddChild( popup )
		self.Subscribe( vp.MOUSE_DOWN, self.MouseDownEH, None )


	def AddPopup( self, popup ): self.popup.Set( popup )


	# ------------------------------------------------------------------
	# Drawing methods:

	def Show( self ):               # make visible (manage, popup)
		if not self.visible:
			self.visible = TRUE
			self.ExecuteCallback( 'visible', self )


	def Hide( self ):               # make invisible (unmanage, popdown)
		if self.visible:
			self.visible = FALSE
			self.ExecuteCallback( 'invisible', self )


	def ShowControlPoints( self ):  # turn on editing 
		self.show_cntrl_pnts = TRUE
		self._DrawControlPoints()
		self.Redraw = self._RedrawWithControlPoints

		
	def HideControlPoints( self ):  # turn off editing
		self.show_cntrl_pnts = FALSE
		self._DrawControlPoints()
		self.Redraw = self._Redraw
		

	def _DrawControlPoints( self ):
		if not hasattr(self, 'cp_gc'):
			gcres = {}
			gcres['function']   = X.GXxor
			gcres['foreground'] = self.parent.GetWidget().background

			self.cp_gc = self.parent.GetWidget().GetGC( gcres )

		bb    = self._GetWcBBox()
		bsize = vp.CPB
		self.cp_gc.FillRectangle( bb[0]-bsize+1, bb[1]-bsize+1, bsize, bsize )
		self.cp_gc.FillRectangle( bb[2]-1,       bb[1]-bsize+1, bsize, bsize )
		self.cp_gc.FillRectangle( bb[2]-1,       bb[3]-1,       bsize, bsize )
		self.cp_gc.FillRectangle( bb[0]-bsize+1, bb[3]-1,       bsize, bsize )


	def Redraw( self ): pass


	def _Redraw( self ): pass


	def _RedrawWithControlPoints( self ):
		self._DrawControlPoints()
		self._Redraw()


	def RedrawOn( self ): pass


	def RedrawOff( self ): pass


	# ------------------------------------------------------------------
	# Geometry methods:

	def Translate( self, dx, dy ):
		self._Translate( dx, dy )
		for graphic in vp.theSelection.GetSelection():
			if graphic != self and graphic.draggable and \
			   graphic.parent == self.parent:
				graphic._Translate( dx, dy )


	def _Translate( self, dx, dy ):
		self.x.Set( self.x.value + dx )
		self.y.Set( self.y.value + dy )


	def SetPosition( self, x, y ):      
		self.x.Set( x )
		self.y.Set( y )


	def SetCenterPosition( self, cx, cy ):
		self.x.Set( cx - self.width.value/2 )
		self.y.Set( cy - self.height.value/2 )


	def SetBBox( self, xmin, ymin, xmax, ymax ):
		self.x.Set( xmin )
		self.y.Set( ymin )
		self.width.Set( xmax - xmin )
		self.height.Set( ymax - ymin )
		
		
	def SetSize( self, width, height ):
		self.width.Set( width )
		self.height.Set( height )
		if not self.IsRealized(): self.natural_size = (width, height)


	def SetNaturalSize( self, width, height ):
		self.width.Set( width )
		self.height.Set( height )
		self.natural_size = (width, height)

	
	def GetPosition( self ): return (self.x.value, self.y.value)


	def GetAbsolutePosition( self ):
		#
		# Returns the position in absolute coordinates, i.e. relative to the
		# root-window.
		#
		if self.IsRealized():
			return self.GetWidget().TranslateCoords( 0, 0 )
		else:
			return None


	def GetCenterPosition( self ):
		return (self.x.value + self.width.value/2, 
			    self.y.value + self.height.value/2)


	def GetBBox( self ):
		return (self.x.value, 
			    self.y.value, 
			    self.x.value + self.width.value, 
				self.y.value + self.height.value)


	def _GetWcBBox( self ):
		#
		# Return bbox in window (WidgetObject) coordinates
		#
		return (self.wc_x.value, 
			    self.wc_y.value, 
			    self.wc_x.value+self.width.value,
				self.wc_y.value+self.height.value)


	def _SetWcBBox( self, xmin, ymin, xmax, ymax ):
		#
		# The bbox is given in the coordinate system of the WidgetObject while
		# the x and y values should be relative to the parent.
		#
		self.x.Set( xmin - self.parent.wc_x.value )
		self.y.Set( ymin - self.parent.wc_y.value )
		self.width.Set( xmax - xmin )
		self.height.Set( ymax - ymin )

		if not vp.theApplication.IsRunning():
			self.natural_size = (xmax - xmin, ymax - ymin)


	def GetSize( self ): return (self.width.value, self.height.value)

	def GetNaturalSize( self ): return self.natural_size


	# ------------------------------------------------------------------
	# Extra geometry-access methods:


	def GetWidth(self): return self.width.value

	def GetHeight(self): return self.height.value

	def GetXcenter(self): return self.x.value + self.width.value/2

	def GetYcenter(self): return self.y.value + self.height.value/2

	def GetXmin(self): return self.x.value

	def GetYmin(self): return self.y.value

	def GetXmax(self): return self.x.value + self.width.value

	def GetYmax(self): return self.y.value + self.height.value


	# ------------------------------------------------------------------
	# Misc modification (protocol) methods:

	def SetXResources( self, xres_dict ): pass

	def SetHiliXRes( self, res, val ): pass

	def SetHilight( self, bool ): self.hilight = bool


	def SetSelected( self, bool ):
		if debug:
			print 'Graphic.SetSelected called for', self.GetName(), \
				  'bool =', bool, 'self.editmode =', self.editmode

		if bool == TRUE:
			vp.theSelection.Add( self )
			self.ExecuteCallback( 'selected', self )
			if self.editmode: self.ShowControlPoints()

		elif bool == FALSE:
			vp.theSelection.Remove( self )
			if self.editmode: self.HideControlPoints()

			
	def SetStretchability( self, hor, ver ):
		#
		# hor and ver should be either vp.ELASTIC or FIXED
		#
		self.stretchability = (hor, ver)


	def SetDraggable( self, bool ):
		if self.draggable != bool:
			self.draggable = bool

			if bool == TRUE:
				self.Subscribe( vp.MOUSE_DOWN, self.MouseDownEH, None )
				self.Subscribe( vp.MOUSE2_DRAG, self.Mouse2DragEH, None )
				self.Subscribe( vp.MOUSE_UP, self.MouseUpEH, None )
			else:
				self.UnSubscribe( vp.MOUSE_DOWN )
				self.UnSubscribe( vp.MOUSE2_DRAG )
				self.UnSubscribe( vp.MOUSE_UP )


	def SetEditMode( self, bool ):
		if self.editmode != bool:
			self.editmode = bool

			if bool == TRUE:
				self.Subscribe( vp.MOUSE_DOWN, self.MouseDownEH, None )
				self.Subscribe( vp.MOUSE1_DRAG, self.Mouse1DragEH, None )
				self.Subscribe( vp.MOUSE_UP, self.MouseUpEH, None )
			else:
				if self.show_cntrl_pnts: self.HideControlPoints()
				self.UnSubscribe( vp.MOUSE_DOWN )
				self.UnSubscribe( vp.MOUSE1_DRAG )
				self.UnSubscribe( vp.MOUSE_UP )


	# ------------------------------------------------------------------
	# Event handler protocol methods (overridden in WidgetObject and 
	# Primitive classes)

	def MouseDownEH( self, target, client_data, xevent, e ): 
		if debug: print 'Graphic.MouseDownEH called for', self.GetName()

		if e == None: e = xevent          # XButtonEvent

		self.mouse_pos = (e.x_root, e.y_root)

		if e.button == X.Button3 and hasattr( self, '_popup' ):
			if debug: print 'Graphic.MouseDownEH: popup the menu'
			vp.theCurrentPopupNode = self
			self._popup.Show( xevent )

		elif self.draggable and e.button == X.Button2:    # start drag
			EventNotifier.NotifyExpose( FALSE )        
			EventNotifier.GrabFocus( self )       

		elif e.button == X.Button1:
			if self.show_cntrl_pnts:
				EventNotifier.NotifyExpose( FALSE )        
				EventNotifier.GrabFocus( self )           # start resizing
			else:
				vp.theSelection.SetEmpty()          
				self.SetSelected( TRUE )

		
	def Mouse1DragEH( self, target, client_data, xevent, e ): pass


	def Mouse2DragEH( self, target, client_data, xevent, e ): 
		if self.draggable:
			if e == None: e = xevent      # XMotionEvent

			delta_x = e.x_root - self.mouse_pos[0]
			delta_y = e.y_root - self.mouse_pos[1]
			self.Translate( delta_x, delta_y  )
			self.mouse_pos = (e.x_root, e.y_root)
		

	def MouseUpEH( self, target, client_data, xevent, e ): 
		EventNotifier.GrabFocus( None )
		EventNotifier.NotifyExpose( TRUE )
		

	# ------------------------------------------------------------------
	# Access methods:

	def GetWidget(self):
		#
		# The default for Primitives (classes not based on widgets).
		#
		if self.parent != None: 
			return self.parent.GetWidget()
		else:
			return None


	def GetStretchability( self ): return self.stretchability


	# ------------------------------------------------------------------
	# Query methods:

	def IsElastic( self, hor_ver ):
		return self.stretchability[hor_ver] == vp.ELASTIC


	def IsRealized( self ): return self.realized


	def IsVisible( self ): return self.visible


	def IsPicked( self, x, y ): 
		#
		# Must be overridden in the Primitive and WidgetObject classes
		#
		pass


	def _IsControlPointPicked( self, x, y, bb ):
		bsize = vp.CPB
		cp1 = (bb[0]-bsize+1, bb[1]-bsize+1, bb[0]+1,       bb[1]+1)
		cp2 = (bb[2]-1,       bb[1]-bsize+1, bb[2]+bsize-1, bb[1]+1)
		cp3 = (bb[2]-1,       bb[3]-1,       bb[2]+bsize-1, bb[3]+bsize-1)
		cp4 = (bb[0]-bsize+1, bb[3]-1,       bb[0]+1,       bb[3]+bsize-1)
				  
		if cp1[0] < x and cp1[1] < y and cp1[2] > x and cp1[3] > y:
			self._cp = 1
			return TRUE
		elif cp2[0] < x and cp2[1] < y and cp2[2] > x and cp2[3] > y:
			self._cp = 2
			return TRUE
		elif cp3[0] < x and cp3[1] < y and cp3[2] > x and cp3[3] > y:
			self._cp = 3
			return TRUE
		elif cp4[0] < x and cp4[1] < y and cp4[2] > x and cp4[3] > y:
			self._cp = 4
			return TRUE
		else:
			self._cp = 0
			return FALSE
		


