# -------------------------------------------------------------------------
# MODULE:      Frame
#
# DESCRIPTION: 
#     Contains the Frame class
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, <per@cwi.nl>
#

import Xm, Xmd

import vp
from WidgetObject  import WidgetObject
from MiscGraphic   import Label
from Canvas        import Canvas
from Box           import Box, Glue

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         Frame
#
# INHERITS FROM: WidgetObject : Graphic : (TreeNode, DatumDict) : Object 
#
# DESCRIPTION: 
#

class Frame( WidgetObject ):
	
	# ------------------------------------------------------------------
	# Init methods

	def __init__( self, argdict = {} ):
		#
		# Args can be: 
		# - 'label'       : <string>
		# - 'child'       : <any Graphic obj>
		# - 'margin'      : <int> (5=default)
		# - 'label_indent': <int> (10=default)
		#
		self.margin       = 5
		self.label_indent = 10

		WidgetObject.__init__( self, argdict )

		self.top_margin   = 8
		self._margin      = self.margin + 2    # shadowThickness
		self.did_calculate_size = vp.FALSE

		if not hasattr( self, 'label' ):
			self.widget_class = Xm.Frame
			if hasattr( self, 'child' ): self.AddChild( self.child )
			self.SetXResources({ 
				  'marginWidth' : self.margin,
				  'marginHeight': self.margin
				  })
		else:
			#
			# must explicitly manage the layout of the frame and the label; (in
			# version 1.2 of Motif this will all be supported directly by the
			# Frame widget).
			#
			self.widget_class = Xm.DrawingArea
			self.name         = self.label
			self.box          = Box({ 
				  'name'      : 'FrameBox',
				  'alignment' : vp.VCENTER, 
				  'child_list': [Glue(self.top_margin, vp.FIXED)]
				  })
			self.frame        = Frame({ 'child': self.box })
			self.frame.Subscribe( vp.EXPOSE, self.FrameExposedEH, None )
			self.label        = Label({ 'label': self.label })

			if hasattr( self, 'child' ): self.box.AddChild( self.child )

			WidgetObject.AddChild( self, self.label )
			WidgetObject.AddChild( self, self.frame )
			self.SetXResources({ 'marginWidth': 0, 'marginHeight': 0 })


	def ProcessArgs( self, argdict ):
		if debug: print 'Frame.ProcessArgs: argdict =', argdict

		for key in argdict.keys():
			if key == 'size':
				(w,h) = argdict[key]
				self.SetSize( w, h )

			elif key == 'pos':
				(x,y) = argdict[key]
				self.SetPosition( x, y )

			elif key == 'draggable':
				self.SetDraggable( argdict[key] )

			elif key == 'editmode':
				self.SetEditMode( argdict[key] )

			elif key == 'name':
				self.SetName( argdict[key] )

			elif self.datums.has_key( key ):
				self.datums[ key ].Set( argdict[key] )

			else:
				setattr( self, key, argdict[key] )

	def Realize( self ):
		if debug: print 'Frame.Realize called for', self.GetName()

		WidgetObject.Realize( self )
		self.CalculateSizeNeeded()

		if not (self.parent.IsA( Box ) or self.parent.IsA( Frame )):
			self.SetSize( self.natural_size[0], self.natural_size[1] )


	def CalculateSizeNeeded( self ):
		if hasattr( self, 'label' ):
			fsize           = self.frame.GetNaturalSize()
			lsize           = self.label.GetNaturalSize()
			self.frame_ymin = int(lsize[1]*0.4)
			self._margin    = self.margin + self.frame.w.shadowThickness

			w     = max( fsize[0], (self.label_indent + lsize[0] + 5) )
			h     = fsize[1] + self.frame_ymin
		else:
			csize        = self.child.GetNaturalSize()
			self._margin = self.margin + self.w.shadowThickness

			w = csize[0] + 2*self._margin
			h = csize[1] + 2*self._margin + 2
			
		self.did_calculate_size = vp.TRUE
		self.natural_size = (w,h)

			
	def AlignChildren( self ):
		self.label.SetPosition( self.label_indent, 0 )
		self.frame.SetPosition( 0, self.frame_ymin )


	def AddChild( self, child ):
		if hasattr( self, 'box' ) and len(self.box.children) == 1:
			self.box.AddChild( child )
			if self.IsRealized():
				self.CalculateSizeNeeded()

		elif len( self.children ) == 0:
			WidgetObject.AddChild( self, child )
			if self.IsRealized():
				self.CalculateSizeNeeded()

		else:
			print 'Frame.AddChild: only one child allowed!'
			
			
	# ------------------------------------------------------------------
	# Event notification methods:

	def FrameExposedEH( self, frame, client_date, xevent, e ):
		#
		# Used to make sure that the label will be displayed on top 
		# 
		if debug: print 'FrameExposedEH called'
		self.label.w.RaiseWindow()
		self.frame.UnSubscribe( vp.EXPOSE )


	def ResizeEH( self, target, client_data, xevent, e ):
		if debug:print 'Frame.ResizeEH called'

		if self.w.width != self.width.value or \
			  self.w.height != self.height.value:
			WidgetObject.ResizeEH( self, target, client_data, xevent, e )
			(w,h) = self.GetSize()

			if hasattr( self, 'box' ):
				self.box.SetSize( w - 2*self._margin, 
				                  h - (self.frame_ymin + 2*self._margin) )
			else:
				self.child.SetSize( w - 2*self._margin, 
				                    h - 2*self._margin )


	# ------------------------------------------------------------------
	# Geometry methods:

	def SetSize( self, width, height ):
		if debug:
			print 'Frame(', self.GetName(),')SetSize: (width, height) =', \
				  (width, height)

		WidgetObject.SetSize( self, width, height )

		if not self.IsRealized() or not self.did_calculate_size: return

		if hasattr( self, 'box' ):
			self.box.SetSize( width - 2*self._margin, 
							  height - (self.frame_ymin + 2*self._margin) )
			self.frame.width.UpdateInternal()
			self.frame.height.UpdateInternal()
		else:
			self.child.SetSize( width - 2*self._margin, 
								height - 2*self._margin )
			self.width.UpdateInternal()
			self.height.UpdateInternal()

		if hasattr( self, 'label' ):
			self.AlignChildren()


	# ------------------------------------------------------------------
	# Query methods:

	def GetNaturalSize( self ):
		if not self.did_calculate_size:
			self.CalculateSizeNeeded()
		return self.natural_size
