# -------------------------------------------------------------------------
# MODULE:      EventNotifier
#
# DESCRIPTION: 
#     Contains the EventNotifier module. This module encapsulates the lower-
#     level details of the Xt/Xlib event-handling interface. The graphic
#     classes uses this module to subscribe and unsubscribe events.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import X
import vp

from X  import KeyPress, KeyRelease, ButtonPress, ButtonRelease, \
	 MotionNotify, EnterNotify, LeaveNotify, Expose, NoExpose

from vp import ENTER, LEAVE, MOUSE_DOWN, MOUSE2_DRAG, MOUSE1_DRAG, MOUSE_UP, \
	 KEYPRESS, RESIZE, EXPOSE, WIDGET_DESTROYED, TRUE, FALSE

X_TRUE  = X.TRUE
X_FALSE = X.FALSE

debug = FALSE

# -------------------------------------------------------------------------
# Public methods:


def Subscribe( graphic, widget, etype ):
	if debug: 
		print 'EventNotifier.Subscribe: graphic =', graphic.GetClassName(), \
			  'etype ', debugswitch[etype]

	(subscribe, emask, generic_eh) = subscribeswitch[etype] 
	subscribe( widget, emask, graphic, generic_eh )
		

def SubscribeComposite( graphic, widget, etype ): 
	if debug: 
		print 'EventNotifier.SubscribeComposite: graphic =', \
			  graphic.GetClassName(), 'etype ', debugswitch[etype]

	(subscribe, emask, generic_eh) = composite_subscribeswitch[etype]
	subscribe( widget, emask, graphic, generic_eh )
		

def UnSubscribe( graphic, widget, etype ):
	(no_subscribe, emask, generic_eh) = no_subscribeswitch[etype]
	no_subscribe( widget, emask, graphic, generic_eh )


def UnSubscribeComposite( graphic, widget, etype ):
	if debug: 
		print 'EventNotifier.UnSubscribeComposite: graphic =', \
			  graphic.GetClassName(), 'etype ', debugswitch[etype]

	(subscribe, emask, generic_eh) = composite_no_subscribeswitch[etype]
	subscribe( widget, emask, graphic, generic_eh )


event_focus = None   # <prim.id>

def GrabFocus( prim_id ):
	if debug: print 'EventNotifier.GrabFocus called by', prim_id
	global event_focus
	event_focus = prim_id


notify_expose = TRUE

def NotifyExpose( bool ):
	if debug: print 'EventNotifier.NotifyExpose: =', bool
	global notify_expose
	notify_expose = bool


# -------------------------------------------------------------------------
# Private methods:

def _AddEH( w, emask, graphic, ehandler ):
	w.AddEventHandler( emask, X_TRUE, ehandler, graphic )


def _RemoveEH( w, emask, graphic, ehandler ):
	w.RemoveEventHandler( emask, X_TRUE, ehandler, graphic )


def _AddCB( w, emask, graphic, ehandler ):
	w.AddCallback( emask, ehandler, graphic )


def _RemoveCB( w, emask, graphic, ehandler ):
	w.RemoveCallback( emask, ehandler, graphic )


# -------------------------------------------------------------------------
# Generic event handlers for the Graphic class:

def _MouseDown( w, graphic, xevent ):
	if graphic.DidSubscribe( MOUSE_DOWN ):
		(eh, client_data, count) = graphic.eh_dict[ MOUSE_DOWN ]
		eh( graphic, client_data, xevent, None )


def _Mouse2Drag( w, graphic, xevent ):
	if graphic.DidSubscribe( MOUSE2_DRAG ):
		(eh, client_data, count) = graphic.eh_dict[ MOUSE2_DRAG ]
		eh( graphic, client_data, xevent, None )


def _Mouse1Drag( w, graphic, xevent ):
	if graphic.DidSubscribe( MOUSE1_DRAG ):
		(eh, client_data, count) = graphic.eh_dict[ MOUSE1_DRAG ]
		eh( graphic, client_data, xevent, None )


def _MouseUp( w, graphic, xevent ):
	if graphic.DidSubscribe( MOUSE_UP ):
		(eh, client_data, count) = graphic.eh_dict[ MOUSE_UP ]
		eh( graphic, client_data, xevent, None )


def _Keypress( w, graphic, xevent ):
	if graphic.DidSubscribe( KEYPRESS ):
		(eh, client_data, count) = graphic.eh_dict[ KEYPRESS ]
		eh( graphic, client_data, xevent, None )


def _Enter( w, graphic, xevent ):
	if graphic.DidSubscribe( ENTER ):
		(eh, client_data, count) = graphic.eh_dict[ ENTER ]
		eh( graphic, client_data, xevent, None )


def _Leave( w, graphic, xevent ):
	if graphic.DidSubscribe( LEAVE ):
		(eh, client_data, count) = graphic.eh_dict[ LEAVE ]
		eh( graphic, client_data, xevent, None )


def _Resize( w, graphic, xevent ):
	if graphic.DidSubscribe( RESIZE ):
		(eh, client_data, count) = graphic.eh_dict[ RESIZE ]
		eh( graphic, client_data, xevent, None )


def _Expose( w, graphic, xevent ):
	if not notify_expose:
		return

	if graphic.DidSubscribe( EXPOSE ):
		e = xevent

		if hasattr(e, 'count') and e.count != 0: # Compress the expose events
			return

		if debug: 
			print 'EventNotifier: dispatching EXPOSE ev. to', graphic.GetName()

		(eh, client_data, count) = graphic.eh_dict[ EXPOSE ]
		eh( graphic, client_data, xevent, e )


def _NewExpose( w, graphic, call_data):
	if not notify_expose:
		return

	if graphic.DidSubscribe( EXPOSE ):
		if call_data:
			e = call_data.event
			if e.count > 0:
				if debug: print `self`+'._expose_callback() -- count > 0'
				return

			if debug:
				print 'EventNotifier: dispatching EXPOSE ev. to', \
					  graphic.GetName()

			(eh, client_data, count) = graphic.eh_dict[ EXPOSE ]
			eh( graphic, client_data, xevent, e )


def _WidgetDestroyed( w, graphic, call_data ):
	graphic.WidgetDestroyed()


# -------------------------------------------------------------------------
# Event handlers for the Composite class:

target = None

def _EventTargetIsFound( comp, graphic, e, etype ): 
	global target

	if comp == graphic:
		if graphic.IsVisible():
			rev = graphic.children[:]
			rev.reverse()
			for child in rev:
				if _EventTargetIsFound( comp, child, e, etype ): 
					return TRUE

			if graphic.DidSubscribe( etype ):
				target = graphic
				return TRUE

	elif graphic.IsVisible() and graphic.IsPicked( e.x, e.y ):
		rev = graphic.children[:]
		rev.reverse()
		for child in rev:
			if _EventTargetIsFound( comp, child, e, etype ): 
				return TRUE

		if graphic.DidSubscribe( etype ):
			target = graphic
			return TRUE

	return FALSE
			

def _CompositeMouseDown( w, composite, xevent ):
	#
	# The following procedure is used to find a target for the event in a 
	# Composite:
	# (1) - check if there is a child of the Composite which has grabbed
	#       event focus
	# (2) - else, check if there is a child object which wants the 
	#       event; if yes then bot the xevent and the unpacked event 'e' are
	#       passed-on
	# (3) - else, check if the Composite itself is interested in the event
	# (4) - else, no target found for the event
	#
	e = xevent

	if debug: 
		print 'EventNotifier._CompositeMouseDown: event_focus =', \
			  event_focus, 'etype', etypeswitch[e.type]

	if  event_focus != None: 
		(eh, client_data, count) = event_focus.eh_dict[MOUSE_DOWN]
		eh( event_focus, client_data, xevent, e )

	elif _EventTargetIsFound( composite, composite, e, MOUSE_DOWN ):
		if debug: print 'EventNotifier._CompositeMouseDown: target =', target
		(eh, client_data, count) = target.eh_dict[MOUSE_DOWN]
		eh( target, client_data, xevent, e )

	else:
		vp.theSelection.SetEmpty()


def _CompositeMouse1Drag( w, composite, xevent ):
	e = xevent

	if  event_focus != None:
		(eh, client_data, count) = event_focus.eh_dict[MOUSE1_DRAG]
		eh( event_focus, client_data, xevent, e )

	elif _EventTargetIsFound( composite, composite, e, MOUSE1_DRAG ):
		(eh, client_data, count) = target.eh_dict[MOUSE1_DRAG]
		eh( target, client_data, xevent, e )


def _CompositeMouse2Drag( w, composite, xevent ):
	e = xevent

	if  event_focus != None:
		(eh, client_data, count) = event_focus.eh_dict[MOUSE2_DRAG]
		eh( event_focus, client_data, xevent, e )

	elif _EventTargetIsFound( composite, composite, e, MOUSE2_DRAG ):
		(eh, client_data, count) = target.eh_dict[MOUSE2_DRAG]
		eh( target, client_data, xevent, e )


def _CompositeMouseUp( w, composite, xevent ):
	e = xevent

	if  event_focus != None:
		(eh, client_data, count) = event_focus.eh_dict[MOUSE_UP]
		eh( event_focus, client_data, xevent, e )

	elif _EventTargetIsFound( composite, composite, e, MOUSE_UP ):
		(eh, client_data, count) = target.eh_dict[MOUSE_UP]
		eh( target, client_data, xevent, e )


def _CompositeKeypress( w, composite, xevent ):
	e = xevent

	if  event_focus != None:
		(eh, client_data, count) = event_focus.eh_dict[KEYPRESS]
		eh( event_focus, client_data, xevent, e )

	elif _EventTargetIsFound( composite, composite, e, KEYPRESS ):
		(eh, client_data, count) = target.eh_dict[KEYPRESS]
		eh( target, client_data, xevent, e )


# -------------------------------------------------------------------------
# Switches:

subscribeswitch = {
	  MOUSE_DOWN:       (_AddEH, X.ButtonPressMask,     _MouseDown),
	  MOUSE1_DRAG:      (_AddEH, X.Button1MotionMask,   _Mouse1Drag),
	  MOUSE2_DRAG:      (_AddEH, X.Button2MotionMask,   _Mouse2Drag),
	  MOUSE_UP:         (_AddEH, X.ButtonReleaseMask,   _MouseUp),
	  KEYPRESS:         (_AddEH, X.KeyPressMask,        _Keypress),
	  ENTER:            (_AddEH, X.EnterWindowMask,     _Enter),
	  LEAVE:            (_AddEH, X.LeaveWindowMask,     _Leave),
	  EXPOSE:           (_AddEH, X.ExposureMask,        _Expose),
	  WIDGET_DESTROYED: (_AddCB, 'destroyCallback',     _WidgetDestroyed),
	  RESIZE:           (_AddEH, X.StructureNotifyMask, _Resize)
}

no_subscribeswitch = {
	  MOUSE_DOWN:       (_RemoveEH, X.ButtonPressMask,     _MouseDown),
	  MOUSE1_DRAG:      (_RemoveEH, X.Button1MotionMask,   _Mouse1Drag),
	  MOUSE2_DRAG:      (_RemoveEH, X.Button2MotionMask,   _Mouse2Drag),
	  MOUSE_UP:         (_RemoveEH, X.ButtonReleaseMask,   _MouseUp),
	  KEYPRESS:         (_RemoveEH, X.KeyPressMask,        _Keypress),
	  ENTER:            (_RemoveEH, X.EnterWindowMask,     _Enter),
	  LEAVE:            (_RemoveEH, X.LeaveWindowMask,     _Leave),
	  EXPOSE:           (_RemoveEH, X.ExposureMask,        _Expose),
	  WIDGET_DESTROYED: (_RemoveCB, 'destroyCallback',    _WidgetDestroyed),
	  RESIZE:           (_RemoveEH, X.StructureNotifyMask, _Resize)
}

composite_subscribeswitch = {
	  MOUSE_DOWN:       (_AddEH, X.ButtonPressMask,     _CompositeMouseDown),
	  MOUSE1_DRAG:      (_AddEH, X.Button1MotionMask,   _CompositeMouse1Drag),
	  MOUSE2_DRAG:      (_AddEH, X.Button2MotionMask,   _CompositeMouse2Drag),
	  MOUSE_UP:         (_AddEH, X.ButtonReleaseMask,   _CompositeMouseUp),
	  KEYPRESS:         (_AddEH, X.KeyPressMask,        _CompositeKeypress),
	  ENTER:            (_AddEH, X.EnterWindowMask,     _Enter),
	  LEAVE:            (_AddEH, X.LeaveWindowMask,     _Leave),
	  EXPOSE:           (_AddEH, X.ExposureMask,        _Expose),
	  WIDGET_DESTROYED: (_AddCB, 'destroyCallback',     _WidgetDestroyed),
	  RESIZE:           (_AddEH, X.StructureNotifyMask, _Resize)
}

composite_no_subscribeswitch = {
	  MOUSE_DOWN:       (_RemoveEH, X.ButtonPressMask,   _CompositeMouseDown),
	  MOUSE1_DRAG:      (_RemoveEH, X.Button1MotionMask, _CompositeMouse1Drag),
	  MOUSE2_DRAG:      (_RemoveEH, X.Button2MotionMask, _CompositeMouse2Drag),
	  MOUSE_UP:         (_RemoveEH, X.ButtonReleaseMask, _CompositeMouseUp),
	  KEYPRESS:         (_RemoveEH, X.KeyPressMask,      _CompositeKeypress),
	  ENTER:            (_RemoveEH, X.EnterWindowMask,   _Enter),
	  LEAVE:            (_RemoveEH, X.LeaveWindowMask,   _Leave),
	  EXPOSE:           (_RemoveEH, X.ExposureMask,      _Expose),
	  WIDGET_DESTROYED: (_RemoveCB, 'destroyCallback',   _WidgetDestroyed),
	  RESIZE:           (_RemoveEH, X.StructureNotifyMask, _Resize)
}

debugswitch = {
	  MOUSE_DOWN:       'MOUSE_DOWN',
	  MOUSE1_DRAG:      'MOUSE1_DRAG',
	  MOUSE2_DRAG:      'MOUSE2_DRAG',
	  MOUSE_UP:         'MOUSE_UP',
	  KEYPRESS:         'KEYPRESS',
	  ENTER:            'ENTER',
	  LEAVE:            'LEAVE',
	  EXPOSE:           'EXPOSE',
	  WIDGET_DESTROYED: 'WIDGET_DESTROYED',
	  RESIZE:           'RESIZE'
}
	  
maskswitch = {
	  X.ButtonPressMask:     'ButtonPressMask',
	  X.Button1MotionMask:   'Button1MotionMask',
	  X.Button2MotionMask:   'Button2MotionMask',
	  X.ButtonReleaseMask:   'ButtonReleaseMask',
	  X.KeyPressMask:        'KeyPressMask',
	  X.EnterWindowMask:     'EnterWindowMask',
	  X.LeaveWindowMask:     'LeaveWindowMask',
	  X.ExposureMask:        'ExposureMask',
	  X.StructureNotifyMask: 'StructureNotifyMask'
}

etypeswitch = {
	X.KeyPress:         'KeyPress',
	X.KeyRelease:       'KeyRelease',
	X.ButtonPress:      'ButtonPress',
	X.ButtonRelease:    'ButtonRelease',
	X.MotionNotify:     'MotionNotify',
	X.EnterNotify:      'EnterNotify',
	X.LeaveNotify:      'LeaveNotify',
	X.FocusIn:          'FocusIn',
	X.FocusOut:         'FocusOut',
	X.KeymapNotify:     'KeymapNotify',
	X.Expose:           'Expose',
	X.GraphicsExpose:   'GraphicsExpose',
	X.NoExpose:         'NoExpose',
	X.VisibilityNotify: 'VisibilityNotify',
	X.CreateNotify:     'CreateNotify',
	X.DestroyNotify:    'DestroyNotify',
	X.UnmapNotify:      'UnmapNotify',
	X.MapNotify:        'MapNotify',
	X.MapRequest:       'MapRequest',
	X.ReparentNotify:   'ReparentNotify',
	X.ConfigureNotify:  'ConfigureNotify',
	X.ConfigureRequest: 'ConfigureRequest',
	X.GravityNotify:    'GravityNotify',
	X.ResizeRequest:    'ResizeRequest',
	X.CirculateNotify:  'CirculateNotify',
	X.CirculateRequest: 'CirculateRequest',
	X.PropertyNotify:   'PropertyNotify',
	X.SelectionClear:   'SelectionClear',
	X.SelectionRequest: 'SelectionRequest',
	X.SelectionNotify:  'SelectionNotify',
	X.ColormapNotify:   'ColormapNotify',
	X.ClientMessage:    'ClientMessage',
	X.MappingNotify:    'MappingNotify'
}
