# -------------------------------------------------------------------------
# MODULE: Datum
#
# DESCRIPTION: 
#     
# USAGE:
#
# AUTHOR: 
#     Dirk Soede, soede@cwi.nl
#

import vp
from TypeInfo import hasclass
from Mixin    import Mixin

debug       = vp.FALSE
debug_final = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         DatumDict
#
# INHERITS FROM: None
#
# DESCRIPTION: 
#     A class which manages Datum intances.
#

class DatumDict( Mixin ):

	# ------------------------------------------------------------------
	# Initialisation and destruction

	def __init__( self ):
		self.datums      = {}
		self.xres_shells = []  # Used by XResourceShell


	def Finalize( self ):
		if debug_final: 
			print 'DatumDict.Finalize called for', self.GetClassName()

		for key in self.datums.keys():
			if debug_final: print 'DatumDict.Finalize: deleting', key
			self.datums[key].Finalize()
			del self.datums[key]
			try:
				del self.__dict__[key]
			except:
				pass

		if debug_final: print 'DatumDict.Finalize: deleting XResShells'

		for item in self.xres_shells:
			item.Finalize()
			del item

		self.xres_shells = None
		self.datums      = None


	# ------------------------------------------------------------------
	# X-resource shell methods

	def AddXResShell( self, xrs ):
		#
		# Called by an XResourceShell when it is created.
		#
		self.xres_shells.append( xrs )


	def UpdateXResShells( self ):
		#
		# Called by the WidgetObject.SetXRes() method
		#
		for shell in self.xres_shells:
			shell.Update()


	def InitXResShells( self, widget ):
		#
		# Fetch or set real X resources.
		#
		for shell in self.xres_shells:
			shell.InitResource( widget )

		
	# ------------------------------------------------------------------
	# Datum management.

	def SetDatum(self, name, value):
		#
		# Assigns datum value.
		# `datum' is a string and creates a dependency-link when `datum' is 
		# a Datum object, otherwise a normal value is assigned.
		# 
		if hasclass( value, Datum ):
			void = self.GetDatum(name).LinkFrom(value)
		else:
			self.GetDatum(name).Set(value)


	def GetDatum(self, name):
		return self.datums[name]


	def AddDatum(self, name, datum):
		self.datums[name] = datum
		setattr(self, name, datum)
		

	def AddFunctionDatum(self, name, function):
		self.AddDatum(name, ExternalDatum(FunctionShell(function)))


	def AddAttribDatum(self, name, object, attrib_name):
		self.AddDatum(name, ExternalDatum(AttribShell(object, attrib_name)))



# -------------------------------------------------------------------------
# CLASS:         Datum
#
# INHERITS FROM: None
#
# DESCRIPTION: 
#     An elementary data unit within a constraint network.
#

class Datum:
	#
	# Class vars:
	#
	msg       = None

	# ------------------------------------------------------------------
	# Constructor & destructor.

	def __init__(self, value):
		self.value     = None
		self.sinks     = []
		self.sources   = []
		self._Set(value, None)


	def Finalize( self ):
		self.UnlinkAll()
		self.sinks     = None
		self.sources   = None


	# ------------------------------------------------------------------
	# Linking Datums

	def LinkTo(self, other):
		self._MakeLink(other)
		other.Update(self.value, self)


	def LinkFrom(self, other):
		other._MakeLink(self)
		self.Update(other.value, other)


	def DoubleLink(self, other):
		#
		# Value of `self' overrides value of `other'.
		#
		self._MakeLink(other)
		other._MakeLink(self)
		if self.value != None:
			other.Update(self.value, self)
		elif other.value != None:
			self.Update(other.value, other)


	def Unlink(self, other):
		try:
			self.sinks.remove(other)
		except:
			pass
		try:
			other.sources.remove(self)
		except:
			pass


	def DoubleUnlink(self, other):
		self.Unlink(other)
		other.Unlink(self)


	def UnlinkAll( self ):
		try:
			for sink in self.sinks[:]:
				self.Unlink( sink )
		except:
			pass

		try:
			for source in self.sources[:]:
				source.Unlink( self )
		except:
			pass


	def _MakeLink(self, other):
		self.sinks.append(other)
		other.sources.append(self)


	# ------------------------------------------------------------------ #
	# Value access and forwarding.


	def Get(self):
		return self.value


	def Set(self, value):
		self.Update(value, None)


	def Update(self, value, source):
		#
		# Method used by other datums to report a change.
		#
		self._Set(value, source)


	def _Set(self, value, source):
		#
		# Set the actual value and forward the new value.
		# 
		if value != self.value:
			if self.msg != None:
				print self.msg, self.value, '->', value
			self.value = value
			self.ForwardValue(value, source)


	def ForwardValue(self, value, source):
		for sink in self.sinks:
			if sink != source:
				sink.Update(value, self)
		

# -------------------------------------------------------------------------
# CLASS:         ExternalDatum
#
# INHERITS FROM: Datum
#
# DESCRIPTION:
#     This Datum is used to link the Datum world to the external world. It
#     communicates via a Shell instance, which is a wrapper around a data
#     unit or function call.
#     External data values may have themselves mutual relations. These relation
#     can be taken care of in the Datum network by making `Update' links:
#     When it is known that value y changes when value x is changed, the datum
#     of y can be instructed with UpdateOnDatum to read its external value
#     when datum x has changed.
#

class ExternalDatum( Datum ):

	def __init__(self, shell):
		Datum.__init__(self, shell.Get())
		self.shell   = shell
		self.update  = UpdateDatum()
		shell.SetDatum( self )
		self.update.LinkTo( self )


	def Finalize(self):
		try:
			self.update.Finalize()
		except:
			pass

		self.update = None
		self.shell  = None
		Datum.Finalize( self )


	def Update(self, value, source):
		if source == self.update:
			value = self.shell.Get()
		else:
			self.shell.Set(value)
		self._Set(value, source)


	def UpdateInternal( self ):
		#
		# Method used to notify the datum that its external data has changed.
		#
		self.update.Set(None)


	def UpdateOnDatum(self, datum):
		datum.LinkTo(self.update)



# -------------------------------------------------------------------------
# CLASS:         UpdateDatum
#
# INHERITS FROM: Datum
#
# DESCRIPTION: 
#     ...
#

class UpdateDatum(Datum):

	def __init__(self):
		Datum.__init__(self, None)
		pass


	def Update(self, value, source):
		self.ForwardValue(None, source)		 # Update MUST cause an update.



# -------------------------------------------------------------------------
# CLASS:         DataShell's (AttribShell, XResourceShell, MethodShell,
#                FunctionShell)
#
# INHERITS FROM: None
#
# DESCRIPTION: 
#     Creates a wrapper around an object attribute
#

class Shell:

	def __init__( self ):
		self.datum = None


	def Finalize(self):
		self.datum = None


	def SetDatum( self, datum ):
		self.datum = datum


	def Set( self, value ):
		pass


	def Get( self ):
		pass


# -------------------------------------------------------------------------

class AttribShell( Shell ):

	def __init__(self, object, attrib_name):
		Shell.__init__( self )
		self.object      = object
		self.attrib_name = attrib_name


	def Finalize(self):
		self.object = None
		Shell.Finalize( self )


	def Set(self, value):
		setattr(self.object, self.attrib_name, value)


	def Get(self):
		return getattr(self.object, self.attrib_name)


# -------------------------------------------------------------------------

class XResourceShell( Shell ):

	def __init__(self, resource, widget_object):
		Shell.__init__( self )
		self.widget        = None
		self.widget_object = widget_object # for debugging 
		self.resource      = resource
		widget_object.AddXResShell( self )


	def Finalize(self):
		self.widget        = None
		self.widget_object = None
		Shell.Finalize( self )


	def __del__(self):
		if debug_final:
			print self.GetClassName(), self.resource, 'deleted'

		
	def Set(self, value):
		if self.widget != None and value != None:
			setattr(self.widget, self.resource, value )
			set_value = getattr(self.widget, self.resource)

			if value != set_value:
				#
				# Did not succeed in setting the value
				#
				if self.widget.IsManaged():
					self.widget.UnmanageChild()
					setattr(self.widget, self.resource, value )
					self.widget.ManageChild()
					

	def Get(self):
		if self.widget != None:
			return getattr(self.widget, self.resource)


	def InitResource(self, widget):
		self.widget = widget
		if self.datum.value != None:
			value = self.datum.value
		else:
			value = self.Get()
		self.datum.Set(value)


	def Update( self ):
		#
		# Called to notify that the Xresource values have changed. 
		#
		self.datum.Set( self.Get() )
		

# -------------------------------------------------------------------------

class MethodShell( Shell ):

	def __init__(self, object, method_name):
		Shell.__init__( self )
		self.object      = object
		self.method_name = method_name


	def Finalize(self):
		self.object = None
		Shell.Finalize( self )


	def Set(self, value):
		getattr( self.object, self.method_name )( value )


# -------------------------------------------------------------------------

class FunctionShell( Shell ):

	def __init__(self, function):
		Shell.__init__( self )
		self.function = function


	def Finalize(self):
		self.function = None
		Shell.Finalize( self )


	def __del__(self):
		if debug_final:
			print self.GetClassName(), 'deleted'

		
	def Set(self, value):
		self.function(value)




# -------------------------------------------------------------------------
# CLASS:         Transform
#
# INHERITS FROM: Datum
#
# DESCRIPTION: 
#     Transforms any incoming data before it is passed on and vice versa.
#     Has exactly one input source (arbitrary descision) and may have 
#     any number of sinks. 
#     

class Transform(Datum):

	def __init__(self, input):
		Datum.__init__(self, None)
		self.input = input
		self.input.LinkTo(self)


	def Finalize(self):
		self.input.Finalize()
		Datum.Finalize( self )


	def Update(self, value, source):
		if source == self.input:
			value = self.Compute(value)
			self._Set(value, source)
		else:
			value = self.InverseCompute(value)
			self.input.Update(value, self)
			
		
	def GetInputValue(self):
		return self.input.Get()

		
# -------------------------------------------------------------------------
# CLASS:         Merge
#
# INHERITS FROM: Datum
#
# DESCRIPTION: 
#     Merges a number of incoming data into a list. The inputs have
#     to be given at initialization.
#     


class Merge(Datum):

	def __init__(self, *inputs):
		Datum.__init__(self, [])
		self.input_datums = []
		self.input_values = []
		apply(self.InitInputDatums, inputs)


	def Finalize(self):
		try:
			for datum in self.input_datums[:]:
				try:
					datum.Unlink(self)
				except:
					pass
		except:
			pass

		self.input_datums = None
		self.input_values = None
		Datum.Finalize( self )


	def InitInputDatums(self, *inputs):
		for input in inputs:
			self.AddInput(input, 0)
		for datum in self.input_datums:
			if datum != None:
				datum.LinkTo(self)


	def AddInput(self, input, *do_link):
		if hasclass(input, Datum):
			self.input_datums.append(input)
			self.input_values.append(input.Get())
			if len(do_link) == 0:
				input.LinkTo(self)
		else:
			self.input_datums.append(None)
			self.input_values.append(input)


	def RemoveInput( self, input ):
		if hasclass(input, Datum):
			i = self.input_datums.index(input)
			input.Unlink(self)
		else:
			i = self.input_values.index(input)
		del self.input_datums[i]
		del self.input_values[i]
		self._Set( self.input_values[:], None )

		
	def Update(self, value, source):
		if source != None and source in self.input_datums:
			#
			# One of the inputs changed.
			# 
			i                    = self.input_datums.index(source)
			self.input_values[i] = value
			self._Set(self.input_values[:], source)
		elif value != None:
			#
			# The merged Datum changed.
			#
			for i in range(len(value)):
				if self.input_values[i] != value[i]:
					self.input_values[i] = value[i]
					if self.input_datums[i] != None:
						self.input_datums[i].Update(value[i], None)
	


		
# -------------------------------------------------------------------------
# CLASS:         Compose
#
# INHERITS FROM: Transform : Datum
#
# DESCRIPTION: 
#     Composes out of several inputs a value transformed by a function,
#     defined by its subclassses.
#     

class Compose(Transform):

	def __init__(self, *inputs):
		self.merge     = apply(Merge, inputs)
		self.var_input = 0
		Transform.__init__(self, self.merge)


	def Finalize( self ):
		self.merge.Finalize()
		Transform.Finalize( self )


	def VA(self, i):
		#
		# Indicates which one of the input arguments should be variable
		#
		self.var_input = i - 1
		return self

	
	def Update(self, value, source):
		if source == self.input:
			if not (None in value):
				value = self.Compute(value)
				self._Set(value, source)
		else:
			if not (None in self.input.Get()):
				value = self.InverseCompute(value)
				self.input.Update(value, self)


	# ------------------------------------------------------------------
	# Managing input arguments.

	def AddInput( self, input ):
		self.merge.AddInput( input )


	def RemoveInput( self, input ):
		self.merge.RemoveInput( input )

		
#-----------------------------------------------------

class Add(Compose):

	def Compute(self, args):
		return args[0] + args[1]

	def InverseCompute(self, result):
		args = self.GetInputValue()[:]	 # Make a copy.
		if self.var_input == 0:
			args[0] = result - args[1]
		else:
			args[1] = result - args[0]
		return args

#-----------------------------------------------------

class Subtract(Compose):


	def Compute(self, args):
		return args[0] - args[1]


	def InverseCompute(self, result):
		args = self.GetInputValue()[:]	 # Make a copy.
		if self.var_input == 0:
			args[0] = result + args[1]
		else:
			args[1] = args[0] - result
		return args

#-----------------------------------------------------

class Multiply(Compose):

	def Compute(self, args):
		return args[0] * args[1]


	def InverseCompute(self, result):
		args = self.GetInputValue()[:]	 # Make a copy.
		if self.var_input == 0:
			args[0] = result / args[1]
		else:
			args[1] = result / args[0]
		return args

#-----------------------------------------------------

class Divide(Compose):

	def Compute(self, args):
		return args[0] / args[1]


	def InverseCompute(self, result):
		args = self.GetInputValue()[:]	 # Make a copy.
		if self.var_input == 0:
			args[0] = result * args[1]
		else:
			args[1] = args[0] / result
		return args


#-----------------------------------------------------

class Max( Compose ):

	def Compute( self, args ):
		arg_list = []
		for arg in args:
			if arg != None:
				arg_list.append( arg )
		if len(arg_list) > 0:
			return max ( arg_list )
		else:
			return 0

	def InverseCompute( self, result ):
		pass


#-----------------------------------------------------

class Min( Compose ):

	def Compute( self, args ):
		arg_list = []
		for arg in args:
			if arg != None:
				arg_list.append( arg )
		if len(arg_list) > 0:
			return min ( arg_list )
		else:
			return 0

	def InverseCompute( self, result ):
		pass




#-----------------------------------------------------

# Equate classes Merge and Tuple for backward compatibility.

Tuple = Merge



#-----------------------------------------------------
# BBoxPos and BBoxSize can be used to determine the bounding-box-position
# (upper-left) and bounding-box-size given two points. These classes are
# used in the Line class

class BBoxPos( Compose ):

	def __init__( self, begin, end ):
		#
		# begin = bx|ex, end = by|ey
		#
		void = Compose.__init__(self, begin, end)


	def Compute( self, args ):
		return min( args )


	def InverseCompute( self, result):
		#
		# Translate
		#
		begin = self.GetInputValue()[0]
		end   = self.GetInputValue()[1]
		delta = result - min( [begin, end] )
		begin = begin + delta
		end   = end + delta
		return [begin,end]					 # Make a copy.

#-----------------------------------------------------

class BBoxSize( Compose ):

	def __init__( self, begin, end ):
		#
		# begin = bx|ex, end = by|ey
		#
		void = Compose.__init__(self, begin, end)


	def Compute( self, args):
		return max(args) - min(args)


	def InverseCompute( self, result ):
		#
		# Change the max val
		#
		begin = self.GetInputValue()[0]
		end   = self.GetInputValue()[1]
		if begin > end:
			begin = end + result
		else:
			end = begin + result
		return [begin,end]					 # Make a copy.





######################################################################
#  Test Functions
######################################################################
#


def test1():

	class Window:
		def __init__(self):
			self.x = 10
			self.y = 10

	# Writing to external value
	_win = Window()
	x = ExternalDatum(AttribShell(_win, 'x'))
	xi = Datum(50)
	xi.LinkTo(x)
	xi.Set(100)
	print _win.x

	# Updating when requested
	l = Datum('lab')
	x.UpdateOnDatum(l)
	_win.x = 115
	l.Set('label')
	print x.Get()

def test2():
	x = Datum(50)
	y = Datum(30)
	w = Datum(200)
	h = Datum(100)

	lhook = Tuple( x, Add(y, Divide(h, 2) ))
	rhook = Tuple(Add(x, w), Add(y, Divide(h, 2)))
	print rhook.Get()
	print lhook.Get()
	y.Set(15)
	print rhook.Get()
	print lhook.Get()


def test3():
	# Test two-way constraints.
	# s = x + y
	# t = s * 3
	x = Datum(5)
	y = Datum(10)
	s = Add(x, y).VA(2)

	print   x.Get(), y.Get(), s.Get()
	s.Set(20)
	print   x.Get(), y.Get(), s.Get()

	t = Divide(s, 3)
	print t.Get()
	t.Set(50)
	print   x.Get(), y.Get(), s.Get(), t.Get()
	u = Tuple(s, t)
	print u.Get()
	u.Set((100, 100))
	print   x.Get(), y.Get(), s.Get(), t.Get(), u.Get()


def test():
	test3()
