# -------------------------------------------------------------------------
# MODULE:      Command
#
# DESCRIPTION: 
#     Contains various Command classes.
#
# ACKNOWLEDGEMENT:
#     The design of these classes have more or less been taken from the book:
#     "OOP w. C++ & Motif", D. Young, Prentice-Hall, 1992.
#
# AUTHOR:
#     Per Spilling, CWI, Amsterdam, per@cwi.nl

import vp

from Object import Object

debug = vp.FALSE

# -------------------------------------------------------------------------
# CLASS:         Command
#
# INHERITS FROM: Object
#
# COLLABORATORS: Control
#
# DESCRIPTION: 
#     An abstract class providing the protocol for all command classes. It 
#     supports the following features:
#
#     - execute an action
#     - undo the action (1 level)
#     - enable and disable the command view(s) (the Control classes)
#     - maintain a list of dependent objects 
#     - activate and deactivate dependent objects when executed
#     - remember last command
#
#     The undo mechanism is supported via a central undo facility in the 
#     "vp" module: > theUndoCommand.
#
#     Args can be the following: 
#		 - 'name'    : <string>
#        - 'enabled' : [TRUE=default|FALSE] 
#        - 'has_undo': [TRUE|FALSE=default]

class Command( Object ):

	# ------------------------------------------------------------------
	# Constructor & destructor

	def __init__( self, argdict = {} ):
		self.name         = ''
		self.enabled      = vp.TRUE
		self.has_undo     = vp.FALSE

		Object.__init__( self, argdict )

		self.viewlist     = []
		self.prev_enabled = 0
		self.enable_list  = []
		self.disable_list = []


	def Finalize( self ):
		for view in self.viewlist: view.Finalize()

		self.viewlist     = None
		self.enable_list  = None
		self.disable_list = None

		Object.Finalize( self )


	# ------------------------------------------------------------------
	# Misc. public methods:

	def Register( self, command_view ):
		#
		# The command_view will usually be a button
		# 
		self.viewlist.append( command_view )

		if self.enabled:
			command_view.Enable()
		else:
			command_view.Disable()


	def UnRegister( self, command_view ):
		self.viewlist.remove( command_view )


	def Enable( self ):
		#
		# Enable the command views
		#
		for view in self.viewlist: view.Enable()

		self.prev_enabled = self.enabled      # save current state
		self.enabled      = vp.TRUE


	def Disable( self ):
		for view in self.viewlist: view.Disable()

		self.prev_enabled = self.enabled      # save current state
		self.enabled = vp.FALSE


	def Revert( self ):
		#
		# Return to previous state
		#
		if self.prev_enabled:
			self.Enable()
		else:
			self.Disable()


	def AddToEnableList( self, command ):
		self.enable_list.append( command )
		

	def AddToDisableList( self, command ):
		self.disable_list.append( command )


	def Execute( self ):
		#
		# If a command is disabled then it cannot be executed
		#
		if not self.enabled:
			return

		# Call the "Doit" method of the derived class to perform the action

		self.Doit()

		if self.has_undo:
			vp.theLastCommand = self
			vp.theUndoCommand.Enable()
		else:
			vp.theLastCommand = None
			vp.theUndoCommand.Disable()

		# Process the commands that depend on this one

		for com in self.enable_list:
			com.Enable()

		for com in self.disable_list:
			com.Disable()


	def Undo( self ):
		#
		# Call the Undoit method of the derived class
		#
		self.Undoit()

		# The vpApp supports only one level so disable the undo facility

		vp.theUndoCommand.Disable()

		# Reverse the effect of the Execute() method by reverting all
		# dependent commands to their previous states

		for com in self.enable_list:
			com.Revert()

		for com in self.disable_list:
			com.Revert()

	# ------------------------------------------------------------------
	# Access functions:

	def IsEnabled( self ):
		return self.enabled


	def HasUndo( self ):
		return self.has_undo


	# ------------------------------------------------------------------
	# Methods which should be overridden by subclasses

	def Doit( self ):
		print 'Command.Doit called for command:', self.GetName()


	def Undoit( self ):
		print 'Command.Undoit called for command:', self.GetName()



# -------------------------------------------------------------------------
# CLASS:         AksFirstCommand
#
# INHERITS FROM: Command : Object
#
# DESCRIPTION: 
#     Abstract class which uses the QuestionDialog to ask the user a 
#     question before proceeding.
#
#     Args can be the following: 
#		 - 'question'    : <string>
#

default_question = 'Do you really want to execute this command?'

class AskFirstCommand( Command ):
	
	def __init__( self, argdict = {} ):
		self.question = default_question
		Command.__init__( self, argdict ) 


	def SetQuestion( self, text ):
		self.question = text


	def Execute( self ):            # o.r.m from Command class
		vp.theQuestionDialog.Post( self.question, self.OkCB, None, None )


	def OkCB( self, question_dialog ):
		Command.Execute( self )
		

# -------------------------------------------------------------------------
# CLASS:         WarnNoUndoCommand
#
# INHERITS FROM: AskFirstCommand : Command : Object
#
# DESCRIPTION: 
#     Abstract class which which warns the user that the command he wants
#     to execute cannot be undone. The Doit() method must be defined in
#     a subclass.
#

default_warning = 'This command cannot be undone. Proceed anyway?'

class WarnNoUndoCommand( AskFirstCommand ):
	
	def __init__( self, argdict = {} ):
		#
		# Args can be:
		# - 'name'   : <string> (required)
		# - 'enabled': [TRUE=default|FALSE]
		#
		argdict = self.MergeDefaults( argdict, {'has_undo': vp.FALSE} )
		AskFirstCommand.__init__( self, argdict ) 
		self.SetQuestion( default_warning )



