#!/usr/bin/python

"""
	DTM (Dynamic Terminating Multicast) is the basis for 
	Mobile agents using Group Communication Services. 

	Protocol:

	Message M is sent to nodes p0,p1,p2 ... pN.
	A message must be recieved all nodes from p0...pN at some
        indeterminate time in the future for the message to be considered
	a success. If, after an elapsed time, a response has not been
	recieved by a some of the nodes (a subset of p0...pN) then
	a copy of the original message is sent to the subset until 
	all nodes respond, the orignal is deleted. 

	All succsesful messages are flushed. 

	There are then two types of incoming data - messages and responses.
		     
	Each message as an order number. If it is greater than 
	1 plus the current order number for that address than it is places
	in storage until a message with an order number of current+1 is
	recieved. 

	Each response carries the order number and sender address. Nothing
	more.

	Features:
	
		We must a have a way of giving each node
	in the network a virtual static IP address. This
	is because of the following problem: 
	
	If machine A has a dynamic IP and looses its
	network connection, when it reaquires it it will
	have a different IP address when it comes up and
	thus any job going between it and another node
	will be lost.
	
	Therefore, when machine A re-aquires its network 
	connection it must broadcast its IP address to
	every node which has come in contact with to update
	the ip address. 
	
		



"""
from socket import *
from signal import *
import time, shelve, rand, strop

import cPickle

# --- exceptions -----
DatagramTooLarge = "dtm.DatagramTooLarge"
UnhandledClass = "dtm.UnhandledClass"
InternalError = "dtm.InternalError"
NetworkDown = "dtm.NetworkDown"

# constants and other things 
from config import * 


# shells out to a utility called "iphunter" 
# that gets the ip address of the host machine 
# (static or dynamic)
def getIP():
	import os, regsub, tempfile
	
	tfilename = tempfile.mktemp()
	# try to create it to make sure
	try:
		open(tfilename,"w")
		os.remove(tfilename)
	except:
		tfilename = "%d.tmp" % (rand.rand())
	
	print ""
	print "Probing for IP connection ..."
	
	for mode in ("ppp","slip"):
		cmd = "./iphunter %s > %s" % (mode,tfilename)
		print "Tesing wether or not you have a ",mode," connection ... "
		error = os.system(cmd)
		
		if error == 0:
			text = open(tfilename).read()
			text = regsub.gsub("\n","",text)
			os.remove(tfilename)
			print mode," connection found, getting IP address",text

			return text
		else:
			print "Test failed trying next protocol ... "	
			
	msg = "Unable to aquire ip address"
	raise NetworkDown, msg			
					
"""
	Repesents a message sent to a host.
	Contains data, an idnumber and a list 
	of destinations.	
"""
class message:
	#def __init__(self, orderkey = None, data = None, source = None):
	def __init__(self, data = None, source = None):
		if orderkey == None: return

		#self.data = cPickle.dumps(data)
		self.source = source
		self.timestamp = time.time()
		self.lifecount = LIFECOUNT
		self.orderkey = unique_key()
		
	def addressChange(sef, oldIp, newIp):
		if self.source == oldIp:
			self.source = newIp	
		
	def getId(self):
		return self.orderkey
		
	def decLifeCount(self):
		self.lifecount = self.lifecount - 1

	def isAlive(self):
		if self.lifecount > 0: return true
		return false		

	def getData(self):
		return cPickle.loads(self.data)

# abstract base of all orders
class order:
	pass
	
# dispatch a command to the dtmMgr
class sendOrder(order):
	def __init__(self, rawData=None, destinations=None, orderkey=None):
		self.rawData = rawData
		self.destinations = destinations		
		self.orderkey = orderkey


	
#respresents a response to a message	
class response:
	def __init__(self, destination = None, order = None, source = None):
		self.destination,self.orderkey = destination, order
		self.source = source
		
	def addressChange(sef, oldIp, newIp):
		if self.source == oldIp:
			self.source = newIp	
		if self.destination == oldIp:
			self.destination = newIp	

# no response from an address after max retries
class suspect:
	def __init__(self, destination=None, addr=None, timestamp=None):
		self.addr, self.timestamp = addr, timestamp
		self.destination = destination
		
	def addressChange(sef, oldIp, newIp):
		if self.destination == oldIp:
			self.destination = newIp	
		if self.addr == oldIp:
			self.addr = newIp	
		

# there has been a change in our local ip addresses
"""
class update:
	def __init__(self, oldIp=None, newIp=None):
		self.oldIp,self.newIp=oldIp,newIp
"""	
				

#*****
# creates a udp socket for a datagram 
def _make_socket():
	# create DATAGRAM 		
	proto = getprotobyname("udp")
	type = SOCK_DGRAM
	family = AF_INET		
	return socket( family, type, proto )


#*****
# creates a unique identifier
def unique_key():
	key = "%s_%s" % (time.time(),rand.rand())		
	return key

# primitive fo send commands
def send_data( data ):
	# make sure we can send
	if len(data) > MAXSIZE:
		print "Data length of ",len(data)," exceeds maximum limit of ",MAXSIZE 
		raise DatagramTooLarge 
		
	s = _make_socket()
	s.connect( "", DTM_PORT )				
	s.send( data )
	s.close()


#*****
# sends raw data to a a list of hosts
def send(rawData, destinations = [""]):
	orderkey = unique_key() 
	
	# check for work around for cPickle 
	order = sendOrder(rawData,destinations,orderkey)
	_order = cPickle.dumps( order )
	send_data( _order )	 
	return orderkey

#*****
# Used to construct a class on the fly. The cPickle module
# needs to have the module with the class declaration in it loaded
# prior to calling cPickle.loads(). In order to get around this I
# pack the name of the class and the module along with the
# class's data dictionary then reesemble on the client
class cacoon:
	def __init__(self, rawData=None, module=None, className=None):
		if rawData == None: return

		self.data = cPickle.dumps( vars(rawData) )

		if module == None and className == None:
			x = str( rawData )[1:]	
			x = strop.split(x)[0]
			l = strop.splitfields(x,".")
		
			self.module = l[0]
			self.className = l[1]
		else:
			self.module = module
			self.className = className
		


	# compose object without using normal pickling mechanism		
	def unpickle(self):
		cmd = """
from %(m)s import %(s)s
obj = %(s)s()
"""
		
		cmd = cmd % {
			"m" : self.module,
			"s" : self.className
			}
	
		# get methods	
		exec(compile(cmd,"<string>","exec"))
		
		# get fields
		dict = cPickle.loads(self.data)
		for (attr,val) in dict.items():
			setattr(obj,attr,val)

		return obj
		
class update:
	def __init__(self, oldIp=None, newIp=None):
		self.oldIp,self.newIp=oldIp,newIp

		
class netalias:
	def __init__(self):
		self.d = {}
		
	def __getitem__(self, ip):
		if self.d.has_key(ip):
			return self.d[ip]
		else:
			return ip
		
	def __setitem__(self, oldIp, newIp):
		# see if the new ip already exists
		for (o,n) in self.d.items():
			if oldIp == n:
				self.d[o] = newIp
				self.d[oldIp] = newIp
		self.d[oldIp] = newIp		
		
			
		
#***** 
# dtmMgr - Dynamic Terminating Multicast Manager.
#	   Responsible for sending and recieving messages and sending
#          responses.
class dtmMgr:
	def __init__(self, verbose = false, testmode = false):
		self.msgTable = shelve.open(MSGTABLE)
		self.failures = {}
		# list of all ip addresses we have had
		# contact with. This is used if our ip address
		# has changed due to a loss of network connection
		# (i.e. we have a dynamic ip and our address
		# changed when we logged back in 
		self.contact_set = {}

		# to prevent duplicate messages
		# due to slow netorks
		self.receipts = {}
		
		# if our ip address has changed 
		# (We have a dynamic ip and the network
		#  has restarted) we treat the old address
		# as an alias for the new one
		self.netalias = netalias()
		
		self.verbose = verbose
		self.testmode = testmode

		self.localhost = getIP()
		self.netstat = true
	
	def __del__(self):
		self.s.close()	

	#----- public interface -------------
	
	def addressChange(self, x):
		self.netalias[x.oldIp] = x.newIp
	
	# We have just come up check to see if our ip address
	# has changed, if it has add our old ip address to the 
	# netalias table along with our new address then broadcast to
	# everybody who knows us our new address
	def updateIp(self):
		newIp = getIp()
		if newIp != self.localhost:
			u = update(self.localhost,newIp)
			# every address we have come in contact
			# with is kept here, broadcast to all of them
			dest = self.contact_set.keys()
			
			self.addressChange(u)
			self.localhost = newIp

			# broadcast to everybody that
			# our address has changed
			send(u, dest)	

	# handles bookkeeping tasks
	def backgroundTask(self, *args):
		n = self.netStatus()
				
		# did our connection go down go down ?
		if n == false:
			print "Network down!"		
			self.netstat = false
			# can't resend messages
			return
	
		# test if we were doen and just came up
		elif n == true and self.netstat == false:
			self.netstat = true
			print "Network is back up !"
			self.updateIp() 
			
		
		self._checkMessages(args)


	# bogus destination ip address
	def UnableToFindIP(self, destination):
		pass

	#*****
	# Pure virtual used by derived class to process messages
	def messageHandler(self, m):
		pass

	#*****
	# signal that message m has failed. Destination has 
	# not responded.
	def failedMessage(self, m):
		pass


		



	# loop forever handling messages. Interupt every 5
	# seconds to check on the status of messages already 
	# sent
	def mainloop(self):
		import sys					

		# setup signal handler
		signal( SIGALRM, self.backgroundTask )
		
		while 1:
			#if self.testmode == true:	
			#	mp = self.s.recv(MAXSIZE)
			#	self._proc(mp)
			#	continue

			# every five seconds call background task
			alarm(5)
			
			try:
				# receive next udp datagram 
				mp = self.s.recv(MAXSIZE)
				# handles all incoming messages	
				try:
				
					self._proc(mp)
				except:
					print "---     exception      ---"
					print "Error in dtm._proc(): "	
					print sys.exc_type, sys.exc_value
					print "--------------------------"
			except:
				pass				

	#*****
	# Creates a server for handling incoming messages
	def server(self):
		self.s = _make_socket()
		self.s.bind("", DTM_PORT)
		self.s.listen(1)
		self.mainloop()
		

	#------------  private methods -------------		


	# message never made it
	def _del_message(self, m):
		del self.msgTable[ m.orderkey ]


	# adds a message to a dictionary of message 
	# dictionaries. The first key is the destination 
	# the next is the orderkeyber
	def _add_message(self, destination, m):	
		# place message into a table in case we have to resend			
		self.msgTable[ m.orderkey ] = (destination, m)

	# source refers to response source
	def _remove_message(self, source, orderkey):
		del self.msgTable[ orderkey ]

	# decode class			
	def decode(self, x):		
		# determine if its a message or a response to a message
		if   message().__class__ == x.__class__:
			self._proc_message(x)
		elif response().__class__ == x.__class__:
			self._proc_response(x)
		elif sendOrder().__class__ == x.__class__:
			self._proc_sendOrder(x)
		elif suspect().__class__ == x.__class__:
			self._proc_suspect(x)
		elif update().__class__ == x.__class__:	
			self.addressChange(x)
		else:
			msg = "Unhandled class %s in stmMgr._proc()" % (x.__class__)
			raise UnhandledClass, msg

	# process message
	def _proc(self, mp):
		# unpickle here
		try:
			x = cPickle.loads(mp)
		except cPickle.UnpicklingError:
			# garbled message, return without 
			# response so we force a resend
			if self.verbose == true:
				print "Garbled transmition", mp
			return
		self.decode( x )
	
	# send message m to multiple addresses
	def _proc_sendOrder(self, order):
		rawData, destinations, orderkey = order.rawData, order.destinations, order.orderkey	
	
		# create new message
		#m = message(orderkey, rawData, self.localhost)

		# pickle message into a string to be sent as a datagram
		mp = cPickle.dumps(m)	

		# create a udp socket	
		s = _make_socket()
		
		# send the datagram to all destinations
		for destination in destinations:
			# create new message
			m = message(rawData, self.localhost)

			# pickle message into a string to be sent as a datagram
			mp = cPickle.dumps(m)	
			
			# see if destination is on the list of dynamic ip's
			# that have changed.
			destination = self.netalias[destinaion]

			if destination in (self.localhost,"",gethostname()):
				# if we are sending to localhost, 
				# then don't add to message table
				self.messageHandler(m)
			else:	
				# place destination into a set of all destinations
				# that have come in contact with up besides ourselves
				if destination != self.localhost:
					self.contact_set[destination] = None

				# add to message table
				self._add_message(destination,m)
				
				# if the network is down don't bother
				# sending we will resend later, just continue
				# in this loop until we have recorded all the destinations
				if self.netstat == false: continue
					
				# connect to remote host						
				try:
					s.connect(destination, DTM_PORT)
				except:
					self.netstat = self.netStatus()
					if self.netstat == true:
						# Unknown ip address network is fine
						self.UnableToFindIP( destination )
						continue
							

				if self.verbose == true:
					print "Sending ",len(mp)," bytes to ", destination
					
				# fire off a packet
				s.send( mp )

		# close socket
		s.close()

	# remove message from table, response recieved
	def _proc_response(self, r):
		if self.verbose == true:
			print "response from ",r.source
		self._remove_message(r.source, r.orderkey)

	# process a message, send a response to sender and call handler
	def _proc_message(self, m): 		
		if self.verbose == true:
			print "recieved message from ",m.source

		# check to see if this is a duplicate
		# This can happen on extremely slow networks
		# and the sender thinks we haven't responded so
		# it has resent the message. But we actually *did*
		# recieve the message and the response hasn't gotten 
		# back in time.
		if not self.receipts.has_key( m.getId() ): 
			# call handler for all messages
			self.messageHandler(m)
			self.receipts[ m.getId() ] = None
		# ... else
		#	we already processed this message, the
		#	resend was not nessessary.	
			
				
		# send response back to source
		r = response(m.source, m.orderkey, self.localhost)
		if self.verbose == true:
			print "Sending response to ",r.destination
		self._resend(r)

	# called periodically to resend a message to a remote site
	def _checkMessages(self, *args):
		# get the current time
		c = time.time()
		
		# examine each message still waiting for a 
		# response
		for key in self.msgTable.keys():
			(dest, m) = self.msgTable[ key ]

			t = c - m.timestamp

			# is this message too old ?
			if t > RESEND_THRESHOLD:
				# determine if this message is still alive.
				m.decLifeCount()
				if m.isAlive() == true:	
					# resend message to destination
					m.destination = dest
					if self.verbose == true:
						print "Resending message to ", m.destination, " ... no response yet"
					self._resend(m)
					self.msgTable[key] = (dest, m)
				else:
					# remove message and signal that the message
					# has failed
					print "Message failed node ",dest," suspect."
					
					
					# remove from table
	 				self._remove_message(dest, key)
					# send to handler
					self.failedMessage(m)
		
		# reset signal		
		signal( SIGALRM, self.backgroundTask )


	# resend a message to a remote host
	def _resend(self, m):		
		# pickle message into a string to be sent as a datagram
		mp = cPickle.dumps(m)

		# create datagram 		
		s = _make_socket()
		# send datagram
		s.connect( m.destination, DTM_PORT)
		s.send( mp )

		# close socket
		s.close()

	# test to see if we are still on the web			
	def netStatus(self):		
		# Try to reach well known addresses on the web.
		# If these guys are all down we assume that there has
		# been a neuclear war.
		for address in ("www.microsoft.com","www.w3c.org","www.oracle.com"):
			try:
				s = socket(AF_INET, SOCK_STREAM)
				s.connect(address, 80)
				s.close()
				return true
			except:
				pass
		return false
		

										
"""
	test harness -
	
	1) Send a message in loopback and recieve a response.
	2) Send to csmctmto.interpoint.net 
"""
class testMgr(dtmMgr):
	def __init__(self):
		dtmMgr.__init__(self, true)
	
	def messageHandler(self, m):
		print m

 	 
def test():
	addr = "205.152.185.214"
	for i in range(0,100):
		send("Hello",[addr])


	
if __name__ == "__main__":
	import sys
	
	if sys.argv[1] == "start_server":
		testMgr().server()
	else:
		test()





