/***********************************************************
Copyright 1991-1995 by Stichting Mathematisch Centrum, Amsterdam,
The Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI or Corporation for National Research Initiatives or
CNRI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission.

While CWI is the initial source for this software, a modified version
is made available by the Corporation for National Research Initiatives
(CNRI) at the Internet address ftp://ftp.python.org.

STICHTING MATHEMATISCH CENTRUM AND CNRI DISCLAIM ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH
CENTRUM OR CNRI BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

#include "Python.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>


/* Declarations for objects of type production */

typedef struct {
	PyObject_HEAD
	char *raw;
	int   istag;	
} productionobject;

staticforward PyTypeObject Productiontype;



/* ---------------------------------------------------------------- */


static PyObject *ErrorObject;
#define onError(message) \
	{ PyError_SetString(ErrorObject, message); return NULL; }



/* ----------------------------------------------------- */


void replace(char *source, char *pattern, char replace, char **buffer)
{
	char *pos;
	
	if ((pos = strstr(source,pattern)) != NULL)
	{
		char *sptr, *tptr;
		
		/* one time creation cost */
		if (*buffer == NULL)
			*buffer = strdup(source);

		/* copy upto position */		
		for (tptr = *buffer, sptr = source;
		     sptr < pos;
		     tptr++, sptr++)
		{
			*tptr = *sptr;	
		}
		/* insert replacement*/
		*tptr = replace;
		tptr++;
		
		/* copy till end*/
		for (sptr = (char *) pos + strlen(pattern); 
			*sptr != '\0';
			sptr++, tptr++)
		{
			*tptr = *sptr;
		}
		*tptr = '\0';

		strcpy(source, *buffer);
	}
}

void eval_ascii(char *source, char **buffer)
{
	char *pos;
	
	if ((pos = strstr(source,"&#")) != NULL)
	{
		char digits[4], ascii;
		char *sptr, *tptr;
		int i;
		
		/* one time creation cost */
		if (*buffer == NULL)
			*buffer = strdup(source);

		/* copy upto position */		
		for (tptr = *buffer, sptr = source;
		     sptr < pos;
		     tptr++, sptr++)
		{
			*tptr = *sptr;	
		}

		sptr++; /* point past & */
		sptr++; /* point past # */
		
		i = 0;
		while (isdigit(*sptr)){
			digits[i++] = *sptr;
			sptr++;	
			}
		digits[i] = '\0';
		
		if (i > 0){
			*tptr = (char) atoi(digits);
			tptr++;
			}

		/* copy till end*/
		for (; 
			*sptr != '\0';
			sptr++, tptr++)
		{
			*tptr = *sptr;
		}
		*tptr = '\0';
		strcpy(source, *buffer);		
	}
}

/* save on stack creation costs by making global */
struct _aliases
{
	char *pattern;
	char *replacement;
}aliases[] = {
{"&quot", "\""},
{"&lt"  , "<"},
{"&gt", ">"},
{"&amp", "&"},
{0,0} 	
};		


void production_translate(productionobject *self, char *raw)
{
	int i;
	char *buffer;
	
	buffer = NULL;
	
	/* replace aliases */	
	for (i = 0; aliases[i].pattern != 0; i++)
		replace(raw,
			aliases[i].pattern,
			aliases[i].replacement[0],
			&buffer);

	eval_ascii(raw, &buffer);
	
	/* if the buffer was used then it must be equal to raw, don't
	   free it. use it as apposed to a wasted strdup*/		
	if (buffer != NULL){ 
		self->raw = buffer;
		} else {
		self->raw = strdup(raw);
		}
}





/* ----------------------------------------------------- */


static productionobject *
newproductionobject()
{
	productionobject *self;
	
	self = PyObject_NEW(productionobject, &Productiontype);
	if (self == NULL)
		return NULL;
	self->raw = NULL;
	return self;
}



static char production_raw__doc__[] = 
"get raw text that was eaither inside or outside '<' '>'"
;

static PyObject *
production_raw(self, args)
	productionobject *self;	
	PyObject *args;
{
	PyObject *res;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyString_FromString((const char *)self->raw);
}


/* istag method */
static char production_istag__doc__[] =
"Returns a flag that determines whether or not this production was inside or outside of <>"
;

static PyObject *
production_istag(self, args)
	productionobject *self;	
	PyObject *args;
{
	PyObject *res;

	if (!PyArg_ParseTuple(args, ""))
		return NULL;
	return PyInt_FromLong((long) self->istag);
}

/* init method */
static char production_init__doc__[] =
"initialize production"
;

static PyObject *
production_init(self, args)
	productionobject *self;
	PyObject *args;
{
	char *raw;
	int   istag;
	productionobject *nobj;

	if (!PyArg_ParseTuple(args, "si", &raw, &istag))
		return NULL;
	
	/* create a new production object */
	nobj = newproductionobject();
	
	/* set values */
	nobj->istag = istag;
	production_translate(nobj, raw);

        return (PyObject *) nobj;
}





static struct PyMethodDef production_methods[] = {
	{"istag", (PyCFunction)production_istag,	METH_VARARGS,	production_istag__doc__},
 	{"raw",	(PyCFunction)production_raw,	METH_VARARGS,	production_raw__doc__},
 	{"init",  (PyCFunction)production_init, METH_VARARGS, production_init__doc__},

	{NULL,		NULL}		/* sentinel */
};

/* -------------------------------------------------------------------- */



static PyObject *
production_dealloc(self)
	productionobject *self;
{
	if (self->raw != NULL) free(self->raw);
	PyMem_DEL(self);
} 

static PyObject *
production_getattr(self, name)
	productionobject *self;
	char *name;
{
	/* XXXX Add your own getattr code here */
	return Py_FindMethod(production_methods, (PyObject *)self, name);
}

static char Productiontype__doc__[] = 
""
;

static PyTypeObject Productiontype = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,				/*ob_size*/
	"production",			/*tp_name*/
	sizeof(productionobject),		/*tp_basicsize*/
	0,				/*tp_itemsize*/
	/* methods */
	(destructor) production_dealloc,	/*tp_dealloc*/
	(printfunc)0,		/*tp_print*/
	(getattrfunc)production_getattr,	/*tp_getattr*/
	(setattrfunc)0,	/*tp_setattr*/
	(cmpfunc)0,		/*tp_compare*/
	(reprfunc)0,		/*tp_repr*/
	0,			/*tp_as_number*/
	0,		/*tp_as_sequence*/
	0,		/*tp_as_mapping*/
	(hashfunc)0,		/*tp_hash*/
	(ternaryfunc)0,		/*tp_call*/
	(reprfunc)0,		/*tp_str*/

	/* Space for future expansion */
	0L,0L,0L,0L,
	Productiontype__doc__ /* Documentation string */
};

/* End of code for production objects */
/* -------------------------------------------------------- */


/* Initialization function for the module (*must* be called initproduction) */

static char production_module_documentation[] = 
""
;

void
initproduction()
{
	PyObject *m, *d;

	/* Create the module and add the functions */
	m = Py_InitModule4("production", production_methods,
		production_module_documentation,
		(PyObject*)NULL,PYTHON_API_VERSION);

	/* Add some symbolic constants to the module */
	d = PyModule_GetDict(m);
	ErrorObject = PyString_FromString("production.error");
	PyDict_SetItemString(d, "error", ErrorObject);

	/* XXXX Add constants here */
	
	/* Check for errors */
	if (PyErr_Occurred())
		Py_FatalError("can't initialize module production");
}

