#
# Copyright 1995 Carlos Maltzahn
# 
# Permission to use, copy, modify, distribute, and sell this software
# and its documentation for any purpose is hereby granted without fee,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of Carlos Maltzahn or 
# the University of Colorado not be used in advertising or publicity 
# pertaining to distribution of the software without specific, written 
# prior permission.  Carlos Maltzahn makes no representations about the 
# suitability of this software for any purpose.  It is provided "as is" 
# without express or implied warranty.
# 
# CARLOS MALTZAHN AND THE UNIVERSITY OF COLORADO DISCLAIMS ALL WARRANTIES 
# WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF 
# MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF COLORADO
# BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY 
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
# IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING 
# OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# 
# Author:
# 	Carlos Maltzahn
# 	Dept. of Computer Science
# 	Campus Box 430
# 	Univ. of Colorado, Boulder
# 	Boulder, CO 80309
# 
# 	carlosm@cs.colorado.edu
#

import sys
import shelve
import string
from types import *
import copy
import Utilities


### Initialization ### 

# The database consists of the following entries:
# db['__id__'] = lowest object id that has never been used
# db[<class name>] = list of db_ids of objects which belong to this class or 
#                    its subclasses
# db[`<db_id>`] = - object with db_id references instead of object references
#                 (Note: format of a db_id *reference* is ('__db', db_id);
#                  a db_id is a unique integer number)
#                 (Note: db_id is an integer. The corresponding key 
#                  to the database is a string)
#
# write locks are kept in two dictionaries: 
#   locks_by_client: client -> list of locked db_ids
#   locks_by_db_id: db_id -> client
#
# notification requests are stored as a list
# called request_list. Each query is a tuple consisting of 
# (request_id, client, scope, prop_list)

# create name for dbfile (default: 'database')
# the name can be specified with the second command line argument

if len(sys.argv) == 3:
  dbfile = sys.argv[2]
else:
  dbfile = 'database'
  
# open database
db = shelve.open(dbfile)

# set unique id counter if new database
if not db.has_key('__id__'):
  db['__id__'] = 1

# initialize lock tables
locks_by_client = {}
locks_by_db_id = {}

# initialize request_list
request_list = []


### Public functions ###

# load objects for client. if 'rw' then client wants to lock all loaded objects.
# Scope is defined either by a class name or a list of db_id refs. Properties 
# are defined as a list of label-value tuples with db_ids as references to 
# objects.
# Releases all locks of client if client tries but cannot lock all objects
# requested
# Returns a list of dissassembled objects when successful, otherwise
# an error is raised
def load(client, r_w, scope, properties):
  global db

  # evaluate query
  obj_list = eval_query(db, scope, properties)

  # if write access is requested try to lock selected objects 
  if r_w == 'rw':
    if not_locked_by_other(client, obj_list):
      lock(client, obj_list)
      print obj_list, "locked"
    else:
      release_all_locks(client)
      print 'Store: all locks released of client', client
      raise RuntimeError, 'some requested objects are blocked'
  elif r_w != 'r':
    raise ValueError, 'needs either \'r\' or \'rw\''

  return obj_list

# commit:
# commits only those objects that are in obj_list and locked by client
def commit(client, obj_list):
  global db, request_list

  # check type of obj_list
  if type(obj_list) is not ListType:
    raise TypeError, "needs *list* of objects"

  # determine which objects are locked by client. 
  # readonly objects are ignored. 
  writable_obj = []
  for obj in obj_list:
    if has_lock(client, obj):
      writable_obj.append(obj)

  # write objects to the database and the new_object_dict
  print 'Store: writing', writable_obj
  new_object_dict = {}
  for obj in writable_obj:
    db[`obj.db_id`] = obj
    new_object_dict[`obj.db_id`] = obj

    # Update class lists in database and in new_object_dict 
    for class_obj in Utilities.classes(obj):
      class_name = string.split(`class_obj`)[1]
      if db.has_key(class_name): 
	list = db[class_name]
	if `obj.db_id` not in list: 
	  list.append(`obj.db_id`)
	  db[class_name] = list
      else:
	db[class_name] = [`obj.db_id`]
      if new_object_dict.has_key(class_name): 
	if `obj.db_id` not in new_object_dict[class_name]:
	  new_object_dict[class_name].append(`obj.db_id`)
      else:
	new_object_dict[class_name] = [`obj.db_id`]

  # evaluate registered notification queries on new_object_dict
  notifications = []
  for (req_id, r_client, scope, properties) in request_list:
    print 'Store5 request:', (req_id, r_client, scope, properties)
    matched_objs = eval_query(new_object_dict, scope, properties)
    print 'Store5 matched_objs:', matched_objs
    if len(matched_objs) > 0:
      notifications.append((r_client, req_id, matched_objs))

  must_flush = 0

  # remove all objects with '_delete' attribute
  for obj in writable_obj:
    if hasattr(obj,'_delete'):
      del db[`obj.db_id`]
      must_flush = 1
      for class_obj in Utilities.classes(obj):
	class_name = string.split(`class_obj`)[1]
	if db.has_key(class_name): 
	  list = db[class_name]
	  if `obj.db_id` in list: 
	    list.remove(`obj.db_id`)
	    db[class_name] = list
      print 'Store: removed', obj 

  # flush database to file
  if must_flush == 1:
    db.close()
    db = shelve.open(dbfile)

  # release all locks of client to ensure two phase locking protocol
  release_all_locks(client)
  print 'commit releases all locks of', client

  #print 'db =', db
  return ('ok', notifications)

def release_all_locks(client):
  global locks_by_db_id, locks_by_client
  if locks_by_client.has_key(client):
    for db_id in locks_by_client[client]:
      locks_by_db_id[db_id] = ''
    print 'Store:', locks_by_client[client], 'released'
    del locks_by_client[client]

def register(client, scope, prop_list):
  global request_list
  request_id = getNewId()
  request_list.append((request_id, client, scope, prop_list))
  return request_id

def unregister(client, request_id):
  global request_list
  for (i, c, s, p) in request_list[:]:
    if c == client and i == request_id:
      request_list.remove((i, c, s, p))
      break
  else:
    raise ValueError, 'no such request_id with this client'
  return 'ok'

def delete_all_requests(client):
  global request_list
  for (i, c, s, p) in request_list[:]:
    if c == client:
      request_list.remove((i,c,s,p))

def public_lock(client, db_id_list):
  global locks_by_db_id
  db_id_refs = []
  for db_id in db_id_list:
    if locks_by_db_id.has_key(db_id) and\
       locks_by_db_id[db_id] not in ['', client]:
      return None
    db_id_refs.append(('__db', db_id))
  return load(client, 'rw', db_id_refs, [])

def get_new_ids(client, count):
  global db, locks_by_db_id, locks_by_client

  # get new id
  new_id = db['__id__']
  db['__id__'] = new_id + count

  # lock new ids for client so objects can be committed
  if not locks_by_client.has_key(client):
    locks_by_client[client] = []
  for db_id in range(new_id, new_id + count):
    locks_by_client[client].append(db_id)
    locks_by_db_id[db_id] = client
  
  # return new db_ids
  return range(new_id, new_id + count)


### Private functions ###

# checks whether all objects within obj_list are not locked by someone else
def not_locked_by_other(client, obj_list):
  global locks_by_db_id
  for obj in obj_list:
    if locks_by_db_id.has_key(obj.db_id) and \
       locks_by_db_id[obj.db_id] not in ['', client]:
      return 0
  return 1

def lock(client, obj_list):
  global locks_by_db_id, locks_by_client
  if not locks_by_client.has_key(client):
    locks_by_client[client] = []
  for obj in obj_list:
    locks_by_client[client].append(obj.db_id)
    locks_by_db_id[obj.db_id] = client

def has_lock(client, obj):
  global locks_by_db_id
  if hasattr(obj, 'db_id'):
    if locks_by_db_id.has_key(obj.db_id):
      return locks_by_db_id[obj.db_id] == client
    else:
      return 0
  else:
    return 1 # in this case the client introduces a new object

def getNewId():
  global db
  id = db['__id__']
  db['__id__'] = id + 1
  if db['__id__'] <= id:
    raise RuntimeError, "__id__ field was not incremented"
  return id

def eval_query(database, scope, properties):

  # determine scope of objects which can be either a class or a list of db_id
  # references
  scope_list = []
  if type(scope) is StringType:
    if database.has_key(scope):
      for db_id_str in database[scope]:
	scope_list.append(database[db_id_str])
  elif type(scope) is ListType:
    for (x, db_id) in scope:
      if database.has_key(`db_id`):
	scope_list.append(database[`db_id`])
#      else:
#	raise ValueError, "non valid db_id"
  else:
    raise TypeError, "scope needs string or list"

  # select those objects from the scope by the given properties
  # and build a list of objects (obj_list) 
  obj_list = []
  if len(properties) == 0: 
    obj_list = scope_list
  else:
    for object in scope_list:
      for (label, op, value) in properties:
	if not hasattr(object, label):
	  break
	else:
	  if op == '==':
	    if getattr(object, label) != value:
	      break
	  elif op == '!=':
	    if getattr(object, label) == value:
	      break
	  elif op == 'in':
	    if getattr(object, label) not in value:
	      break
	  elif op == 'not in':
	    if getattr(object, label) in value:
	      break
	  elif op == 'has':
	    if value not in getattr(object, label):
	      break
	  elif op == 'has not':
	    if value in getattr(object, label):
	      break
	  elif op == 'all in':
	    foundall = 1
	    for item in getattr(object, label):
	      if item not in value:
		foundall = 0
		break
	    if not foundall:
	      break
	  elif op == 'not all in':
	    foundall = 1
	    for item in getattr(object, label):
	      if item not in value:
		foundall = 0
		break
	    if foundall:
	      break
	  elif op == 'some in':
	    found = 0
	    for item in getattr(object, label):
	      if item in value:
		found = 1
		break
	    if not found:
	      break
	  elif op == 'none in':
	    found = 0
	    for item in getattr(object, label):
	      if item in value:
		found = 1
		break
	    if found:
	      break
	  else:
	    raise ValueError, 'Unknown relation'
      else:
	obj_list.append(object)
  return obj_list
