#
# Copyright 1995 Carlos Maltzahn
# 
# Permission to use, copy, modify, distribute, and sell this software
# and its documentation for any purpose is hereby granted without fee,
# provided that the above copyright notice appear in all copies and that
# both that copyright notice and this permission notice appear in
# supporting documentation, and that the name of Carlos Maltzahn or 
# the University of Colorado not be used in advertising or publicity 
# pertaining to distribution of the software without specific, written 
# prior permission.  Carlos Maltzahn makes no representations about the 
# suitability of this software for any purpose.  It is provided "as is" 
# without express or implied warranty.
# 
# CARLOS MALTZAHN AND THE UNIVERSITY OF COLORADO DISCLAIMS ALL WARRANTIES 
# WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF 
# MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF COLORADO
# BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY 
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER 
# IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING 
# OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
# 
# Author:
# 	Carlos Maltzahn
# 	Dept. of Computer Science
# 	Campus Box 430
# 	Univ. of Colorado, Boulder
# 	Boulder, CO 80309
# 
# 	carlosm@cs.colorado.edu
#

import os
import sys
import socket
#import pickle
import cPickle
pickle = cPickle

import copy
import signal
from types import *
import traceback
import time
import fcntl
import Utilities

class Connection:
  def __init__(self, host_name, port, application_name, callback = None):
    self.client = (socket.gethostname(), os.getpid(),
		   os.getuid(), application_name)
    #print self.client
    self.host_name = host_name
    self.port = port
    if callback == None:
      self.callback = None
    elif type(callback) is StringType:
      if callback == '':
	self.callback = None
      else:
	self.callback = callback
    elif type(callback) is TupleType:
      self.callback = callback
    elif type(callback) is FunctionType:
      self.callback = callback
    else:
      raise ValueError, "callback has to be either function, pipe or string type"
    self.mailbox = []
    self.s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    self.s.connect(host_name, port)
    self.cache = {}
    self.query_cache = {}


  # commit:
  # returns either 'ok' or None (indicates error).
  # Attempts to write each object to the database. Objects that are
  # not write locked by this client are ignored.
  # Invalidates the query cache because serialization (pickle) of objects 
  # changes attribute values.
  def commit(self, obj_list):

    # invalidate query_cache because attribute values in 
    # objects are changed to db_id refs in pickle.dumps(...)
    self.query_cache = {}

    # send and receive
    data = pickle.dumps((self.client, 'commit', obj_list))
    Utilities.SEND(self.s, data)
    data = Utilities.RECV(self.s, 20000)
    return pickle.loads(data)


  # get:
  # returns a list of objects which refer to other objects by db_id refs.
  # r_w specifies read_only ('r') or read and write ('rw'). In the second
  # case get tries to lock all matching objects. On failure to do so get returns
  # None (no blocking call). The scope can be either a string specifying 
  # class name or a list of db_id references. In the first case a class name
  # matches all objects of the corresponding class and its subclasses. 
  # proplist is a list of 3-tuple: (attr_name, cmp, expression). These tuples
  # specify properties which restrict the scope. 
  # cmp in {'==', '!=', 'in', 'not in', 'has', 'has not', 'some of', 'none of',
  #	    'all of', 'not all of'}
  # updates the cache 
  def get(self, r_w, scope, proplist):

    if type(scope) is ListType and scope == []:
      return []

    # convert all object references in proplist into db_id references
    if proplist == []:
      dbproplist = []
    else:
      def obj2db(value):
	if type(value) is InstanceType:
	  if hasattr(value, 'db_id'):
	    return ('__db', value.db_id)
	  else:
	    return ('__db', 0) # make sure new objects do not match
	else:
	  return value

      (dbproplist, visited) = Utilities.complex_map(obj2db, proplist)

    # send query to database
    data = pickle.dumps((self.client, 'load', r_w, scope, dbproplist))
    Utilities.SEND(self.s, data)

    # receive answer from database
    data = Utilities.RECV(self.s, 20000)
    result = pickle.loads(data)

    # evaluate answer
    if type(result) is TupleType:
      return None
    else:
      # update cache
      self.register_objs(result)
      return result

  # getexpr:
  # returns an expression isomorphic to expression. All db_id_refs are
  # replaced by objects. All invalidated objects are replaced with valid
  # ones. Loads all needed objects without context
  def getexpr(self, expression):

    # find out whether expression is in query_cache
    if self.query_cache.has_key(id(expression)):
      return self.query_cache[id(expression)]

    # replace all object references with db_id references
    (expression, visited) =\
      Utilities.complex_map(Utilities.obj2db, expression, 'no_instance')

    # collect all db_id references from visited list
    def is_db_id(value):
      return type(value) is TupleType and value != () and value[0] == '__db'
    db_id_refs = filter(is_db_id, visited)
    #print 'collected:', db_id_refs

    # if it is empty then there is nothing left to do
    if db_id_refs == []:
      # update query_cache
      self.query_cache[id(expression)] = expression
      return expression

    # find out which objects are not in the cache yet
    new_db_id_refs = []
    for (x, db_id) in db_id_refs:
      if not self.cache.has_key(db_id):
	new_db_id_refs.append((x, db_id))

    #print 'still need to get:', new_db_id_refs

    if new_db_id_refs != []:
      # get objects from the database and update cache
      new_obj_list = self.get('r', new_db_id_refs, [])

    # define db/obj mapping within the context of the updated cache
    def db2obj(value, context = self.cache):
      if type(value) is TupleType and value != () and \
         value[0] == '__db' and context.has_key(value[1]):
        return context[value[1]]
      else:
        return value

    # save and return mapped expression
    (ret, visited) = Utilities.complex_map(db2obj, expression, 'no_instance')
    self.query_cache[id(expression)] = ret
    return ret

  # register:
  # returns a notification request id
  # requests notification for all objects that match scope and proplist
  # (see get)
  # Installs child process which is receiving notifications from the server
  # and interupts parent process. Also installs interrupt handler of parent
  # process
  def register(self, scope, proplist):

    if type(scope) is ListType and scope == []:
      return 0

    if proplist == []:
      dbproplist = []
    else:
      def obj2db(value):
	if type(value) is InstanceType:
	  if hasattr(value, 'db_id'):
	    return ('__db', value.db_id)
	  else:
	    return ('__db', 0) # make sure new objects do not match
	else:
	  return value

      # maps all object references of proplist to db_ids
      (dbproplist, visited) = Utilities.complex_map(obj2db, proplist)

    if not hasattr(self, 'child'):
      # create child for receiving notifications:
      # create pipes to and from the child
      # if callback is pipe then use that for pipe from child
      (from_parent, self.to_child) = os.pipe()
      if type(self.callback) is TupleType:
	(self.from_child, to_parent) = self.callback
      else:
	(self.from_child, to_parent) = os.pipe()

      # store parent pid in variable so it is accessible to child
      parent_pid = os.getpid()

      # create child
      self.child = os.fork()
      if self.child == 0:
	#print 'Child:', os.getpid()

	# close parent's pipe ends
	os.close(self.to_child)
	os.close(self.from_child)

	# create own connection to server
        conn = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        conn.connect(self.host_name, self.port)

	# make sure that SIGUSR1 signals are ignored by child
	# unless callback is a pipe: no interrupt handling 
	if type(self.callback) is not TupleType:
	  signal.signal(signal.SIGUSR1, signal.SIG_IGN)

	# listen on pipe from_parent and server connection
	import select
	while 1:
	  (ready_conn_list, x, y) = select.select([from_parent, conn], [], [])
	  for ready_conn in ready_conn_list:

	    if ready_conn is from_parent:
	      # read data from_parent
	      #print 'Child: read from_parent'
	      data = Utilities.READ(from_parent, 20000)

	      if len(data) == 0:
		# connection from_parent died: exit child
		print 'Child: lost connection from_parent: exiting now'
		sys.exit(1)

	      # relay data to server
	      #print 'Child: send to connection'
	      Utilities.SEND(conn, data)

	      # get answer from server
	      #print 'Child: recv from connection'
	      data = Utilities.RECV(conn, 20000)

	      if len(data) == 0:
		# connection to server died: exit child
		print 'Child: lost connection to server: exiting now'
		sys.exit(2)

	      # relay answer to parent
	      #print 'Child: write to_parent'
	      Utilities.WRITE(to_parent, data)

	    elif ready_conn is conn:
	      #print 'Child: recv from connection (notification?)'
	      data = Utilities.RECV(conn, 20000)

	      if len(data) == 0:
		# connection to server died: exit child
		print 'Child: lost connection to server: exiting now'
		sys.exit(3)

	      # notification: send signal SIGUSR1 to parent
	      # and write notification data on pipe to_parent 
	      # NOTE: sequence is important since write blocks if
	      # message length exceeds block size and parent does not
	      # read on pipe (i.e. parent needs to be notified before writing
	      # to pipe)
	      # If callback is pipe: don't use interrupts
	      if type(self.callback) is not TupleType:
		#print 'Child: send signal to parent'
		os.kill(parent_pid, signal.SIGUSR1)

	      #print 'Child: write to_parent:', pickle.loads(data)
	      Utilities.WRITE(to_parent, data)

	# make sure child exits if loop breaks
	print 'left while loop of child %d !!!' % os.getpid()
	sys.exit(1)

      elif self.child > 0:
	# parent process
	# reconstruct client name of child
	self.child_client = \
	  (self.client[0], self.child, self.client[2], self.client[3])

	# close child's ends of pipes 
	os.close(from_parent)
	os.close(to_parent)

	# if callback is '__C_handler' then the signal handler is
	# entirely defined by the user
	# if callback is pipe then no interrupt handling at all
	if self.callback != '__C_handler' and \
	   type(self.callback) is not TupleType:
	  # define notification handler
	  def notification_handler(sig_nr, frame_obj, conn = self):
	    # read from_child pipe and 
	    # which is a map (request_id -> list of matched objects)
	    # notification format: (request_id, matched_obj_list)
	    # return notification
	    data = Utilities.READ(conn.from_child, 20000)

	    if len(data) == 0:
	      print 'Parent(notification_handler): lost connection from_child'
	      return None
	    notification = pickle.loads(data)
	    #print 'Parent: notification from child = ', notification
	    conn.mailbox.append(notification)

	    if type(conn.callback) is StringType and conn.callback != '':
	      #print 'notification_handler: apply', conn.callback
	      apply(frame_obj.f_locals[conn.callback],())
	    elif conn.callback != None:
	      conn.callback()

	  # declare notification handler as signal handler for SIGUSR1 
	  signal.signal(signal.SIGUSR1, notification_handler)

    # write register order to child (use child_client as client name!)
    #print 'Parent: write register order to_child'
    Utilities.WRITE(self.to_child, 
	     pickle.dumps((self.child_client, 'register', scope, dbproplist)))

    # receive answer and return it (if error then return None)
    #print 'Parent: read from_child'
    answer = pickle.loads(Utilities.READ(self.from_child, 20000))

    if type(answer) is IntType:
      #print 'Parent: register call returns', answer
      return answer
    else:
      #print 'Parent: read from child:', answer, 'return None'
      return None


  # unregister:
  # returns 'ok' for success or None for failure
  # removes a notificatoin request specified by request_id from the server
  def unregister(self, request_id):
    if hasattr(self, 'child'):
      Utilities.WRITE(self.to_child, 
	       pickle.dumps((self.child_client, 'unregister', request_id)))
      answer = pickle.loads(Utilities.READ(self.from_child, 20000))
      if type(answer) is StringType:
	return answer
      else:
	#print 'Client: unregister call returns', answer
	return None
	

  # get_raw_notification:
  # returns None or one notification which is a 3-tuple: 
  # (request_id, matched_objs, client). request_id is the corresponding 
  # notification request id, matched_objs is a list of objects which match
  # the notification request and client is the client who issued the commit
  # that triggered the notification.
  def get_raw_notification(self):

    if not hasattr(self, 'child'):
      # no register was issued - so there won't be any notifications
      return None

    if type(self.callback) is StringType and \
       self.callback == '__C_handler':

      # get_notification is called from an external interrupt handler
      # (no internal interrupt handler is installed)
      # get notification directly from_child

      # set reading from_child to non-blocking
      #fcntl.fcntl(self.from_child, 4, 4)

      try:
	data = Utilities.READ(self.from_child, 20000)
      except os.error, err_val:
	if err_val[0] == 35:
	  # error is EWOULDBLOCK; return None
	  return None
	else:
	  # something else; re-raise error
	  raise sys.exc_type, err_val

      # set reading from_child blocking again
      #fcntl.fcntl(self.from_child, 4, 16)

      if len(data) == 0:

	# lost connection from_child
	print 'Parent(get_notification): lost connection from_child'
	return None

      else:

	# unserialize data
	notification = pickle.loads(data)

	#print 'Parent: notification from child = ', notification
	return notification

    else:

      # an internal interrupt handler is installed which stores all
      # notifications in self.mailbox
      # get oldest notification from self.mailbox (FCFS queue)
      if self.mailbox == []: 
	return None
      else:
	notification = self.mailbox[0]
	self.mailbox.remove(notification)

	return notification


  # get_notification:
  # like get_raw_notification but updates the cache
  def get_notification(self):

    notification = self.get_raw_notification()

    if notification != None:
      self.register_objs(notification[1])

    return notification


  # lock:
  # returns a 2-tuple (error type, error msg) or a list of objects
  # attempts to lock all objects in obj_list
  # if client cannot lock all objects then lock returns a 2-tuple (see above)
  # and releases all locks (to prevent deadlock)
  # if client is successful, then lock returns a list of locked objects
  # Updates the cache with returned objects
  def lock(self, obj_list):

    # convert all objects into db_id numbers
    db_id_list = []
    for obj in obj_list:
      if hasattr(obj, 'db_id'):
	db_id_list.append(obj.db_id)

    if db_id_list != []:
      data = pickle.dumps((self.client, 'lock', db_id_list))
      Utilities.SEND(self.s, data)

      data = Utilities.RECV(self.s, 20000)
      result = pickle.loads(data)

      # update cache if successful
      if type(result) is ListType:
	self.register_objs(result)

      return result
    else:
      return []


  # register_objs:
  # returns a list of db_id numbers which correspond to obj_list 
  # Gets new db_id numbers from the server and assigns them to new objects
  # in obj_list
  # Installs attribute read and write functions and seals all object referencing
  # attributes (by converting '<attr name>' to ' <attr name>')
  # updates cache
  def register_objs(self, obj_list):

    # define attribute read and write functions
    def rd(obj, name, conn = self):
      return conn.getexpr(obj.__dict__[' '+name])

    def wr(obj, name, value):
      obj.__dict__[' '+name] = value
      
    ret = []
    new_obj_count = 0
    for obj in obj_list:
      if hasattr(obj,'db_id'):
	ret.append(obj.db_id)
      else:
	ret.append(0)
	new_obj_count = new_obj_count + 1

    # invalidate query_cache only if the cache is going to be updated with old
    # objects
    if new_obj_count < len(ret):
      self.query_cache = {}

    # get new db_id numbers from server 
    if new_obj_count > 0:
      data = pickle.dumps((self.client, 'get_new_ids', new_obj_count))
      Utilities.SEND(self.s, data)
      data = Utilities.RECV(self.s, 20000)
      new_db_ids = pickle.loads(data)
      if type(new_db_ids) is not ListType:
	return None
    else:
      new_db_ids = []

    # assign new db_ids, install attribute access, and update cache
    j = 0
    for i in range(len(ret)):
      obj = obj_list[i]

      if ret[i] == 0:
	# object is new: assign new db_id  
	ret[i] = new_db_ids[j]
	j = j+1
	obj = obj_list[i]
	obj.db_id = ret[i]

      # seal all attr names from object referencing attrs 
      for attr_name in obj._refs:
	if obj.__dict__.has_key(' '+attr_name) and \
	   obj.__dict__.has_key(attr_name):
	  del obj.__dict__[attr_name]
	elif obj.__dict__.has_key(attr_name):
	  obj.__dict__[' '+attr_name] = obj.__dict__[attr_name]
	  del obj.__dict__[attr_name]

      # install rd and wr functions
      obj._read = rd
      obj._write = wr

      # update cache
      self.cache[obj.db_id] = obj

    return ret
      

  def close(self):
    # close pipes and kill child if it exists
    if hasattr(self, 'child'):
      os.close(self.from_child)
      os.close(self.to_child)
      os.kill(self.child, signal.SIGKILL)
#      os.wait()
      del self.child

    self.s.close()

