/* oramodule.c
 *
 * Copyright 1994, 1995 Thomas J. Culliton
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  
 * Thomas J. Culliton makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.  By use of this software the user agrees to 
 * indemnify and hold harmless Thomas J. Culliton from any  claims or
 * liability for loss arising out of such use.
 */
/*
 * Modified by Alan Geller 9/95 to support 'OUT' parameters in 
 * PL/SQL procedures, using the (formerly stubbed out) execpl methd.
 */

#ifdef __OS2__
// #include <myoci.h>
#include <string.h>
#include <ocidfn.h>
#include <ociapr.h>
#define strcasecmp stricmp
#endif

#include "allobjects.h"
#include "modsupport.h"		/* For getargs() etc. */

#include <malloc.h>
#include <errno.h>

#include "orastuff.h"

#ifdef __OS2__
#include <oratypes.h>    /* Oracle types */
#else
#include "rdbms/demo/oratypes.h"    /* Oracle types */
#endif


char bogus_hack_for_osntab()	/* never called! */
{				/* this is to force load osntab.o */
	extern volatile char *osntab;	/* liar, liar, pants on fire! */
	char x = osntab[0];	/* now use it in a way that can't be */
	return x;		/* optimized away */
}


static object *OracleError;
static long maxlong = MAX_LONG_SIZE;

/* ----------------------------------------------------- */
/* Declarations for objects of type oraconnection */

typedef struct {
	OB_HEAD
	/* XXXX Add your own stuff here */
	LDA *lda;
	HDA *hda;
	int connected;
} oracoobject;

staticforward typeobject Oracotype;

#define is_oracoobject(v)		((v)->ob_type == &Oracotype)

/* ---------------------------------------------------------------- */
/* Declarations for objects of type oracursor */

typedef struct CursorObject {
	OB_HEAD
	CDA *cda;
	ColBufs *cols;
	int open;
	oracoobject *conn;
	struct CursorObject *next;
} oracuobject;

staticforward typeobject Oracutype;

#define is_oracuobject(v)		((v)->ob_type == &Oracutype)

static oracuobject *cursor_list = 0;	/* list of all cursor objects */

/* UTILITY FUNCTIONS */
/* ---------------------------------------------------------------- */


static char *zalloc(size_t sz)		/* allocate and zero */
{
	char *space;

	if ((space = malloc(sz)) != NULL)
		memset(space, 0, sz);

	return space;
}

static int
ora_err_check(lda)
	LDA *lda;
{
	char buf[ORA_MSG_SIZE];
	if (lda->rc != 0)
	{
		oerhms(lda, lda->rc, buf, ORA_MSG_SIZE);
		err_setstr(OracleError, buf);
		return 1;
	}
	return 0;
}

static char *
addVariableToProc(plsql, len, max, name, namelen)
char *plsql;
int *len;
int *max;
char *name;
int namelen;
{
  int varLen;

  varLen = namelen + namelen + 6;
  if (varLen + *len >= *max) {
    *max += varLen + 64;
    plsql = realloc(plsql, *max);
  }
  
  /* Build "arg => :arg" */
  memcpy(plsql + *len, name, namelen);
  *len += namelen;
  plsql[(*len)++] = ' ';
  plsql[(*len)++] = '=';
  plsql[(*len)++] = '>';
  plsql[(*len)++] = ' ';
  plsql[(*len)++] = ':';
  memcpy(plsql + *len, name, namelen);
  *len += namelen;
  return plsql;   /*@SB*/
}

static void
free_cols(self)
	oracuobject *self;
{
	ColBufs  *col = self->cols;
	ColBufs  *next;

	while (col != NULL)
	{
		next = col->next_buf;
		if (col->col_data != NULL)
			free(col->col_data);
		free(col);
		col = next;
	}
	self->cols = NULL;
}

/* Returns 0 for non-strings */
static int
getStringType(type)
int type;
{
  switch (type) {
  case 1:	/* char *//* VERSION 7 varchar2 */
  case 11:	/* rowid */
  case 12:	/* date */
  case 96:	/* VERSION 7 char */
    return EXT_VARCHAR_TYPE;
    break;
  case 23:	/* raw (bytes) */
    return EXT_VARRAW_TYPE;
    break;
  case 8:	/* long (char) */
    return EXT_LONG_VARCHAR_TYPE;
    break;
  case 24:	/* long raw (bytes) */
    return EXT_LONG_VARRAW_TYPE;
    break;
  default:
    break;
  }
  return 0;
}

static char *
setStringValue(obj, len, max, type)
object *obj;
sword *len;
int max;
sword type;
{
  char *param;
  object *rep;
  char *data;
  int size;
  int headerSize;

  if ((type == EXT_LONG_VARCHAR_TYPE) || (type == EXT_LONG_VARRAW_TYPE)) {
    headerSize = 4;
    if (max <= 0)
      max = maxlong;
  } else {
    headerSize = 2;
    if (max <= 0)
      max = MAX_CHAR_SIZE;
  }

  *len = max + headerSize;
  param = malloc(*len);

  if ((obj != NULL) && (obj != None)) {
    rep = PyObject_Str(obj);
    data = GETSTRINGVALUE((stringobject *)rep);
    size = getstringsize(rep);
    if (size > max)
      size = max;
    memcpy(param + headerSize, data, size);
    DECREF(rep);
  } else
    size = 0;

  if ((type == EXT_LONG_VARCHAR_TYPE) || (type == EXT_LONG_VARRAW_TYPE))
    *(long *)param = size;
  else
    *(short *)param = size;
  return param;
}

/* Returns -1 on error, 0 otherwise */
static int
setFloatValue(param, obj)
double *param;
object *obj;
{
  if ((obj == NULL) || (obj == None))
    *param = 0.0;
  else {
    if (is_floatobject(obj))
      *param = GETFLOATVALUE((floatobject *)obj);
    else if (is_intobject(obj))
      *param = GETINTVALUE((intobject*)obj);
    else
      return -1;
  }

  return 0;
}

/* Returns -1 on error, 0 otherwise */
static int
setIntValue(param, obj)
     int *param;
     object *obj;
{
  if ((obj == NULL) || (obj == None))
    *param = 0;
  else {
    if (is_floatobject(obj))
      *param = GETFLOATVALUE((floatobject *)obj);
    else if (is_intobject(obj))
      *param = GETINTVALUE((intobject*)obj);
    else
      return -1;
  }

  return 0;
}


static size_t bufsize(ColBufs *col);
static int buftype(ColBufs *col);
static object *str_to_number(char *str);

static int
get_cols(self)
     oracuobject *self;
{
  int pos, i;
  ColBufs *col;
  ColBufs **link = &self->cols;
  
  free_cols(self);	/* just in case... */
  
  for (pos = 1; ; ++pos)
    {
      *link = col = (ColBufs *)zalloc(sizeof(ColBufs));
      if (col == NULL)
	{
	  free_cols(self);
	  err_setstr(MemoryError, "no space for description");
	  return 0;
	}
      col->dbname_len = sizeof(col->dbname);
      
      /* This is dumb, it doesn't work right under Oracle V6 */
      /* because it's a macro which drops, prec, scale and nullok. */
      /* Whats even dumber, is that theres apparently no way to */
      /* get this information using the V6 OCI!!! */
      odescr(self->cda, pos, &col->dbsize, &col->dbtype, col->dbname,
	     &col->dbname_len, &col->disp_size, &col->prec, &col->scale,
	     &col->nullok);

      if (self->cda->rc == NO_MORE_COLUMNS)
	{
	  *link = NULL;
	  free(col);
	  break;
	}
      link = &col->next_buf;
      
      if (ora_err_check(self->cda))
	{
	  free_cols(self);
	  return 0;
	}
      
      /* trim trailing blanks from dbname */
      for (i = sizeof(col->dbname) - 1; i >= 0; --i)
	{
	  if (col->dbname[i] != ' ')
	    {
	      col->dbname[i+1] = '\0';
	      break;
	    }
	}
      
      /* allocate the data buffers */
      col->col_data = (char *)zalloc(bufsize(col));
      if (col->col_data == NULL)
	{
	  free_cols(self);
	  err_setstr(MemoryError, "no space for buffers");
	  return 0;
	}
    }
  /* Now bind the damned things to the parsed statement */
  for (col = self->cols, pos = 1; col != NULL; col = col->next_buf, ++pos)
    {
      odefin(self->cda, pos, col->col_data, bufsize(col),
	     buftype(col), 0, col->ind, NULL, 0, 0, col->rlen,
	     col->rcode);
    }
  return pos;
}

static void close_my_cursors(conn)
	oracoobject *conn;
{
	oracuobject *p;

	for (p = cursor_list; p != 0; p = p->next)
		if (p->conn == conn)
		{
			if (p->open)
				oclose(p->cda);
			p->open = 0;
			p->conn = 0;		/* prevent accidents */
		}
}
	
/* ---------------------------------------------------------------- */

static object *
oraco_logoff(self, args)
	oracoobject *self;
	object *args;
{
	if (!newgetargs(args, ""))
		return NULL;

	/* close all the associated cursors, ignore any errors */
	close_my_cursors(self);

/*	if (self->connected) */
		ologof(self->lda);
	self->connected = 0;

	if (ora_err_check(self->lda))
		return NULL;

	INCREF(None);
	return None;
}

static object *
oraco_opencursor(self, args)
	oracoobject *self;
	object *args;
{
  //	static oracuobject *neworacuobject(oracoobject *conn);
	oracuobject *neworacuobject(oracoobject *conn);

	if (!self->connected)
	{
		err_setstr(OracleError, "no connection");
		return NULL;
	}
	if (!newgetargs(args, ""))
		return NULL;
	return (object *)neworacuobject(self);
}

static object *
oraco_commit(self, args)
	oracoobject *self;
	object *args;
{
	if (!self->connected)
	{
		err_setstr(OracleError, "no connection");
		return NULL;
	}
	if (!newgetargs(args, ""))
		return NULL;

	ocom(self->lda);

	if (ora_err_check(self->lda))
		return NULL;

	INCREF(None);
	return None;
}

static object *
oraco_rollback(self, args)
	oracoobject *self;
	object *args;
{
	if (!self->connected)
	{
		err_setstr(OracleError, "no connection");
		return NULL;
	}
	if (!newgetargs(args, ""))
		return NULL;

	orol(self->lda);

	if (ora_err_check(self->lda))
		return NULL;

	INCREF(None);
	return None;
}

static object *
oraco_autocommit(self, args)
	oracoobject *self;
	object *args;
{
	char *setting;

	if (!self->connected)
	{
		err_setstr(OracleError, "no connection");
		return NULL;
	}
	if (!newgetargs(args, "s", &setting))
		return NULL;
	if (strcasecmp(setting, "on") == 0)
		ocon(self->lda);
	else if (strcasecmp(setting, "off") == 0)
		ocof(self->lda);
	else
	{
		err_setstr(ValueError, "setting must be \"on\" or \"off\"");
		return NULL;
	}

	if (ora_err_check(self->lda))
		return NULL;

	INCREF(None);
	return None;
}


static struct methodlist oraco_methods[] = {
	{"logoff",	(PyCFunction)oraco_logoff,	1},
	{"opencursor",	(PyCFunction)oraco_opencursor,	1},
	{"commit",	(PyCFunction)oraco_commit,	1},
	{"rollback",	(PyCFunction)oraco_rollback,	1},
	{"autocommit",	(PyCFunction)oraco_autocommit,	1},

	{NULL,		NULL}		/* sentinel */
};

/* ---------- */

static oracoobject *
neworacoobject(char *conn_str)
{
	oracoobject *self;
	//@SB LDA lda_;  //@SB
	//@SB ub1 hda_[512]; //@SB

	self = NEWOBJ(oracoobject, &Oracotype);
	if (self == NULL)
		return NULL;

	self->lda = (LDA *)zalloc(sizeof(LDA));
	self->hda = (HDA *)zalloc(sizeof(HDA));
	if (self->lda == NULL || self->hda == NULL)
	{
		DECREF(self);
		err_setstr(MemoryError, "no space for connection");
		return NULL;
	}
	/* log on to the database and raise OracleError if we can't */
#ifdef __OS2__
	if (orlon(self->lda, *(self->hda), conn_str, -1, (char *)0, -1, 0) != 0)
#else
	if (orlon(self->lda, self->hda, conn_str, -1, (char *)-1, -1, 0) != 0)
#endif
	{
		DECREF(self);
		err_setstr(OracleError, "oracle logon failed");
/*		ora_err_check(self->lda); */
		return NULL;
	}
	self->connected = 1;

	return self;
}

static void
oraco_dealloc(self)
	oracoobject *self;
{
	/* close all the associated cursors, ignore any errors */
	close_my_cursors(self);

	if (self->connected)	/* force a log-off if it hasn't been done yet */
		ologof(self->lda);

	ora_err_check(self->lda);

	free(self->lda);
	free(self->hda);
	DEL(self);
}

static object *
oraco_getattr(self, name)
	oracoobject *self;
	char *name;
{
	if (strcmp(name, "valid") == 0)
		return newintobject((long)self->connected);
	else if (strcmp(name, "code") == 0)
		return newintobject((long)self->lda->rc);
	else
		return findmethod(oraco_methods, (object *)self, name);
}

static typeobject Oracotype = {
	OB_HEAD_INIT(&Typetype)
	0,				/*ob_size*/
	"oraconnection",			/*tp_name*/
	sizeof(oracoobject),		/*tp_basicsize*/
	0,				/*tp_itemsize*/
	/* methods */
	(destructor)oraco_dealloc,	/*tp_dealloc*/
	(printfunc)0,		/*tp_print*/
	(getattrfunc)oraco_getattr,	/*tp_getattr*/
	(setattrfunc)0,	/*tp_setattr*/
	(cmpfunc)0,		/*tp_compare*/
	(reprfunc)0,		/*tp_repr*/
	0,			/*tp_as_number*/
	0,		/*tp_as_sequence*/
	0,		/*tp_as_mapping*/
	(hashfunc)0,		/*tp_hash*/
};

/* End of code for oraconnection objects */
/* -------------------------------------------------------- */

static object *
oracu_close(self, args)
	oracuobject *self;
	object *args;
{
	if (!newgetargs(args, ""))
		return NULL;
/*	if (self->open) */
		oclose(self->cda);
	self->open = 0;
	self->conn = 0;		/* prevent accidents */

	if (ora_err_check(self->cda))
		return NULL;

	INCREF(None);
	return None;
}

static object *
oracu_execsql(self, args)
	oracuobject *self;
	object *args;
{
	char *sql_stmt;
	object *vars;
	object *(*getitem) PROTO((object *, int));
	int len;
	int i;

	if (!self->open)
	{
		err_setstr(OracleError, "not open");
		return NULL;
	}
	if (newgetargs(args, "sO", &sql_stmt, &vars))
	{
		if (is_tupleobject(vars))
		{
			len = gettuplesize(vars);
			getitem = gettupleitem;
		}
		else if (is_listobject(vars))
		{
			len = getlistsize(vars);
			getitem = getlistitem;
		}
		else
		{
			err_setstr(TypeError, "2nd argument must be a tuple or list");
			return NULL;
		}
	}
	else if (newgetargs(args, "s", &sql_stmt))
	{
		vars = 0;
		len = 0;
	}
	else
		return NULL;

	oparse(self->cda, sql_stmt, -1L, DEF_FLAG, LNG_FLAG);

	if (ora_err_check(self->cda))
		return NULL;

	/* bind the source variables */
	for (i = 0; i < len; ++i)
	{
		object *obj;
		void *data;
		int type;
		int size;

		obj = getitem(vars, i);
		if (is_stringobject(obj))
		{
			/* BEWARE OF FUNKY STRINGS WHICH CONTAIN NULLS! */

			data = GETSTRINGVALUE((stringobject *) obj);
			type = EXT_CHAR_TYPE;
			size = getstringsize(obj);
			if (size > MAX_CHAR_SIZE)
				type = EXT_LONG_TYPE;
		}
		else if (is_intobject(obj))
		{
			data = &GETINTVALUE((intobject*) obj);
			type = EXT_INT_TYPE;
			size = sizeof(int);
		}
		else if (is_floatobject(obj))
		{
			data = &GETFLOATVALUE((floatobject *) obj);
			type = EXT_FLOAT_TYPE;
			size = sizeof(double);
		}
		else if (is_longobject(obj))
		{
			obj = reprobject(obj);	/* EEK!  A leak! */
			if (!obj)
				return NULL;
			/* WE NEED TO ADD THIS NEW OBJECT TO A KILL LIST... */

			data = GETSTRINGVALUE((stringobject *) obj);
			type = EXT_CHAR_TYPE;
			size = getstringsize(obj);
		}
		else
		{
			err_setstr(TypeError, "can only bind string, or numeric values");
			return NULL;
		}

		obndrn(self->cda, i+1, data, size, type, -1, 0, 0, -1, -1);

		if (ora_err_check(self->cda))
			return NULL;
	}

	/* parse the description and bind buffers to it */
	if (get_cols(self) == 0)
		return NULL;

	/* execute it */
	oexec(self->cda);

	if (ora_err_check(self->cda))
		return NULL;

	INCREF(None);
	return None;
}

static object *
oracu_fetch(self, args)
	oracuobject *self;
	object *args;
{
	object *tuple;
	object *obj;
	ColBufs *col;
	int cnt;

	if (!self->open)
	{
		err_setstr(OracleError, "not open");
		return NULL;
	}
	if (!newgetargs(args, ""))
		return NULL;

	ofen(self->cda, 1);	/* fetch one for now */

	if (self->cda->rc == NO_DATA_FOUND)
	{
		INCREF(None);
		return None;
	}
	if (ora_err_check(self->cda))
		return NULL;

	for (cnt = 0, col = self->cols; col != NULL; ++cnt, col = col->next_buf)
		;			/* find out how long the list is */

	if ((tuple = newtupleobject(cnt)) == NULL)
		return NULL;

	for (cnt = 0, col = self->cols; col != NULL; ++cnt, col = col->next_buf)
	{
		if (col->rcode[0] == NULL_VALUE && col->rlen[0] == 0)
		{
			INCREF(None);
			obj = None;
		}
		else switch (col->dbtype)
		{
		case 2:		/* number */
#ifdef VERSION6
			obj = str_to_number(col->col_data);
#else
			if (col->scale == 0)
			{
				if (col->prec == 0 || col->prec > 9)
				{
					/* do we need to null terminate data? */
					obj = long_scan(col->col_data, 10);
				}
				else
					obj = newintobject(*(int *)col->col_data);
			}
			else
				obj = newfloatobject(*(double *)col->col_data);
#endif
			break;

		case 1:		/* char *//* VERSION 7 varchar2 */
		case 8:		/* long (char) */
		case 11:	/* rowid */
		case 12:	/* date */
		case 23:	/* raw (bytes) */
		case 24:	/* long raw (bytes) */
		case 96:	/* VERSION 7 char */
			/* do we need to null terminate data? */
			obj = newsizedstringobject(col->col_data, col->rlen[0]);
			break;

		default:
/*			INCREF(None); */
/*			obj = None;   */
			obj = newstringobject("<Unkown type>");
			break;
		}
		if (obj == NULL)
		{
			DECREF(tuple);
			return NULL;
		}
		settupleitem(tuple, cnt, obj);
	}
	return tuple;
}

static object *
oracu_columns(self, args)
	oracuobject *self;
	object *args;
{
	int cnt;
	ColBufs *col;
	object *tuple;
	object *obj;

	if (!self->open)
	{
		err_setstr(OracleError, "not open");
		return NULL;
	}

	if (!newgetargs(args, ""))
		return NULL;
	for (cnt = 0, col = self->cols; col != NULL; ++cnt, col = col->next_buf)
		;			/* find out how long the list is */

	if ((tuple = newtupleobject(cnt)) == NULL)
		return NULL;

	for (cnt = 0, col = self->cols; col != NULL; ++cnt, col = col->next_buf)
	{
		obj = mkvalue("(siiiiii)", col->dbname, col->dbtype,
		  col->disp_size, col->dbsize, col->prec, col->scale,
		  col->nullok);
		if (obj == NULL)
		{
			DECREF(tuple);
			return NULL;
		}
		settupleitem(tuple, cnt, obj);
	}
	return tuple;
}

static object *
oracu_cancel(self, args)
	oracuobject *self;
	object *args;
{
	if (!self->open)
	{
		err_setstr(OracleError, "not open");
		return NULL;
	}
	if (!newgetargs(args, ""))
		return NULL;
	ocan(self->cda);

	if (ora_err_check(self->cda))
		return NULL;

	INCREF(None);
	return None;
}

/* Maximum number of parameters for a stored procedure */
#define MAXPARAMS    16

/*
 * execpl(proc, varDict) will execute the PL/SQL procedure named
 * proc. varDict must be a dictionary that is used to map parameters
 * to values, both IN and OUT. Mapping is done by name.
 * Note that, unlike execsql(), this method completely executes the
 * procedure. There is no need to call fetch() afterward.
 *
 * KNOWN LIMITATIONS:
 * - This implementation does not handle overloaded procedures.
 * - This implementation does not handle record parameters.
 */
static object *
oracu_execpl(self, args)
     oracuobject *self;
     object *args;
{
  char *procName;
  object *varDict;
  int i, j, k;
  int first;
  /* Stored procedure binding data: */
  ub2 ovrld[MAXPARAMS];
  ub2 pos[MAXPARAMS];
  ub2 level[MAXPARAMS];
#ifdef __OS2__
  text argname[MAXPARAMS][30]; /* 30 bytes max per name */
#else
  text argname[MAXPARAMS * 30]; /* 30 bytes max per name */
#endif
  ub2 arnlen[MAXPARAMS];
  ub2 datatype[MAXPARAMS];
  ub1 defval[MAXPARAMS];
  ub1 mode[MAXPARAMS];
  ub4 length[MAXPARAMS];
  sb2 prec[MAXPARAMS];
  sb2 scale[MAXPARAMS];
  ub1 radix[MAXPARAMS];
  ub4 spare[MAXPARAMS];
  ub4 arraySize = MAXPARAMS;
  char *paramName[MAXPARAMS];
  void *paramValue[MAXPARAMS];
  sb2 nullFlag[MAXPARAMS];
  sword paramLen[MAXPARAMS];
  ub2 paramType[MAXPARAMS];
  char *plsql;
  int plsqllen;
  int plsqlmax;
  object *obj;
  void *data;
  int type;
  int size;
  int varLen;
  char *name;
  int argOffset;

  if (!self->open) {
    err_setstr(OracleError, "cursor not open");
    return NULL;
  }
  
  if (newgetargs(args, "sO", &procName, &varDict)) {
    if (!is_dictobject(varDict)) {
      err_setstr(TypeError, "2nd argument must be a dictionary");
      return NULL;
    }
  } else if (newgetargs(args, "s", &procName)) {
    varDict = 0;
  } else
    return NULL;

  i = odessp(self->conn->lda, procName, -1, (ub1 *)0, 0, (ub1 *)0, 0,
#ifdef __OS2__
	     ovrld, pos, level, (text**)argname, arnlen, datatype, defval, mode,
#else
	     ovrld, pos, level, argname, arnlen, datatype, defval, mode,
#endif
	     length, prec, scale, radix, spare, &arraySize);
  if (ora_err_check(self->cda))
    return NULL;
  
  if (arraySize > MAXPARAMS) {
    err_setstr(OracleError, "Too many procedure parameters");
    return NULL;
  }

  plsqlmax = 128 + strlen(procName);
  plsql = malloc(plsqlmax);
  sprintf(plsql, "BEGIN\n                 %s", procName);
  plsqllen = strlen(plsql);
  for (i = 22; i < plsqllen; i++)
    plsql[i] = toupper(plsql[i]);
  
  first = 1;
  argOffset = 0;

  /* Format the PL/SQL block */
  for (i = 0; i < arraySize; ++i) {
    if (datatype[i] == 0) {
      paramName[i] = malloc(1);
      /* A zero datatype means skip this entry */
      continue;
    }

    if (ovrld[i] != 0) {
      err_setstr(OracleError, "Can't handle overloaded procedures");
      free(plsql);
      for (j = 0; j < i; j++)
	free(paramValue[j]);
      return NULL;
    }

    if (level[i] != 0) {
      err_setstr(OracleError, "Can't handle record parameters");
      free(plsql);
      for (j = 0; j < i; j++)
	free(paramValue[j]);
      return NULL;
    }

    if (pos[i] != 0) {
      /* Insert opening '(' */
      if (first)
	plsql[plsqllen++] = '(';
      /* Insert ',' between parameters */
      if (!first)
	plsql[plsqllen++] = ',';
      plsql = addVariableToProc(plsql, &plsqllen, &plsqlmax,
#ifdef __OS2__
				argname[i], arnlen[i]);
#else
				argname + argOffset, arnlen[i]);
#endif
      first = 0;
      name = malloc(arnlen[i] + 2);
      name[0] = ':';
#ifdef __OS2__
      memcpy(name + 1, argname[i], arnlen[i]);
#else
      memcpy(name + 1, argname + argOffset, arnlen[i]);
#endif
      name[arnlen[i] + 1] = '\0';
    } else {  /* Handle return value */
      memcpy(plsql + 6, ":return_value :=", 16);
      name = malloc(14);
      strcpy(name, ":return_value");
    }
    argOffset += 30;
    paramName[i] = name;
  }

  /* Format the end of the block; add an ) only if there are params */
  if (plsqllen + 10 >= plsqlmax) {
    plsqlmax = plsqllen + 10;
    plsql = realloc(plsql, plsqlmax);
  }
  if (first)
    strcpy(plsql + plsqllen, ";\nEND;\n");
  else
    strcpy(plsql + plsqllen, ");\nEND;\n");

  oparse(self->cda, plsql, -1L, DEF_FLAG, LNG_FLAG);
  if (ora_err_check(self->cda)) {
    free(plsql);
    for (i = 0; i < arraySize; i++)
      free(paramName[i]);
    return NULL;
  }

  /* Bind the variables */
  for (i = 0; i < arraySize; ++i) {
    if (datatype[i] == 0) {
      paramValue[i] = malloc(1);
      /* A zero datatype means skip this entry */
      continue;
    }

    if (varDict == 0)
      obj = NULL;
    else
      obj = dictlookup(varDict, paramName[i] + 1); /* +1 to skip ':' */

    if ((obj == NULL) || (obj == None))
      nullFlag[i] = -1;
    else
      nullFlag[i] = 0;

    paramType[i] = getStringType(datatype[i]);
    if ((paramType[i] == EXT_VARCHAR_TYPE) || 
	(paramType[i] == EXT_VARRAW_TYPE))
      paramValue[i] = setStringValue(obj, &paramLen[i], length[i],
				     paramType[i]);
    else if ((paramType[i] == EXT_LONG_VARCHAR_TYPE) || 
	     (paramType[i] == EXT_LONG_VARRAW_TYPE))
      paramValue[i] = setStringValue(obj, &paramLen[i], maxlong, paramType[i]);
    else if (datatype[i] == 2) {  /* NUMBER */
      if ((obj != NULL) && (is_stringobject(obj))) {
	paramType[i] = EXT_VARCHAR_TYPE;
	paramValue[i] = setStringValue(obj, &paramLen[i], 50,
				       paramType[i]);
      } else {
	if (scale[i] == -127) { /* Float */
	  paramLen[i] = sizeof(double);
	  paramType[i] = EXT_FLOAT_TYPE;
	  paramValue[i] = malloc(paramLen[i]);
	  if (setFloatValue(paramValue[i], obj)) {
	    err_setstr(OracleError, "Type mismatch (number required)");
	    free(plsql);
	    for (j = 0; j <= i; j++)
	      free(paramValue[j]);
	    for (k = 0; k < arraySize; k++)
	      free(paramName[k]);
	    return NULL;
	  }
	} else {
	  paramLen[i] = sizeof(int);
	  paramType[i] = EXT_INT_TYPE;
	  paramValue[i] = malloc(paramLen[i]);
	  if (setIntValue(paramValue[i], obj)) {
	    err_setstr(OracleError, "Type mismatch (number required)");
	    free(plsql);
	    for (j = 0; j <= i; j++)
	      free(paramValue[j]);
	    for (k = 0; k < arraySize; k++)
	      free(paramName[k]);
	    return NULL;
	  }
	}
      }
    }

    obndrv(self->cda, paramName[i], -1, paramValue[i], paramLen[i],
	   paramType[i], -1, &nullFlag[i], (char *)0, -1, -1);
    if (ora_err_check(self->cda)) {
      free(plsql);
      for (j = 0; j <= i; j++)
	free(paramValue[j]);
      for (k = 0; k < arraySize; k++)
	free(paramName[k]);
      return NULL;
    }
  }
  
  /* execute it */
  oexec(self->cda);
  if (ora_err_check(self->cda)) {
    for (k = 0; k < arraySize; k++) {
      free(paramName[k]);
      free(paramValue[k]);
    }
    return NULL;
  }

  /* Copy out all of the OUT and IN/OUT parameters */
  for (i = 0; i < arraySize; i++) {
    if ((varDict != 0) && (datatype[i] != 0) && ((mode[i] == 1) ||
						 (mode[i] == 2))) {
      switch (paramType[i]) {
      case EXT_VARCHAR_TYPE:
      case EXT_VARRAW_TYPE:
	data = (char*)paramValue[i] + 2;
	size = *(short *)paramValue[i];
	obj = newsizedstringobject(data, size);
	break;
      case EXT_LONG_VARCHAR_TYPE:
      case EXT_LONG_VARRAW_TYPE:
	data = (char*)paramValue[i] + 4;
	size = *(long *)paramValue[i];
	obj = newsizedstringobject(data, size);
	break;
      case EXT_INT_TYPE:
	obj = newintobject(*(long *)paramValue[i]);
	break;
      case EXT_FLOAT_TYPE:
	obj = newfloatobject(*(double *)paramValue[i]);
	break;
      default:
	obj = newstringobject("<Unkown type>");
	break;
      }
      dictinsert(varDict, paramName[i] + 1, obj);
    }

    free(paramName[i]);
    free(paramValue[i]);
  }

  INCREF(None);
  return None;
}

static struct methodlist oracu_methods[] = {
	{"close",	(PyCFunction)oracu_close,	1},
	{"execsql",	(PyCFunction)oracu_execsql,	1},
	{"fetch",	(PyCFunction)oracu_fetch,	1},
	{"columns",	(PyCFunction)oracu_columns,	1},
	{"describe",	(PyCFunction)oracu_columns,	1},
	{"cancel",	(PyCFunction)oracu_cancel,	1},
	{"execpl",	(PyCFunction)oracu_execpl,	1},

	{NULL,		NULL}		/* sentinel */
};

/* ---------- */

static oracuobject *
neworacuobject(oracoobject *conn)
{
	oracuobject *self;

	self = NEWOBJ(oracuobject, &Oracutype);
	if (self == NULL)
		return NULL;
	self->cda = (CDA *)zalloc(sizeof(CDA));
	if (self->cda == NULL)
	{
		DECREF(self);
		err_setstr(MemoryError, "no space for cursor");
		return NULL;
	}
	self->cols = NULL;
	if (oopen(self->cda, conn->lda, NULL, -1, -1, NULL, -1) != 0)
	{
		DECREF(self);
		err_setstr(OracleError, "can not open cursor");
		return NULL;
	}
	self->open = 1;
	self->conn = conn;
	self->next = cursor_list;
	cursor_list = self;
	return self;
}

static void
oracu_dealloc(self)
	oracuobject *self;
{
	oracuobject *p, **pn;

	if (self->open)
		oclose(self->cda);

	ora_err_check(self->cda);

	for (pn = &cursor_list, p = *pn; p != 0; pn = &(p->next), p = *pn)
		if (p == self)
			*pn = p->next;

	free(self->cda);
	free_cols(self);

	DEL(self);
}

static object *
oracu_getattr(self, name)
	oracuobject *self;
	char *name;
{
	if (strcmp(name, "valid") == 0)
		return newintobject((long)self->open);
	else if (strcmp(name, "code") == 0)
		return newintobject((long)self->cda->rc);
	else if (strcmp(name, "rows") == 0)
		return newintobject((long)self->cda->rpc);
	else
		return findmethod(oracu_methods, (object *)self, name);
}

static typeobject Oracutype = {
	OB_HEAD_INIT(&Typetype)
	0,				/*ob_size*/
	"oracursor",			/*tp_name*/
	sizeof(oracuobject),		/*tp_basicsize*/
	0,				/*tp_itemsize*/
	/* methods */
	(destructor)oracu_dealloc,	/*tp_dealloc*/
	(printfunc)0,		/*tp_print*/
	(getattrfunc)oracu_getattr,	/*tp_getattr*/
	(setattrfunc)0,	/*tp_setattr*/
	(cmpfunc)0,		/*tp_compare*/
	(reprfunc)0,		/*tp_repr*/
	0,			/*tp_as_number*/
	0,		/*tp_as_sequence*/
	0,		/*tp_as_mapping*/
	(hashfunc)0,		/*tp_hash*/
};

/* End of code for oracursor objects */
/* -------------------------------------------------------- */

static object *
orapy_newconnection(self, args)
	object *self;	/* Not used */
	object *args;
{
	char *conn_str;

	if (!newgetargs(args, "s", &conn_str))
		return NULL;
	return (object *)neworacoobject(conn_str);
}

static object *
orapy_maxlong(self, args)
	object *self;	/* Not used */
	object *args;
{
	long oldmax = maxlong;

	if (!newgetargs(args, "l", &maxlong))
		return NULL;
	if (maxlong < 0)		/* just a query? */
		maxlong = oldmax;
	if (maxlong > MAX_LONG_SIZE)
		maxlong = MAX_LONG_SIZE;

	return mkvalue("l", oldmax);
}

/* List of methods defined in the module */

static struct methodlist orapy_methods[] = {
	{"newconnection",	(PyCFunction)orapy_newconnection,	1},
	{"logon",		(PyCFunction)orapy_newconnection,	1},
	{"maxlong",		(PyCFunction)orapy_maxlong,		1},

	{NULL,			NULL}		/* sentinel */
};


/* Initialization function for the module (*must* be called initoracle) */

void
initoracle()
{
	object *m, *d;

	/* Create the module and add the functions */
	m = initmodule("oracle", orapy_methods);

	/* Add some symbolic constants to the module */
	d = getmoduledict(m);
	OracleError = newstringobject("oracle.error");
	dictinsert(d, "error", OracleError);

	/* XXXX Add constants here */

	/* Check for errors */
	if (err_occurred())
		fatal("can't initialize module oracle");
}

static size_t bufsize(ColBufs *col)
{
	size_t size;

	switch (col->dbtype)
	{
	case 2:		/* number */
#ifdef VERSION6
		size = col->disp_size + 1;
#else
		if (col->scale == 0)		/* integer */
		{
			if (col->prec == 0 || col->prec > 9)
				size = col->disp_size + 1;
			else
				size = sizeof(long);
		}
		else
			size = sizeof(double);
#endif
		break;

	case 8:		/* long (char) */
	case 24:	/* long raw (bytes) */
		size = maxlong;		/* there may be a bug here... */
		break;

	case 1:		/* char *//* VERSION 7 varchar2 */
	case 11:	/* rowid */
	case 12:	/* date */
	case 23:	/* raw (bytes) */
	case 96:	/* VERSION 7 char */
	default:
		size = col->disp_size + 1;
		break;
	}
	return size;
}

static int buftype(ColBufs *col)
{
	int type;

	switch (col->dbtype)
	{
	case 2:		/* number */
#ifdef VERSION6
		type = EXT_STRING_TYPE;
#else
		if (col->scale == 0)		/* integer */
		{
			if (col->prec == 0 || col->prec > 9)
				type = EXT_STRING_TYPE;
			else
				type = EXT_INT_TYPE;
		}
		else
			type = EXT_FLOAT_TYPE;
#endif
		break;

	case 1:		/* char *//* VERSION 7 varchar2 */
	case 8:		/* long (char) */
	case 11:	/* rowid */
	case 12:	/* date */
	case 23:	/* raw (bytes) */
	case 24:	/* long raw (bytes) */
	case 96:	/* VERSION 7 char */
	default:
		type = EXT_STRING_TYPE;
		break;
	}
	return type;
}

#ifdef VERSION6

static object *
str_to_number(char *str)
{
	object *obj;
	char *end;
	long ival;
	double fval;

	errno = 0;
	ival = mystrtol(str, &end, 10);
	if (*end == '\0')		/* it looks like an integer */
	{
		if (errno != 0)		/* Did it overflow ? */
			obj = long_scan(str, 10);
		else
			obj = newintobject(ival);
	}
	else				/* maybe a floating point number? */
	{
		errno = 0;
		fval = strtod(str, &end);
		if (*end != '\0')
		{
			obj = NULL;
			err_setstr(ValueError, "not a valid numeric literal");
		}
		else if (errno != 0)
		{
			obj = NULL;
			err_setstr(OverflowError, "numeric literal is too large");
		}
		else
		{
			obj = newfloatobject(fval);
		}
	}
	return obj;
}

#endif
