/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/


/*
 * Take a buffer and a (pointer to) a public key, and produce
 * a buffer which contains a RSA encrypted DEK followed by the
 * original data encrypted (with IDEA) by the DEK.
 *
 * On error, return -1, else return 0;
 */



/* From PGFORMAT.DOC
 * ============================
 * 
 * Conventional Data Encryption Key (DEK) "packet"
 * -----------------------------------------------
 * 
 * The DEK has no CTB packet framing.  The DEK is stored packetless and
 * naked, with padding, encrypted inside the MPI in the RSA
 * public-key-encrypted packet.
 * 
 * PGP versions 2.3 and later use a new format for encoding the message
 * digest into the MPI in the signature packet.  (This format is not
 * presently based on any RFCs due to the use of the IDEA encryption
 * system.)  This format is accepted but not written by version 2.2.
 * older format used by versions 2.2 and earlier is also accepted by
 * versions up to 2.4, but the RSAREF code in 2.5 is unable to cope
 * with it.
 * 
 * PGP versions 2.3 and later encode the DEK into the MPI as follows:
 * 
 *         MSB                     .   .   .                   LSB
 *
 *        0   2   RND(n bytes)   0   1   DEK(16 bytes)   CSUM(2 bytes)
 *
 * CSUM refers to a 16-bit checksum appended to the high end of the DEK.
 * RND is a string of NONZERO pseudorandom bytes, enough to make the length
 * of this whole string equal to the number of bytes in the modulus.
 * 
 * For both versions, the 16-bit checksum is computed on the rest of the
 * bytes in the DEK key material, and does not include any other material
 * in the calculation.  In the above MSB-first representation, the
 * checksum is also stored MSB-first.  The checksum is there to help us
 * determine if we used the right RSA secret key for decryption.
 * 
 * (end PGFORMAT.DOC)
 * 
 * The 1 in the DEK format (to the left of the DEK proper) is the constant 
 * IDEA_ALGORITHM_BYTE.
 * =====================
 */

#include <string.h>
#include <stdio.h>

/* SSLeay */
#include <rsa.h>

#include "pgplib.h"
#include "pgp_m.h"

int
pub_encrypt_buffer(PGPpubkey_t	*key,
		   u_char	*inbuf,
		   int		inbuflen,
		   int		do_compress,
		   u_char	digest[16],
		   int		*outbuflen,
		   u_char 	**outbuf)
{
    u_char 	*enc_packet, *dekptr;
    int		buflen;
    int		keylen;
    int		header_len;
    u_char	header[5];
    RSA		rsa_info;
    u_char	dek[16+2];
    u_short	cksum, us;
    /* scratch */
    int		i;

    if (key == NULL || inbuf == NULL || inbuflen <= 0 || 
	outbuflen == 0 || outbuf == NULL) 
    {
	fprintf(stderr, "Error in parameter\n");
	return(-1);
    }

    i = conventional_key_encrypt(inbuf,	inbuflen, digest, do_compress,
				 &buflen, &enc_packet);
    if ( i == -1 ) {	
	fprintf(stderr, "Failed conventional encrypt\n");
	return(-1);
    }

    cksum = checksum(digest, 16);
    us = htons(cksum);
    memcpy(dek, digest, 16);
    memcpy(dek+16, &us, 2);
    
    /* Then, encrypt */
    rsa_info.n = key->N;
    rsa_info.e = key->E;

    /* The 2 is the MPI prefix used by PGP */
    keylen = BN_num_bytes(key->N) + 2;
    dekptr = malloc(keylen);
    if ( dekptr == NULL ) {
	fprintf(stderr, "OUt of memory\n");
	return(-1);
    }

    i = RSA_public_encrypt(18, dek, dekptr+2, &rsa_info);
    memset(dek, 0, 18);
    if ( i == -1 ) {
	free(dekptr);
	free(enc_packet);
	fprintf(stderr, "Failed to RSA encrypt the DEK\n");
	return(-1);
    }
    assert(i = keylen);
    /* MPI prefix */
    us = htons( (u_short)BN_num_bits(key->N));
    memcpy(dekptr, &us, 2);
    
    /* Then, make a PGP packet from the entire thing, the 1's are
     * version and algorithm byte, respectively. 
     */
    header_len = make_ctb(CTB_PKE_TYPE, 1+KEYFRAGSIZE+1+keylen, 0, header);
    /* Allocate space for two packets */
    *outbuf = malloc(header_len+1+KEYFRAGSIZE+1 + keylen + buflen);
    if ( *outbuf == NULL ) {
	fprintf(stderr, "Out of memory\n");
	free(enc_packet);
	free(dekptr);
	return(-1);
    }
    memcpy(*outbuf, header, header_len);

    (*outbuf)[header_len] = VERSION_BYTE_NEW;
    copy_quad(BN_get_keyid(key->N), (*outbuf)+header_len+1);
    (*outbuf)[header_len+9] =  RSA_ALGORITHM_BYTE;
    /* And the rest of this packet*/
    memcpy((*outbuf)+header_len+KEYFRAGSIZE+2, dekptr, keylen);
    free(dekptr);
    /* and the encrypte packet */
    memcpy((*outbuf)+header_len+KEYFRAGSIZE+2+keylen, enc_packet, buflen);
    free(enc_packet);

    *outbuflen = keylen + header_len+KEYFRAGSIZE+2+ buflen;
    return(0);
}
