/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/

#include <assert.h>

#include <stdio.h>
#include <sys/param.h>			/* includes sys/types */

#include <malloc.h>			/* for dbmalloc */

/* SSLeay */
#include <bn.h>

#include "conf.h"

typedef struct {
    long	length;			/* of the original packet */
    u_char	version;
    u_long	created;
    u_short	valid;
    BIGNUM	*N;
    BIGNUM	*E;
    u_char	pk_alg;
    u_char	iv[8];
    BIGNUM	*D;
    BIGNUM	*P;
    BIGNUM	*Q;
    BIGNUM	*U;
    u_short	cksum;			/* 16 bit */
} PGPseckey_t;

typedef struct {
    long	length;			/* Of the original packet */
    u_char	version;
    u_long	created;
    u_short	valid;			/* in days */
    u_char	algorithm;
    BIGNUM	*N;
    BIGNUM	*E;
} PGPpubkey_t;
    
typedef struct {
    int		length;			/* of total packet  */
    u_char	version;
    u_char	included;
    u_char	classification;
    u_long	timestamp;
    u_quad_t	key_id;
    u_char	pk_alg;
    u_char	hash_alg;
    u_char	ck[2];
    BIGNUM	*I;
} PGPsig_t;

/* We null-terminate the name, although PGP does not. */
typedef struct {
    char	*name;
} PGPuserid_t;

typedef struct {
    int		length;
    u_char	mode;
    char 	*filename;
    u_long	timestamp;
    int		datalen;
    u_char	*data;
} PGPliteral_t;

/* What is returned from buffers containing keyrings ? */
typedef enum {NONE, SIG, PUBKEY, UID, SECKEY} res_t;

/* How to open a database */
typedef enum {NOMODE, READER, WRITER, NEWDB} openmode_t;

/* What to do with data in databases */
typedef enum {NOACTION, INSERT, REPLACE} repl_t;


/*
 * armor.c
 */

int
dearmor_buffer(char	*armor_buf, 
	       long	armor_len,
	       u_char	**buffer, 
	       int	*buf_len);
int
dearmor_file(FILE	*fin,
	     FILE	*fout);

int
armor_buffer(u_char	*buf,
	     int	buf_len,
	     char	**a_buf,
	     int	*a_len);

/*
 * conventional.c
 */

int
conventional_key_decrypt(u_char	*enc_p,
			 u_char	digest[16],
			 u_char	**data);

int
conventional_key_encrypt(u_char	*inbuf,
			 int	inbuflen,
			 u_char digest[16],
			 int	do_compress,
			 int	*outbuflen,
			 u_char	**outbuf);

/*
 * decompress.c
 */

int
decompress_buffer(u_char	*buf,
		  int		buf_len,
		  int		*data_len,
		  u_char	**data);

int
decompress_file(FILE	*fin,
		FILE	*fout);

int
compress_buffer(u_char	*inbuf,
		int	inbuflen,
		u_char	**outbuf,
		int	*outbuflen);


/*
 * literal.c
 */
void
free_literal(PGPliteral_t *literal);

int
parse_literal_buffer(u_char	*data,
		     int  	length,
		     PGPliteral_t *literal);
int
parse_literal_file(FILE		*fin,
		   PGPliteral_t	*lit);

int
literal_data_file(FILE		*fin,
		  PGPliteral_t	*literal);


int
literal_to_buf(PGPliteral_t	*literal,
	       u_char		**buf);

int
create_literal(char		*filename,
	       PGPliteral_t	*literal);

/*
 * keybase.c
 */

extern int dbase_open;

int
keydb_init(char		*caller_name,
	   openmode_t	caller_mode);
void
keydb_end(void);

void
keydb_sync(void);

int
find_keys_from_string(char	*string,
		      int	save_case,
		      u_quad_t	**keys);
int
find_shortid(char 	*short_id,
	     u_quad_t	*keyid);
int
find_uids_with_key(u_quad_t	key_id,
		   PGPuserid_t	***uidpp);

int	
store_pub_key(PGPpubkey_t 	*pubkey, 
	      repl_t 		replace);
int
key_exists(u_quad_t	key_id);

int
get_pub_key_record(u_quad_t	key_id, 
		   u_char 	**buf, 
		   int 		*len);
int
store_pub_key_record(u_quad_t 	key_id, 
		     repl_t 	replace, 
		     u_char 	*buf, 
		     int 	size);

int
store_user_id(PGPuserid_t 	*user_id, 
	      u_quad_t 		key_id);

/*
 * Keymgnt.c
 */

typedef struct {
    res_t		what;
    union {
	PGPpubkey_t	*key;
	PGPsig_t	*sig;
	PGPuserid_t	*uid;
    } u;
} keyent_t;

int
add_user_id(u_quad_t 	key_id,
	    PGPuserid_t	*user_id);

int
add_sig_to_keyid(u_quad_t	key_id,
		 PGPsig_t	*sig);
int
buf2seckey(u_char	*buf,
	   int		buflen,
	   PGPseckey_t	*key);
void
free_pubkey(PGPpubkey_t	*key);

int
fetch_only_pubkey(u_quad_t 	key_id,
		  PGPpubkey_t 	**key);
void
free_keyrecords(keyent_t	**records);

int
get_keycontents(u_quad_t	key_id,
		keyent_t	***arr);

int
print_keyrecords(u_char		*buffer, 
		 int 		buf_len, 
		 FILE 		*fd);
int
pubkey2buf(PGPpubkey_t	*key, 
	   u_char 	**buf);

int
buf2pubkey(	u_char 		*buf, 
		int 		len, 
		PGPpubkey_t 	*key);
int
decrypt_seckey(	PGPseckey_t 	*key, 
		u_char 		*digest);

/*
 * keyring.c
 */

u_quad_t 
BN_get_keyid(BIGNUM *a);

u_quad_t 
BN_get_quad(BIGNUM *a);

int
get_entry_in_keyring(	FILE		*fd,
			res_t		*what,
			PGPsig_t	*sig,
			PGPpubkey_t	*key,
			PGPuserid_t	*uid);

int
get_keyentry_from_buffer(u_char		*buf,
			 int		buf_len,
			 res_t		*what,
			 PGPsig_t	**sig,
			 PGPpubkey_t	**pubkey,
			 PGPuserid_t	**uid,
			 PGPseckey_t	**seckey,
			 int		*used);
int
get_seckey_from_buffer(	u_char		*buf, 
			int 		buflen, 
			PGPseckey_t 	**key);
int
get_seckey_from_file(	FILE 		*keyring, 
			PGPseckey_t 	**key);

int
get_keyentry_from_file(	FILE		*fd,
			res_t		*what,
			PGPsig_t	**sig,
			PGPpubkey_t	**pubkey,
			PGPuserid_t	**uid,
			PGPseckey_t	**seckey);

/*
 * misc.c
 */

int
bufgets(u_char 	*line, 
	int 	linelen, 
	u_char 	*buf, 
	int buflen);

int
packet_length(u_char	*p);

int
make_ctb(u_char		ctb_type, 
	 u_long 	length, 
	 int 		force, 
	 u_char 	*buf);

#ifdef __FreeBSD__
unsigned long long
atoq(const char *s);
#endif

u_quad_t
to_quad(u_char buf[8]);

void
copy_quad(u_quad_t, u_char *);

u_short 
checksum(u_char	*, int);

int
mpi2bignum(u_char	*buf,
	   int		buflen,
	   BIGNUM	**N);

/*
 * packet.c
 */

int
read_packet(FILE	*fd,
	    int		*length,
	    u_char	**packet);
int
find_packet_in_buffer(u_char	*buffer,
		      int	buf_len,
		      u_char	**packet,
		      int	*plen);
/*
 * pubencrypt.c
 */

int
pub_encrypt_buffer(PGPpubkey_t	*key,
		   u_char	*inbuf,
		   int		inbuflen,
		   int		do_compress,
		   u_char	digest[16],
		   int		*outbuflen,
		   u_char 	**outbuf);

/*
 * signature.c
 */

void
free_signature(PGPsig_t *sig);

int
signature2buf(PGPsig_t	*sig, 
	      u_char	**buf);

int
buf2signature(u_char 	*buf, 
	      int 	len, 
	      PGPsig_t 	*sig);

int
add_sig_to_key(u_quad_t	key_id, 
	       PGPsig_t *sig);

int
check_signature(u_char 	CBT, 
		u_char 	*file_ptr, 
		int 	*length, 
		u_char 	**data);

int
verify_signature(u_char		*data,
		 int		dlen,
		 PGPsig_t	*sig,
		 PGPpubkey_t	*signkey);
int
verify_keysig(	PGPpubkey_t	*pubkey,
		PGPuserid_t	*uid,
		PGPsig_t	*sig,
		PGPpubkey_t	*signkey);
int
sign_buffer(	u_char		*buffer, 
		int 		bufsize,
		PGPseckey_t	*key, 
		PGPsig_t 	*sig);

/*
 * userid.c
 */

void
free_uid(PGPuserid_t	*uid);

int
uid2buf(PGPuserid_t	*uid,
	u_char		**buf);

int
buf2userid(u_char 	*buf, 
	   int		length, 
	   PGPuserid_t	*uid);
