/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.
*/

#include <stdio.h>
#include <stdlib.h>

#include "pgplib.h"
#include "pgp_m.h"

u_quad_t 
BN_get_quad(BIGNUM *a)
{
    int i;
    u_quad_t ret;

    i = BN_num_bytes(a);
    if ( i > sizeof(u_quad_t)) 
	return ( ~0 );

    if ( a->top == 1 )
	return(a->d[0]);

    assert( a->top == 2 );

    ret = a->d[0];
    ret |= ( (u_quad_t) a->d[1] << 32);

    return(ret);
}

/* 
 * This is how we obtain the PGP KeyID from N in a RSA public key.
 */
u_quad_t 
BN_get_keyid(BIGNUM *a)
{
    u_quad_t ret;

    if ( BN_is_zero(a))
	return(0);

    if ( a->top == 1 )
	return(a->d[0]);

    ret = a->d[0];
    ret |= ( (u_quad_t) a->d[1] << 32);

    return(ret);
}



/*
 * Parsing a BUFFER (presumaly holding a keyring) and return public 
 * keys, certificates and UserIDs as we find them.  One packet is 
 * returned on each call.  
 * If a packet was indeed found, we return 0 and *used shows how much
 * we consumed.  If there was a pointer into which we could store data,
 * we do so.  We don't return Key Control Packets---they are _too_ boring. 
 * If we don't find a packet or the buffer  is exhausted, we return 1.
 * On error, we return -1;
 *
 * Although some checks are performed, sending in random data is probably
 * a reliable way to obtain a coredump.  In other words, quite a few
 * checks for sanity should be inserted.
 */

int
get_keyentry_from_buffer(u_char		*buf,
			 int		buf_len,
			 res_t		*what,
			 PGPsig_t	**sig,
			 PGPpubkey_t	**pubkey,
			 PGPuserid_t	**uid,
			 PGPseckey_t	**seckey,
			 int		*used)
{
    int		how_far;
    int		p_len;
    u_char	*packet;
    int		ret;
    /* scratch */
    int		i;

    if ( buf == NULL || buf_len == 0 || used == NULL )
	return(-1);
	
    *what = NONE;
    how_far = 0;
    p_len = 0;
    while ( how_far <= buf_len ) { 
	ret = find_packet_in_buffer(buf+how_far, buf_len-how_far, 
				  &packet, &p_len);
	*used = how_far + p_len;
	switch ( ret ) {
	case -1:
	    return(-1);
	case 1:
	    /* Syntax error _or_ packet without length (compressed) */
	    return(1);
	case 0:
	    /* Some packet, p_len bytes long */
	    switch ( (packet[0] & CTB_TYPE_MASK) >> 2) {
	    default:
		free(packet);
		how_far += p_len;
		break;
	    case CTB_USERID_TYPE:
		if ( uid != NULL ) {
		    *uid = malloc(sizeof(PGPuserid_t));
		    if (*uid == NULL ) {
			fprintf(stderr, "Out of memory\n");
			return(-1);
		    }
		    i = buf2userid(packet, p_len, *uid);
		    free(packet);
		    if ( i != 0 ){
			return(-1);
		    }
		    else {
			*what = UID;
			return(0);
		    }
		}
		else {
		    free(packet);
		    return(0);
		}
	    case CTB_KEYCTRL_TYPE:
		/* We don't care */
		free(packet);
		how_far += p_len;
		break;
	    case CTB_CERT_PUBKEY_TYPE:
		if ( pubkey != NULL ) {
		    *pubkey = malloc(sizeof(PGPpubkey_t));
		    if (*pubkey == NULL ) {
			fprintf(stderr, "Out of memory\n");
			return(-1);
		    }
		    i = buf2pubkey(packet, p_len, *pubkey);
		    free(packet);
		    if ( i != 0 )
			return(-1);
		    else {
			*what = PUBKEY;
			return(0);
		    }
		}
 		else {
		    free(packet);
		    return(0);
		}
	    case CTB_CERT_SECKEY_TYPE:
		if ( seckey != NULL ) {
		    *seckey = malloc(sizeof(PGPseckey_t));
		    if (*seckey == NULL ) {
			fprintf(stderr, "Out of memory\n");
			return(-1);
		    }
		    i = buf2seckey(packet, p_len, *seckey);
		    free(packet);
		    if ( i != 0 )
			return(-1);
		    else {
			*what = SECKEY;
			return(0);
		    }
		}
	    case CTB_SKE_TYPE:
		if ( sig != NULL ) {
		    *sig = malloc(sizeof(PGPsig_t));
		    if (*sig == NULL ) {
			fprintf(stderr, "Out of memory\n");
			return(-1);
		    }
		    i = buf2signature(packet, p_len, *sig);
		    free(packet);
		    if ( i != 0 )
			return(-1);
		    else {
			*what = SIG;
			return(0);
		    }
		}
		else {
		    free(packet);
		    return(0);
		}
	    } /* switch CTB */
	} /* switch find_packet_in_buffer */
    } /* while buffer */

    return(1);
}

/*
 * Find a secret key in a buffer.  Return the number of bytes consumed.
 * If there is an error (or no seckey) return -1 or 0, respectively.
 */

int
get_seckey_from_buffer(u_char	*buf,
		       int 	buflen, 
		       PGPseckey_t **key)
{ 
    res_t	what;
    int		used, tot_used;
    int		ret;
    PGPseckey_t	*tmpkey;

    if ( buf == NULL || buflen < 3 || key == NULL || *key == NULL) {
	fprintf(stderr, "Incorrect parameters\n");
	return(-1);
    }

    *key = NULL;
    used = tot_used = 0;

    while ( buflen > 0 ) {
	ret = get_keyentry_from_buffer(buf, buflen, &what, NULL, NULL, 
				       NULL, &tmpkey, &used);
	buflen -= used;
	buf += used;
	tot_used += used;

	if ( what != SECKEY )
	    continue;
	
	*key = tmpkey;
	return(tot_used);
    }
    return(0);
}
	    

/* 
 * Parsing a FILE (presumably a keyring) and return public keys
 * certificates and UserID's as we find them.  One is returned on 
 * each call.  That is, if you don't like the key you get, throw it away 
 * and get a new one.
 * If something is found, 0 is returned and *what is set to the type
 * in question; if a pointer was provided, we create a struct of the
 * correct type.  Upon EOF or syntax error 1 is returned.
 * An (real) error will give you -1.
 * 
 */

int
get_keyentry_from_file(FILE		*fd,
		       res_t		*what,
		       PGPsig_t		**sig,
		       PGPpubkey_t 	**pubkey,
		       PGPuserid_t 	**uid,
		       PGPseckey_t	**seckey)
{
    u_char	*buf;
    int		length, ret;
    /* scratch */
    int		i,j;

    *what = NONE;
    
    i = read_packet(fd, &length, &buf);
    switch ( i ) {
    case -1:
	return(-1);
    case 1:
	/* Syntax error, EOF or packet without length (compressed) */
	return(1);
    case 0:
	break;
    default:
	assert(0);
    }

    ret = get_keyentry_from_buffer(buf, length, what, sig, 
				   pubkey, uid, seckey, &j);
    return(ret);
}

/*
 * The idea is that you can scan a file of secret keys, obtaining 
 * one for each call.  The return codes are inherited from 
 * get_keyentry_from_file: 0 on success, 1 on EOF -1 on error.
 */

int
get_seckey_from_file(FILE *keyring, PGPseckey_t **key)
{
    res_t	what;
    PGPseckey_t *tmpkey;
    int		ret;

    if ( keyring == NULL || key == NULL ) {
	fprintf(stderr, "Incorrect parameter\n");
	return(-1);
    }
    while ( ( ret = get_keyentry_from_file(keyring, &what, NULL, NULL,
					   NULL, &tmpkey)) == 0) {
	if ( what == SECKEY ) {
	    *key = tmpkey;
	    return(0);
	}
	else
	    continue;
    }	
    
    /* no seckey */
    return(ret);
}
