/*  Hey Emacs, switch to -*-C-*-, please 
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.]

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This
implementation was a non-trivial and unpaid effort.

*/


/*
 * In order to keep track of encryption keys we store them in a set 
 * of databases.  The "real" index is the KeyID (64 bit long) and
 * you can use this to find a key based on the KeyID.
 * 
 * Then, the string(s) associated with each key ("User ID Packet" in
 * PGP terms) are stored in a separate database for faster lookup.
 */

/* A name must be provided.  From it, we will create two file-names.
 * If these files exists, we will use them, if not, we will create them.
 */


#include <ctype.h>
#include <db.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>

extern char * strerror(int );

#include "pgplib.h"
#include "pgp_m.h"

#define KEYEXT		".key"
#define STRINGEXT	".string"
#define	REVSTRINGEXT	".revstring"
#define KEYIDEXT	".keyid"

static DB	*keydb;
static DB	*stringdb;
static DB	*revstringdb;
static DB	*keyiddb;

int	dbase_open = 0;


void
close_databases()
{
    fprintf(stderr, "Fatal error detected by DB -- exit\n");
    exit(1);
}

int
keydb_init(char 	*caller_name,
	   openmode_t	caller_mode)
{
    char 	filename[MAXPATHLEN];
    char 	*p;
    int		mode;
    HASHINFO	hinfo;

    if ( caller_name == NULL ) {
	fprintf(stderr, "keydb_init called with NULL");
	return(-1);
    }
    switch(caller_mode) {
    case READER: 
	mode = O_RDONLY;
	break;
    case WRITER: 
	mode = O_RDWR; 
	break;
    case NEWDB:  
	mode = O_CREAT | O_TRUNC | O_RDWR ; 
	break;
    default: 
	return(-1);
    }

    memset(&hinfo, 0, sizeof(hinfo));
    hinfo.cachesize = 5*CACHE_SIZE;
    hinfo.bsize     = BSIZE;
    hinfo.nelem	    = NELEM;

    filename[0] = NULL;
    strcat(filename, caller_name);
    strcat(filename, KEYEXT);
    keydb = dbopen (filename, mode, 0600, DB_HASH, &hinfo);
    if ( keydb == NULL ) {
	fprintf(stderr, "Error from dbopen %s (%s)\n", 
		filename, strerror(errno));
	return(-1);
    }
    
    hinfo.cachesize = CACHE_SIZE;
    hinfo.bsize     = BSIZE;

    p = rindex(filename, '.');
    *p = NULL;
    strcat(filename, STRINGEXT);
    stringdb = dbopen(filename, mode, 0600, DB_HASH, &hinfo);
    if ( stringdb == NULL ) {
	fprintf(stderr, "Error from dbopen %s (%s)\n",
		filename, strerror(errno));
	return(-1);
    }

    p = rindex(filename, '.');
    *p = NULL;
    strcat(filename, REVSTRINGEXT);
    revstringdb = dbopen (filename, mode, 0600, DB_HASH, &hinfo);
    if (revstringdb == NULL ) {
	fprintf(stderr, "Error from open %s (%s)\n",
		filename, strerror(errno));
	return(-1);
    }

    p = rindex(filename, '.');
    *p = NULL;
    strcat(filename, KEYIDEXT);
    keyiddb = dbopen (filename, mode, 0600, DB_HASH, &hinfo);
    if ( keyiddb == NULL ) {
	fprintf(stderr, "Error from dbopen %s (%s)\n",
		filename, strerror(errno));
	return(-1);
    }
    dbase_open = 1;

    return(0);
}


void
keydb_end(void)
{
    if ( keydb != NULL )
	(keydb->close)(keydb);
    if ( stringdb !=  NULL )
	(stringdb->close)(stringdb);
    if ( revstringdb != NULL )
	(revstringdb->close)(revstringdb);
    if ( keyiddb != NULL )
	(keyiddb->close)(keyiddb);
}

void
keydb_sync(void)
{
    if ( keydb != NULL )
	(keydb->sync)(keydb, 0);
    if ( stringdb !=  NULL )
	(stringdb->sync)(stringdb, 0);
    if ( revstringdb != NULL )
	(revstringdb->sync)(revstringdb, 0);
    if ( keyiddb != NULL )
	(keyiddb->sync)(keyiddb, 0);
}

/*  * Below is all the routines to store, fetch, alter and delete
 * PGP compatible keys and certificates.
 * All these routines take and return nicely formatted keys and certificates
 * as pointers to struct's.  The internal format (in the databases)
 * are less nice since it has to be marshalled (and, urk, compatible).
 *
 * In the Key database, the keys (together with UserID and signatures)
 * are stored in PGP format.  This way we can print the
 * key fast by simply "armour" the buffer directly.  It takes some
 * effort to convert the buffer to the correct type of C-struct's,
 * but that is the price we have to pay.
 *
 * The strings (UserID) are stored in a separate database.  This
 * way we can locate one _very_ fast
 *
 * The KeyID that pgp presents (only 32 bit) are stored in a third
 * database.
 */

/* 
 * Take a string ("0x12345678") and convert it into the corresponding
 * number, and find the key.  No key: return 1 and set keyid to 0.
 * On error return -1;
 */

int
find_shortid(char 	*short_string,
	     u_quad_t	*keyid)
{

    DBT		key, contents;
    u_long	short_id;
    int 	i;
    char	*p;

    *keyid = 0LL;

    if ( short_string == NULL || *short_string == 0 )
	return(-1);
    
    short_id = strtoul(short_string, &p, 0);
    if ( p == NULL || *p != 0 )
	return(-1);

    key.data = (void *)&short_id;
    key.size = sizeof(short_id);

    i = (keyiddb->get)(keyiddb, &key, &contents, 0);
    if ( i == -1 ) {
	perror("GET");
	return(-1);
    }
    if ( i == 0 ) {
	assert(contents.size == sizeof(u_quad_t));
	memcpy(keyid, contents.data, sizeof(u_quad_t));
    }
    return(0);
}

int
store_pub_key(PGPpubkey_t	*pubkey, 
	      repl_t		replace)
{
    DBT		key, contents;
    u_quad_t	key_id;
    u_long	short_id;
    int		key_size;
    u_char	*key_buf;
    /* scratch */
    int 	i;

    assert ( replace == INSERT || replace == REPLACE);

    if ( dbase_open == 0 )
	return(-1);

    key_id = BN_get_keyid(pubkey->N);
    key_size = pubkey2buf(pubkey,  &key_buf);
    if ( key_size < 0 ) {
	return(-1);
    }
    i = store_pub_key_record(key_id, replace, key_buf, key_size);
    free(key_buf);
    switch (i) {
    case -1:
	return(-1);
    case 1:
	fprintf(stdout, "Key exists (asked for insert and NO-OVERWRITE)\n");
	return(1);
    case 0:
	break;
    default:
	fprintf(stderr, "Unknown return from put\n");
	assert(0);
    }

    /* The store went well.  We then proceed by adding the shorter
     * id presented by "pgp -kv", a string representing a 32 bit
     * number.  However, we store the string as the number it is
     * supposed to represent (occupies less than half the space)
     */
    
    short_id = (u_long) key_id;

    key.data = (void *)&short_id;
    key.size = sizeof(short_id);
    
    contents.data = (void *) &key_id;
    contents.size= sizeof(key_id);

    i = (keyiddb->put)(keyiddb, &key, &contents, 0);
    if ( i == -1 )
	perror("PUT 2");
    else 
	if ( i == 1 )
	    /* We just hit 1/12.000 probability.  Warn someone. */
	    fprintf(stderr, "1/12.000 just stroke.\n");
    return(i);
}

int
get_pub_key_record(u_quad_t	key_id,
		   u_char 	**buf, 
		   int 		*len)
{

    DBT		key, contents;	
    int		i;

    if ( dbase_open == 0 )
	return(-1);

    key.data = (void *)&key_id;
    key.size = sizeof(key_id);
    
    i = (keydb->get)(keydb, &key, &contents, 0);
    if ( i == -1 ) {
	perror("GET");
	return(-1);
    }
    if ( i == 1 )
	return(1);

    *buf = malloc(contents.size);
    if ( *buf == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    memcpy(*buf, contents.data, contents.size);
    *len = contents.size;

    return(0);
}

int
store_pub_key_record(u_quad_t	key_id, 
		     repl_t 	user_replace, 
		     u_char 	*buf, 
		     int 	size)
{
    DBT 	key, contents;
    int 	i,j;

    assert ( user_replace == REPLACE || user_replace == INSERT);

    if ( dbase_open == 0 )
	return(-1);

    key.data = (void *)&key_id;
    key.size = sizeof(key_id);

    contents.data = (void *)buf;
    contents.size = size;

    if ( user_replace == REPLACE )
	j = 0;
    else
	j = R_NOOVERWRITE;
    i = (keydb->put)(keydb, &key, &contents, j);
    if ( i == -1 ) {
	perror("PUT 3");
    }
    
    return(i);
}
 
/* 
 * If the key exists, return 1.  If not, return 0.  Error: -1
 */

int
key_exists(u_quad_t	key_id)
{
    DBT		key, contents;
    int 	i;

    if ( dbase_open == 0 )
	return(-1);

    key.data = (void *)&key_id;
    key.size = sizeof(key_id);

    i = (keydb->get)(keydb, &key, &contents, 0);
    if ( i == -1 ) {
	perror("GET");
	return(-1);
    }
    if ( i == 0) {
	return(0);
    }
    else
	return (1);
}

/*
 * Storing a UID consists of two actions.  First, store the KeyID
 * indexed by the UID.  Then, store the UID indexed by the KeyID.
 */

int
store_user_id(PGPuserid_t	*user_id, 
	      u_quad_t 		key_id)
{
    DBT		key, contents;	
    res_t	what;
    int		used, how_far, found, newlen;
    PGPuserid_t	*uid;
    u_char	*buf;
    /* scratch */
    int 	i;
    u_char	*p;


    if ( dbase_open == 0 )
	return(-1);

    key.data = (void *)user_id->name;
    key.size = strlen(user_id->name)+1;
    contents.data = (void *) &key_id;
    contents.size = sizeof(key_id);
    i = (stringdb->put)(stringdb, &key, &contents, 0);
    if ( i == -1 ) {
	perror("PUT 4");
	return(-1);
    }
    assert ( i != 1 );

    key.data= (void *) &key_id;
    key.size = sizeof(key_id);
    i = (revstringdb->get)(revstringdb, &key, &contents, 0);
    if ( i == 1 ) {
	/* This is the first UID for this key */
	contents.size = uid2buf(user_id, (u_char **) &contents.data);
	i = (revstringdb->put)(revstringdb, &key, &contents, 0);
	free(contents.data);
	if ( i != 0 ) {
	    perror("PUT 5");
	    return(-1);
	}
	else 
	    return(0);
    }
    else {
	/* There was already (at least) one UID associated with
	 * this key.  We have to walk through them to check if
	 * the one we have already is here.
	 * Note that we have a PGP compatibale buffer at hand.
	 */
	how_far = 0;
	found = 0;
	while( how_far < contents.size  && found == 0) {

	    i = get_keyentry_from_buffer( contents.data  + how_far,
					  contents.size - how_far,
					  &what, NULL, NULL, &uid, NULL,
					  &used);
	    if ( i != 0 ) {
		return(-1);
	    }
	    assert ( what == UID );

	    how_far += used;
	    
	    if ( strcmp(user_id->name, uid->name) == 0 ) {
		/* It's already here */
		found = 1;
		free_uid(uid);
		break;
	    }
	    else{
		/* Not this one */
		free_uid(uid);
		continue;
	    }		
	}
	if ( found == 1 )
	    return(1);

	newlen = uid2buf(user_id, &buf);
	if ( newlen == 0 )
	    return(-1);

	p = malloc(contents.size + newlen);
	if ( p == NULL ) {
	    fprintf(stderr, "Out of memory\n");
	    return(-1);
	}
	memcpy(p, contents.data, contents.size);
	memcpy(p+contents.size, buf, newlen);
	free(buf);
	contents.data = p;
	contents.size += newlen;

	i = (revstringdb->put)(revstringdb, &key, &contents, 0);
	free(p);
	if ( i != 0 )
	    return(-1);
	return(0);
    }
    /*NOTREACHED*/
    assert(0);
}


/*
 * Find a string in the database.  We start by checking if we can
 * locate the exact string.  If not, we scan the database looking
 * for matches.
 * If save_case is TRUE we preserve case, if not, everything 
 * is converted to UPPER.
 * Return an the number of KeyID's we found with a match, and an 
 * array (at least) that big of the numbers.
 */
 
int
find_keys_from_string(char	*string,
		      int	save_case,
		      u_quad_t	**keys)
{
    DBT		key, contents;
    u_quad_t	*keyp, tmpkey;
    int		num_left, num_alloc, offset;
    char	upstring[256];
    int		already;
    int		ret_value;

    char	*tmp;
    int		i;

    if ( dbase_open == 0 ) {
	fprintf(stderr, "Database not opened\n");
	return(-1);
    }

    if ( string == NULL || *string == 0 )
	return(0);
    key.data = (void *)string;
    key.size = strlen(string) + 1;
    i = (stringdb->get)(stringdb, &key, &contents, 0);
    if ( i == -1 ) {
	perror("GET");
	return(-1);
    }
    if ( i == 0 ) {
	/* Perfect match ! */
	keyp = malloc(sizeof(u_quad_t));
	if ( keyp == NULL ) {
	    fprintf(stderr, "Out of memory");
	    return(0);
	}
	memcpy(keyp, contents.data, sizeof(u_quad_t));
	*keys = keyp;
	return(1);
    }

    /* Hmmmm, we will have to scan the database */

#define	ALLOCATE	100

    offset = num_alloc = num_left = 0;
    keyp = NULL;

    while ( 
	(ret_value = (stringdb->seq)(stringdb, &key, &contents, R_NEXT)) == 0)
    {
	if (save_case == 0) {
	    /* Need this trick since sizeof(*void) != 1 */
	    char *_p;
	    
	    _p = (char *)key.data;
	    memset(upstring, 0, 256);
	    for(i=0; i < key.size; i++)
		upstring[i] = toupper(_p[i]);
	    for( tmp = string ; *tmp != NULL; tmp += 1)
		*tmp = toupper(*tmp);
	    tmp = strstr(upstring, string);
	}
	else
	    tmp = strstr((char *)key.data, string);
	
	if ( tmp != NULL ) {
	    /* The string matches.  Get the KeyID */
	    if (num_left == 0 ) {
		tmp = realloc(keyp, (num_alloc+1)*(ALLOCATE*sizeof(u_quad_t)));
		if ( tmp == NULL ) {
		    free(keyp);
		    fprintf(stderr, "Out of memory\n");
		    return(0);
		}
		else
		    keyp = (u_quad_t *)tmp;
		num_alloc += 1;
		num_left = ALLOCATE;
	    }
	    memcpy(&tmpkey, contents.data, sizeof(u_quad_t));
	    assert(tmpkey != 0);
	    /* Check that it isn't already there */
	    already = 0;
	    for(i=0; i<offset; i++) {
		if ( keyp[i] == tmpkey) {
		    already = 1;
		    break;
		}
	    }
	    if ( already != 1 ) {
		/* Add this key */
		keyp[offset] = tmpkey;
		offset += 1;
		num_left -= 1;
	    }
	}
    } /* while */
    if ( ret_value == -1 ) {
	perror("SEQ");
	free(keyp);
	return(-1);
    }
    assert(ret_value == 1);
    *keys = keyp;
    return(offset);
}

/*
 * Find the UIDs attached to this key.  Return of 1 means no UIDs
 * (or possibly unknown key ?), -1 is error, 0 is OK.
 * The array of pointers to UID's is NULL terminated.
 */

int
find_uids_with_key(u_quad_t	key_id,
		   PGPuserid_t	***uidpp)
{
    DBT		key, contents;
    int		offset;
    res_t	what;
    int		ent;
    int		used;
    PGPuserid_t	**arr, **tmp;
    /* scratch */
    int		i,j;

    if ( dbase_open == 0 )
	return(-1);

    key.data = (void *)&key_id;
    key.size = sizeof(key_id);
    
    i = (revstringdb->get)(revstringdb, &key, &contents, 0);
    if ( i == -1 ) {
	perror("GET");
	return(-1);
    }
    if ( i == 1 )
	return(1);
    
    /* Ensure our array is NULL terminated */
    arr = calloc(1, sizeof(PGPuserid_t *));
    if ( arr == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }

    ent = 1;
    offset = 0;
    do {
	tmp = realloc(arr, (ent+1) * sizeof(PGPuserid_t *));
	if ( tmp == NULL ) {
	    free(arr);
	    fprintf(stderr, "Out of memory\n");
	    return(-1);
	}
	arr = tmp;
	arr[ent] = NULL;
	i = get_keyentry_from_buffer((char*)contents.data + offset,
				     contents.size - offset,
				     &what,
				     NULL, NULL, &arr[ent-1], NULL,
				     &used);
	switch ( i ) {
	case -1:
	    for(j=0; j<ent-2; j++)
		free(arr[j]);
	    free(arr);
	    return(-1);
	case 1:
	    *uidpp = arr;
	    return(0);
	case 0:
	    /* the revstring database is suppoed to only hold UID's */
	    assert (what == UID);
	    offset += used;
	    ent += 1;
	    break;
	default:
	    assert(0);
	    /*NOTREACHED*/
	}
    } while (offset < contents.size);

    *uidpp = arr;
    return(0);
}
