/* Emacs, please go into -*-C-*- mode
------------

Copyright (C) 1997 Tage Stabell-Kul (tage@acm.org).  All rights
reserved.

This is a package, containing a library, that makes it possible for
programmers to include security functionality in their programs, in a
format compatible with PGP.

This library is free for commercial and non-commercial use as long as
the following conditions are aheared to:

Copyright remains Tage Stabell-Kul's, and as such any Copyright
notices in the code are not to be removed.
If this package is used in a product, Tage Stabell-Kul should be given
attribution as the author of the parts of the library used.
This can be in the form of a textual message at program startup or
in documentation (online or textual) provided with the package.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.
3. All advertising materials mentioning features or use of this software
   must display the following acknowledgement:
   "This product includes software written by
    Tage Stabell-Kul (tage@acm.org)"

THIS SOFTWARE IS PROVIDED BY TAGE STABELL-KUL ``AS IS'' AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

The licence and distribution terms for any publically available
version or derivative of this code cannot be changed.  i.e. this code
cannot simply be copied and put under another distribution licence
[including the GNU Public Licence.] 

The reason behind this being stated in this direct manner is past
experience in code simply being copied and the attribution removed
from it and then being distributed as part of other packages. This 
implementation was a non-trivial and unpaid effort.

*/


#include <stdlib.h>
#include <string.h>

#include "pgplib.h"
#include "pgp_m.h"


/* I have taken some lines of code directly from PGP.  It didn't
 * feel quite right to paste it in underneath my Copyright so I
 * include it In Verbatim instead.
 *
 * Can someone please tell me if this is sane ?
 *
 */

#include "armor.inc"

/* 
 * Take a buffer and parse it.  Dearmor as we go.
 *
 * We return 0 if there is an (format) error in the input.  If there
 * is an (real) error, we return -1.  If all is well, we return
 * the number of bytes we consumed of the input buffer.
 */

/* How much memory we malloc each time we iterate.  Notice that it is
 * assumed that we can fit the header of a PGP armored message into
 * this amount of memory.
 */

#define	CHUNK		1024		/* bytes */

int
dearmor_buffer(char	*a_buf,
	       long	a_len,
	       u_char	**buffer, 
	       int	*buf_len)
{
    char	line[MAXLINE];
    u_char	buf[MAXLINE];
    int		len, ret, num_read;
    int		found_checksum;
    int		mem_left, mem_sofar, mem_chunks;
    u_long	found_crc, made_crc;
    int		a_off;
    /* scratch */
    u_char	*p;

    a_off = 0;
    while ((ret = bufgets(line, MAXLINE, a_buf+a_off, a_len-a_off)) != 0) {
	a_off += ret;
	if ( strncmp(line, "-----BEGIN PGP ", 14) == 0)
	    break;
    }
    if ( ret == 0 )
	/* incorrect format */
	return(0);

    /* Start of something.  Let us check the alternatives.  We don't
     * care what it is, as long as it seems to be correct.
     */
    if ( strncmp(line+15,    "PUBLIC KEY BLOCK-----", 21) != 0
	 && strncmp(line+15, "MESSAGE-----", 12) != 0
	 && strncmp(line+15, "SIGNED MESSAGE-----", 19) != 0)
	return(0);
    
    /* Skip any header lines.  We simply don't care whether
     * they are valid or not (see armor.c in PGP to understand how
     * validity is decided.)
     */
    while ((ret = bufgets(line, MAXLINE, a_buf+a_off, a_len-a_off)) != 0 ) {
	a_off += ret;
	if ( strlen(line) > 1 )
	    continue;
	if (line[0] == '\n')
	    break;
    }
    if ( ret == 0 )
	return(0);

    mem_left = mem_sofar = mem_chunks = 0;
    *buffer = NULL;
    *buf_len = 0;
    found_checksum = 0;

    init_crc();
    made_crc = CRCINIT;

    /* Read and read, until we reach the checksum */
    for( ;; ) {
	num_read = bufgets(line, MAXLINE, a_buf+a_off, a_len-a_off);
	if ( num_read == 0 ) {
	    if ( found_checksum ) {
		fprintf(stderr, "WARNING: No ASCII armor `END'.\n");
		break;
	    }
	    /* Not missing END, then it is an error */
	    return(0);
	}
	a_off += num_read;

	/* Get rid of trailing whitespace (and things) */
	p = rindex(line, 0);
	/* This shouldn't be possible (I think) */
	if ( strlen(line) == 1) 
	    return(-1);
	while( *(p-1) == ' ' || *(p-1) == '\n' ) {
	    *(p-1) = NULL;
	    if (p == (u_char *)line) {
		/* a line with just whitespace */
		fprintf(stderr, "WARNING: A line with just whitespace.\n");
		continue;
	    }
	    else
		p -= 1;
	}

	if ( found_checksum == 1 ) {
	    /* This line should be the end of things */
	    if ( strncmp(line, "-----END PGP", 12) ) {
		fprintf(stderr, "WARNING: No ASCII armor `END'\n");
	    }

	    /* 
	     * This is how we normally depart from here (the only break)
	     */
	    break;
	}
	if ( line[0] == PAD ) {
	    /* Here comes the checksum. */
	    if ( strlen(line) != 5 ) {
		/*fprintf(stderr, "Malformed checksum (not 5 chars)\n");*/
		return(0);
	    }
	    memset(buf, 0, 4);
	    if (darmor_line(line+1, buf, &len) != 0 ) {
		/*fprintf(stderr, "Format Error in CRC\n");*/
		return(0);
	    }
	    assert(len == 3);
	    found_crc = buf[0] << 16;
	    found_crc |= buf[1] << 8;
	    found_crc |= buf[2];
	    /* We will read one more line, looking for the END */
	    found_checksum = 1;
	    continue;
	}
	
	if ( darmor_line(line, buf, &len) ) {
	    return(0);
	}
	/* A new line has been decoded.  Copy it into memory */
	if ( len > mem_left ) {
	    p = realloc(*buffer, (mem_chunks+1)*CHUNK);
	    if ( p == 0 ) {
		fprintf(stderr, "Out of memory\n");
		free(*buffer);
		return(-1);
	    }
	    *buffer = p;
	    mem_chunks += 1;
	    mem_left += CHUNK;
	}
	memcpy(*buffer+mem_sofar, buf, len);
	mem_sofar += len;
	mem_left  -= len;
	/* accumulate the crc */
	made_crc = crcbytes(buf, len, made_crc);
	continue;
    } /* while */
    
    if ( found_checksum == 0 ) {
	/* fprintf(stderr, "Incorrect termination of package\n");*/
	return(0);
    }

    if ( made_crc != found_crc ) {
	/* fprintf(stderr, "Checksum incorrect\n");*/
	return(0);
    }

    *buf_len = mem_sofar;
    return(a_off);
}

/* 
 * Dearmor a FILE into another FILE.
 * On sucess, we return 0.  If the format is incorrect, we
 * return 1.  Error returns -1.
 */

int
dearmor_file(FILE	*fin,
	     FILE	*fout)
{
    char	line[MAXLINE];
    u_char	buf[MAXLINE];
    int		len;
    int		found_checksum;
    u_long	found_crc, made_crc;
    /* scratch */
    u_char	*p;
    int		i;


    while ( (p=fgets(line, MAXLINE, fin)) != NULL) {
	if ( strncmp(line, "-----BEGIN PGP ", 14) == 0)
	    break;
    }

    if ( p == NULL )
	return(1);

    /* Start of something.  Let us check the alternatives.  We don't
     * care what it is, as long as it seems to be correct */
    if ( strncmp(line+15,    "PUBLIC KEY BLOCK-----", 21) != 0
	 && strncmp(line+15, "MESSAGE-----", 12) != 0
	 && strncmp(line+15, "SIGNED MESSAGE-----", 19) != 0)
	return(1);
    
    /* Skip any header lines.  We simply don't care whether
     * they are valid or not (see armor.c in PGP to understand how
     * validity is decided.)
     */
    while ( (p=fgets(line, MAXLINE, fin)) != NULL ) {
	if ( strlen(line) > 1 )
	    continue;
	if (line[0] == '\n')
	    break;
    }
    if ( i == 0 )
	return(1);

    found_checksum = 0;

    init_crc();
    made_crc = CRCINIT;

    /* Read and read, until we reach the checksum */
    for ( ;; ) {
	p=fgets(line, MAXLINE, fin);
	if ( p == NULL ) {
	    if ( found_checksum ) {
		fprintf(stderr, "WARNING: No ASCII armor `END'.\n");
		break;
	    }
	    /* Not missing END, then it is an error */
	    return(1);
	}

	/* Get rid of trailing whitespace (and things) */
	p = rindex(line, 0);
	/* This shouldn't be possible (I think) */
	if ( strlen(line) == 1) 
	    return(-1);
	while( *(p-1) == ' ' || *(p-1) == '\n' ) {
	    *(p-1) = NULL;
	    if (p == (u_char *)line) {
		/* a line with just whitespace */
		fprintf(stderr, "WARNING: A line with just whitespace.\n");
		continue;
	    }
	    else
		p -= 1;
	}

	if ( found_checksum == 1 ) {
	    /* This line should be the end of things */
	    if ( strncmp(line, "-----END PGP", 12) ) {
		fprintf(stderr, "WARNING: No ASCII armor `END'\n");
	    }
	    /* This is how we normally depart from here (the only break)*/
	    break;
	}
	if ( line[0] == PAD ) {
	    /* Here comes the checksum. */
	    if ( strlen(line) != 5 ) {
		/*fprintf(stderr, "Malformed checksum (not 5 chars)\n");*/
		return(1);
	    }
	    memset(buf, 0, 4);
	    if (darmor_line(line+1, buf, &len) != 0 ) {
		/*fprintf(stderr, "Format Error in CRC\n");*/
		return(1);
	    }
	    assert(len == 3);
	    found_crc = buf[0] << 16;
	    found_crc |= buf[1] << 8;
	    found_crc |= buf[2];
	    /* We will read one more line, looking for the END */
	    found_checksum = 1;
	    continue;
	}
	
	if ( darmor_line(line, buf, &len) ) {
	    return(1);
	}
	/* A new line has been decoded.  Write it out */
	i = fwrite(buf, len, 1, fout);
	if ( i == 1 )
	    return(-1);
	/* accumulate the crc */
	made_crc = crcbytes(buf, len, made_crc);
	continue;
    } /* while */
    
    if ( found_checksum == 0 ) {
	/* fprintf(stderr, "Incorrect termination of package\n");*/
	return(1);
    }

    if ( made_crc != found_crc ) {
	/* fprintf(stderr, "Checksum incorrect\n");*/
	return(1);
    }
    return(0);
}


/* 
 * Read a FILE and look for what might be an PGP armored package.
 * Read it in and return it in a buffer.  Return 1 if the FILE doesn't
 * seem to contain a packet, -1 on error and 0 if nothing went wrong.
 *
 * This code is almost identical to the preamble in dearmor_buffer;
 * can't be helped.
 */

int
read_armored_package(FILE	*fd,
		     int	*buf_len,
		     char	**buffer)
{
    char	line[MAXLINE];
    int		mem_left, mem_sofar, mem_chunks;
    int		found_checksum;
    int		len;
    u_char	*p;
    
    while ((p = fgets(line, MAXLINE, fd)) != NULL) {
	if ( strncmp(line, "-----BEGIN PGP ", 14) == 0)
	    break;
    }
    if ( p == NULL )
	return(1);
    
    /* Start of something.  Let us check the alternatives.  We don't
     * care what it is, as long as it seems to be correct 
     */
    if ( strncmp(line+15,    "PUBLIC KEY BLOCK-----", 21) != 0
	 && strncmp(line+15, "MESSAGE-----", 12) != 0
	 && strncmp(line+15, "SIGNED MESSAGE-----", 19) != 0)
	return(1);

    mem_left = mem_sofar = mem_chunks = 0;
    *buffer = NULL;
    *buf_len = 0;
    found_checksum = 0;

    /* Save the first line. */
    *buffer = malloc(CHUNK);
    if ( *buffer == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    mem_chunks = 1;
    mem_left = CHUNK;

    len = strlen(line);
    memcpy(*buffer, line, len);
    mem_sofar += len;
    mem_left  -= len;

    /* Copy in header lines */
    while ( (p = fgets(line, MAXLINE, fd)) != NULL ) {
	len = strlen(line);
	if ( len > mem_left ) {
	    p = realloc(*buffer, (mem_chunks+1)*CHUNK);
	    if ( p == NULL ) {
		free(*buffer);
		fprintf(stderr, "Out of memory\n");
		return(-1);
	    }
	    *buffer = p;
	    mem_chunks += 1;
	    mem_left += CHUNK;
	}
	memcpy((*buffer)+mem_sofar, line, len);
	mem_sofar += len;
	mem_left  -= len;
	if ( len == 1 )
	    if ( line[0] != '\n') {
		/*fprintf(stderr, "Incorrect format\n"); */
		free(*buffer);
		return(1);
	    }
	else
	    break;
    }
    if ( p == NULL ) {
	free(*buffer);
	return(1);
    }
    
    /* Read until we reach the checksum.  A line is MAXLINE long.
     */
    for ( ;; ) {
	p=fgets(line, MAXLINE, fd);
	if ( p == NULL ) {
	    if ( found_checksum ) {
		fprintf(stderr, "WARNING: No ASCII armor `END'.\n");
		break;
	    }
	    /* Not "just" missing END, then it is an error */
	    free(*buffer);
	    return(1);
	}
	
	/* Get rid of trailing whitespace (and things) */
	p = rindex(line, 0);
	/* This shouldn't be possible (I think) */
	if ( strlen(line) == 1) 
	    return(-1);
	while( *(p-1) == ' ' || *(p-1) == '\n' ) {
	    *(p-1) = NULL;
	    if (p == (u_char *)line) {
		/* a line with just whitespace */
		fprintf(stderr, "WARNING: A line with just whitespace.\n");
		continue;
	    }
	    else
		p -= 1;
	}
	
	if ( found_checksum == 1 ) {
	    /* This line should be the end of things */
	    if ( strncmp(line, "-----END PGP", 12) ) {
		fprintf(stderr, "WARNING: No ASCII armor `END'\n");
		/* Don't copy. We break and leave this mess to others.
		 * Note that the packet might be OK */
		break;
	    }
	    else {
		len = strlen(line);
		if ( len > mem_left ) {
		    p = realloc(*buffer, (mem_chunks+1)*CHUNK);
		    if ( p == NULL ) {
			free(*buffer);
			fprintf(stderr, "Out of memory\n");
			return(-1);
		    }
		    *buffer = p;
		    mem_chunks += 1;
		    mem_left += CHUNK;
		}
		memcpy((*buffer)+mem_sofar, line, len);
		mem_sofar += len;
		mem_left  -= len;
		/* This is how we normally depart from here */
		break;
	    }
	}
	if ( line[0] == PAD ) {
	    /* Here comes the checksum. */
	    len = strlen(line);
	    if ( len != 5 ) {
		/*fprintf(stderr, "Malformed checksum (not 5 chars)\n");*/
		free(*buffer);
		return(1);
	    }
	    if ( len > mem_left ) {
		p = realloc(*buffer, (mem_chunks+1)*CHUNK);
		if ( p == NULL ) {
		    free(*buffer);
		    fprintf(stderr, "Out of memory\n");
		    return(-1);
		}
		*buffer = p;
		mem_chunks += 1;
		mem_left += CHUNK;
	    }
	    memcpy(*buffer+mem_sofar, line, len);
	    mem_sofar += len;
	    mem_left  -= len;

	    /* We will read one more line, looking for the END */
	    found_checksum = 1;
	    continue;
	}
	
	/* A (normal) new line has been found.  Copy it into memory */
	len = strlen(line);
	if ( len > mem_left ) {
	    p = realloc(*buffer, (mem_chunks+1)*CHUNK);
	    if ( p == NULL ) {
		free(*buffer);
		fprintf(stderr, "Out of memory\n");
		return(-1);
	    }
	    *buffer = p;
	    mem_chunks += 1;
	    mem_left += CHUNK;
	}
	memcpy(*buffer+mem_sofar, line, len);
	mem_sofar += len;
	mem_left  -= len;

	continue;
    } /* while */    
    
    return(0);
}


/*
 * Take a buffer containing a PGP package and armor it.
 * All OK : return 0;
 * Not a PGP packet or internal prblem, return -1;
 */

int
armor_buffer(u_char	*buf,
	     int	buf_len,
	     char	**a_buf,
	     int	*a_len)
{
    char	*what;
    char	*tmp;
    u_char	*p;
    int		mem_sofar, mem_left, mem_chunks;
    int		off;
    int		remain;
    int		i,j;
    u_char	t_arr[DATALEN+3];
    u_char	crcbuf[3];
    u_long	crc;

    assert(buf != NULL);
    assert(buf_len > 2); 
    assert(is_ctb(*buf));
    assert(packet_length(buf) <= buf_len);

    switch ( (*buf & CTB_TYPE_MASK) >> 2) {
    case CTB_CERT_SECKEY_TYPE:
	what = "SECRET KEY BLOCK-----";
	break;
    case CTB_CERT_PUBKEY_TYPE:
    case CTB_SKE_TYPE:
	what = "PUBLIC KEY BLOCK-----";
	break;
    default :
	what = "MESSAGE-----";
	break;
    }

    tmp = calloc(CHUNK, 1);
    if ( tmp == NULL ) {
	fprintf(stderr, "Out of memory\n");
	return(-1);
    }
    mem_chunks = 1;
    tmp[0] = NULL;

    strcat(tmp, "-----BEGIN PGP ");
    strcat(tmp, what);
    strcat(tmp, "\n");
    strcat(tmp, PGP_VERSION);
    strcat(tmp, "\n");
    strcat(tmp, PGP_COMMENT);
    strcat(tmp, "\n");
    strcat(tmp, "Comment: created by ");
    strcat(tmp, VERSION);
    strcat(tmp, "\n");
    /* End of headers */
    strcat(tmp, "\n");
    i = strlen(tmp);
    mem_sofar = i;
    mem_left  = CHUNK - i;

    init_crc();
    crc = CRCINIT;
    crc = crcbytes( buf, buf_len, crc);

    off = 0;
    while (off < buf_len-DATALEN) {
	if (mem_left <= ARMORLINE+1 ) {
	    p = realloc(tmp, (mem_chunks+1)*CHUNK);
	    if ( p == NULL ) {
		free(tmp);
		fprintf(stderr, "Out of memory\n");
		return(-1);
	    }
	    tmp = p;
	    mem_chunks += 1;
	    mem_left += CHUNK;
	}

	for(j=0, i=0; j < DATALEN; j+=3, i+=4)
	    outdec(tmp+mem_sofar+i, buf+off+j, 3);
	off += j;
	mem_sofar += i+1;
	mem_left -= i+1;
	tmp[mem_sofar-1] = '\n';
    }
    /* Anything left ? */
    remain = buf_len % DATALEN;
    if ( remain == 0 )
	remain = DATALEN;
    /* We need memory for the last pice of data, the CRC and the
     * ending string (---END...)
     */
    if (mem_left <= (remain*4/3+1)+(7)+(13+strlen(what))) {
	p = realloc(tmp, (mem_chunks+1)*CHUNK);
	if ( p == 0 ) {
	    free(tmp);
	    fprintf(stderr, "Out of memory\n");
	    return(-1);
	}
	tmp = p;
	mem_chunks += 1;
	mem_left += CHUNK;
    }
    memset(t_arr, 0, sizeof(t_arr));
    memcpy(t_arr, buf + off, remain);
    for(i=0, j=mem_sofar; i < remain-3; i += 3, j+= 4)
	outdec(tmp+j, t_arr+i, 3);
    outdec(tmp+j, t_arr+i, remain-i);
    tmp[j+4] = '\n';
    mem_sofar = j+5;
    
    /* Then the CRC */
    crcbuf[0] = (crc >> 16) & 0xff;
    crcbuf[1] = (crc >> 8) & 0xff;
    crcbuf[2] = crc & 0xff;
    tmp[mem_sofar] = PAD;
    mem_sofar +=1;
    outdec(tmp+mem_sofar, crcbuf, 3);
    mem_sofar += 4;
    tmp[mem_sofar] = '\n';
    tmp[mem_sofar+1] = 0;
    mem_sofar += 2;

    strcat(tmp, "-----END PGP ");
    strcat(tmp, what);
    strcat(tmp, "\n");

    mem_sofar += 13+strlen(what);

    *a_buf = tmp;
    *a_len = mem_sofar;
    return(0);
}
