(***********************************************************************)
(*                                                                     *)
(*                               Ledit                                 *)
(*                                                                     *)
(*       Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt      *)
(*                                                                     *)
(*  Copyright 1997 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* $Id: go.ml,v 1.1 2002/12/26 08:46:01 weis Exp $ *)

#open "ledit";;
#open "sys";;

let version = "1.6";;

let usage () =
  prerr_string "Usage: ";
  prerr_string command_line.(0);
  prerr_endline " [options] [comm [args]]";
  prerr_endline " -h file : history file";
  prerr_endline " -x  : don't remove old contents of history";
  prerr_endline " -l len : line max length";
  prerr_endline " -v : prints ledit version and exit";
  prerr_endline "Exec comm [args] as child process";
  exit 1
;;

let get_arg i = if i >= vect_length command_line then usage ()
 else command_line.(i);;

let histfile = ref "";;
let trunc = ref true;;
let comm = ref "cat";;
let args = ref [| "cat" |];;

let rec arg_loop i =
  if i < vect_length command_line then
    arg_loop
      (match command_line.(i) with
       | "-h" -> histfile := get_arg (i + 1); i + 2
       | "-l" ->
           let x = get_arg (i + 1) in
           begin try set_max_len (int_of_string x) with
             _ -> usage ()
           end;
           i + 2
       | "-x" -> trunc := false; i + 1
       | "-v" ->
           printf__printf "Ledit version %s\n" version; flush stdout; exit 0
       | _ ->
           let i = if command_line.(i) = "-c" then i + 1 else i in
           if i < vect_length command_line then
             begin
               comm := command_line.(i);
               args := sub_vect command_line i (vect_length command_line - i);
               vect_length command_line
             end
           else vect_length command_line)
in
arg_loop 1;;

#open "unix";;

let string_of_signal =
  function
  | 2 -> "Interrupted"
  | 3 -> "Quit"
  | 10 -> "Bus error"
  | 11 -> "Segmentation fault"
  | x -> "Signal " ^ string_of_int x
;;

let rec read_loop () =
  begin try
    match input_char std_in with
    | `\n` -> print_newline ()
    | x -> print_char x
  with
    Break -> ()
  end;
  read_loop ()
;;

let go () =
  let (id, od) = pipe () in
  let pid = fork () in
  if pid < 0 then failwith "fork"
  else if pid > 0 then
    begin
      dup2 od stdout;
      close id;
      close od;
      set_son pid;
      signal unix__SIGCHLD
        (Signal_handle
           (fun _ ->
              match snd (waitpid [WNOHANG] pid) with
              | WSIGNALED (sign, _) ->
                  prerr_endline (string_of_signal sign);
                  flush std_err;
                  raise End_of_file
              | _ -> raise End_of_file));
      try
        if !histfile <> "" then open_histfile !trunc !histfile;
        catch_break true;
        read_loop ();
        if !histfile <> "" then close_histfile ();
        ()
      with
      | x ->
          signal unix__SIGCHLD Signal_ignore;
          begin
           try close stdout; let _ = wait () in () with
           | Unix_error (_, _, _) -> ()
          end;
          match x with
          | End_of_file -> ()
          | _ -> prerr_string "(ledit) "; flush std_err; raise x
    end
  else
    begin
      dup2 id stdin; close id; close od; execvp !comm !args; failwith "execv"
    end
;;

let handle f a =
  try f a with
  | unix__Unix_error (code, fname, param) ->
      printf__eprintf "Unix error: %s\nOn function %s %s\n"
        (unix__error_message code) fname param;
      flush std_err;
      exit 2
  | e -> printexc__f raise e
;;

handle go ();;

(*printexc__f go ();;*)
