(***********************************************************************)
(*                                                                     *)
(*                           Caml Light                                *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*                                                                     *)
(*  Copyright 2000 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  Distributed only by permission.                   *)
(*                                                                     *)
(***********************************************************************)

(* [save_image filename image] saves an image from the graphics
   library as a PPM image in file [filename].
   [load_image filename] loads a PPM image stored in file [filename] as
   a graphics image. *)

(* Loading and saving images in portable format: PPM, PGM, and PBM.

PPM: portable pixmap (pixels (picture element) map).
PGM: portable greymap (grey scale map).
PBM: portable bitmap (binary digit map (black & white images)).

*)
#open "pixmap";;

value load_pixmap : string -> pixmap;;
 (* [load_pixmap filename] reads the image contained in the file [filename],
    and returns an array of integers (representing the RGB colors of pixels).
    Assumes that the image is stored in PPM (Portable Pixmap) or PGM
    (Portable Greymap), or PBM (Portable Bitmap) formats.
    PPM format is 24 bits per pixel, and both ASCII and raw encoding
    are supported.
    PGM format is 8 bits per pixel, and both ASCII and raw encoding
    are supported.
    PBM is 1 bit per pixel, and both ASCII and raw encoding
    are supported.
    If a PPM image is followed by a PBM image in the same file, the
    PBM image is supposed to be the transparency mask of the first
    image : pixels set to [1] in the mask receive the [transp] color
    into the resulting image (see [save_pixmap] below).
 *)

value save_pixmap : string -> pixmap -> unit;;
 (* [save_pixmap filename img] stores image [img], on file [filename].
    Image is supposed to be represented as an array of colors in the
    spirit of images of the graphics library.
    The image is saved as a portable pixmap, in raw encoding mode.
    If the image contains transparent pixels (i.e. pixels with color
    [graphics__transp]), an extra transparency mask is saved 
    in the file [filename], as a PBM image. (See [load_pixmap] above.)
 *)

value save_bitmap : string -> pixmap -> unit;;
value load_bitmap : string -> pixmap;;
 (* Same as above for portable bitmaps. *)

value load_ppm : string -> pixmap;;
value save_ppm : string -> pixmap -> unit;;
 (* Equivalent to [load_pixmap] and [save_pixmap]. *)

value load_grimage : string -> grimage__image;;
value save_grimage : string -> grimage__image -> unit;;

value load_image : string -> graphics__image;;
 (* [load_image filename] loads the PPM image in file [filename] as an
    image of the graphics library. *)

value save_image : string -> graphics__image -> unit;;
 (* [save_image filename img] saves [img] as a PPM image in file [filename]. *)
