(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Franois Pessaux, projet Cristal, INRIA Rocquencourt     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*            Jun Furuse, projet Cristal, INRIA Rocquencourt           *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* Generic image format. Used to read and write image files, in *)
(* various proprietary formats. *)

#open "pixmap";;

type rgb = {r : int; g : int; b : int};;
 (* Colormap items : R, G, B values. *)

type colormap == rgb vect;;
 (* Color maps. *)

type bpp8descr = {
 data : string ;
 colormap : colormap ;
 transparent : int
 };;
 (* 8 bits per pixels images: each pixel of the image is represented by
    its color, a byte of the [data] field string, which is an index
    to the color map vector, that contains the set of colors used in the
    image. *)

type bitmap =
| Bpp8 of bpp8descr
| Bpp24 of string;;
  (* Bitmap values are mainly big strings, that can be of two flavours
     - 8 bits per pixels bitmaps: these one use a color map, each byte
       of the string is an index in the color map, that encodes the color
       of the corresponding pixel.
     - 24 bits per pixel or true color: no color map, the color of each
       pixel is given as its R, G, B values. Hence, bytes in the string
       must be taken 3 by 3 to give RGB values of pixels. *)

type image = {
 bitmap : bitmap ;
 width : int ;
 height : int
 };;
  (* Generic image type used in this library. *)

(** Interface of images of this library to images of the Caml
   [Graphics] library *)

value pixmap_of_grimage : grimage__image -> pixmap;;
 (* Convert an [grimage__image] value to an array of colors. *)
value grimage_of_pixmap : pixmap -> grimage__image;;
 (* Convert a pixmap to a [grimage] image. *)

value image_of_grimage : grimage__image -> graphics__image;;
 (* Convert an [grimage__image] value to a graphic image. *)
value grimage_of_image : graphics__image -> grimage__image ;;
 (* The converse convertion. *)

value draw_grimage : grimage__image -> int -> int -> unit;;
 (* Draw an image from this library (i.e. a [grimage__image] value)
    into the graphic window at the specified coordinates. *)

value load_ppm_grimage : string -> grimage__image;;
  (* Loads a  ppm image as a Caml value of type [grimage__image]. *)
value load_bmp_grimage : string -> grimage__image;;
  (* Loads a  bmp image as a Caml value of type [grimage__image]. *)
value save_bmp_grimage : string -> grimage__image -> unit;;
  (* Saves a [grimage__image] Caml value as an image in the bmp format. *)
value save_ppm_grimage : string -> grimage__image -> unit;;
  (* Saves a [grimage__image] Caml value as an image in the ppm format. *)
