(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Pierre Weis, projet Cristal, INRIA Rocquencourt          *)
(*            Franois Pessaux, projet Cristal, INRIA Rocquencourt     *)
(*            Jun Furuse, projet Cristal, INRIA Rocquencourt           *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

#open "graphics";;

let pixmap_of_grimage = function
 {bitmap = bm ; width = w ; height = h } ->
  match bm with
  | Bpp8 {data = b; colormap = colormap; transparent = t} -> 
     let cmap = map_vect (fun {r = r; g = g; b = b} -> rgb r g b) colormap in
     init_vect h (fun i ->
     init_vect w (fun j -> cmap.(int_of_char b.[i * w + j])))
  | Bpp24 b ->
     init_vect h (fun i ->
     init_vect w (fun j ->
       let c = 3 * (i * w + j) in
       let r = int_of_char b.[c] in
       let g = int_of_char b.[c + 1] in
       let b = int_of_char b.[c + 2] in
       rgb r g b));;

#open "color";;

let rgb_of_color c =
 let r = red_of_color c
 and g = green_of_color c
 and b = blue_of_color c in
 char_of_int r, char_of_int g, char_of_int b;;

(* Trivial conversion: should analyse the pixmap to create a grimage
   with a [Bpp8] bitmap when it is possible. *)
let grimage_of_pixmap img =
 let h = vect_length img in
 let w = if h = 0 then 0 else vect_length img.(0) in
 let bitmap =
  if h = 0 then "" else begin
   let s = create_string (3 * h * w) in
   let k = ref 0 in
   for i = 0 to h - 1 do
    let line = img.(i) in
    for j = 0 to w - 1 do
     let r, g, b = rgb_of_color line.(j) in
     s.[!k] <- r; incr k;
     s.[!k] <- g; incr k;
     s.[!k] <- b; incr k;
    done;
   done;
   s end in
 { bitmap = Bpp24 bitmap;
   width = w; height = h };;

let grimage_of_image img = grimage_of_pixmap (graphics__dump_image img);;
let image_of_grimage img = graphics__make_image (pixmap_of_grimage img);;

let draw_grimage img x y = graphics__draw_image (image_of_grimage img) x y;;

let load_ppm_grimage = ppm__load_grimage;;
let load_bmp_grimage = bmp__load_grimage;;

let save_ppm_grimage = ppm__save_grimage;;
let save_bmp_grimage = bmp__save_grimage;;
