(* lexer.mll *)

{
#open "parser";;

(* A little hash table to recognize keywords *)
let kwtable = (hashtbl__new 127: (string, token) hashtbl__t);;
do_list (fun (k, t) -> hashtbl__add kwtable k t)
  [ "begin", BEGIN; "end", END; "var", VAR; "print", PRINT;
    "if", IF; "then", THEN; "else", ELSE; "while", WHILE; 
    "do", DO; "proc", PROC; "ref", REF ];;

let lookup s = try hashtbl__find kwtable s with Not_found -> IDENT s;;

let lineno = ref 1;;
}

rule token = parse
  [`A`-`Z``a`-`z`][`A`-`Z``a`-`z``0`-`9``_`]*
			{ lookup (get_lexeme lexbuf) }
| [`0`-`9`]+		{ NUM (int_of_string (get_lexeme lexbuf)) }
| "+"			{ PLUS }
| "-"			{ MINUS }
| "*"			{ TIMES }
| "/"			{ DIVIDE }
| "<"			{ LESS }
| "<="			{ LEQ }
| "="			{ EQUAL }
| "<>"			{ NEQ }
| ">="			{ GEQ }
| ">"			{ GREATER }
| "("			{ LPAR }
| ")"			{ RPAR }
| ","			{ COMMA }
| ";"			{ SEMI }
| ":="			{ ASSIGN }
| [` ``\t`]+		{ token lexbuf }
| `\n`			{ incr lineno; token lexbuf }
| _			{ failwith ("bad token on line " 
					^ string_of_int !lineno) }
| eof			{ EOF }
;;
