#open "windows";;
#open "camlwin";;

#open "g_global";;
#open "g_button";;
#open "g_string";;
#open "g_scroll";;




let debug = false;;



let make_text Text =
{
  tx_window=Text.t_window;
  tx_top=Text.t_top;
  tx_left=Text.t_left;
  tx_width=Text.t_width;
  tx_height=Text.t_height;
  tx_name=Text.t_name;
  tx_cursor=0;
  tx_line=0;
  tx_1st_line=0;
  tx_1st_row=0;
  tx_state=Text.t_state;
  tx_modified=false;
  tx_scroll=Text.t_scroll;
  tx_vscroll=0;
  tx_hscroll=0;
  tx_s_select=(0,0);
  tx_e_select=(0,0)
};;



let gr_line_number String =
  let rec gr_line_number_loop Pos Line =
    if Pos <= -1
    then (Line + 1)
    else if nth_char String Pos = `\n`
         then  gr_line_number_loop (Pos - 1) (Line + 1)
         else gr_line_number_loop (Pos - 1) Line
  in
    let n = ((string_length String) - 1) in
      gr_line_number_loop n 0
;;





let gr_1st_line String =
  let rec gr_1st_line_loop String Pos =
    let n = string_length String in
      if n = 0
      then ""
      else if Pos >= n
           then String
           else if (nth_char String Pos)=`\n`
                then (sub_string String 0 Pos)
                else gr_1st_line_loop String (Pos + 1)
  in
    gr_1st_line_loop String 0
;;




let gr_without_1st_line String =
  let rec gr_without_1st_line_loop String Pos =
    let n = string_length String in
      if n = 0
      then ""
      else if Pos >=n 
           then ""
           else if nth_char String Pos = `\n`
                then (sub_string String (Pos + 1) (n - Pos -1))
                else gr_without_1st_line_loop String (Pos + 1)
  in
    gr_without_1st_line_loop String 0
;;




let replace_tab String =
  let rec replace_loop pos String len =
    if pos>=(len-1)
    then String
    else if nth_char String pos = `\t`
         then 
         (
           let str_begin = sub_string String 0 pos 
           and str_end   = sub_string String (pos+1) (len-pos-1) in
             replace_loop (pos+5) (str_begin^"    "^str_end) (len+3)
         )
         else replace_loop (pos+1) String len
  in
    replace_loop 0 String (string_length String)
;;
  







let lines_of_string String =
  let rec lines_of_string_loop Text Vect Pos =
    if Pos=((vect_length Vect) - 1)
    then
    (
      Vect.(Pos) <- gr_1st_line Text;
      Vect
    )
    else
    (
      Vect.(Pos) <- gr_1st_line Text;
      lines_of_string_loop (gr_without_1st_line Text) Vect (Pos + 1)
    )
  in
    let n=(gr_line_number String) in
      if n=0
      then [| |]
      else 
      (
        let lines = lines_of_string_loop String (make_vect n "") 0 in
          let nb_lines = vect_length lines in
            for i = 0 to (nb_lines - 1) do
              lines.(i) <- replace_tab lines.(i)
            done;
        lines
      )
;;





let string_of_lines Lines =
  let rec string_of_lines_loop Vect String Pos =
    if Pos=((vect_length Vect) - 1)
    then String ^ Vect.(Pos) ^ "\n"
    else string_of_lines_loop Vect (String ^ Vect.(Pos) ^ "\n") (Pos + 1)
  in
    let n=(vect_length Lines) in
      if n=0
      then ""
      else string_of_lines_loop Lines "" 0
;;


let lines_del_nth Lines n =
  let before=(sub_vect Lines 0 n) and
      after =(sub_vect Lines (n+1) ((vect_length Lines) - n - 1)) in
    concat_vect before after
;;

let lines_add_nth Lines n Add=
  let before=(sub_vect Lines 0 n) and
      after =(sub_vect Lines n ((vect_length Lines) - n)) in
    concat_vect before (concat_vect [| Add |] after)
;;






(***************************************************************************)
(*                                                                         *)
(*                                                                         *)
(***************************************************************************)
let gr_text_get_selected_area Text =
  if Text.tx_s_select = Text.tx_e_select
  then ""
  else
  (
    let x1 = fst Text.tx_s_select
    and y1 = snd Text.tx_s_select
    and x2 = fst Text.tx_e_select
    and y2 = snd Text.tx_e_select in
      let part=sub_vect Text.tx_name y1 (y2-y1+1) in
        let str=string_of_lines part in
          let len=string_length str
          and after=(string_length Text.tx_name.(y2))-x2 in
              (* erase begin of 1st line and end of last line. *)
            sub_string str x1 (len-x1-after-1)
  )
;;



let gr_text_add_text Text String =
  if string_length String > 0
  then
  (
    let Line=Text.tx_name.(Text.tx_line)
    and line=Text.tx_line
    and pos=Text.tx_cursor in
      let len=string_length Line in
        let b_line=sub_string Line 0 pos
        and e_line=sub_string Line pos (len-pos) in
          let add=lines_of_string (b_line ^ String ^ e_line) in
            let add_len=vect_length add
            and b_vect = sub_vect Text.tx_name 0 line
            and e_vect = sub_vect Text.tx_name (line+1) 
                                  ((vect_length Text.tx_name)-line-1) in
              Text.tx_name <- concat_vect b_vect (concat_vect add e_vect);
              Text.tx_s_select <- (pos,line);
              Text.tx_e_select <- (string_length add.(add_len-1)-len+pos,
                                   line+add_len-1);
              Text.tx_line <- snd Text.tx_e_select;
              Text.tx_cursor <- fst Text.tx_e_select;
              Text.tx_modified <- true
  )
;;



let gr_text_del_selected_area Text =
  if not Text.tx_e_select = Text.tx_s_select
  then
  (
    let x1 = fst Text.tx_s_select
    and y1 = snd Text.tx_s_select
    and x2 = fst Text.tx_e_select
    and y2 = snd Text.tx_e_select
    and nb_lines = vect_length Text.tx_name
    and Lines = Text.tx_name in
      let b = sub_string Lines.(y1) 0 x1
      and e = sub_string Lines.(y2) x2 ((string_length Lines.(y2))-x2) in
        let before = (sub_vect Lines 0 y1)
        and after = if y2 < (nb_lines-1)
                    then (sub_vect Lines (y2+1) (nb_lines - y2 - 1))
                    else [| |] in
          Text.tx_name <- concat_vect before 
                                      (concat_vect [| (b ^ e) |] after);
          Text.tx_s_select <- Text.tx_e_select;
          Text.tx_modified <- true;
          Text.tx_line <- Text.tx_line-(y2-y1)
  )
;;





(***************************************************************************)
(*                                                                         *)
(*                                                                         *)
(***************************************************************************)
let gr_text_width Text =
  match Text.tx_scroll with
    Vscroll  -> (to_real_coord Text.tx_width) - scroll_width - short_space
  | Noscroll -> to_real_coord Text.tx_width
  | Dscroll  -> (to_real_coord Text.tx_width) - scroll_width - short_space
;;


let gr_text_height Text =
  match Text.tx_scroll with
    Vscroll  -> to_real_coord Text.tx_height
  | Noscroll -> to_real_coord Text.tx_height
  | Dscroll  -> (to_real_coord Text.tx_height) - scroll_width - short_space
;;




(***************************************************************************)
(*                                                                         *)
(*                                                                         *)
(***************************************************************************)
let std_gr_draw_text_cursor Text Color =
  if debug
  then
  (
    print_string "std_gr_draw_text_cursor";
    print_newline ()
  );
  let top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left
  and width = gr_text_width Text
  and height = gr_text_height Text in
    set_clip_area (left + 1) (top - (height - 1)) (width - 2) (height - 2);
    if Text.tx_cursor >= Text.tx_1st_row
    then
    (
      let line_before=sub_string Text.tx_name.(Text.tx_line)
                                 Text.tx_1st_row
                                 (Text.tx_cursor-Text.tx_1st_row) in
        let cursor_pos = fst (text_size (line_before))
        and cursor_height =line_height+short_space in
          (
            match Text.tx_state with
              Edited ->
                (
                  set_color Color;
                  let x = left + short_space + cursor_pos - 1
                  and y = top - short_space - 
                          (cursor_height * (Text.tx_line-Text.tx_1st_line)) in
                    moveto x y;
                    lineto x (y - short_space - line_height)
                )
            | Editable ->
                (
                  set_color grey;
                  let x = left + short_space + cursor_pos - 1
                  and y = top - short_space - 
                          (cursor_height * (Text.tx_line-Text.tx_1st_line)) in
                    moveto x y;
                    lineto x (y - short_space - line_height)
                )
            | _      -> ()
          )
    );
    set_clip_area 0 0 (screen_width()) (screen_height ())
;;



let open_gr_draw_text_cursor Text Color =
  if debug
  then
  (
    print_string "open_gr_draw_text_cursor";
    print_newline ()
  );
  let width = gr_text_width Text
  and height = gr_text_height Text
  and top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left in
    set_clip_area (left + 1) (top - (height - 1)) (width - 2) (height - 2);

    set_color Color;
    (
      match Text.tx_state with
        Edited ->
          if Text.tx_cursor >= Text.tx_1st_row
          then
          (
            let line_before=sub_string Text.tx_name.(Text.tx_line)
                                       Text.tx_1st_row
                                       (Text.tx_cursor-Text.tx_1st_row) in
              let cursor_pos = fst (text_size (line_before))
              and cursor_height =line_height+short_space in
               draw_open_cursor (left + short_space + cursor_pos - 1)
                                (top - (cursor_height * 
                                       (Text.tx_line - Text.tx_1st_line+1)))
          )
      | _      -> ()
    );
    set_clip_area 0 0 (screen_width()) (screen_height ())
;;




let gr_draw_lines Lines Coord High Begin_line Begin_row =
  if debug
  then
  (
    print_string "gr_draw_lines";
    print_newline ()
  );
  let dy=(line_height+short_space) in
    let n=((High-2*short_space) / dy) and
        nb_lines=((vect_length Lines) - 1)in
      for i=0 to n do
      (
       moveto (Coord.x + short_space)
              (Coord.y - (i + 1) * dy - short_space);
       let nu_line=(Begin_line + i) in
          if nu_line<=nb_lines
          then
          (
            let len=string_length Lines.(nu_line) in
              if Begin_row<len
              then draw_string (sub_string Lines.(nu_line)
                                           Begin_row
                                           (len-Begin_row)
                               )
          )
      )
      done
;;


let update_cursor_text_line Text =
  if debug
  then
  (
    print_string "update_cursor_text_line";
    print_newline ()
  );
  if Text.tx_window.win_state = Created
  then
  (
    set_draw_window Text.tx_window.win_id;
    let top  = to_real_coord Text.tx_top
    and left = to_real_coord Text.tx_left
    and width = (gr_text_width Text) - 2*short_space
    and height = gr_text_height Text in
      set_clip_area (left + short_space)
                    (top - (height - short_space))
                    (width - 2*short_space)
                    (height - 2*short_space);
      let dy = (line_height+short_space) in
        let n = ((height - 2*short_space) / dy)
        and nb_lines = max 0 ((vect_length Text.tx_name) - 1)
        and x = (left + short_space)
        and y = (top - short_space -
               (Text.tx_line-Text.tx_1st_line + 1) * dy )
        and nb_char = string_length Text.tx_name.(Text.tx_line) in
          if not Text.tx_e_select = Text.tx_s_select
          then
          (
            if Text.tx_line = snd Text.tx_e_select
              then if nb_char < (fst Text.tx_e_select)
                     then Text.tx_e_select <- (Text.tx_line,nb_char);
            if Text.tx_line = snd Text.tx_s_select
              then if nb_char<(fst Text.tx_s_select)
                     then Text.tx_s_select <- (Text.tx_line,nb_char)
          );
          if Text.tx_line <= nb_lines
          then
          (
            set_color white;
            fill_rect x y width dy;
            moveto x y;
            set_color black;
            let len = string_length Text.tx_name.(Text.tx_line) in
              if Text.tx_1st_row < len
              then draw_string (sub_string Text.tx_name.(Text.tx_line)
                                           Text.tx_1st_row
                                           (len-Text.tx_1st_row)
                               );


            match (Text.tx_e_select=Text.tx_s_select,
                   (snd Text.tx_s_select=Text.tx_line),
                   (snd Text.tx_e_select=Text.tx_line)) with
              (true, _, _)          -> (* no selected area *)
                ()

            | (false, false, false) -> (* all the line selected *)
                set_color black;
                fill_rect x y width dy;
                moveto x y;
                set_color white;
                let len=string_length Text.tx_name.(Text.tx_line) in
                  if Text.tx_1st_row<len
                    then draw_string (sub_string Text.tx_name.(Text.tx_line)
                                      Text.tx_1st_row
                                      (len-Text.tx_1st_row)
                                     )

            | (false, false, true)  -> (* begining of line selected *)
                let pos=fst Text.tx_e_select
                and len=string_length Text.tx_name.(Text.tx_line) in
                  let select_string=sub_string Text.tx_name.(Text.tx_line)
                                               0
                                               pos in
                    let select_width=fst (text_size select_string) in
                      fill_rect x y select_width dy;
                      moveto x y;
                      set_color white;
                      draw_string select_string

            | (false, true, false)  -> (* end of line selected *)
                let pos=fst Text.tx_s_select
                and len=string_length Text.tx_name.(Text.tx_line) in
                  let select_string=sub_string Text.tx_name.(Text.tx_line)
                                               pos
                                               (len-pos)
                  and unselect_string=sub_string Text.tx_name.(Text.tx_line)
                                                 0
                                                 pos in
                    let unselect_width=fst (text_size unselect_string) in
                      fill_rect (x+unselect_width) y
                                (width-unselect_width) dy;
                      moveto (x+unselect_width) y;
                      set_color white;
                      draw_string select_string

            | (false, true, true)   -> (* midle of line selected *)
                let pos1=fst Text.tx_s_select
                and pos2=fst Text.tx_e_select
                and len=string_length Text.tx_name.(Text.tx_line) in
                  let select_string=sub_string Text.tx_name.(Text.tx_line)
                                               pos1
                                               (pos2-pos1)
                  and unselect_string=sub_string Text.tx_name.(Text.tx_line)
                                                 0
                                                 pos1 in
                    let unselect_width=fst (text_size unselect_string)
                    and select_width  =fst (text_size select_string) in
                      fill_rect (x+unselect_width) y select_width dy;
                      moveto (x+unselect_width) y;
                      set_color white;
                      draw_string select_string

          )
  )
;;



let gr_draw_selected Text =
  if debug
  then
  (
    print_string "gr_draw_selected_text";
    print_newline ()
  );
  set_draw_window Text.tx_window.win_id;
  let tmp=Text.tx_line
  and nb_lines=vect_length Text.tx_name in
    let y1=snd Text.tx_s_select
    and y2=snd Text.tx_e_select in
      if y2<nb_lines & y1<nb_lines
      then
      (
        for i=y1 to y2 do
        (
          Text.tx_line <- i;
          update_cursor_text_line Text
        )
        done;
        Text.tx_line <- tmp
      )
;;











let std_draw_gr_text Text =
  let top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left
  and width = gr_text_width Text
  and height = gr_text_height Text in
    draw_button left top width height Down;

    set_color white;
    set_clip_area (left + short_space)
                  (top - (height - short_space))
                  (width - 2*short_space)
                  (height - 2*short_space);
    fill_rect (left + short_space)
              (top - (height - short_space))
              (width - 2*short_space)
              (height - 2*short_space);

    set_color black;
    gr_draw_lines Text.tx_name
                  { x=left; y=top }
                  height 
                  Text.tx_1st_line
                  Text.tx_1st_row;
    gr_draw_selected Text;
    std_gr_draw_text_cursor Text black
;;



let open_col_draw_gr_text Text =
  let top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left
  and width = gr_text_width Text
  and height = gr_text_height Text in
    set_color white;
    set_clip_area left (top - height) width height;

    fill_rect left (top - height) width height;

    set_color black;
    gr_draw_lines Text.tx_name
                  { x=left; y=top }
                  height
                  Text.tx_1st_line
                  Text.tx_1st_row;
    gr_draw_selected Text;
    open_gr_draw_text_cursor Text black
;;



let open_mono_draw_gr_text Text =
  let top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left
  and width = gr_text_width Text
  and height = gr_text_height Text in
    set_color white;
    fill_rect left (top - height) width height;

    set_color black;
    draw_rect left (top - height) width height;

    set_clip_area left (top - height) width height;

    gr_draw_lines Text.tx_name
                  { x=left; y=top }
                  height
                  Text.tx_1st_line
                  Text.tx_1st_row;
    gr_draw_selected Text;
    open_gr_draw_text_cursor Text black
;;




let gr_draw_text_cursor Text Color =
  if Text.tx_window.win_state=Created
  then
  (
    if Text.tx_cursor <= string_length Text.tx_name.(Text.tx_line)
    then
    (
      set_draw_window Text.tx_window.win_id;
      match (get_gr_look()) with
        Std_color       -> std_gr_draw_text_cursor Text Color
      | Std_mono        -> std_gr_draw_text_cursor Text Color
      | Open_look_color -> open_gr_draw_text_cursor Text Color
      | Open_look_mono  -> open_gr_draw_text_cursor Text Color
      | _               -> std_gr_draw_text_cursor Text Color
    )
  )
;;


let draw_gr_text Text =
   if Text.tx_window.win_state=Created
  then
  (
    set_draw_window Text.tx_window.win_id;
    let top  = to_real_coord Text.tx_top
    and left = to_real_coord Text.tx_left
    and width= to_real_coord Text.tx_width
    and height=to_real_coord Text.tx_height in

    let n=vect_length Text.tx_name in
    if Text.tx_line >= n
    then
    (
      Text.tx_line <- 0;
      Text.tx_1st_line <- 0;
      Text.tx_1st_row <- 0;
      Text.tx_cursor <- 0;
      Text.tx_vscroll <- 0;
      Text.tx_hscroll <- 0
    );
    if Text.tx_line < 0
      then Text.tx_line <- 0;

    if Text.tx_1st_line >= n
      then Text.tx_1st_line <- max 0 (n-5);
    if Text.tx_1st_line < 0
      then Text.tx_1st_line <- 0;


    let len=string_length Text.tx_name.(Text.tx_line) in
      if Text.tx_cursor>len
        then Text.tx_cursor <- len;
    if Text.tx_cursor<0
      then Text.tx_cursor <- 0;

    (
      match (get_gr_look()) with
        Std_color       -> std_draw_gr_text Text
      | Std_mono        -> std_draw_gr_text Text
      | Open_look_color -> open_col_draw_gr_text Text
      | Open_look_mono  -> open_mono_draw_gr_text Text
      | _               -> std_draw_gr_text Text
    );
    (
      match Text.tx_scroll with
        Noscroll -> ()

      | Vscroll  -> draw_vscrollbar (left-scroll_width+width) top
                                    height Text.tx_vscroll

      | Dscroll  -> draw_vscrollbar (left+width-scroll_width) top
                                    (height-scroll_width)
                                    Text.tx_vscroll;
                    draw_hscrollbar left (top+scroll_width-height)
                                    (width-scroll_width)
                                    Text.tx_hscroll
    )
  )
;;








let CursorPos Text x y =
  if debug
  then
  (
    print_string "CursorPos";
    print_newline ()
  );
  let top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left in

  let dy = line_height + short_space
  and n2 = vect_length Text.tx_name in
    let line = (min (n2-1) (Text.tx_1st_line+(top-2*short_space-y)/dy)) in
      let String = Text.tx_name.(line) in
        let n1 = (string_length String) in
          if n1 > Text.tx_1st_row
          then
          (
            let text = (sub_string String
                                   Text.tx_1st_row
                                   (n1 - Text.tx_1st_row)) in
              let Left = left+short_space
              and n = (string_length text) in
                let dot_length = max 0 (x - Left) in
                  ((Text.tx_1st_row +gr_string_pos_loop text dot_length 0 n),
                   line)
          )
          else (n1,line)
;;


let update_scroll Text =
  let top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left in

  let nb_lines = ((vect_length Text.tx_name)-1) in
    let height = gr_text_height Text
    and dy = (line_height+ short_space) in
      let nb_display=((height / dy)-1) in
         Text.tx_vscroll <- ((Text.tx_1st_line * (scroll_course height))/
                             (max 1 (nb_lines - nb_display)));
         Text.tx_hscroll <- ((Text.tx_1st_row * (scroll_course height))/
                             256)
;;


let rec gr_edit_text Text Add =
  if debug
  then
  (
    print_string "gr_edit_text: ";
    print_int Add;
    print_newline ()
  );
  let nb_lines=(vect_length Text.tx_name)
  and top  = to_real_coord Text.tx_top
  and left = to_real_coord Text.tx_left
  and height = gr_text_height Text
  and width = gr_text_width Text
  and dy = (line_height+ short_space) in
    let nb_display=((height / dy)-1) in


  let Pos=(Text.tx_cursor) and
      Line=(Text.tx_line) and
      String=(Text.tx_name.(Text.tx_line)) and
      n=(string_length Text.tx_name.(Text.tx_line)) in
    match Add with
        0x008         ->  (* Backspace *)
          if not Pos=0
          then (
            let string_start=(sub_string String 0 (Pos - 1)) and
                string_end=  (sub_string String Pos (n - Pos))in
              gr_draw_text_cursor Text white;
              Text.tx_name.(Text.tx_line) <- (string_start ^ string_end);
              Text.tx_modified <- true;
              gr_edit_text Text Left_key;
              update_cursor_text_line Text;
              gr_draw_text_cursor Text black
               )
          else if Line=0
               then ()
               else (
                 let line_before=(Text.tx_name.(Text.tx_line - 1)) in
                   Text.tx_name.(Text.tx_line - 1) <- line_before ^ String;
                   Text.tx_name <- lines_del_nth Text.tx_name Line;
                   Text.tx_modified <- true;
                   Text.tx_line <- (Line - 1);
                   Text.tx_cursor <- (string_length line_before);
                   draw_gr_text Text
                     )

      | 0x007f         -> (* Del *)
          gr_edit_text Text Del_key

      | 0xff09         -> (* Del *)
          if not Pos = n
          then
            (
              let string_start=(sub_string String 0 Pos)
              and string_end=(sub_string String (Pos+1) (n - Pos-1)) in
                Text.tx_name.(Text.tx_line) <- (string_start ^ string_end);
                Text.tx_modified <- true;
                update_cursor_text_line Text;
                gr_draw_text_cursor Text black
            )


      | 0xff00         ->  (* Cursor to right *)
          if Pos < n
          then
            (              
              gr_draw_text_cursor Text white;
              Text.tx_cursor <- (Pos + 1);
              if Text.tx_1st_row > Text.tx_cursor
              then 
              (
                Text.tx_1st_row <- Text.tx_cursor - 1;
                update_scroll Text;
                draw_gr_text Text
              );
              let str = sub_string String
                                 Text.tx_1st_row
                                 (Text.tx_cursor-Text.tx_1st_row) in
                let len = fst (text_size str) in
                  if len > width
                  then
                  (
                    Text.tx_1st_row <- Text.tx_1st_row + 1;
                    update_scroll Text;
                    draw_gr_text Text
                  );
              gr_draw_text_cursor Text black
            )

      | 0xff01         ->  (* Cursor to left *)
          if Pos > 0
          then
            (
              gr_draw_text_cursor Text white;
              Text.tx_cursor <- (Pos - 1);
              if Text.tx_cursor < Text.tx_1st_row
              then
              (
                Text.tx_1st_row <- Text.tx_cursor;
                update_scroll Text;
                draw_gr_text Text
              );
              gr_draw_text_cursor Text black
            )

      | 0xff02         ->  (* Cursor up *)
          if Line > 0
          then
          (
            gr_draw_text_cursor Text white;
            Text.tx_line <- (Line - 1);
            let p=(string_length Text.tx_name.(Text.tx_line)) in
              if Pos>p
              then Text.tx_cursor <- p;
            if Text.tx_1st_row>Text.tx_cursor
            then
            (
              Text.tx_1st_row <- Text.tx_cursor;
              update_scroll Text;
              draw_gr_text Text
            );
            if Text.tx_1st_line > Text.tx_line
            then
            (
              Text.tx_1st_line <- Text.tx_line;
              update_scroll Text;
              draw_gr_text Text
            );
            gr_draw_text_cursor Text black
          )

      | 0xff03         ->  (* Cursor down *)
          if Line<((vect_length Text.tx_name) - 1)
          then
          (
            gr_draw_text_cursor Text white;
            Text.tx_line <- (Line + 1);
            let p=(string_length Text.tx_name.(Text.tx_line)) in
              if Pos>p
              then Text.tx_cursor <- p;
            if Text.tx_1st_row>Text.tx_cursor
            then
            (
              Text.tx_1st_row <- Text.tx_cursor;
              update_scroll Text;
              draw_gr_text Text
            );
            if (Text.tx_1st_line+nb_display)< Text.tx_line
            then
            (
              Text.tx_1st_line <- Text.tx_line-nb_display;
              update_scroll Text;
              draw_gr_text Text
            );
            gr_draw_text_cursor Text black
          )

      | 0xff04         -> (* Page up*)
          Text.tx_line <- (Line-nb_display);
          Text.tx_1st_line <- (Text.tx_1st_line-nb_display);
          if Text.tx_line <0
          then Text.tx_line <- 0;
          if Text.tx_1st_line < 0
          then Text.tx_1st_line <- 0;
          let p=(string_length Text.tx_name.(Text.tx_line)) in
            if Pos>p
            then Text.tx_cursor <- p;
          if Text.tx_1st_row>Text.tx_cursor
          then Text.tx_1st_row <- Text.tx_cursor;
          update_scroll Text;
          draw_gr_text Text

      | 0xff05         -> (* Page down*)
          Text.tx_line <- (Line+nb_display);
          Text.tx_1st_line <- (Text.tx_1st_line+nb_display);
          if Text.tx_line > nb_lines
          then Text.tx_line <- (nb_lines-1);
          if Text.tx_1st_line > (nb_lines-nb_display)
          then Text.tx_1st_line <- (nb_lines-nb_display-1);
          let p=(string_length Text.tx_name.(Text.tx_line)) in
            if Pos>p
            then Text.tx_cursor <- p;
          if Text.tx_1st_row>Text.tx_cursor
          then Text.tx_1st_row <- Text.tx_cursor;
          update_scroll Text;
          draw_gr_text Text


      | 0xff06         ->  (* Home *)
          gr_draw_text_cursor Text white;
          Text.tx_cursor <- 0;
          if Text.tx_1st_row > 0
          then
          (
            Text.tx_1st_row <- 0;
            update_scroll Text;
            draw_gr_text Text
          );
          gr_draw_text_cursor Text black

      | 0xff07         ->  (* End *)
          gr_draw_text_cursor Text white;
          Text.tx_cursor <- n;
          let str = sub_string String
                                 Text.tx_1st_row
                                 (n-Text.tx_1st_row) in
                let len = fst (text_size str) in
                  if len > width
                  then
                  (
                    Text.tx_1st_row <- Text.tx_1st_row + 1 +
                                       ((len-width) / (fst (text_size "a")));
                    update_scroll Text;
                    draw_gr_text Text
                  );
          gr_draw_text_cursor Text black

      | 0xff20         ->  (* Copy *)
          if not Text.tx_e_select=Text.tx_s_select
          then gr_copy_text (gr_text_get_selected_area Text)

      | 0xff21         ->  (* Cut *)
          if not Text.tx_e_select=Text.tx_s_select
          then
          (
            gr_copy_text (gr_text_get_selected_area Text);
            gr_text_del_selected_area Text;
            Text.tx_modified <- true;
            draw_gr_text Text
          )

      | 0xff22         ->  (* Past *)
          gr_text_add_text Text (gr_past_text ());
          Text.tx_modified <- true;
          draw_gr_text Text

      | 0x000d         ->  (* LF *)
          let string_start=(sub_string String 0 Pos) and
              string_end  =(sub_string String Pos (n - Pos)) in
            (
              Text.tx_name.(Text.tx_line) <- string_start;
              Text.tx_name <- lines_add_nth Text.tx_name
                                               (Line + 1)
                                               string_end;
              Text.tx_cursor <- 0;
              Text.tx_line <- (Line + 1);
              if (Text.tx_1st_line+nb_display) < Text.tx_line
              then Text.tx_1st_line <- Text.tx_line-nb_display;
              Text.tx_1st_row <- 0;
              update_scroll Text;
              Text.tx_modified <- true;
              draw_gr_text Text
            )

      | _              -> (* Ascii character *)
          if Add <0xff & Add >0x1f
          then
            (
              gr_draw_text_cursor Text white;
              let Char = char_of_int Add in
                if Char = `\t`
                then
                (
                  for i = 0 to 3 do
                  (
                    Text.tx_name.(Text.tx_line) <- 
                                     add_char Text.tx_name.(Text.tx_line)
                                              ` `
                                              Text.tx_cursor;
                    gr_edit_text Text Right_key
                  )
                  done
                )
                else
                (
                  Text.tx_name.(Text.tx_line) <- add_char String Char Pos;
                  gr_edit_text Text Right_key
                );
              Text.tx_modified <- true;
              if Text.tx_e_select = Text.tx_s_select
              then update_cursor_text_line Text
              else
              (
                Text.tx_e_select <- Text.tx_s_select;
                draw_gr_text Text
              );
              gr_draw_text_cursor Text black
            )
;;








type local_dir = To_up
               | To_bottom;;

let gr_text_callback_loop Text Event =
  if debug
  then
  (
    print_string "gr_text_callback_loop";
    print_newline ()
  );
  if not Text.tx_e_select=Text.tx_s_select
  then
  (
    Text.tx_s_select <- Text.tx_e_select;
    draw_gr_text Text;
    gr_draw_text_cursor Text white
  );

  let nb_lines=((vect_length Text.tx_name)-1) in
    let top  = to_real_coord Text.tx_top
    and left = to_real_coord Text.tx_left
    and height = gr_text_height Text
    and dy=(line_height+ short_space) in
      let nb_display=((height / dy)-1) in


  let Coord = CursorPos Text Event.mouse_x Event.mouse_y
  and dir = ref To_bottom in
    Text.tx_s_select <- Coord;
    Text.tx_e_select <- Coord;
    Text.tx_cursor <- fst Text.tx_e_select;
    Text.tx_line   <- snd Text.tx_e_select;

  while
  (
    let Event=(get_event([Button_up; Mouse_motion])) in
      let mouse_coord={x=Event.mouse_x; y=Event.mouse_y}
      and Text_area={x1=left;
                     y1=top-(to_real_coord Text.tx_height);
                     x2=left+(to_real_coord Text.tx_width);
                     y2=top} in
        if (inside mouse_coord Text_area) & Event.id_event=Mouse_motion
        then
        (
          let Pos=CursorPos Text Event.mouse_x Event.mouse_y in
            if !dir =To_bottom
            then
            (
              if not Pos=Text.tx_e_select
              then
              (
                let s=snd Text.tx_e_select
                and e=snd Pos in
                  if e>s
                  then
                  (
                    Text.tx_e_select <- Pos;
                    for i=s to e do
                    (
                      Text.tx_line <- i;
                      update_cursor_text_line Text
                    )
                    done
                  );
                  if e<s
                  then
                  (
                    Text.tx_e_select <- Text.tx_s_select;
                    for i=e to s do
                    (
                      Text.tx_line <- i;
                      update_cursor_text_line Text
                    )
                    done
                  );
                Text.tx_e_select <- Pos;
                Text.tx_cursor <- fst Text.tx_e_select;
                Text.tx_line   <- snd Text.tx_e_select;
                if (snd Text.tx_e_select) < (snd Text.tx_s_select) or
                   ( (snd Text.tx_e_select) = (snd Text.tx_s_select) &
                     (fst Text.tx_e_select) < (fst Text.tx_s_select)
                   )
                then
                (
                  dir := To_up;
                  let tmp=Text.tx_e_select in
                    Text.tx_e_select <- Text.tx_s_select;
                    Text.tx_s_select <- tmp;
                    for i=(snd Text.tx_s_select) to (snd Text.tx_e_select)
                    do
                    (
                      Text.tx_line <- i;
                      update_cursor_text_line Text
                    )
                    done
                );
                update_cursor_text_line Text
              )
            )
            else
            (
              if not Pos=Text.tx_s_select
              then
              (
                let s=snd Text.tx_s_select
                and e=snd Pos in
                  if e<s
                  then
                  (
                    Text.tx_s_select <- Pos;
                    for i=s downto e do
                    (
                      Text.tx_line <- i;
                      update_cursor_text_line Text
                    )
                    done
                  );
                  if e>s
                  then
                  (
                    Text.tx_s_select <- Text.tx_e_select;
                    for i=e downto s do
                    (
                      Text.tx_line <- i;
                      update_cursor_text_line Text
                    )
                    done
                  );
                Text.tx_s_select <- Pos;
                Text.tx_cursor <- fst Text.tx_s_select;
                Text.tx_line   <- snd Text.tx_s_select;
                if (snd Text.tx_e_select) < (snd Text.tx_s_select) or
                   ( (snd Text.tx_e_select) = (snd Text.tx_s_select) &
                     (fst Text.tx_e_select) < (fst Text.tx_s_select)
                   )
                then
                (
                  dir := To_bottom;
                  let tmp=Text.tx_e_select in
                    Text.tx_e_select <- Text.tx_s_select;
                    Text.tx_s_select <- tmp;
                    for i=(snd Text.tx_s_select) to (snd Text.tx_e_select)
                    do
                    (
                      Text.tx_line <- i;
                      update_cursor_text_line Text
                    )
                    done
                );
                update_cursor_text_line Text
              )
            )
        )
        else
        ( (* mouse outside *)
          if Event.mouse_y<top-(to_real_coord Text.tx_height)
          then
          (
            gr_edit_text Text Down_key;
            Text.tx_e_select <- (Text.tx_cursor,Text.tx_1st_line+nb_display)
          );
          if Event.mouse_y>top
          then
          (
            gr_edit_text Text Up_key;
            Text.tx_s_select <- (Text.tx_cursor,Text.tx_1st_line)
          )
        );

      Event.button
  )
  do ()
  done;
  gr_draw_text_cursor Text black;
  true;;



let gr_text_callback Text Event =
  let nb_lines=(vect_length Text.tx_name) in
    let top  = to_real_coord Text.tx_top
    and left = to_real_coord Text.tx_left
    and height = gr_text_height Text
    and dy = (line_height + short_space) in
      let nb_display=((height / dy)-1) in
        let scroll_delta=(nb_display*(scroll_course height) /
                          (nb_lines+1)) in

        let scrollevent=
             (
               match Text.tx_scroll with
                 Vscroll  ->
                   let scrollbar={
                         sb_window=Text.tx_window;
                         sb_left=left + (to_real_coord Text.tx_width) - 
                                  scroll_width;
                         sb_top=top;
                         sb_length=Text.tx_height;
                         sb_pos=Text.tx_vscroll;
                         sb_dir=Vertical;
                         sb_delta=scroll_delta;
                         sb_callback= do_nothing
                         } in
                     if (gr_vscrollbar_callback scrollbar Event)
                     then
                       (
                         Text.tx_vscroll <- scrollbar.sb_pos;
                         Text.tx_1st_line <- ((scrollbar.sb_pos * 
                                              (nb_lines - nb_display)) /
                                              (scroll_course height));
                         draw_gr_text Text;
                         true
                       )
                     else false
               | Noscroll -> false
               | Dscroll  ->
                   let scroll1={
                         sb_window=Text.tx_window;
                         sb_left=left + (to_real_coord Text.tx_width) - 
                                 scroll_width;
                         sb_top=top;
                         sb_length=(to_real_coord Text.tx_height)-scroll_width;
                         sb_pos=Text.tx_vscroll;
                         sb_dir=Vertical;
                         sb_delta=scroll_delta;
                         sb_callback= do_nothing
                         }
                   and scroll2={
                         sb_window=Text.tx_window;
                         sb_left=left;
                         sb_top = top - (to_real_coord Text.tx_height) + 
                                  scroll_width;
                         sb_length = (to_real_coord Text.tx_width) - 
                                     scroll_width;
                         sb_pos=Text.tx_hscroll;
                         sb_dir=Horizontal;
                         sb_delta=10;
                         sb_callback= do_nothing
                         } in
                     if (gr_vscrollbar_callback scroll1 Event)
                     then
                       (
                         Text.tx_vscroll <- scroll1.sb_pos;
                         Text.tx_1st_line <- ((scroll1.sb_pos * 
                                              (nb_lines - nb_display)) /
                                              (scroll_course height));
                         draw_gr_text Text;
                         true
                       )
                     else if (gr_hscrollbar_callback scroll2 Event)
                          then
                            (
                              Text.tx_hscroll <- scroll2.sb_pos;
                              Text.tx_1st_row <- ((scroll2.sb_pos * 256) /
                                                   (scroll_course height));
                              draw_gr_text Text;
                              true
                            )
                          else false
  ) in
  if not scrollevent
  then
    (
  let mouse_coord={x=Event.mouse_x; y=Event.mouse_y} in
    let Text_area={x1=left;
                   y1=top-(to_real_coord Text.tx_height);
                   x2=left+(to_real_coord Text.tx_width);
                   y2=top} in
      match (Event.id_event,
             Text.tx_state,
             (inside mouse_coord Text_area )) with
          (Key_pressed , Edited, _) ->
              gr_edit_text Text Event.key;
              true

        | (_, View_only, _) -> true

        | (Button_down, _ , true) ->
              Text.tx_state <- Edited;
              gr_draw_text_cursor Text white;
              gr_text_callback_loop Text Event

        | (_, Edited, false) ->
              Text.tx_state <- Editable;
              draw_gr_text Text;
              true

        | _ -> false
  )
  else true
;;






let goto_text Text Coord =
  let height = gr_text_height Text
  and dy=(line_height + short_space) in
    let nb_display=((height / dy)-1) in
      Text.tx_cursor <- fst Coord;
      Text.tx_line   <- snd Coord;
      if Text.tx_line < Text.tx_1st_line or
         Text.tx_line > (Text.tx_1st_line+nb_display)
      then Text.tx_1st_line <- snd Coord;
      if Text.tx_cursor < Text.tx_1st_row
      then Text.tx_1st_row <- fst Coord;
      update_scroll Text;
      Text.tx_state <- Edited
;;



let rec search_string_loop Text String TxtPos TxtEnd StrPos Strlen =
  if Strlen=StrPos
  then
  (
    Text.tx_s_select <- ((fst TxtPos)-Strlen,snd TxtPos);
    Text.tx_e_select <- TxtPos;
    TxtPos
  )
  else if UpperPos TxtPos TxtEnd
       then (-1,-1)
       else if (string_length Text.tx_name.(snd TxtPos))=(fst TxtPos)
            then search_string_loop Text
                                    String
                                    (0,(snd TxtPos)+1)
                                    TxtEnd
                                    0
                                    Strlen
            else if (nth_char Text.tx_name.(snd TxtPos) (fst TxtPos))=
                    (nth_char String StrPos)
                 then search_string_loop Text
                                         String
                                         ((fst TxtPos)+1,snd TxtPos)
                                         TxtEnd
                                         (StrPos+1)
                                         Strlen
                 else search_string_loop Text
                                         String
                                         ((fst TxtPos)+1,snd TxtPos)
                                         TxtEnd
                                         0
                                         Strlen
;;



let search_string Text String Search =
  let n=string_length String
  and nb_line=(vect_length Text.tx_name)-1 in
    let lastlen=string_length Text.tx_name.(nb_line) in
    match Search with
      FromCursor -> search_string_loop Text
                                       String
                                       (Text.tx_cursor,Text.tx_line)
                                       (lastlen,nb_line)
                                       0
                                       n
    | FullText   -> search_string_loop Text
                                       String
                                       (0,0)
                                       (lastlen,nb_line)
                                       0
                                       n
    | FullSelected -> search_string_loop Text
                                       String
                                       Text.tx_s_select
                                       Text.tx_e_select
                                       0
                                       n
;;

let rec replace_string_loop Text FindStr NewStr  TxtPos TxtEnd Strlen Found =
 let pos=search_string_loop Text FindStr TxtPos TxtEnd 0 Strlen in
   if fst pos = -1
   then Found
   else
   (
     let Line=Text.tx_name.(snd pos) in
       let len=string_length Line
       and fst_char=fst pos in
         let b_line=sub_string Line 0 (fst_char-Strlen)
         and e_line=sub_string Line fst_char (len-fst_char)
         and len=string_length NewStr in
           Text.tx_name.(snd pos) <- b_line ^ NewStr ^ e_line;
           Text.tx_line <- (snd pos);
           Text.tx_cursor <- (fst pos);
           Text.tx_modified <- true;
           replace_string_loop Text
                               FindStr
                               NewStr
                               (fst_char-Strlen+len,snd pos)
                               TxtEnd
                               Strlen
                               true
   )
;;

let replace_string Text FindStr NewStr Search =
  let n=string_length FindStr
  and nb_line=(vect_length Text.tx_name)-1 in
    let lastlen=string_length Text.tx_name.(nb_line) in
    match Search with
      FromCursor -> replace_string_loop Text
                                        FindStr
                                        NewStr
                                        (Text.tx_cursor,Text.tx_line)
                                        (lastlen,nb_line)
                                        n
                                        false
    | FullText   -> replace_string_loop Text
                                        FindStr
                                        NewStr
                                        (0,0)
                                        (lastlen,nb_line)
                                        n
                                        false
    | FullSelected -> replace_string_loop Text
                                          FindStr
                                          NewStr
                                          Text.tx_s_select
                                          Text.tx_e_select
                                          n
                                          false
;;




let gr_init_text Text =
  Text.tx_cursor <- 0;
  Text.tx_line <- 0;
  Text.tx_1st_line <- 0;
  Text.tx_1st_row <- 0;
  Text.tx_modified <- false;
  Text.tx_vscroll <- 0;
  Text.tx_hscroll <- 0;
  Text.tx_s_select <- (0,0);
  Text.tx_e_select <- (0,0);;



let save_gr_text Text Channel =
  save_lines Channel Text.tx_name;
  Text.tx_modified <- false;;

let load_gr_text Text Channel =
  Text.tx_name <- load_lines Channel;
  gr_init_text Text;
  draw_gr_text Text
;;
  









