#open "sys";;

#open "windows";;
#open "camlwin";;

#open "g_config";;
#open "g_global";;
#open "g_button";;
#open "g_string";;
#open "g_scroll";;
#open "g_text";;



let shell_max_line=300;;
let camlwin_env=
  try
    (getenv "CAMLWIN") ^ gr_directory_separator
  with _ -> output_string stderr "CAMLWIN is not set.\n";
            flush stderr;
            raise(Failure "Can't get CAMLWIN variable.\n")
  ;;


let make_shell Shell =
{
  sh_window=Shell.s_window;
  sh_top=Shell.s_top;
  sh_left=Shell.s_left;
  sh_width=Shell.s_width;
  sh_height=Shell.s_height;
  sh_name=[| ">" |];
  sh_cursor=1;
  sh_line=0;
  sh_1st_line=0;
  sh_1st_row=0;
  sh_state=Shell.s_state;
  sh_vscroll=0;
  sh_hscroll=0;
  sh_s_select=(0,0);
  sh_e_select=(0,0)
};;








(***************************************************************************)
(*                                                                         *)
(*                                                                         *)
(***************************************************************************)
let gr_text_of_shell Shell =
{
  tx_window=Shell.sh_window;
  tx_top=Shell.sh_top;
  tx_left=Shell.sh_left;
  tx_width=Shell.sh_width;
  tx_height=Shell.sh_height;
  tx_name=Shell.sh_name;
  tx_cursor=Shell.sh_cursor;
  tx_line=max 0 ((vect_length Shell.sh_name)-1);
  tx_1st_line=Shell.sh_1st_line;
  tx_1st_row=Shell.sh_1st_row;
  tx_state=Shell.sh_state;
  tx_modified=false;
  tx_scroll=Dscroll;
  tx_vscroll=Shell.sh_vscroll;
  tx_hscroll=Shell.sh_hscroll;
  tx_s_select=Shell.sh_s_select;
  tx_e_select=Shell.sh_e_select
};;

let draw_gr_shell Shell =
  let n=(vect_length Shell.sh_name) in
    if n>shell_max_line
    then Shell.sh_name <- sub_vect Shell.sh_name
                                   (n-shell_max_line)
                                   shell_max_line;
  let Text=gr_text_of_shell Shell in
    update_scroll Text;
    Shell.sh_vscroll <- Text.tx_vscroll;
    Shell.sh_hscroll <- Text.tx_hscroll;
  draw_gr_text Text;;


let rec found_next_command Lines Pos =
  let Len=vect_length Lines in
    if Pos<Len
    then
    (
      let len=string_length Lines.(Pos) in
        if len=0
        then found_next_command Lines (Pos+1)
        else if (nth_char Lines.(Pos) 0)=`>`
             then Pos
             else found_next_command Lines (Pos+1)
    )
    else (Len-1);;


let rec found_prec_command Lines Pos =
  if Pos>=0
  then
  (
    let len=string_length Lines.(Pos) in
      if len=0
      then found_prec_command Lines (Pos-1)
      else if (nth_char Lines.(Pos) 0)=`>`
           then Pos
           else found_prec_command Lines (Pos-1)
  )
  else 0;;

let rec gr_edit_shell Shell Add =
  let line=((vect_length Shell.sh_name)-1)
  and dy=(line_height + short_space) in
    let nb_display=((((to_real_coord Shell.sh_height) - scroll_width - short_space) / dy)-1)
    and nb_lines=line+1 in

    let Pos=(Shell.sh_cursor)
    and String=(Shell.sh_name.(line))
    and n=(string_length Shell.sh_name.(line))
    and Text=gr_text_of_shell Shell in
    match Add with
        0x008         ->  (* Backspace *)
          if not Pos=1
          then
          (
            let string_start=(sub_string String 0 (Pos - 1)) and
                string_end=  (sub_string String Pos (n - Pos))in
              gr_draw_text_cursor Text white;
              Shell.sh_name.(line) <- (string_start ^ string_end);
              Shell.sh_cursor <- (Pos - 1);
              update_cursor_text_line (gr_text_of_shell Shell)
          )

      | 0xff09         -> (* Del *)
          if not (Pos=n)
          then
            (
              let string_start=(sub_string String 0 Pos)
              and string_end=(sub_string String (Pos+1) (n - Pos-1)) in
                Shell.sh_name.(line) <- (string_start ^ string_end);
                update_cursor_text_line (gr_text_of_shell Shell)
            )


      | 0xff00         ->  (* Cursor to right *)
          if Pos<n & Pos>1
          then
            (
              gr_draw_text_cursor Text white;
              Shell.sh_cursor <- (Pos + 1);
              gr_draw_text_cursor (gr_text_of_shell Shell) black
            )

      | 0xff01         ->  (* Cursor to left *)
          if Pos>1
          then
            (
              gr_draw_text_cursor Text white;
              Shell.sh_cursor <- (Pos - 1);
              gr_draw_text_cursor (gr_text_of_shell Shell) black
            )

      | 0xff02         ->  (* Cursor up *)
          if Shell.sh_line>0
          then Shell.sh_line <-
            found_prec_command Shell.sh_name (Shell.sh_line-1);
          Shell.sh_name.(line) <- Shell.sh_name.(Shell.sh_line);
          Shell.sh_cursor <- string_length Shell.sh_name.(line);
          update_cursor_text_line (gr_text_of_shell Shell)

      | 0xff03         ->  (* Cursor down *)
          if Shell.sh_line<line
          then
          (
            Shell.sh_line <-
              found_next_command Shell.sh_name (Shell.sh_line+1);
            Shell.sh_name.(line) <- Shell.sh_name.(Shell.sh_line)
          )
          else
          (
            Shell.sh_name.(line) <- ">";
            Shell.sh_line <- line
          );
          Shell.sh_cursor <- string_length Shell.sh_name.(line);
          update_cursor_text_line (gr_text_of_shell Shell)

      | 0xff04         -> (* Page up*)
          Shell.sh_1st_line <- (Shell.sh_1st_line-nb_display);
          if Shell.sh_1st_line < 0
          then Shell.sh_1st_line <- 0;
          let Texts=gr_text_of_shell Shell in
                update_scroll Texts;
                Shell.sh_vscroll <- Texts.tx_vscroll;
                Shell.sh_hscroll <- Texts.tx_hscroll;
                Shell.sh_1st_line <- Texts.tx_1st_line;
          draw_gr_shell Shell

      | 0xff05         -> (* Page down*)
          Shell.sh_1st_line <- (Shell.sh_1st_line+nb_display);
          if Shell.sh_1st_line > (nb_lines-nb_display)
          then Shell.sh_1st_line <- (nb_lines-nb_display-1);
          let Texts=gr_text_of_shell Shell in
                update_scroll Texts;
                Shell.sh_vscroll <- Texts.tx_vscroll;
                Shell.sh_hscroll <- Texts.tx_hscroll;
                Shell.sh_1st_line <- Texts.tx_1st_line;
          draw_gr_shell Shell

      | 0xff06         ->  (* Home *)
          gr_draw_text_cursor Text white;
          Shell.sh_cursor <- 1;
          gr_draw_text_cursor (gr_text_of_shell Shell) black

      | 0xff07         ->  (* End *)
          gr_draw_text_cursor Text white;
          Shell.sh_cursor <- n;
          gr_draw_text_cursor (gr_text_of_shell Shell) black

      | 0x000d         ->  (* LF *)
            (
              system (sub_string String 1 (n-1));
              let err_file=open_in (camlwin_env ^ "camlwin.err")
              and out_file=open_in (camlwin_env ^ "camlwin.out") in
                let txt1=load_lines out_file
                and txt2=load_lines err_file in
                  let txt=concat_vect txt1 txt2 in
              close_in err_file;
              close_in out_file;
              Shell.sh_name <- concat_vect Shell.sh_name txt;
              Shell.sh_name <- concat_vect Shell.sh_name [| ">" |];
              Shell.sh_line <- (vect_length Shell.sh_name)-1;
              Shell.sh_cursor <- 1;
              if (Shell.sh_1st_line+nb_display)< Shell.sh_line
              then Shell.sh_1st_line <- Shell.sh_line-nb_display+1;
              let Texts=gr_text_of_shell Shell in
                update_scroll Texts;
                Shell.sh_vscroll <- Texts.tx_vscroll;
                Shell.sh_hscroll <- Texts.tx_hscroll;
                Shell.sh_1st_line <- Texts.tx_1st_line;
              draw_gr_shell Shell
            )

      | _              -> (* Ascii character *)
          if Add <0xff
          then
            (
              gr_draw_text_cursor Text white;
              let Char=(char_of_int Add) in
                if Char=`\t`
                then
                (
                  for i=0 to 3 do
                  (
                    Shell.sh_name.(line) <- add_char Shell.sh_name.(line)
                                                     ` `
                                                     Shell.sh_cursor;
                    Shell.sh_cursor <- (Shell.sh_cursor + 1)
                  )
                  done
                )
                else
                (
                  Shell.sh_name.(line) <- add_char String Char Pos;
                  Shell.sh_cursor <- (Pos + 1)
                );
              let Texts=gr_text_of_shell Shell in
                update_scroll Texts;
                if Texts.tx_1st_line = Shell.sh_1st_line
                then update_cursor_text_line (gr_text_of_shell Shell)
                else
                (
                  Shell.sh_vscroll <- Texts.tx_vscroll;
                  Shell.sh_hscroll <- Texts.tx_hscroll;
                  Shell.sh_1st_line <- Texts.tx_1st_line;
                  draw_gr_shell Shell
                )
             )
;;




let rec gr_shell_callback Shell Event =
  if Event.id_event=Key_pressed & Shell.sh_state=Edited
  then
  (
    gr_edit_shell Shell Event.key;
    true
  )
  else
  (
    let Text=gr_text_of_shell Shell in
      let ans=gr_text_callback Text Event in
      Shell.sh_vscroll <- Text.tx_vscroll;
      Shell.sh_hscroll <- Text.tx_hscroll;
      Shell.sh_1st_line <- Text.tx_1st_line;
      Shell.sh_state <- Text.tx_state;
      ans
  )
;;



