(* general functions and type for windows management *)
#open "windows";;
#open "sys";;
#open "camlwin";;
#open "g_config";;


open_graph ();;


let std_win = 
{
      win_id = default_window ();
      win_top = 0;
      win_left = 0;
      win_width = 0;
      win_height = 0;
      win_name = "Camlwin default windows";
      win_state = Destroyed
};;



let undef_window = default_window ();;



let grey      = 0xAFAFAF;;
let darkgrey  = 0x8F8F8F;;
let nextgrey  = 0x505050;;
let winblue   = 0x101080;;
let motifblue = 0x6482B0;;
let darkblue  = 0x446290;;


let look = ref Std_color;;
let gr_bkground = ref grey;;

let set_gr_look Look =
  look := Look;
  match Look with
      Std_color       -> gr_bkground := grey
    | Std_mono        -> gr_bkground := white
    | Window_color    -> gr_bkground := grey
    | Os2_color       -> gr_bkground := grey
    | Motif_color     -> gr_bkground := motifblue
    | Open_look_color -> gr_bkground := grey
    | Open_look_mono  -> gr_bkground := white
    | Next_color      -> gr_bkground := grey
    | Mac_color       -> gr_bkground := grey
    | Mac_mono        -> gr_bkground := white
;;



let default_look () =
  match (gr_operating_system,gr_screen_col) with
    (Unix, MonochromeScreen) -> set_gr_look Open_look_mono
  | (Unix, ColorScreen)      -> set_gr_look Open_look_color
  | (Dos, _)                 -> set_gr_look Std_color
  | (MsWindows, _)           -> set_gr_look Window_color
  | (OS2, _)                 -> set_gr_look Os2_color
  | (WindowsNT, _)           -> set_gr_look Window_color
  | (MacOs, ColorScreen)     -> set_gr_look Mac_color
  | (MacOs, MonochromeScreen)-> set_gr_look Mac_mono
  | (NextStep, _)            -> set_gr_look Next_color
;;



try
  match (getenv "CAMLLOOK") with
    "OpenLookColor" -> set_gr_look Open_look_color
  | "OpenLookMono"  -> set_gr_look Open_look_mono
  | "Windows"       -> set_gr_look Window_color
  | "Mono"          -> set_gr_look Std_mono
  | "Next"          -> set_gr_look Next_color
  | "Motif"         -> set_gr_look Motif_color
  | _               -> set_gr_look Std_color
with Not_found      -> default_look ()
;;



let is_color_look () =
match !look with
    Std_color       -> true
  | Std_mono        -> false
  | Window_color    -> true
  | Os2_color       -> true
  | Motif_color     -> true
  | Open_look_color -> true
  | Open_look_mono  -> false
  | Next_color      -> true
  | Mac_color       -> true
  | Mac_mono        -> false
;;




let get_gr_look ()= !look;;
let backcolor ()= !gr_bkground;;



let surround x y = (x / y) + (if x mod y <=(y/2) then 0 else 1);;

let line_height= snd (text_size "Ap");;


let coord_extent = surround line_height  4;;

let to_real_coord x = (x*line_height)/4;;
let to_win_coord x = (4*x)/line_height;;

let long_space = 2*coord_extent;;
let short_space = coord_extent;;


let get_move = ref false;;


let g_debug = ref false;;
let set_debug ()= g_debug := true;;
let unset_debug ()= g_debug := false;;
let gr_debug () = !g_debug;;





let do_nothing x y = true;;
let empty_image = create_image 1 1;;
let empty_line = "";;





let class x =
    if(snd x > fst x) then x else (snd x, fst x);;

let inside coord rect =
    let couplex=class (rect.x1, rect.x2) in
      let coupley=class (rect.y1, rect.y2) in
          if (((fst couplex) < coord.x) & ((snd couplex) > coord.x) &
              ((fst coupley) < coord.y) & ((snd coupley) > coord.y))
          then true else false
;;



let draw_rect x y w h =
  moveto x y;
  lineto x (y+h);
  lineto (x+w) (y+h);
  lineto (x+w) y;
  lineto x y
;;

let color_draw_triangle  x y w =
  moveto (x+w) y;
  set_color black;
  lineto x y;
  lineto (x+(w/2)) (y-w);
  set_color white;
  lineto (x+w) y
;;


let mono_draw_triangle  x y w =
  moveto (x+w) y;
  set_color black;
  lineto x y;
  lineto (x+(w/2)) (y-w);
  lineto (x+w) y
;;




let draw_triangle  x y w =
  if is_color_look ()
  then color_draw_triangle x y w
  else mono_draw_triangle x y w
;;








(*****************************************************************************)
(*                                                                           *)
(*****************************************************************************)
let rec gr_nth_elem List Pos =
  match List with
    [] -> raise (Failure "Bad element number")
  | x::y -> if Pos=0
            then x
            else gr_nth_elem y (Pos-1);;


let rec set_nth_elem_loop Pos Elem List Result=
  match List with
    [] -> Result
  | x::y -> if Pos=0
            then set_nth_elem_loop (Pos-1) Elem y (Result @ [ Elem ])
            else set_nth_elem_loop (Pos-1) Elem y (Result @ [ x ]);;

let gr_set_nth_elem List Pos Elem =
  set_nth_elem_loop Pos Elem List [];;


let rec del_nth_elem_loop Pos List Result=
  match List with
    [] -> Result
  | x::y -> if Pos=0
            then (Result @ y)
            else del_nth_elem_loop (Pos-1) y (Result @ [ x ]);;

let gr_del_nth_elem List Pos =
  del_nth_elem_loop Pos List [];;



let rec end_list List Nb =
  match (List, Nb) with
    ([], _)  -> []
  | (_, 0)   -> List
  | (x::y, _) -> end_list y (Nb-1)
;;


(*****************************************************************************)
(*                                                                           *)
(*****************************************************************************)
let rotate Array =
  let h = vect_length Array
  and w = vect_length Array.(0) in
    let ret = make_matrix w h 0 in
      for i = 0 to (w-1) do
        for j = 0 to (h-1) do
          ret.(i).(j) <- Array.(j).(w-i-1)
        done
      done;
  ret
;;




(*****************************************************************************)
(*                                                                           *)
(*****************************************************************************)
let add_char Str c n =
  let len = string_length Str in
    let NewStr = make_string (len+1) ` ` in
      blit_string Str 0 NewStr 0 n;
      if n < len
      then blit_string Str n NewStr (n+1) (len-n);
      set_nth_char NewStr n c;
    NewStr
;;



(*****************************************************************************)
(*                                                                           *)
(*****************************************************************************)
let save_lines file Vect =
  let n=vect_length Vect in
  if not n=0
  then
    for i=0 to (n-1) do
      output_string file (Vect.(i) ^ "\n")
    done;;

let really_input_line Channel =
  let Line=ref ""
  and n=ref 0
  and c= ref ` ` in
  try
  (
    while(not !c=`\n`)
    do
    (
      c:=input_char Channel;
      if not !c=`\n`
      then Line:= add_char !Line !c !n;
      n:=!n+1
    )
    done;
    !Line
  )
  with End_of_file -> if !Line=""
                      then raise End_of_file
                      else !Line;;



let load_lines Channel =
  let Lines=ref [| "" |] in
  Lines:=[| |];
  try
  (
    while(true)
    do
    (
      Lines:= (concat_vect !Lines [| (really_input_line Channel) |])
    )
    done;
    [| |]
  )
  with End_of_file -> !Lines;;



let load_file Channel =
  let Line=ref ""
  and n=ref 0
  and c= ref ` ` in
  try
  (
    while(true)
    do
    (
      c:=input_char Channel;
      Line:= add_char !Line !c !n;
      n:=!n+1
    )
    done;
    !Line
  )
  with End_of_file -> !Line;;

let UpperPos Pos1 Pos2 =
  if snd Pos1 > snd Pos2
  then true
  else if snd Pos1 < snd Pos2
       then false
       else if fst Pos1 > fst Pos2
            then true
            else false
;;

