#!/usr/bin/env python

RCSID="$Id: hostbup.py,v 1.46 2007/04/26 15:38:00 andreas Exp $"

import shlex, sys, string, os, pwd, getopt, time, types, time, smtplib
import thread, Queue, signal, select, statvfs, resource


MAXWAIT=45*3600		# wait 45 hours for output from a child process
#
# utitlity functions
# 
def timestamp(t=None):
	if t == None:
		now=time.time()
	else:
		now=t
	return time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(now))
	

#
def sq(s):
	if len(s) >= 2 and s[0] == '"' and s[-1] == '"':
		return s[1:-1]
	else:
		return s
#
#
def dbgprint(s,lvl=1):
	if debugflag >= lvl:
		print "dbg: %s" % s

#
# 
def logopen(logname, reopen=0):
	global logf, logbuf, currlogfname
	if reopen:
		try:
			logname=currlogfname
		except:
			return
	else:
		if not logname:
			logname=general.logname
		currlogfname=logname
	logf=open(logname,"a")
	logbuf=[]

#
#
def logflush():
	global logf
	if logf:
		logf.flush()
#
#
def logclose():
	global logf
	if logf:
		logprint("%s\n" % RCSID)
		logf.close()
#
# 
def logprint(s = ""):
	global logf, logbuf
	if logf:
		logf.write(s)
	if not logf or verboseflag:
		print s,
	logbuf.append(s)


def logdump():
	global logbuf
	return string.join(logbuf,'')

#
#
def human(n):
	if n >= 1099511627776L:
		div=1099511627776L
		sufix="Tb"
	elif n >= 1073741824L:
		div=1073741824L
		sufix="Gb"
	elif n >= 1048576L:
		div=1048576L
		sufix="Mb"
	elif n >= 1024L:
		div=1024L
		sufix="Kb"
	else:
		div=1L
		sufix=""
	return "%0.0f%s" % ((n + (div / 2)) / div, sufix)
#
#
def gt(p,expect=[],typ=None):
	global EOF
	r=p.get_token()
	if r == "":
		if EOF == 1:
			print "error on line %s: EOF" % pp.lineno
			if debugflag: a=aa
			sys.exit(1)
		EOF=1
		return None
	dbgprint("token: %s" % r,lvl=2)
	if len(expect) > 0 and not r in expect:	
		print "error on line %s: expected '%s', got '%s'" % (pp.lineno, expect, r)
		if debugflag: a=aa
		sys.exit(1)
	r=sq(r)
	if typ:
		if typ == types.IntType:
			fac=1
			# recognize nnnsss where nnn is a number and sss is one of kb, mb, gb or tb
			if r[-1] in ['b','B']:
				if len(r) > 2:
					if r[-2] in ['k','K']:
						fac=1024L
					elif r[-2] in ['m','M']:
						fac=1048576L
					elif r[-2] in ['g','G']:
						fac=1073741824L
					elif r[-2] in ['t','T']:
						fac=1099511627776L
				r=r[:-2]
			try:
				r=int(r)
			except:
				print "error on line %s: expected <int>, got '%s'" % (pp.lineno, r)
				if debugflag: a=aa
				sys.exit(1)
			r=r*fac
				

	return r

#
#
def bool(t):
	if t in ['y','yes','1','true','ok']:
		return 1
	if t in ['n','no','0','false','notok']:
		return 0
	print "error on line %s: expected <bool>, got '%s'" % (pp.lineno, t)
	if debugflag: a=aa
	sys.exit(1)

#
#
def ttype(t):
	if t[0] in string.ascii_letters:
		return "string"
	elif t[0] in string.digits:
		return "num"
	else:
		return "unknown"

#
# UU type locking functions
#
class Uulock:
	def __del__(self):
		if self.LF != "":
			os.remove(self.LF)
			if verboseflag: print "uu: removed lock"
			self.LF=""
	
	def __init__(self, pid, resource, lockfile="/var/spool/lock/LCK"):
		LFtmp=lockfile+'..%d' % pid
		self.LF=lockfile+'..%s' % resource
		try:
			lf=open(LFtmp,'w')
		except:
			if verboseflag: print 'cannot create lock file %s' % LFtmp
			raise 'cannot create lock file %s' % LFtmp
			return
		lf.write("%10s\n" % pid)
		lf.close()
		while 1:
			try:
				os.link(LFtmp,self.LF)
			except:
				if verboseflag: print "lock: cannot link %s to %s" % (LFtmp, self.LF)
			else:
				if verboseflag: print "lock: pid %s locked %s" % (pid, self.LF)
				os.remove(LFtmp)
				break

			try:
				ll=open(self.LF,'r')
				pid=string.atoi(string.strip(ll.readline()[:-1]))
				if verboseflag: print "lock: lock exists, pid is %s" % pid
				os.kill(pid,0)
				if verboseflag: print "lock: process exists"
			except:
				os.remove(self.LF)
				if verboseflag: print "lock: removed stale pidfile"
			else:
				self.LF=""
				os.remove(LFtmp)
				if verboseflag: print "lock: (%s) in use" % resource
				raise "lock (%s) in use" % resource
				break
		

	def unlock(self):
		if self.LF != "":
			try:
				os.remove(self.LF)
				if verboseflag: print "lock: removed lock"
			except:
				pass
			self.LF=""

#
# signal handlers
#
def sighuphandler(signum, frame):
	logflush()
	logclose()
	logopen(None,1)

# 
# config file handling function
#
def load_list():
	dbgprint("load_list",lvl=2)
	r=[]
	tok=gt(pp,['{'])
	tok=gt(pp)
	while tok != '}':
		r.append(tok)
		tok=gt(pp,[';'])
		tok=gt(pp)
#?	tok=gt(pp, [';'])
	dbgprint("load_list exit: %s" % r,lvl=2)
	return r


#
# --- Classes ----------
#
class fsC:
  def __init__(self, name):
	self.name=name
	self.type=None
	self.backup=1
	self.keep=None
	self.bkpserver=None
	self.bkppath=None
	self.exclude=[]

  def __str__(self):
	a=''
	if self.type: a+='		type "%s";\n' % self.type
	if not self.backup: a+='		backup %s;\n' % self.backup
	if self.keep: a+='		keep "%s";\n' % self.keep
	if self.bkpserver: a+='		backupto "%s:%s";\n' % (self.bkpserver, self.bkppath)
	if len(self.exclude) > 0:
		a+='		exclude { "%s"; };\n' % string.join(self.exclude,'"; "')
	if a == '':
		return ''
	return " {\n%s	}" % a

#
#
class clientC:
  def __init__(self, name):
	self.name=name
	self.access="ssh"
	self.sysctl=[]
	self.os="**unknown**"
	self.version="**unknown**"
	self.kerberos=0
	self.wakeup=None
	self.keep=None
	self.rdiffbin=None
	self.bkpserver=None
	self.bkppath=None
	self.fs={}
	self.fslist=[]
	self.exclude=[]
	self.uulock=None

  def addfs(self, fs):
	self.fs[fs.name]=fs
	self.fslist.append(fs.name)
	if fs.name in ['/dev','/kern','/proc']:
		fs.backup=0;

  def __str__(self):
	a=''
	if self.access != "ssh": a+='	access "%s";\n' % self.access
	if self.os != "**unknown**": a+='	os "%s";\n' % self.os
	if self.version != "**unknown**": a+='	version "%s";\n' % self.version
	if self.kerberos != 0: a+='	kerberos %s;\n' % self.kerberos
	if self.wakeup: a+='	wakeup "%s";\n' % self.wakeup
	if self.keep: a+='	keep "%s";\n' % self.keep
	if self.rdiffbin: a+='	rdiffbin "%s";\n' % self.rdiffbin
	if self.bkpserver: a+='	backupto "%s:%s";\n' % (self.bkpserver, self.bkppath)
	if len(self.sysctl) > 0:
		a+='	sysctl { "%s"; };\n' % string.join(self.sysctl,'"; "')
	if len(self.exclude) > 0:
		a+='	exclude { "%s";	};\n' % string.join(self.exclude,'"; "')
	for fsys in self.fslist:
		a+='	fs "%s"%s;\n' % (fsys, str(self.fs[fsys]))
	return a

  def Lock(self):
	if self.uulock:		#NB. already locked 
		return
	if self.bkpserver:
		lockdir=self.bkppath
	else:
		lockdir=general.bkppath
	self.uulock=Uulock(os.getpid(), self.name, lockdir+"/LCK")


  def UnLock(self):
	self.uulock.unlock()

#
#
class directoryC:
  def __init__(self, name):
	self.name=name
	self.removable=None
	self.mountscript=None
	self.unmountscript=None
	self.statusscript=None
	self.usablesize=None
	self.default=None

  def __str__(self):
	a=''
	if self.removable:
		a+='		removable;\n'
	if self.mountscript:
		a+='		mountscript "%s";\n' % self.mountscript
	if self.unmountscript:
		a+='		unmountscript "%s";\n' % self.unmountscript
	if self.statusscript:
		a+='		statusscript "%s";\n' % self.statusscript
	if self.usablesize:
		a+='		use %s;\n' % human(self.usablesize)
	if self.default:
		a+='		default;\n'
	if a == '':
		return ''
	return " {\n%s	}" % a

#
#
class serverC:
  def __init__(self, name):
  	self.name=name
	self.directory={}
	self.access="ssh"
	self.sysctl=[]
	self.stdrdiffbin="rdiff-backup"
	self.rdiffbin=self.stdrdiffbin
	self.datalimit=None
	self.queues=1

  def __str__(self):
  	a=''
	if self.rdiffbin != self.stdrdiffbin: a+='	rdiffbin "%s";\n' % self.rdiffbin
	if self.queues != 1: a+='	queues %s;\n' % self.queues
	if self.access != "ssh": a+='	access "%s";\n' % self.access
	if len(self.sysctl) > 0:
		a+='	sysctl { "%s"; };\n' % string.join(self.sysctl,'"; "')
	if self.datalimit != None: a+='	datalimit %s;\n' % human(self.datalimit)
	dirkeys=self.directory.keys()
	dirkeys.sort()

	for dir in dirkeys:
		a+='	directory "%s"%s;\n' % (dir, str(self.directory[dir]))
	return a

  def space(self, dir):
	try:
		stat=os.statvfs(dir)
	except:
		return [0, 0]
	usedB=stat[statvfs.F_BFREE]*stat[statvfs.F_FRSIZE]
	totB=stat[statvfs.F_BLOCKS]*stat[statvfs.F_FRSIZE]
	return [usedB, totB]
#
#
class generalC:
  def __init__(self):
	self.stdoptions="--force --no-file-statistics --exclude-other-filesystems --print-statistics --override-chars-to-quote '' --no-compression"
	self.options=self.stdoptions
	self.bkpserver=None
	self.bkppath=None
	self.exclude=[]
	self.stdrdiffbin="rdiff-backup"
	self.rdiffbin=self.stdrdiffbin
	self.stdkeep="999D"
	self.keep=self.stdkeep
	self.mailto=None
	self.stdmailfrom="%s@%s" % (ouruser, ourhostname)
	self.mailfrom=self.stdmailfrom
	self.stdsmtpserver="localhost"
	self.smtpserver=self.stdsmtpserver
	self.stdlogname="/var/log/hostsbup.log"
	self.logname=self.stdlogname

  def __str__(self):
  	a=''
	a+='	backupto "%s:%s";\n' % (self.bkpserver, self.bkppath)
	if self.options != self.stdoptions:
		a+='	options "%s";\n' % self.options
	if self.keep != self.stdkeep:
		a+='	keep "%s";\n' % self.keep
	if self.rdiffbin != self.stdrdiffbin:
		a+='	rdiffbin "%s";\n' % self.rdiffbin
	if self.mailto:
		a+='	mailto "%s";\n' % self.mailto
	if self.mailfrom != self.stdmailfrom:
		a+='	mailfrom "%s";\n' % self.mailfrom
	if self.smtpserver != self.stdsmtpserver:
		a+='	smtpserver "%s";\n' % self.smtpserver
	if self.logname != self.stdlogname:
		a+='	logname "%s";\n' % self.logname
	if len(self.exclude) > 0:
		a+='	exclude { "%s"; };\n' % string.join(self.exclude,'"; "')
	return a

#
# --- CONFIG releated routines and functions ----------
#
def load_backupto(check=1):
	tok=gt(pp)
	backupto=string.split(tok,':')
	if len(backupto) != 2:
		print "error on line %s: expected '<server:/directory>', got '%s'" % (pp.lineno, tok)
		if debugflag: a=aa
		sys.exit(1)
	bkpserver=backupto[0]
	bkppath=backupto[1]
	if check:
		if servers.has_key(bkpserver):
			if not servers[bkpserver].directory.has_key(bkppath):
				print "error on line %s: server '%s' has no directory '%s'" % (pp.lineno, bkpserver, bkppath)
				if debugflag: a=aa
				sys.exit(1)
		else:
			print "error on line %s: no server '%s'" % (pp.lineno, bkpserver)
			if debugflag: a=aa
			sys.exit(1)
	return bkpserver, bkppath
#
#
#
def load_fs():
	fs=fsC('new')
	keys=['type','backup','keep','backupto','exclude']
	tok=gt(pp)
	if tok[0] != "/":
		print "error on line %s: expected <fsname>, got '%s'" % (pp.lineno, tok)
		if debugflag: a=aa
		sys.exit(1)
	fs.name=tok
	tok=gt(pp)
	if tok == '{':
		tok=gt(pp)
		while tok != '}':
			if tok == 'type': fs.type=gt(pp)
			elif tok == 'backup': fs.backup=bool(gt(pp))
			elif tok == 'backupto': 
				fs.bkpserver,fs.bkppath=load_backupto()
			elif tok == 'keep': fs.keep=gt(pp).upper()
			elif tok == 'exclude': fs.exclude=load_list()
			else:
				print "error on line %s: expected on of '%s', got '%s'" % \
					(pp.lineno, keys, tok)
				if debugflag: a=aa
				sys.exit(1)
			tok=gt(pp,[';'])
			tok=gt(pp)
		dbgprint("load_fs while_end",lvl=2)
	else:
		pp.push_token(tok)
	dbgprint("load_fs exit",lvl=2)
	return fs

#
#
def load_directory():
	directory=directoryC('new')
	keys=['removable','use','mount','unmount','status','default']
	tok=gt(pp)
	if tok[0] != "/":
		print "error on line %s: expected <directoryname>, got '%s'" % (pp.lineno, tok)
		if debugflag: a=aa
		sys.exit(1)
	directory.name=tok
	tok=gt(pp)
	if tok == '{':
		tok=gt(pp)
		while tok != '}':
			if tok == 'removable': directory.removable=1
			elif tok == 'use': directory.usablesize=gt(pp,typ=types.IntType)
			elif tok == 'mount': directory.mountscript=gt(pp,typ=types.StringType)
			elif tok == 'unmount': directory.unmountscript=gt(pp,typ=types.StringType)
			elif tok == 'status': directory.statusscript=gt(pp,typ=types.StringType)
			elif tok == 'default': directory.default=1
			else:
				print "error on line %s: expected on of '%s', got '%s'" % \
					(pp.lineno, keys, tok)
				if debugflag: a=aa
				sys.exit(1)
			tok=gt(pp,[';'])
			tok=gt(pp)
		dbgprint("load_directory while_end",lvl=2)
	else:
		pp.push_token(tok)
	dbgprint("load_directory exit",lvl=2)
	return directory
#
#	
def load_server():
	dbgprint("load_server",lvl=2)
	server=serverC("new")
	keys=['directory','access','sysctl','rdiffbin','queues','datalimit']
	tok=gt(pp)
	if ttype(tok) != "string":
		print "error on line %s: expected <servername>, got '%s'" % (pp.lineno, tok)
		if debugflag: a=aa
		sys.exit(1)
	server.name=tok
	tok=gt(pp,'{')
	tok=gt(pp)
	while tok != '}':
		dbgprint("load_server while",lvl=2)
		if tok == 'access': server.access=gt(pp)
		elif tok == 'rdiffbin': server.rdiffbin=gt(pp)
		elif tok == 'queues': server.queues=gt(pp,typ=types.IntType)
		elif tok == 'sysctl': server.sysctl=load_list()
		elif tok == 'datalimit': server.datalimit=gt(pp,typ=types.IntType)
		elif tok == 'directory':
			dir=load_directory()
			if server.directory.has_key(dir.name):
				print "error on line %s: server %s already has directory %s" (pp.lineno, server.name, dir.name)
				if debugflag: a=aa
				sys.exit(1)
			server.directory[dir.name]=dir
		else:
			print "error on line %s: expected on of '%s', got '%s'" % (pp.lineno, keys, tok)
			if debugflag: a=aa
			sys.exit(1)
		tok=gt(pp,[';'])
		tok=gt(pp)
	tok=gt(pp)
	if tok != ';':
		print "error on line %s: expected ';', got '%s'" % (pp.lineno, tok)
		if debugflag: a=aa
		sys.exit(1)

	dbgprint("load_server exit",lvl=2)
	return server
#
#
def load_client():
	dbgprint("load_client",lvl=2)
	client=clientC("new")
	keys=['os','version','kerberos','wakeup','access','sysctl','exclude','backupto','keep','rdiffbin','fs']
	tok=gt(pp)
	if ttype(tok) != "string":
		print "error on line %s: expected <hostname>, got '%s'" % (pp.lineno, tok)
		if debugflag: a=aa
		sys.exit(1)
	client.name=tok
	tok=gt(pp,'{')
	tok=gt(pp)
	while tok != '}':
		dbgprint("load_client while",lvl=2)
		if tok == 'backupto': client.bkpserver,client.bkppath=load_backupto()
		elif tok == 'os': client.os=gt(pp)
		elif tok == 'version': client.version=gt(pp)
		elif tok == 'kerberos': client.kerberos=1
		elif tok == 'wakeup': client.wakeup=gt(pp)
		elif tok == 'access': client.access=gt(pp)
		elif tok == 'sysctl': client.sysctl=load_list()
		elif tok == 'keep': client.keep=gt(pp).upper()
		elif tok == 'rdiffbin': client.rdiffbin=gt(pp)
		elif tok == 'exclude': client.exclude=load_list()
		elif tok == 'fs':
			fs=load_fs()
			if client.fs.has_key(fs.name):
				print "error on line %s: client %s already has fs %s" (pp.lineno, client.name, fs.name)
				if debugflag: a=aa
				sys.exit(1)
			client.addfs(fs)
		else:
			print "error on line %s: expected on of '%s', got '%s'" % (pp.lineno, keys, tok)
			if debugflag: a=aa
			sys.exit(1)
		tok=gt(pp,[';'])
		tok=gt(pp)
	tok=gt(pp)
	if tok != ';':
		print "error on line %s: expected ';', got '%s'" % (pp.lineno, tok)
		if debugflag: a=aa

	dbgprint("load_client exit",lvl=2)
	return client

#
#
def load_general():
	dbgprint("load_general",lvl=2)
	keys=['options','exclude','keep','rdiffbin','mailto','mailfrom','smtpserver','logname']
	tok=gt(pp,'{')
	tok=gt(pp)
	while tok != '}':
		if tok == 'options': general.options=gt(pp)
		elif tok == 'exclude': general.exclude=load_list()
		elif tok == 'keep': general.keep=gt(pp).upper()
		elif tok == 'rdiffbin': general.rdiffbin=gt(pp)
		elif tok == 'mailto': general.mailto=gt(pp)
		elif tok == 'mailfrom': general.mailfrom=gt(pp)
		elif tok == 'smtpserver': general.smtpserver=gt(pp)
		elif tok == 'logname': general.logname=gt(pp)
		else:
			print "error on line %s: expected on of '%s', got '%s'" % (pp.lineno, keys, tok)
			sys.exit(1)
		tok=gt(pp,[';'])
		tok=gt(pp)
	tok=gt(pp)
	if tok != ';':
		print "error on line %s: expected ';', got '%s'" % (pp.lineno, tok)
	dbgprint("load_general exit",lvl=2)
	return general

#
#
def loadconfig():
	dbgprint("loading config")
	while 1:
		tok=gt(pp)
		if tok == None:
			break
		if tok == "general":
			general=load_general()
		elif tok == "server":
			server=load_server()
			servers[server.name]=server
			serverlist.append(server.name)
		elif tok == "client":
			client=load_client()
			clients[client.name]=client
			clientlist.append(client.name)
		else:
			print "error on line %s: unknown token: %s" % (pp.lineno, tok)
			sys.exit(1)

	# some checking of config file sanity
	if len(servers) == 0:
		print "error: no 'servers' specified'"
		sys.exit(1)

	if len(clients) == 0:
		print "error: no 'client' specified'"
		sys.exit(1)

	for server in serverlist:
		for dir in servers[server].directory.keys():
			if servers[server].directory[dir].default:
				if general.bkpserver:
					print "error: default backup directory set twice" 
					sys.exit(1)
				general.bkpserver=servers[server].name
				general.bkppath=dir
	if verboseflag:
		print "default backupto is %s:%s" % (general.bkpserver,general.bkppath)
	
	if not general.bkpserver:
		print "error: no default backup server:/directory specified"
		sys.exit(1)
	if debugflag > 1:
		dumpconfig()
	dbgprint("loading config finished")
#

def dumpconfig():
	print "# hostbup config %s dump on %s" % (cfgname, time.asctime(time.localtime(time.time())))
	print "#"
	print "general {\n", general,"\n};\n"
	for server in serverlist:
		print 'server "%s" {\n%s};\n'  % (server, servers[server])
	for client in clientlist:
		print 'client "%s" {\n%s};\n'  % (client, clients[client])

#
# --- Command releated functions ----------
#
def rdiffcmd(hostname):
	client=clients[hostname]
	exfs={}
	cmd={}
	fslist=[]
	# construct list of fs's to be backed up
	for fs in client.fslist:
		if client.fs[fs].backup:
			fslist.append(fs)
	
	if client.keep:
		bkpkeep=client.keep
	else:
		bkpkeep=general.keep

	if client.bkpserver:
		bkpserver=client.bkpserver
		bkppath=client.bkppath
	else:
		bkpserver=general.bkpserver
		bkppath=general.bkppath

	if client.rdiffbin:
		rdiffbin=client.rdiffbin
	else:
		rdiffbin=general.rdiffbin

	if client.access == 'rsh':
		schema='rsh %s '+rdiffbin+' --server'
	else:
		schema='ssh -c arcfour %s '+rdiffbin+' --server'

	for ex in client.exclude+general.exclude:
		infs=""
		for fs in fslist:
			if string.find(ex, fs) == 0:
				if len(infs) < len(fs):
					infs=fs
		if exfs.has_key(infs):
			exfs[infs].append(ex)
		else:
			exfs[infs]=[ex]

	for fs in fslist:
		if fs == '/':
			tfs='/root'
		else:
			tfs=fs
		exc=[]
		if exfs.has_key(fs):
			exc+=exfs[fs]
		exc+=client.fs[fs].exclude

		if  client.fs[fs].bkpserver:
			srvr=client.fs[fs].bkpserver
			dir=client.fs[fs].bkppath
		else:
			srvr=bkpserver
			dir=bkppath

		if  client.fs[fs].keep:
			keep=client.fs[fs].keep
		else:
			keep=bkpkeep

		cmd[fs]={}
		cmd[fs]['client']=hostname
		cmd[fs]['options']=general.options
		cmd[fs]['excludelist']=exc
		cmd[fs]['fs']=fs
		cmd[fs]['tfs']=tfs
		cmd[fs]['server']=srvr
		cmd[fs]['srvdir']=dir
		cmd[fs]['schema']=schema
		cmd[fs]['keep']=keep
	dbgprint("rdiffcmd returns: %s" % cmd)
	return cmd

#
#

def runcmd(host, fs, cmd, ts=0):
	if verboseflag:
		logprint("%s runcmd: %s\n" % (timestamp(), cmd))
	res=[None,[],[]]
	if noflag:
		if ts:
			res[1]=[["now", "simulated output\n"]]
		else:
			res[1]=["simulated output\n"]
		res[2]=["simulated error\n"]
		code=0
	else:
		hd=os.popen3(cmd)
		ilist=[hd[1],hd[2]]
		lines=[None,[],[]]
		while 1:
			if len(ilist) == 0:
				break
			r=select.select(ilist,[],[],MAXWAIT)
			stamp=timestamp()
			if r == ([], [], []):
				res[2].append("%s timeout!" % stamp)
				break
			for h in r[0]:
				if h == hd[1]:
					fn=1
				elif h == hd[2]:
					fn=2
				else:
					continue
				c=hd[fn].read(1)

				if len(c) == 0:
					hd[fn].close()
					ilist.remove(hd[fn])
					if len(lines[fn]) == 0:
						continue
				else:
					lines[fn].append(c)
					if c != '\n':
						continue

				l=string.join(lines[fn],'')
				if verboseflag:
					print '%s:%s %s' % (host, fs, l),
				lines[fn]=[]
				if ts:
					res[fn].append([stamp,l])
				else:
					res[fn].append(l)

		code=hd[0].close()
		if code == None:
			code=0
	return (res[1], res[2], code)

#
#
def mkconfcmd(hostname, access):
	cmd=access+' '+hostname+' "'+'uname -n;uname -s;uname -r;df -l"'
	return cmd

#
#r
def genconf(hostname, access="rsh"):
	cmd=mkconfcmd(hostname, access)
	res,err,code=runcmd(hostname, 'conf',  cmd)
	if len(res) < 6:
		print "error: not enough output"
		sys.exit(1)
	node=res[0][:-1]
	os=res[1][:-1]
	ver=res[2][:-1]
	client=clientC(node)
	for l in res[4:]:
		x=string.split(l[:-1])
		client.addfs(fsC(x[5]))
	client.os=os
	client.version=ver
	client.access=access
	return client

#
def catchres(lines):
	res={}
	leftover=[]
	for tup in lines:
		l=tup[1]
		if l.find('--------------') == 0:
			continue
		r=string.split(l[:-1])
		if l.find('Executing') == 0:
			continue
		if len(r) < 2: 
			leftover.append(tup)
		elif len(r) == 2:
			res[r[0]]=r[1]
		elif len(r) > 2 and r[2][0] == '(' and r[-1][-1] == ')': 
			res[r[0]]=r[1]
		else:
			leftover.append(tup)
	return [res, leftover]

def prettytime(ts):
	t=float(ts)
	if t > 86400:
		return '>>:>>'
	return time.strftime("%H:%M", time.gmtime(t))

#
# thread runner
#
def runjob(threadno):
	while taskQ.qsize():
		client, cmd=taskQ.get(1)

		cmd['v']=3+verboseflag
		if cmd['v'] > 9:
			cmd['v']=9

		if clients[client].wakeup:
			logprint("%s   waking client\n" % timestamp())
			dbgprint("/usr/pkg/bin/wol %s" % clients[client].wakeup)
			runcmd(client, "wol", "/usr/pkg/bin/wol %s" % clients[client].wakeup)
			time.sleep(5)
			clients[client].wakeup=0

		if cmd['cmd'] == 'rdiff':
			exc=""
			for ex in cmd['excludelist']:
				exc+=' --exclude "%s"' % ex
			cmd['exclude']=exc
			cmdarg='-b -v%(v)d --remote-schema "%(schema)s" %(options)s %(exclude)s %(client)s::"%(fs)s" %(srvdir)s/%(client)s"%(tfs)s"' % cmd
			cmdbin=ourserver.rdiffbin
		elif cmd['cmd'] == 'rsync':
			exc=""
			for ex in cmd['excludelist']:
				if string.find(ex, cmd['fs']) == 0:
					ex=ex[len(cmd['fs']):]
				exc+=' --exclude="%s"' % ex
			cmd['exclude']=exc
			cmdarg='-axH --delete %(exclude)s %(client)s:"%(fs)s"/ "%(ltfs)s"' % cmd
			cmdbin="/usr/pkg/bin/rsync"

		elif cmd['cmd'] == 'cleanup':
			exc=""
			cmdarg='--force -v%(v)d --remove-older-than %(keep)s %(srvdir)s/%(client)s"%(tfs)s"' % cmd
			cmdbin=ourserver.rdiffbin

		if verboseflag:
			logprint("%s Starting thread %d for %s:%s\n" % (timestamp(), threadno, cmd['client'], cmd['fs']))
			logflush()
		res,err,code=runcmd(client, cmd['fs'], "%s %s" % (cmdbin, cmdarg),ts=1)
		resultQ.put([cmd, res,err,code])
	if verboseflag:
		logprint("%s Thread %d done\n" % (timestamp(), threadno))

#
# Main backup function
#
def do_backup():
	lockedClients=[]
	if not ourserver:
		logprint("This server (%s) not listed in config's server section\n" % ourhostname)
		logprint("Nothing to do, exiting\n")
		logclose()
		maillog()
		sys.exit(1)
	if len(args) > 1:
		if not args[1] in clients.keys():
			print "error: client %s not found in config" % args[1]
			sys.exit(1)
		backupclients=[args[1]]
	else:
		backupclients=clientlist

	if len(args) > 2:
		fslist=[args[2]]
	else:
		fslist=None

	ourpid="%s" % os.getpid()
	sysctls={}
	if len(ourserver.sysctl) > 0:
		for ctl in ourserver.sysctl:
			ctl=ctl.replace("$$",ourpid)	
			r=string.split(ctl,'=')
			if len(r) != 2:
				print "sysctl for server format incorrect"
				sys.exit(1)
			cmd="/sbin/sysctl -n %s" % r[0]
			res,err,code=runcmd("server","none",cmd)
			dbgprint("sysctl cmd: res %s err %s code %s" % (res,err,code))
			if len(res) != 1:
				print "sysctl for server result unexpected"
				sys.exit(1)
			oldval=res[0][:-1]
			if oldval != r[1]:		# sysctl is not what we want
				sysctls[r[0]]=oldval	# remember what it was
				cmd="/sbin/sysctl -w %s=%s" % (r[0],r[1])
				res,err,code=runcmd("server","none",cmd)
				dbgprint("sysctl cmd2: res %s err %s code %s" % (res,err,code))
				
	if ourserver.datalimit != None:
		lsoft, lhard=resource.getrlimit(resource.RLIMIT_DATA)
		if lsoft > ourserver.datalimit:
			logprint("warning: Data limit requested (%s), is greater than current (%s)\n" % (human(ourserver.datalimit), human(lsoft)))
		else:
			resource.setrlimit(resource.RLIMIT_DATA, (ourserver.datalimit, ourserver.datalimit))
			logprint("Data limit set to %s, was %s\n" % (human(ourserver.datalimit), human(lsoft)))

	starttime=time.time()
	logprint("HostBUP backup starting at %s for client(s) %s\n" % (timestamp(starttime), string.join(backupclients,', ')))
	for client in backupclients:
		cmd=rdiffcmd(client)
		if len(cmd) == 0:
			logprint("Client %s: nothing to backup\n" % client)
			continue

		if not fslist:
			fss=cmd.keys()
		else:
			fss=fslist

		for fs in fss:
			if cmd[fs]['server'] != ourhostname:
				dbgprint("Skipping client %s fs %s for server %s\n" % (cmd[fs]['client'], fs, cmd[fs]['server']))
				continue

			tfs="%(srvdir)s/%(client)s%(tfs)s" % cmd[fs]
			cmd[fs]['ltfs']=tfs
			if not os.path.exists(tfs):
				os.makedirs(tfs)
				logprint("%s created backup directory %s\n" % (timestamp(), tfs))
				cmd[fs]['cmd']=initialsync
			else:
				cmd[fs]['cmd']='rdiff'

			taskQ.put([client, cmd[fs]])
			if not client in lockedClients:
				clients[client].Lock()
				lockedClients+=[client]
	jobs=taskQ.qsize()
	threads=min(jobs, ourserver.queues)
	if verboseflag:
		logprint("%s jobs, starting %s threads\n" % (jobs, threads))
	logflush()

	for x in xrange(threads):
		thread.start_new_thread(runjob,(x,))

	# wait for all threads to finish
	while resultQ.qsize() < jobs:
		time.sleep(1)

	# restore sysctl values
	for k in sysctls.keys():
		cmd="/sbin/sysctl -w %s=%s" % (k,sysctls[k])
		res,err,code=runcmd("server","none",cmd)
		dbgprint("sysctl cmd3: res %s err %s code %s" % (res,err,code))
		
	cleanupdirs=[]
	sum={}
	sum['SourceFiles']=0
	sum['SourceFileSize']=0
	sum['TotalDestinationSizeChange']=0
	sum['Errors']=0
	logprint("\nSummary Report\n")
	logprint("              host:filesystem                   time  #files     Size Change err n\n")
	logprint("------------------:--------------------------- ----- ------- -------- ------ --- -\n")
	while jobs > 0:
			jobs-=1
			reportit=0
			cmd, res, err, code=resultQ.get()
			errors=len(err)
			if len(err) > 0:
				reportit=1

			if len(res) > 10:
				r, leftover=catchres(res)
				res=leftover 
			else:
				r=None
			if len(res) > 0 or r == None:
				reportit=1		

			logprint("%18s:%-27s" % (cmd['client'], cmd['fs']))
			if r:
				if reportit:
					Rep="*"
				else:
					Rep=""
				logprint(" %5s %7s %8.1f %6.1f %3s %1s\n" % \
					(prettytime(r['ElapsedTime']),r['SourceFiles'],float(r['SourceFileSize'])/1048576.0,\
					float(r['TotalDestinationSizeChange'])/1048576.0, r['Errors'], Rep))
				for n in ['SourceFiles', 'SourceFileSize', 'TotalDestinationSizeChange','Errors']:
					sum[n]+=float(r[n])
			else:
				logprint(" %5s %7s %8s %6s %3s %1s\n" % ('-', '-', '-','-',code,'*'))
			 
			if reportit:
				reportQ.put([cmd, res, err, code])
			cmd['cmd']='cleanup'
			cleanupdirs+=[cmd]
			logflush()
	logprint("------------------:--------------------------- ----- ------- -------- ------ --- -\n")
	endtime=time.time()
	logprint("%46s %5s %7s %8.1f %6.1f %3s\n" % \
		('*TOTAL*', prettytime(endtime-starttime), int(sum['SourceFiles']), float(sum['SourceFileSize'])/1048576.0,\
		float(sum['TotalDestinationSizeChange'])/1048576.0, sum['Errors']))
	

	if reportQ.empty():
		logprint("No Error Reports\n")
	else:
		logprint("Error Reports\n")
		while not reportQ.empty():
			cmd, res, err, code=reportQ.get()			
			logprint("%s:%s\n" % (cmd['client'], cmd['fs']))
			for l in err:
				logprint("%s err: %s" % (l[0], l[1]))
			if len(res) > 0:
				for l in res:
					logprint("%s log: %s" % (l[0], l[1]))
	
			if code != 0:
				logprint("%s exit code %s\n" % (timestamp(endtime), code))
			logprint("\n")

	endtime=run_cleanup(cleanupdirs)
	for client in lockedClients:
		clients[client].UnLock()

	try:
		x=os.statvfs('.')
	except:
		pass
	else:
		dirkeys=ourserver.directory.keys()
		dirkeys.sort()
		logprint("Backup Volume Status\n")
		logprint(" cap   free  total directory\n")
		logprint("---- ------ ------ -------------------------------------------------\n")
		for dir in dirkeys:
			usedB,totB=ourserver.space(dir)
			if totB == 0:
				spac=0
			else:
				spac=100.0*usedB / totB
			logprint("%3d%% %6d %6d %s\n" % (spac, usedB / 1048576L, totB / 1048576L, dir))
		logprint("---- ------ ------ -------------------------------------------------\n")
		logprint("\n")

	logprint("\nHostBUP backup finished at %s\n" % timestamp(endtime))
	logclose()
	maillog()

#
#  mail a report
# 
def maillog():
	if general.mailto:
		emsg="From: HostBUP on %s<%s>\nTo: %s\nSubject: hostbup report from %s\nMessage-Id: <%s@%s>\n\n%s\n" % \
					 (ourhostname, general.mailfrom, general.mailto, ourhostname, time.time(), ourhostname, logdump())
		if general.smtpserver == "viamail":
			x=os.popen("/usr/sbin/sendmail -t", "w")
			x.writelines(emsg)
			x.close()
		else:
			server=smtplib.SMTP(general.smtpserver)
			try:
				server.sendmail(general.mailfrom, [general.mailto], emsg)
			except:
				print "cannot send to server"
			server.quit() 
#
# cleanup function
#
def do_cleanup():
	if not ourserver:
		logprint("This server (%s) not listed in config's server section\n" % ourhostname)
		logprint("Nothing to do, exiting\n")
		logclose()
		maillog()
		sys.exit(1)
	if len(args) > 1:
		if not args[1] in clients.keys():
			print "error: client %s not found in config" % args[1]
			sys.exit(1)
		cleanupclients=[args[1]]
	else:
		cleanupclients=clientlist

	if len(args) > 2:
		fslist=[args[2]]
	else:
		fslist=None

	ourpid="%s" % os.getpid()

	cleanupdirs=[]
	starttime=time.time()
	logprint("HostBUP cleanup starting at %s for client(s) %s\n" % (timestamp(starttime), string.join(cleanupclients,', ')))
	for client in cleanupclients:
		cmd=rdiffcmd(client)
		if len(cmd) == 0:
			logprint("Client %s: nothing to cleanup\n" % client)
			continue

		if not fslist:
			fss=cmd.keys()
		else:
			fss=fslist

		for fs in fss:
			if cmd[fs]['server'] != ourhostname:
				dbgprint("Skipping client %s fs %s for server %s\n" % (cmd[fs]['client'], fs, cmd[fs]['server']))
				continue

			tfs="%(srvdir)s/%(client)s%(tfs)s" % cmd[fs]
			cmd[fs]['ltfs']=tfs
			if not os.path.exists(tfs):
				logprint("%s no backup directory %s\n" % (timestamp(), tfs))
				continue
			else:
				cmd[fs]['cmd']='cleanup'

			cleanupdirs+=[cmd[fs]]
			
	endtime=run_cleanup(cleanupdirs)
	dirkeys=ourserver.directory.keys()
	dirkeys.sort()
	logprint("Backup Volume Status\n")
	logprint(" cap   free  total directory\n")
	logprint("---- ------ ------ -------------------------------------------------\n")
	for dir in dirkeys:
		usedB,totB=ourserver.space(dir)
		spac=100.0*usedB / totB
		logprint("%3d%% %6d %6d %s\n" % (spac, usedB / 1048576L, totB / 1048576L, dir))
	logprint("---- ------ ------ -------------------------------------------------\n")
	logprint("\n")


	logprint("\nHostBUP cleanup finished at %s\n" % timestamp(endtime))
	logclose()
	maillog()


def run_cleanup(cleanupdirs):
	for  cmd in cleanupdirs:
		#print "cleanup: taskQ %s " % cmd['client']
		taskQ.put([cmd['client'], cmd])

	jobs=taskQ.qsize()
	threads=ourserver.queues
	if verboseflag:
		logprint("%s jobs, starting %s threads\n" % (jobs, threads))
	logflush()

	for x in xrange(threads):
		thread.start_new_thread(runjob,(x,))

	# wait for all threads to finish
	while resultQ.qsize() < jobs:
		time.sleep(1)

	#print "cleanup: %s " % cmd['client']
	while resultQ.qsize() > 0:
		cmd, res, err, code=resultQ.get()
		#print "resultQ.get  %s" % cmd['client']
#		for l in res:
#			logprint("%s" % l)
			#print "cleanupL: %s " % l
		reportQ.put([cmd, res, err, code])
		logflush()
	endtime=time.time()

	if reportQ.empty():
		logprint("\nNo Cleanup Reports\n")
	else:
		logprint("\nCleanup Reports\n")
		while not reportQ.empty():
			cmd, res, err, code=reportQ.get()			
			logprint("%s:%s\n" % (cmd['client'], cmd['fs']))
			for l in err:
				logprint("%s err: %s" % (l[0], l[1]))
			if len(res) > 0:
				for l in res:
					logprint("%s log: %s" % (l[0], l[1]))
	
			if code != 0:
				logprint("%s exit code %s\n" % (timestamp(endtime), code))
			logprint("\n")


	return endtime
	
#
# Main
#

taskQ=Queue.Queue()  
resultQ=Queue.Queue()
reportQ=Queue.Queue()

ouruid=os.getuid()

# N.B. AEW!! work around bug in ssh - it takes HOME over ~user
os.environ['HOME']=pwd.getpwuid(ouruid)[5]
ouruser=pwd.getpwuid(ouruid)[0]
ourhostname=os.uname()[1]

invokecmd=sys.argv[0]
cfgname="/etc/hostbup.conf"
logname=None
general=generalC()
clients={}
clientlist=[]
servers={}
serverlist=[]

verboseflag=0 
noflag=0
debugflag=0 
helpflag=0
initialsync="rdiff"
optlist=[]  
sig=[]

try:
	optlist, args = getopt.getopt(sys.argv[1:], 'c:dhl:n?rv')
except:         
	helpflag=1

  
for o,a in optlist:
	if o == '-c':
		cfgname=a
	if o == '-d':
		debugflag+=1 
	if o == '-h' or o == '-?':
		helpflag=1 
	if o == '-l':
		logname=a
	if o == '-n':
		noflag=1
	if o == '-r':
		initialsync="rsync"
	if o == '-v':
		verboseflag+=1 

if len(args) == 0:
	print "error: command missing"
	helpflag=1

if helpflag:
	print "usage: hostbup [-ndv] [-c configfile] [-l logfile] cmd [options]"
	print "	-c configfile	default: /etc/hostbup.conf"
	print "	-l logfile		default: /var/log/hostbup.log"
	print "	-n don't run, just print commands"
	print "	-d debug"
	print "	-r 				use rsync for the initial run"
	print "	-v verbose 	- make rdiff-backup more verbose"
	print "		backup [client [fs]]  - run backup for all clients, or just 'client' and 'fs'"
	print "		genconf client [rsh]  - contact client and generate a config entry"
	print "		dumpconf            - parse and dump current config file"
	print "		testaccess          - test access to each client"
	print "		testmail	        - test mail message to admin"
	print ""
	sys.exit(1)

command=args[0]
    
oldsigHUP=signal.signal(signal.SIGHUP, sighuphandler)

if command == 'genconf':
	if len(args) < 2:
		print "genconfig needs hostname argument"
		sys.exit(1)
	client=args[1]
	if len(args) == 3:
		access=args[2]
	else:
		access="ssh"
	client=genconf(client, access)
	print "# prune fs's"
	print 'client "%s" {\n%s};\n' % (client.name, client)
	sys.exit(0)
	

try:
	cfg=open(cfgname,"r")
except:
	print "error: cannot open config file '%s'" % cfgname
	sys.exit(1)

pp=shlex.shlex(cfg)
pp.commenters="#"
EOF=0

if command == 'dumpconf':
	loadconfig()
	dumpconfig()

elif command == 'testmail':
	loadconfig()
	logopen(logname)
	logprint("testing mail\n")
	logclose()
	maillog()

elif command == 'backup':
	loadconfig()
	
	if servers.has_key(ourhostname):
		ourserver=servers[ourhostname]
	else:
		ourserver=None
	logopen(logname)
	do_backup()

elif command == 'cleanup':
	loadconfig()
	
	if servers.has_key(ourhostname):
		ourserver=servers[ourhostname]
	else:
		ourserver=None
	logopen(logname)
	do_cleanup()

elif command == 'testaccess':
	loadconfig()
	logopen(logname)
	error=0
	for client in clientlist:
		print "Client %s" % client
		if clients[client].wakeup:
			if verboseflag:
				print "waking %s" % client
			dbgprint("/usr/local/bin/wol %s" % clients[client].wakeup)
			runcmd(client, 'wol',  "/usr/local/bin/wol %s" % clients[client].wakeup)
			time.sleep(5)

		if clients[client].rdiffbin:
			rdiffbin=clients[client].rdiffbin
		else:
			rdiffbin=general.rdiffbin
		res,err,code=runcmd(client, 'test',  "%s %s %s --version" % (clients[client].access, client, rdiffbin))
		if not code and len(res) > 0:
				print "  %s" % (string.join(res,'')),
		else:
				print  "  FAILED connect"
				error=1
				continue
	sys.exit(error)

else:
	print "error: unknown command %s" % command
