// Filter for display access key tree walker
function webdeveloper_accessKeyFilter(node)
{
    // If the node has an accesskey attribute
    if(node.hasAttribute("accesskey"))
    {
        return NodeFilter.FILTER_ACCEPT;
    }
    else
    {
        return NodeFilter.FILTER_SKIP;
    }
}

// Filter for display anchors tree walker
function webdeveloper_anchorFilter(node)
{
    // If the node has an id attribute or is an a tag with a name attribute
    if(node.hasAttribute("id") || (node.nodeName && node.nodeName.toLowerCase() == "a" && node.hasAttribute("name")))
    {
        return NodeFilter.FILTER_ACCEPT;
    }
    else
    {
        return NodeFilter.FILTER_SKIP;
    }
}

// Displays all access keys
function webdeveloper_displayAccessKeys(element, applyStyle)
{
    const mainTabBox   = getBrowser().mTabBox;
    const documentList = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());

    var accessKeyElement     = null;
    var accessKeyElementList = null;
    var pageDocument         = null;
    var spanElement          = null;
    var spanElementList      = null;

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument     = documentList[i];
        spanElementList  = pageDocument.getElementsByTagName("span");

        // Loop through all the span tags
        for(var j = 0; j < spanElementList.length; j++)
        {
            spanElement = spanElementList[j];

            // If the class exists and is set to webdeveloper-access-key-span
            if(spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-access-key-span")
            {
                spanElement.parentNode.removeChild(spanElement);
                j--;
            }
        }

        // If the element is checked
        if(element.getAttribute("checked"))
        {
            const treeWalker = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_accessKeyFilter, false);

            accessKeyElementList = new Array();

            // Loop through the access key elements
            while((accessKeyElement = treeWalker.nextNode()) != null)
            {
                accessKeyElementList.push(accessKeyElement);
            }

            // Loop through all the access key elements
            for(j = 0; j < accessKeyElementList.length; j++)
            {
                accessKeyElement = accessKeyElementList[j];
                spanElement      = pageDocument.createElement("span");

                spanElement.setAttribute("class", "webdeveloper-access-key-span");
                spanElement.appendChild(pageDocument.createTextNode("Accesskey=" + accessKeyElement.getAttribute("accesskey")));
                accessKeyElement.parentNode.insertBefore(spanElement, accessKeyElement);
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_access_keys.css", "webdeveloper-display-access-keys", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-access-keys-tooltips", ".webdeveloper-access-key-span");
}

// Displays all anchors
function webdeveloper_displayAnchors(element, applyStyle)
{
    const mainTabBox   = getBrowser().mTabBox;
    const documentList = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());

    var anchorElement     = null;
    var anchorElementList = null;
    var anchorText        = null;
    var pageDocument      = null;
    var spanElement       = null;
    var spanElementList   = null;

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument     = documentList[i];
        spanElementList  = pageDocument.getElementsByTagName("span");

        // Loop through all the span tags
        for(var j = 0; j < spanElementList.length; j++)
        {
            spanElement = spanElementList[j];

            // If the class exists and is set to webdeveloper-anchor-span
            if(spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-anchor-span")
            {
                spanElement.parentNode.removeChild(spanElement);
                j--;
            }
        }

        // If the element is checked
        if(element.getAttribute("checked"))
        {
            const treeWalker = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_anchorFilter, false);

            anchorElementList = new Array();

            // Loop through the anchor elements
            while((anchorElement = treeWalker.nextNode()) != null)
            {
                anchorElementList.push(anchorElement);
            }

            // Loop through all the anchor elements
            for(j = 0; j < anchorElementList.length; j++)
            {
                anchorElement = anchorElementList[j];
                spanElement   = pageDocument.createElement("span");

                // If the anchor element has an id attribute
                if(anchorElement.hasAttribute("id"))
                {
                    anchorText = "#" + anchorElement.getAttribute("id");
                }
                else if(anchorElement.hasAttribute("name"))
                {
                    anchorText = "#" + anchorElement.getAttribute("name");
                }

                spanElement.setAttribute("class", "webdeveloper-anchor-span");
                spanElement.appendChild(pageDocument.createTextNode(anchorText));
                anchorElement.parentNode.insertBefore(spanElement, anchorElement);
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_anchors.css", "webdeveloper-display-anchors", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-anchors-tooltips", ".webdeveloper-anchor-span");
}

// Displays all block sizes
function webdeveloper_displayBlockSize(element, applyStyle)
{
    const display      = element.getAttribute("checked");
    const mainTabBox   = getBrowser().mTabBox;
    const documentList = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());

    var pageDocument = null;

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        var blockSizeElement     = null;
        var blockSizeElementList = null;

        pageDocument = documentList[i];

        // If displaying
        if(display)
        {
            const blockSizeElementTypes = new Array("div", "form", "table");

            var blockSize         = null;
            var height            = null;
            var left              = null;
            var top               = null;
            var width             = null;

            // Loop through the block size elements
            for(var j = 0; j < blockSizeElementTypes.length; j++)
            {
                blockSizeElementList = pageDocument.getElementsByTagName(blockSizeElementTypes[j]);

                // Loop through the block size elements
                for(var k = 0; k < blockSizeElementList.length; k++)
                {
                    blockSize        = pageDocument.createElement("span");
                    blockSizeElement = blockSizeElementList[k];
                    height           = blockSizeElement.offsetHeight;
                    left             = blockSizeElement.offsetLeft;
                    top              = blockSizeElement.offsetTop;
                    width            = blockSizeElement.offsetWidth;

                    blockSize.appendChild(pageDocument.createTextNode(width + "x" + height));
                    blockSize.setAttribute("class", "webdeveloper-block-size");
                    blockSize.style.left = left + "px";
                    blockSize.style.position = "absolute";
                    blockSize.style.top = top + "px";

                    blockSizeElement.appendChild(blockSize);
                }
            }
        }
        else
        {
            const treeWalker = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_displayBlockSizeFilter, false);

            blockSizeElementList = new Array();

            // Loop through the block size elements
            while((blockSizeElement = treeWalker.nextNode()) != null)
            {
                blockSizeElementList.push(blockSizeElement);
            }

            // Loop through the block size elements
            for(j = 0; j < blockSizeElementList.length; j++)
            {
                blockSizeElement = blockSizeElementList[j];
                blockSizeElement.parentNode.removeChild(blockSizeElement);
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_block_size.css", "webdeveloper-display-block-size", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-block-size-tooltips", ".webdeveloper-block-size");
}

// Filter for display block size tree walker
function webdeveloper_displayBlockSizeFilter(node)
{
    // If the class of the node is webdeveloper block size
    if(node.getAttribute("class") == "webdeveloper-block-size")
    {
        return NodeFilter.FILTER_ACCEPT;
    }
    else
    {
        return NodeFilter.FILTER_SKIP;
    }
}

// Displays ID and class details for the page
function webdeveloper_displayIDClassDetails(element, applyStyle)
{
    // If the menu is checked
    if(element.getAttribute("checked"))
    {
        const mainTabBox         = getBrowser().mTabBox;
        const documentList       = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());
        const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");

        var headElementList = null;
        var pageDocument    = null;
        var styleElement    = null;

        // Loop through the documents
        for(var i = 0; i < documentList.length; i++)
        {
            pageDocument    = documentList[i];
            headElementList = pageDocument.getElementsByTagName("head");
            styleElement    = pageDocument.createElement("style");

            styleElement.setAttribute("id", "webdeveloper-display-id-class-details");
            styleElement.setAttribute("type", "text/css");

            // If the class colors are set
            if(preferencesService.prefHasUserValue("webdeveloper.color.class.background") && preferencesService.prefHasUserValue("webdeveloper.color.class.text"))
            {
                styleElement.appendChild(pageDocument.createTextNode("*[class]:before { background-color: " + preferencesService.getComplexValue("webdeveloper.color.class.background", Components.interfaces.nsISupportsString).data.trim() + " !important; color: " + preferencesService.getComplexValue("webdeveloper.color.class.text", Components.interfaces.nsISupportsString).data.trim() + ' !important; content: "Class=" attr(class) !important; }'));
            }

            // If the ID colors are set
            if(preferencesService.prefHasUserValue("webdeveloper.color.id.background") && preferencesService.prefHasUserValue("webdeveloper.color.id.text"))
            {
                styleElement.appendChild(pageDocument.createTextNode("*[id]:before { background-color: " + preferencesService.getComplexValue("webdeveloper.color.id.background", Components.interfaces.nsISupportsString).data.trim() + " !important; color: " + preferencesService.getComplexValue("webdeveloper.color.id.text", Components.interfaces.nsISupportsString).data.trim() + ' !important; content: "Id=" attr(id) !important; }'));
            }

            // If the ID and class colors are set
            if(preferencesService.prefHasUserValue("webdeveloper.color.id.class.background") && preferencesService.prefHasUserValue("webdeveloper.color.id.class.text"))
            {
                styleElement.appendChild(pageDocument.createTextNode("*[id][class]:before { background-color: " + preferencesService.getComplexValue("webdeveloper.color.id.class.background", Components.interfaces.nsISupportsString).data.trim() + " !important; color: " + preferencesService.getComplexValue("webdeveloper.color.id.class.text", Components.interfaces.nsISupportsString).data.trim() + ' !important; content: "Id=" attr(id) ", Class=" attr(class) !important; }'));
            }

            // If there is a head element
            if(headElementList.length > 0)
            {
                headElementList[0].appendChild(styleElement);
            }
            else
            {
                pageDocument.documentElement.appendChild(styleElement);
            }
        }

        // If applying styles
        if(applyStyle)
        {
            webdeveloper_addAppliedStyle("webdeveloper-display-id-class-details");
        }
    }
    else
    {
        webdeveloper_removeStyleSheet("webdeveloper-display-id-class-details", true);
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/imports/before.css", "webdeveloper-display-id-class-details-before", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-id-class-details-tooltips", "*:before");
}

// Displays all link paths
function webdeveloper_displayLinkPaths(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_link_paths.css", "webdeveloper-display-link-paths", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-link-paths-tooltips", "*:before");
}

// Displays all tab indices
function webdeveloper_displayTabIndex(element, applyStyle)
{
    const mainTabBox   = getBrowser().mTabBox;
    const documentList = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());

    var pageDocument        = null;
    var spanElement         = null;
    var spanElementList     = null;
    var tabIndexElement     = null;
    var tabIndexElementList = null;

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument     = documentList[i];
        spanElementList  = pageDocument.getElementsByTagName("span");

        // Loop through all the span tags
        for(var j = 0; j < spanElementList.length; j++)
        {
            spanElement = spanElementList[j];

            // If the class exists and is set to webdeveloper-tab-index-span
            if(spanElement.hasAttribute("class") && spanElement.getAttribute("class") == "webdeveloper-tab-index-span")
            {
                spanElement.parentNode.removeChild(spanElement);
                j--;
            }
        }

        // If the element is checked
        if(element.getAttribute("checked"))
        {
            const treeWalker = pageDocument.createTreeWalker(pageDocument.body, NodeFilter.SHOW_ELEMENT, webdeveloper_tabIndexFilter, false);

            tabIndexElementList = new Array();

            // Loop through the tab index elements
            while((tabIndexElement = treeWalker.nextNode()) != null)
            {
                tabIndexElementList.push(tabIndexElement);
            }

            // Loop through all the tab index elements
            for(j = 0; j < tabIndexElementList.length; j++)
            {
                tabIndexElement = tabIndexElementList[j];

                spanElement = pageDocument.createElement("span");

                spanElement.setAttribute("class", "webdeveloper-tab-index-span");
                spanElement.appendChild(document.createTextNode("Tabindex=" + tabIndexElement.getAttribute("tabindex")));
                tabIndexElement.parentNode.insertBefore(spanElement, tabIndexElement);
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_tab_index.css", "webdeveloper-display-tab-index", applyStyle);
    webdeveloper_toggleFeatureTooltipStyles(element, "webdeveloper-display-tab-index-tooltips", ".webdeveloper-tab-index-span");
}

// Displays the topographic information for all elements
function webdeveloper_displayTopographicInformation(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/display_topographic_information.css", "webdeveloper-display-topographic-information", applyStyle);
}

// Sorts two links
function webdeveloper_sortLinks(linkOne, linkTwo)
{
    const linkOneHref = linkOne.href;
    const linkTwoHref = linkTwo.href;

    var sortValue = 1;

    // If the links are equal
    if(linkOneHref == linkTwoHref)
    {
        sortValue = 0;
    }
    else if(linkOneHref < linkTwoHref)
    {
        sortValue = -1;
    }

    return sortValue;
}

// Filter for display tab index tree walker
function webdeveloper_tabIndexFilter(node)
{
    // If the node has an tabindex attribute
    if(node.hasAttribute("tabindex"))
    {
        return NodeFilter.FILTER_ACCEPT;
    }
    else
    {
        return NodeFilter.FILTER_SKIP;
    }
}

// Tidies a list of links by removing duplicates and sorting
function webdeveloper_tidyLinks(linkList)
{
    var link              = null;
    var newLinkList       = new Array();
    var newLinkListLength = new Array();
    var tidiedLinkList    = new Array();

    // Loop through the links
    for(var i = 0; i < linkList.length; i++)
    {
        link = linkList[i];

        // If this link is set
        if(link.href)
        {
            newLinkList.push(link);
        }
    }

    newLinkList.sort(webdeveloper_sortLinks);
    newLinkListLength = newLinkList.length;

    // Loop through the links
    for(i = 0; i < newLinkListLength; i++)
    {
        link = newLinkList[i];

        // If this is not the last link and the link is the same as the next link
        if(i + 1 < newLinkListLength && link.href == newLinkList[i + 1].href)
        {
            continue;
        }

        tidiedLinkList.push(link);
    }

    return tidiedLinkList;
}

// Displays all the cookies for the page
function webdeveloper_viewCookieInformation()
{
    const dateService        = Components.classes["@mozilla.org/intl/scriptabledateformat;1"].getService(Components.interfaces.nsIScriptableDateFormat);
    const mainTabBox         = getBrowser().mTabBox;
    const documentList       = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());
    const oldTab             = getBrowser().selectedTab;
    const oldURL             = getBrowser().currentURI.spec;
    const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");
    const request            = new XMLHttpRequest();
    const stringBundle       = document.getElementById("webdeveloper-string-bundle");
    const title              = stringBundle.getFormattedString("webdeveloper_viewCookieInformationTitle", [oldURL]);

    var cellDataElement   = null;
    var cellHeaderElement = null;
    var cookie            = null;
    var cookieExpires     = null;
    var cookieHost        = null;
    var cookieLength      = null;
    var cookieSecure      = null;
    var cookies           = null;
    var date              = null;
    var generatedPage     = null;
    var headerElement     = null;
    var linkElement       = null;
    var pageDocument      = null;
    var pElement          = null;
    var preElement        = null;
    var tableElement      = null;
    var tableRowElement   = null;

    generatedPage = webdeveloper_generatePage("");

    // This must be done to make generated content render
    request.open("get", "about:blank", false);
    request.send("");

    generatedPage.content.document.title = title;

    linkElement = generatedPage.content.document.createElement("link");
    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated_content.css");
    linkElement.setAttribute("media", "all");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedPage.content.document.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement = generatedPage.content.document.createElement("h1");
    headerElement.appendChild(generatedPage.content.document.createTextNode(title));
    generatedPage.content.document.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument = documentList[i];
        cookies      = webdeveloper_getCookies(pageDocument.location.hostname, true);
        cookieLength = cookies.length;

        headerElement = generatedPage.content.document.createElement("h2");
        linkElement   = generatedPage.content.document.createElement("a");
        linkElement.setAttribute("href", pageDocument.documentURI);
        linkElement.appendChild(generatedPage.content.document.createTextNode(pageDocument.documentURI));
        headerElement.appendChild(linkElement);
        generatedPage.content.document.body.appendChild(headerElement);

        // If there are no cookies
        if(cookieLength == 0)
        {
            pElement = generatedPage.content.document.createElement("p");
            pElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_noCookies")));
            generatedPage.content.document.body.appendChild(pElement);
        }
        else
        {
            // Loop through the cookies
            for(var j = 0; j < cookieLength; j++)
            {
                cookie       = cookies[j];
                cookieHost   = cookie.host;
                cookieSecure = cookie.isSecure;
                tableElement = generatedPage.content.document.createElement("table");

                //  Cookie name
                cellDataElement   = generatedPage.content.document.createElement("td");
                cellHeaderElement = generatedPage.content.document.createElement("th");
                tableRowElement   = generatedPage.content.document.createElement("tr");

                cellHeaderElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_viewCookieInformationName")));
                tableRowElement.appendChild(cellHeaderElement);
                cellDataElement.appendChild(generatedPage.content.document.createTextNode(cookie.name));
                tableRowElement.appendChild(cellDataElement);
                tableElement.appendChild(tableRowElement);

                //  Cookie value
                cellDataElement   = generatedPage.content.document.createElement("td");
                cellHeaderElement = generatedPage.content.document.createElement("th");
                tableRowElement   = generatedPage.content.document.createElement("tr");

                tableRowElement.setAttribute("class", "odd");
                cellHeaderElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_viewCookieInformationValue")));
                tableRowElement.appendChild(cellHeaderElement);
                cellDataElement.appendChild(generatedPage.content.document.createTextNode(cookie.value));
                tableRowElement.appendChild(cellDataElement);
                tableElement.appendChild(tableRowElement);

                // If the cookie host starts with '.'
                if(cookieHost.charAt(0) == ".")
                {
                    cookieHost = cookieHost.substring(1);
                }

                //  Cookie host
                cellDataElement   = generatedPage.content.document.createElement("td");
                cellHeaderElement = generatedPage.content.document.createElement("th");
                tableRowElement   = generatedPage.content.document.createElement("tr");

                cellHeaderElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_viewCookieInformationHost")));
                tableRowElement.appendChild(cellHeaderElement);
                cellDataElement.appendChild(generatedPage.content.document.createTextNode(cookieHost));
                tableRowElement.appendChild(cellDataElement);
                tableElement.appendChild(tableRowElement);

                //  Cookie path
                cellDataElement   = generatedPage.content.document.createElement("td");
                cellHeaderElement = generatedPage.content.document.createElement("th");
                tableRowElement   = generatedPage.content.document.createElement("tr");

                tableRowElement.setAttribute("class", "odd");
                cellHeaderElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_viewCookieInformationPath")));
                tableRowElement.appendChild(cellHeaderElement);
                cellDataElement.appendChild(generatedPage.content.document.createTextNode(cookie.path));
                tableRowElement.appendChild(cellDataElement);
                tableElement.appendChild(tableRowElement);

                //  Cookie secure
                cellDataElement   = generatedPage.content.document.createElement("td");
                cellHeaderElement = generatedPage.content.document.createElement("th");
                tableRowElement   = generatedPage.content.document.createElement("tr");

                cellHeaderElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_viewCookieInformationSecure")));
                tableRowElement.appendChild(cellHeaderElement);

                // If the cookie is secure
                if(cookieSecure)
                {
                    cellDataElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_yes")));
                }
                else
                {
                    cellDataElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_no")));
                }

                tableRowElement.appendChild(cellDataElement);
                tableElement.appendChild(tableRowElement);

                // If the cookie has an expiration date
                if(cookie.expires)
                {
                    date          = new Date(cookie.expires * 1000);
                    cookieExpires = dateService.FormatDateTime("", dateService.dateFormatLong, dateService.timeFormatSeconds, date.getFullYear(), date.getMonth()+1, date.getDate(), date.getHours(), date.getMinutes(), date.getSeconds());
                }
                else
                {
                    cookieExpires = stringBundle.getString("webdeveloper_viewCookieInformationSession");
                }

                //  Cookie expires
                cellDataElement   = generatedPage.content.document.createElement("td");
                cellHeaderElement = generatedPage.content.document.createElement("th");
                tableRowElement   = generatedPage.content.document.createElement("tr");

                tableRowElement.setAttribute("class", "odd");
                cellHeaderElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_viewCookieInformationExpires")));
                tableRowElement.appendChild(cellHeaderElement);
                cellDataElement.appendChild(generatedPage.content.document.createTextNode(cookieExpires));
                tableRowElement.appendChild(cellDataElement);
                tableElement.appendChild(tableRowElement);

                generatedPage.content.document.body.appendChild(tableElement);

                pElement = generatedPage.content.document.createElement("p");
                generatedPage.content.document.body.appendChild(pElement);
            }
        }
    }

    // If the open tabs in background preference is set to true
    if(preferencesService.prefHasUserValue("webdeveloper.open.tabs.background") && preferencesService.getBoolPref("webdeveloper.open.tabs.background"))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// View JavaScript
function webdeveloper_viewJavaScript()
{
    const mainTabBox         = getBrowser().mTabBox;
    const documentList       = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());
    const oldTab             = getBrowser().selectedTab;
    const oldURL             = getBrowser().currentURI.spec;
    const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");
    const request            = new XMLHttpRequest();
    const stringBundle       = document.getElementById("webdeveloper-string-bundle");
    const title              = stringBundle.getFormattedString("webdeveloper_viewJavaScriptTitle", [oldURL]);

    var documentURL   = null;
    var generatedPage = null;
    var headerElement = null;
    var linkElement   = null;
    var pageDocument  = null;
    var preElement    = null;
    var scriptElement = null;
    var scriptList    = new Array();
    var scriptSource  = null;

    generatedPage = webdeveloper_generatePage("");

    // This must be done to make generated content render
    request.open("get", "about:blank", false);
    request.send("");

    generatedPage.content.document.title = title;

    linkElement = generatedPage.content.document.createElement("link");
    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated_content.css");
    linkElement.setAttribute("media", "all");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedPage.content.document.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement = generatedPage.content.document.createElement("h1");
    headerElement.appendChild(generatedPage.content.document.createTextNode(title));
    generatedPage.content.document.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument = documentList[i];
        documentURL  = pageDocument.documentURI;
        scriptList   = pageDocument.getElementsByTagName("script");

        headerElement = generatedPage.content.document.createElement("h2");
        linkElement   = generatedPage.content.document.createElement("a");
        linkElement.setAttribute("href", pageDocument.documentURI);
        linkElement.appendChild(generatedPage.content.document.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedPage.content.document.body.appendChild(headerElement);

        // Loop through the scripts
        for(var j = 0; j < scriptList.length; j++)
        {
            scriptElement = scriptList[j];
            scriptSource  = scriptElement.src;

            // If the script is external
            if(scriptSource)
            {
                request.open("get", scriptSource, false);
                request.send("");

                headerElement = generatedPage.content.document.createElement("h2");
                linkElement   = generatedPage.content.document.createElement("a");
                linkElement.setAttribute("href", scriptSource);
                linkElement.appendChild(generatedPage.content.document.createTextNode(scriptSource));
                headerElement.appendChild(linkElement);
                generatedPage.content.document.body.appendChild(headerElement);

                preElement = generatedPage.content.document.createElement("pre");
                preElement.appendChild(generatedPage.content.document.createTextNode(request.responseText));
                generatedPage.content.document.body.appendChild(preElement);
            }
            else
            {
                headerElement = generatedPage.content.document.createElement("h2");
                headerElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getFormattedString("webdeveloper_inlineScript", [documentURL])));
                generatedPage.content.document.body.appendChild(headerElement);

                preElement = generatedPage.content.document.createElement("pre");
                preElement.appendChild(generatedPage.content.document.createTextNode(scriptElement.innerHTML));
                generatedPage.content.document.body.appendChild(preElement);
            }
        }
    }

    // If the open tabs in background preference is set to true
    if(preferencesService.prefHasUserValue("webdeveloper.open.tabs.background") && preferencesService.getBoolPref("webdeveloper.open.tabs.background"))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// Displays all the links for the page
function webdeveloper_viewLinkInformation()
{
    const mainTabBox         = getBrowser().mTabBox;
    const documentList       = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());
    const oldTab             = getBrowser().selectedTab;
    const oldURL             = getBrowser().currentURI.spec;
    const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");
    const request            = new XMLHttpRequest();
    const stringBundle       = document.getElementById("webdeveloper-string-bundle");
    const title              = stringBundle.getFormattedString("webdeveloper_viewLinkInformationTitle", [oldURL]);

    var documentURL       = null;
    var generatedPage     = null;
    var headerElement     = null;
    var link              = null;
    var linkElement       = null;
    var linkHref          = null;
    var linkLength        = null;
    var links             = null;
    var listElement       = null;
    var listItemElement   = null;
    var pageDocument      = null;
    var pElement          = null;

    generatedPage = webdeveloper_generatePage("");

    // This must be done to make generated content render
    request.open("get", "about:blank", false);
    request.send("");

    generatedPage.content.document.title = title;

    linkElement = generatedPage.content.document.createElement("link");
    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated_content.css");
    linkElement.setAttribute("media", "all");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedPage.content.document.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement = generatedPage.content.document.createElement("h1");
    headerElement.appendChild(generatedPage.content.document.createTextNode(title));
    generatedPage.content.document.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument = documentList[i];
        documentURL  = pageDocument.documentURI;
        links        = webdeveloper_tidyLinks(pageDocument.getElementsByTagName("a"));
        linkLength   = links.length;

        headerElement = generatedPage.content.document.createElement("h2");
        linkElement   = generatedPage.content.document.createElement("a");
        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedPage.content.document.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedPage.content.document.body.appendChild(headerElement);

        // If there are no links
        if(linkLength == 0)
        {
            pElement = generatedPage.content.document.createElement("p");
            pElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getString("webdeveloper_noLinks")));
            generatedPage.content.document.body.appendChild(pElement);
        }
        else
        {
            listElement = generatedPage.content.document.createElement("ol");

            // Loop through the links
            for(var j = 0; j < linkLength; j++)
            {
                link     = links[j];
                linkHref = link.href;

                // If the link is set
                if(linkHref)
                {
                    listItemElement = generatedPage.content.document.createElement("li");
                    linkElement     = generatedPage.content.document.createElement("a");
                    linkElement.setAttribute("href", linkHref);
                    linkElement.appendChild(generatedPage.content.document.createTextNode(linkHref));
                    listItemElement.appendChild(linkElement);
                    listElement.appendChild(listItemElement);
                }
            }

            generatedPage.content.document.body.appendChild(listElement);
        }
    }

    // If the open tabs in background preference is set to true
    if(preferencesService.prefHasUserValue("webdeveloper.open.tabs.background") && preferencesService.getBoolPref("webdeveloper.open.tabs.background"))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// View page information
function webdeveloper_viewPageInformation(chromeLocation)
{
    window.openDialog(chromeLocation, "webdeveloper-page-information-dialog", "chrome,dialog=no,resizable");
}

// View the response headers
function webdeveloper_viewResponseHeaders()
{
    const oldTab             = getBrowser().selectedTab;
    const oldURL             = getBrowser().currentURI.spec;
    const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");
    const request            = new XMLHttpRequest();
    const stringBundle       = document.getElementById("webdeveloper-string-bundle");
    const title              = stringBundle.getFormattedString("webdeveloper_viewResponseHeadersTitle", [oldURL]);

    var generatedPage   = null;
    var headerElement   = null;
    var linkElement     = null;
    var preElement      = null;
    var responseHeaders = null;

    generatedPage = webdeveloper_generatePage("");

    // This must be done to make generated content render
    request.open("get", oldURL, false);
    request.send("");
    responseHeaders = request.getAllResponseHeaders();

    generatedPage.content.document.title = title;

    linkElement = generatedPage.content.document.createElement("link");
    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated_content.css");
    linkElement.setAttribute("media", "all");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedPage.content.document.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement = generatedPage.content.document.createElement("h1");
    headerElement.appendChild(generatedPage.content.document.createTextNode(title));
    generatedPage.content.document.body.appendChild(headerElement);

    preElement = generatedPage.content.document.createElement("pre");
    preElement.appendChild(generatedPage.content.document.createTextNode(responseHeaders + "\n" + request.status + " " + request.statusText));
    generatedPage.content.document.body.appendChild(preElement);

    // If the open tabs in background preference is set to true
    if(preferencesService.prefHasUserValue("webdeveloper.open.tabs.background") && preferencesService.getBoolPref("webdeveloper.open.tabs.background"))
    {
        getBrowser().selectedTab = oldTab;
    }
}
