// Adds a user style sheet to the page
function webdeveloper_addUserStyleSheet(element, applyStyle)
{
    const addStyleSheet = element.getAttribute("checked");

    // If adding a style sheet and applying styles
    if(addStyleSheet && applyStyle)
    {
        const filePicker   = Components.classes["@mozilla.org/filepicker;1"].createInstance(Components.interfaces.nsIFilePicker);
        const stringBundle = document.getElementById("webdeveloper-string-bundle");

        filePicker.appendFilter(stringBundle.getString("webdeveloper_styleSheetDescription"), "*.css");
        filePicker.init(window, stringBundle.getString("webdeveloper_addUserStyleSheetTitle"), filePicker.modeOpen);

        // If the user selected a style sheet
        if(filePicker.show() == filePicker.returnOK)
        {
            const inputStream      = Components.classes["@mozilla.org/network/file-input-stream;1"].createInstance(Components.interfaces.nsIFileInputStream);
            const scriptableStream = Components.classes["@mozilla.org/scriptableinputstream;1"].createInstance(Components.interfaces.nsIScriptableInputStream);

            inputStream.init(filePicker.file, 0x01, 0444, null);
            scriptableStream.init(inputStream);

            webdeveloper_userStyleSheet = scriptableStream.read(scriptableStream.available());

            scriptableStream.close();
            inputStream.close();
        }
        else
        {
            addStyleSheet               = false;
            webdeveloper_userStyleSheet = null;
        }
    }

    // If adding a style sheet and the style sheet is not empty
    if(addStyleSheet && webdeveloper_userStyleSheet)
    {
        const mainTabBox   = getBrowser().mTabBox;
        const documentList = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());

        var headElementList = null;
        var pageDocument    = null;
        var styleElement    = null;

        // Loop through the documents
        for(var i = 0; i < documentList.length; i++)
        {
            pageDocument    = documentList[i];
            headElementList = pageDocument.getElementsByTagName("head");
            styleElement    = pageDocument.createElement("style");

            styleElement.setAttribute("id", "webdeveloper-add-user-style-sheet");
            styleElement.setAttribute("type", "text/css");
            styleElement.appendChild(pageDocument.createTextNode(webdeveloper_userStyleSheet));

            // If there is a head element
            if(headElementList.length > 0)
            {
                headElementList[0].appendChild(styleElement);
            }
            else
            {
                pageDocument.documentElement.appendChild(styleElement);
            }
        }

        // If applying styles
        if(applyStyle)
        {
            webdeveloper_addAppliedStyle("webdeveloper-add-user-style-sheet");
        }
    }
    else
    {
        webdeveloper_userStyleSheet = null;
        webdeveloper_removeStyleSheet("webdeveloper-add-user-style-sheet", true);
    }

    webdeveloper_configureEquivalentElements(element, "checked", addStyleSheet);
}

// Display CSS for the given media type
function webdeveloper_displayCSSMediaType(type)
{
    const mainTabBox   = getBrowser().mTabBox;
    const documentList = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());

    var media          = null;
    var mediaText      = null;
    var ownerNode      = null;
    var pageDocument   = null;
    var styleSheet     = null;
    var styleSheetList = null;

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument   = documentList[i];
        styleSheetList = pageDocument.styleSheets;

        // Loop through the style sheets
        for(var j = 0; j < styleSheetList.length; j++)
        {
            styleSheet = styleSheetList[j];

            // If the style sheet is set
            if(styleSheet)
            {
                media     = styleSheet.media;
                ownerNode = styleSheet.ownerNode;

                // If the style sheet has an owner node
                if(ownerNode)
                {
                    // If the owner node is a processing instruction or is an alternate style sheet
                    if(ownerNode.nodeType == 7 || (ownerNode.hasAttribute("rel") && ownerNode.getAttribute("rel").toLowerCase() == "alternate stylesheet"))
                    {
                        styleSheet.disabled = true;
                    }
                    else
                    {
                        // If the owner node has a webdeveloper-added-medium attribute
                        if(ownerNode.hasAttribute("webdeveloper-added-medium"))
                        {
                            media.deleteMedium("screen");
                            ownerNode.removeAttribute("webdeveloper-added-medium");
                        }

                        mediaText = media.mediaText;

                        // If there is no media text and the type is not screen or there is media text and it does not contain all or the type
                        if((!mediaText && type != "screen") || (mediaText && mediaText.indexOf("all") == -1 && mediaText.indexOf(type) == -1))
                        {
                            styleSheet.disabled = true;
                        }
                        else
                        {
                            styleSheet.disabled = false;

                            // If the media text does not contain screen
                            if(mediaText.indexOf("screen") == -1)
                            {
                                ownerNode.setAttribute("webdeveloper-added-medium", "true");
                                media.appendMedium("screen");
                            }
                        }
                    }
                }
                else
                {
                    styleSheet.disabled = true;
                }
            }
        }
    }
}

// Display the disable individual style sheet menu
function webdeveloper_displayDisableIndividualStyleSheetMenu(menu, tooltips)
{
    const mainTabBox      = getBrowser().mTabBox;
    const currentDocument = getBrowser().browsers[mainTabBox.selectedIndex].contentDocument;
    const stringBundle    = document.getElementById("webdeveloper-string-bundle");

    var mediaList       = null;
    var menuItem        = null;
    var ownerNode       = null;
    var styleSheet      = null;
    var styleSheetHref  = null;
    var styleSheetLabel = null;
    var styleSheetList  = currentDocument.styleSheets;

    webdeveloper_removeGeneratedMenuItems(menu);

    // Loop through the style sheets
    for(var i = 0; i < styleSheetList.length; i++)
    {
        styleSheet     = styleSheetList[i];
        mediaList      = styleSheet.media.mediaText;
        ownerNode      = styleSheet.ownerNode;
        styleSheetHref = styleSheet.href;

        // If this is not an inline style sheet, a rule from the browser or an alternate style sheet
        if(styleSheetHref != currentDocument.documentURI && styleSheetHref.indexOf("resource://") != 0 && styleSheetHref.indexOf("about:PreferenceStyleSheet") != 0 && styleSheetHref.indexOf("jar:file://") != 0 && styleSheetHref.indexOf("chrome://") != 0 && (!ownerNode || ownerNode.nodeType == 7 || !ownerNode.hasAttribute("rel") || ownerNode.getAttribute("rel").toLowerCase() != "alternate stylesheet"))
        {
            menuItem = document.createElement("menuitem");

            // If the style sheet is imported from a processing instruction or does not have a title
            if(ownerNode.nodeType == 7 || !ownerNode.hasAttribute("title"))
            {
                styleSheetLabel = styleSheetHref;
            }
            else
            {
                styleSheetLabel = ownerNode.getAttribute("title");
            }

            menuItem.setAttribute("class", "webdeveloper-generated-menu");
            menuItem.setAttribute("label", styleSheetLabel);
            menuItem.setAttribute("oncommand", "webdeveloper_toggleIndividualStyleSheet('" + styleSheetHref + "')");
            menuItem.setAttribute("type", "checkbox");

            // If displaying tooltips
            if(tooltips)
            {
                menuItem.setAttribute("tooltiptext", styleSheetLabel);
            }

            // If the style sheet is disabled
            if(!styleSheet.disabled)
            {
                menuItem.setAttribute("checked", true);
            }

            menu.appendChild(menuItem);
        }
    }

    // If the menu has no children
    if(!menu.hasChildNodes())
    {
        menuItem = document.createElement("menuitem");
        styleSheetLabel = stringBundle.getString("webdeveloper_noStyleSheets");

        menuItem.setAttribute("class", "webdeveloper-generated-menu");
        menuItem.setAttribute("disabled", true);
        menuItem.setAttribute("label", styleSheetLabel);

        // If displaying tooltips
        if(tooltips)
        {
            menuItem.setAttribute("tooltiptext", styleSheetLabel);
        }

        menu.appendChild(menuItem);
    }
}

// Displays a style sheet
function webdeveloper_displayStyleSheet(styleSheet, generatedPage)
{
    const request        = new XMLHttpRequest();
    const styleSheetHref = styleSheet.href;

    var cssRule       = null;
    var headerElement = null;
    var linkElement   = null;
    var preElement    = null;

    request.open("get", styleSheetHref, false);
    request.send("");

    headerElement = generatedPage.content.document.createElement("h2");
    linkElement   = generatedPage.content.document.createElement("a");
    linkElement.setAttribute("href", styleSheetHref);
    linkElement.appendChild(generatedPage.content.document.createTextNode(styleSheetHref));
    headerElement.appendChild(linkElement);
    generatedPage.content.document.body.appendChild(headerElement);

    preElement = generatedPage.content.document.createElement("pre");
    preElement.appendChild(generatedPage.content.document.createTextNode(request.responseText));
    generatedPage.content.document.body.appendChild(preElement);

    // Loop through the the style sheet rules
    for(i = 0; i < styleSheet.cssRules.length; i++)
    {
        cssRule = styleSheet.cssRules[i];

        // If this is an import rule
        if(cssRule.type == 3)
        {
            webdeveloper_displayStyleSheet(cssRule.styleSheet, generatedPage);
        }
    }
}

// Allows the user to edit the CSS of the current page
function webdeveloper_editCSS(force)
{
    toggleSidebar("webdeveloper-edit-css-sidebar", force);
}

// Output style information for an element
function webdeveloper_outputStyleInformationForElement(displayInSidebar, styleSheetArray, title)
{
    const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");

    var generatedDocument = null;
    var headerElement     = null;
    var linkElement       = null;
    var preElement        = null;
    var styleSheet        = null;

    // If displaying in the sidebar
    if(displayInSidebar)
    {
        var parent = null;

        generatedDocument = document.getElementById("sidebar-box").getElementsByAttribute("id", "sidebar")[0].contentDocument.getElementById("webdeveloper-view-style-information-iframe").contentDocument;
        parent            = generatedDocument.getElementsByTagName("head")[0];

        webdeveloper_removeAllChildNodes(parent);

        parent = generatedDocument.body;

        webdeveloper_removeAllChildNodes(parent);
    }
    else
    {
        const request = new XMLHttpRequest();

        var generatedPage = webdeveloper_generatePage("");

        // This must be done to make generated content render
        request.open("get", "about:blank", false);
        request.send("");

        generatedDocument = generatedPage.content.document;
    }

    generatedDocument.title = title;

    linkElement = generatedDocument.createElement("link");
    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated_content.css");
    linkElement.setAttribute("media", "all");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedDocument.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement = generatedDocument.createElement("h1");
    headerElement.appendChild(generatedDocument.createTextNode(title));
    generatedDocument.body.appendChild(headerElement);

    // Loop through the style sheets
    for(styleSheet in styleSheetArray)
    {
        linkElement = generatedDocument.createElement("a");
        linkElement.setAttribute("href", styleSheet);
        linkElement.appendChild(generatedDocument.createTextNode(styleSheet));
        generatedDocument.body.appendChild(linkElement);

        preElement = generatedDocument.createElement("pre");
        preElement.appendChild(generatedDocument.createTextNode(styleSheetArray[styleSheet]));
        generatedDocument.body.appendChild(preElement);
    }
}

// Toggles the default browser styles
function webdeveloper_toggleDefaultStyles(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/disable_default_styles.css", "webdeveloper-disable-default-styles", applyStyle);
}

// Uses the border box model
function webdeveloper_useBorderBoxModel(element, applyStyle)
{
    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/border_box_model.css", "webdeveloper-use-border-box-model", applyStyle);
}

// View CSS
function webdeveloper_viewCSS()
{
    const mainTabBox         = getBrowser().mTabBox;
    const documentList       = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());
    const oldTab             = getBrowser().selectedTab;
    const oldURL             = getBrowser().currentURI.spec;
    const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");
    const request            = new XMLHttpRequest();
    const stringBundle       = document.getElementById("webdeveloper-string-bundle");
    const title              = stringBundle.getFormattedString("webdeveloper_viewCSSTitle", [oldURL]);

    var cssRule          = null;
    var documentURL      = null;
    var generatedPage    = null;
    var headerElement    = null;
    var inlineStylesText = "";
    var linkElement      = null;
    var pageDocument     = null;
    var preElement       = null;
    var styleSheet       = null;
    var styleSheetHref   = null;
    var styleSheetList   = new Array();

    generatedPage = webdeveloper_generatePage("");

    // This must be done to make generated content render
    request.open("get", "about:blank", false);
    request.send("");

    generatedPage.content.document.title = title;

    linkElement = generatedPage.content.document.createElement("link");
    linkElement.setAttribute("href", "chrome://webdeveloper/content/stylesheets/generated_content.css");
    linkElement.setAttribute("media", "all");
    linkElement.setAttribute("rel", "stylesheet");
    linkElement.setAttribute("type", "text/css");
    generatedPage.content.document.getElementsByTagName("head")[0].appendChild(linkElement);

    headerElement = generatedPage.content.document.createElement("h1");
    headerElement.appendChild(generatedPage.content.document.createTextNode(title));
    generatedPage.content.document.body.appendChild(headerElement);

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        inlineStylesText = "";
        pageDocument     = documentList[i];
        documentURL      = pageDocument.documentURI;
        styleSheetList   = pageDocument.getElementsByTagName("style");

        headerElement = generatedPage.content.document.createElement("h2");
        linkElement   = generatedPage.content.document.createElement("a");
        linkElement.setAttribute("href", documentURL);
        linkElement.appendChild(generatedPage.content.document.createTextNode(documentURL));
        headerElement.appendChild(linkElement);
        generatedPage.content.document.body.appendChild(headerElement);

        // Loop through the inline style sheets
        for(var j = 0; j < styleSheetList.length; j++)
        {
            inlineStylesText += styleSheetList[j].innerHTML.trim() + "\n\n";
        }

        // If there are inline styles
        if(inlineStylesText != "")
        {
            headerElement = generatedPage.content.document.createElement("h2");
            preElement    = generatedPage.content.document.createElement("pre");
            headerElement.appendChild(generatedPage.content.document.createTextNode(stringBundle.getFormattedString("webdeveloper_inlineStyles", [documentURL])));
            generatedPage.content.document.body.appendChild(headerElement);
            preElement.appendChild(generatedPage.content.document.createTextNode(inlineStylesText));
            generatedPage.content.document.body.appendChild(preElement);
        }

        styleSheetList = pageDocument.styleSheets;

        // Loop through the style sheets
        for(j = 0; j < styleSheetList.length; j++)
        {
            styleSheet     = styleSheetList[j];
            styleSheetHref = styleSheet.href;

            // If this is an inline style sheet
            if(styleSheetHref == documentURL)
            {
                // Loop through the the style sheet rules
                for(var k = 0; k < styleSheet.cssRules.length; k++)
                {
                    cssRule = styleSheet.cssRules[k];

                    // If this is an import rule
                    if(cssRule.type == 3)
                    {
                        webdeveloper_displayStyleSheet(cssRule.styleSheet, generatedPage);
                    }
                }
            }
            else if(styleSheetHref && styleSheetHref.indexOf("resource://") != 0 && styleSheetHref.indexOf("about:PreferenceStyleSheet") != 0 && styleSheetHref.indexOf("jar:file://") != 0 && styleSheetHref.indexOf("chrome://") != 0)
            {
                webdeveloper_displayStyleSheet(styleSheet, generatedPage);
            }
        }
    }

    // If the open tabs in background preference is set to true
    if(preferencesService.prefHasUserValue("webdeveloper.open.tabs.background") && preferencesService.getBoolPref("webdeveloper.open.tabs.background"))
    {
        getBrowser().selectedTab = oldTab;
    }
}

// View style information
function webdeveloper_viewStyleInformation(element, applyStyle)
{
    const mainTabBox   = getBrowser().mTabBox;
    const documentList = webdeveloper_getDocuments(getBrowser().browsers[mainTabBox.selectedIndex].contentWindow, new Array());

    var checked      = null;
    var pageDocument = null;

    // If the element is not set we are coming from a keyboard shortcut
    if(!element)
    {
        element = document.getElementById("webdeveloper-view-style-information-menu");

        if(element.getAttribute("checked"))
        {
            element.removeAttribute("checked");
        }
        else
        {
            element.setAttribute("checked", true);
        }
    }

    checked = element.getAttribute("checked");

    // Loop through the documents
    for(var i = 0; i < documentList.length; i++)
    {
        pageDocument = documentList[i];

        // If viewing style information
        if(checked)
        {
            pageDocument.addEventListener("mouseover", webdeveloper_displayElementAncestors, true);
            pageDocument.addEventListener("click", webdeveloper_viewStyleInformationForElement, true);
        }
        else
        {
            var sidebar = document.getElementById("sidebar-box");

            pageDocument.removeEventListener("mouseover", webdeveloper_displayElementAncestors, true);
            pageDocument.removeEventListener("click", webdeveloper_viewStyleInformationForElement, true);
            getBrowser().contentWindow.status = null;

            // If the sidebar exists, is visible and is showing view style information
            if(sidebar && !sidebar.getAttribute("hidden") && sidebar.getAttribute("src") == "chrome://webdeveloper/content/sidebar/view_style_information.xul")
            {
                toggleSidebar("webdeveloper-view-style-information-sidebar");
            }
        }
    }

    webdeveloper_toggleStyleSheet(element, "chrome://webdeveloper/content/stylesheets/view_style_information.css", "webdeveloper-view-style-information", applyStyle);
}

// View style information for an element
function webdeveloper_viewStyleInformationForElement(event)
{
    const eventTarget = event.target;

    // If there is an event target and the click was not a right click
    if(eventTarget && event.button != 2)
    {
        const domUtils           = Components.classes["@mozilla.org/inspector/dom-utils;1"].getService(Components.interfaces.inIDOMUtils);
        const oldURL             = getBrowser().currentURI.spec;
        const preferencesService = Components.classes["@mozilla.org/preferences-service;1"].getService(Components.interfaces.nsIPrefService).getBranch("");
        const displayInSidebar   = preferencesService.prefHasUserValue("webdeveloper.view.style.information.sidebar") && preferencesService.getBoolPref("webdeveloper.view.style.information.sidebar");
        const ruleList           = domUtils.getCSSStyleRules(eventTarget);
        const stringBundle       = document.getElementById("webdeveloper-string-bundle");

        var elementRule       = null;
        var headerElement     = null;
        var line              = null;
        var styleRule         = null;
        var styleRuleList     = null;
        var styleSheetArray   = new Array();
        var styleSheet        = null;
        var styleText         = null;

        // Loop through the element rules
        for(var i = 0; i < ruleList.Count(); i++)
        {
            elementRule = ruleList.GetElementAt(i).QueryInterface(Components.interfaces.nsIDOMCSSStyleRule);
            line        = domUtils.getRuleLine(elementRule);

            // If there is a parent style sheet
            if(elementRule.parentStyleSheet)
            {
                styleSheet = elementRule.parentStyleSheet.href;
            }

            // If it's not a rule from the browser
            if(styleSheet && styleSheet.indexOf("resource://") != 0 && styleSheet.indexOf("about:PreferenceStyleSheet") != 0 && styleSheet.indexOf("jar:file://") != 0 && styleSheet.indexOf("chrome://") != 0)
            {
                styleRuleList = elementRule.style;
                styleText     = elementRule.selectorText + "    (" + stringBundle.getString("webdeveloper_viewStyleInformationLine") + " " + line + ")\n{";

                // Loop through the style rules
                for(var j = 0; j < styleRuleList.length; j++)
                {
                    styleRule  = styleRuleList[j];
                    styleText += "\n    " + styleRule + ": " + styleRuleList.getPropertyValue(styleRule) + ";";
                }

                styleText += "\n}\n\n";

                // If this style sheet has rules already stored
                if(styleSheetArray[styleSheet])
                {
                    styleSheetArray[styleSheet] += styleText;
                }
                else
                {
                    styleSheetArray[styleSheet] = styleText;
                }
            }
        }

        // If displaying in the sidebar
        if(displayInSidebar)
        {
            toggleSidebar("webdeveloper-view-style-information-sidebar", true);

            // If the sidebar document is found
            if(document.getElementById("sidebar-box").getElementsByAttribute("id", "sidebar")[0].contentDocument.getElementById("webdeveloper-view-style-information-iframe"))
            {
                webdeveloper_outputStyleInformationForElement(displayInSidebar, styleSheetArray, stringBundle.getFormattedString("webdeveloper_viewStyleInformationTitle", [oldURL]));
            }
            else
            {
                // This sets a small timeout to guarantee the sidebar is open
                window.setTimeout(webdeveloper_outputStyleInformationForElement, 250, displayInSidebar, styleSheetArray, stringBundle.getFormattedString("webdeveloper_viewStyleInformationTitle", [oldURL]));
            }
        }
        else
        {
            const oldTab = getBrowser().selectedTab;

            webdeveloper_outputStyleInformationForElement(displayInSidebar, styleSheetArray, stringBundle.getFormattedString("webdeveloper_viewStyleInformationTitle", [oldURL]));

            // If the open tabs in background preference is set to true
            if(preferencesService.prefHasUserValue("webdeveloper.open.tabs.background") && preferencesService.getBoolPref("webdeveloper.open.tabs.background"))
            {
                getBrowser().selectedTab = oldTab;
            }
        }

        event.preventDefault();
    }
}
