/**
 * Author           : Davide Ficano
 * Date             : 22-Nov-04
 * Fix description  : 15-Jan-05
 * Refactoring      : 13-Feb-05
 */

const VSW_PREF_CONFIG_PATH = "dafi.viewsource.configPath";

function ViewSourceEditorData(isVisible, description, path) {
    this.isVisible = isVisible == "T" || isVisible == "true";
    this.description = description;
    this.path = path;
}

function ViewSourceWithPrefs(forceNew) {
    try {
        if (forceNew) {
            return new ViewSourceWithCachedPrefs();
        } else {
            return ViewSourceWithCachedPrefs.getGlobalPrefs();
        }
    } catch (err) {
        alert("ViewSourceWithPrefs " + err);
    }
}

function ViewSourceWithCachedPrefs() {
    this.prefBranch = Components.classes['@mozilla.org/preferences-service;1']
                      .getService()
                      .QueryInterface(Components.interfaces.nsIPrefBranch);
    this.defaultItem    = -1
    this.frameWarning   = true;
    this._destFolder    = "";
    this._editorData    = new Array();
    this._configPath    =  this.getString(VSW_PREF_CONFIG_PATH, null);
    
    if (!this.isOldSettingImported()) {
        this.readPrefs();
    }
}

// static method
ViewSourceWithCachedPrefs.getGlobalPrefs = function() {
    var app = Components.classes["@mozilla.org/appshell/appShellService;1"]
              .getService(Components.interfaces.nsIAppShellService);
    var prefs = app.hiddenDOMWindow.viewSourcePrefs;

    if (prefs == undefined) {
        prefs = new ViewSourceWithCachedPrefs();
        app.hiddenDOMWindow.viewSourcePrefs = prefs;
    }

    return prefs;
}

// static method
ViewSourceWithCachedPrefs.setGlobalPrefs = function(prefs) {
    ViewSourceWithCachedPrefs.getGlobalPrefs().copy(prefs);
}
    

ViewSourceWithCachedPrefs.prototype = {
    copy : function(prefs) {
        this.defaultItem    = prefs.defaultItem;
        this.frameWarning   = prefs.frameWarning;
        this._destFolder    = prefs._destFolder;
        this._editorData    = prefs._editorData;
        this._configPath    = prefs._configPath;
    },
    
    get destFolder() {
        return this._destFolder;
    },

    set destFolder(folder) {
        this._destFolder = folder;
    },

    get showFrameWarning() {
        return this.frameWarning;
    },

    set showFrameWarning(b) {
        this.frameWarning = b;
    },

    get editorData() {
        return this._editorData;
    },
    
    set editorData(newData) {
        this._editorData = newData;
    },

    get editorDefaultIndex() {
        return this.defaultItem;
    },

    set editorDefaultIndex(newDefaultIndex) {
        this.defaultItem = newDefaultIndex;
    },

    get configPath() {
        return this._configPath;
    },

    set configPath(newConfigPath) {
        this._configPath = newConfigPath;
    },

    getString : function(prefName, defValue) {
        var prefValue;
        try {
            prefValue = this.prefBranch.getCharPref(prefName);
        } catch (ex) {
            prefValue = null;
        }
        return prefValue == null ? defValue : prefValue;
    },

    setString : function(prefName, prefValue) {
        this.prefBranch.setCharPref(prefName, prefValue);
    },

    isOldSettingImported: function() {
        var importSettings = this.configPath == null || this.configPath == "";
        
        if (importSettings) {
            this.importOldSettings();
            var homeDir = ViewSourceWithCommon.getUserHome();
            homeDir.append("viewSource.xml");
            this.configPath = homeDir.path;
            this.savePrefs();
        }
        return importSettings;
    },

    readPrefs : function(configPath) {
        try {
            if (configPath == undefined || configPath == null) {
                configPath = this._configPath;
            } else {
                this._configPath = configPath;
            }
            
            var file = ViewSourceWithCommon.makeLocalFile(configPath);
            if (!file.exists()) {
                return;
            }
            var parser = new DOMParser();
            var doc = parser.parseFromString(ViewSourceWithCommon.read(file), "text/xml");
            if (doc.firstChild.nodeName != "parsererror") {
                this.frameWarning   = this.getFrameWarning(doc);
                this.defaultItem    = this.getDefaultItem(doc);
                this._destFolder    = this.getDestFolder(doc);
                this._editorData    = this.getMainEditors(doc);
            }
        } catch (err) {
            alert("readPrefs: " + err);
        }
    },

    importOldSettings : function() {
        var prefs = new ViewSourceOldPrefs();
        this.defaultItem    = prefs.editorDefaultIndex;
        this.frameWarning   = prefs.showWarning == true;
        this.destFolder     = prefs.destFolder;
        this._editorData    = prefs.editorData;
    },
    
    getVersion : function(doc) {
        return doc.firstChild.attributes.getNamedItem("version").nodeValue;
    },

    getFrameWarning : function(doc) {
        var nl = doc.getElementsByTagName("show-frame-warning");
        var frameWarning = true;
        
        if (nl && nl.item(0).hasChildNodes()) {
            frameWarning = nl.item(0).firstChild.nodeValue == "true";
        }
        return frameWarning;
    },

    getDefaultItem :function(doc) {
        var nl = doc.getElementsByTagName("default-item-index");
        var defItem = "-1";
        
        if (nl && nl.item(0).hasChildNodes()) {
            defItem = nl.item(0).firstChild.nodeValue;
        }
        return defItem;
    },
    
    getDestFolder : function(doc) {
        var nl = doc.getElementsByTagName("destination-folder");
        var defName = "";
        
        if (nl && nl.item(0).hasChildNodes()) {
            defName = nl.item(0).firstChild.nodeValue;
        }
        return defName;
    },

    getMainEditors : function(doc) {
        var nl = doc.getElementsByTagName("main-editor-group");
        var ar = new Array();
        
        if (nl && nl.item(0).hasChildNodes()) {
            var nl = nl.item(0).childNodes;
            for (var i = 0; i < nl.length; i++) {
                var curr = nl.item(i);
                var isValid = curr.nodeType == Node.ELEMENT_NODE;
                
                if (isValid && curr.nodeName == "editor-group-item") {
                    var item = this.createEditorData(curr);
                    if (item) {
                        ar.push(item);
                    }
                }
            }
        }
        return ar;
    },

    createEditorData : function(editorItemNode) {
        var descr = null;
        var enabled = null;
        var path = null;
        
        if (editorItemNode.hasChildNodes()) {
            var nl = editorItemNode.childNodes;

            for (var i = 0; i < nl.length; i++) {
                var curr = nl.item(i);
                var isValid = curr.nodeType == Node.ELEMENT_NODE && curr.hasChildNodes();
                
                if (!isValid) {
                    continue;
                }
                if (curr.nodeName == "description") {
                    descr = curr.firstChild.nodeValue;
                } else if (curr.nodeName == "enabled") {
                    enabled = curr.firstChild.nodeValue;
                } else if (curr.nodeName == "full-path") {
                    path = curr.firstChild.nodeValue;
                }
            }
        }
        
        if (descr == null || enabled == null || path == null) {
            return null;
        }
        return new ViewSourceEditorData(enabled, descr, path);
    },

    // quick and dirty
    savePrefs : function() {
        var str = "";

        str += '<view-source-with version="1.0">\n';
        str += '    <default-item-index>' + this.defaultItem + '</default-item-index>\n';
        str += '    <show-frame-warning>' + this.frameWarning + '</show-frame-warning>\n';
        str += '\n';
        str += '    <destination-info>\n';
        str += '        <destination-folder>' + this.destFolder + '</destination-folder>\n';
        str += '    </destination-info>\n';
        str += '\n';

        str += '    <main-editor-group>\n';
        str += '        <editor-group-name>main</editor-group-name>\n';
        for (var i = 0; i < this.editorData.length; i++) {
            var curr = this.editorData[i];
            var enabled = curr.isVisible ? "true" : "false";
            
            str += '        <editor-group-item>\n';
            str += '            <description>' + curr.description + '</description>\n';
            str += '            <enabled>' + enabled + '</enabled>\n';
            str += '            <full-path>' + curr.path + '</full-path>\n';
            str += '        </editor-group-item>\n';
        }
        str += '    </main-editor-group>        \n';
        str += '\n';
        
        /*
        str += '    <editor-group>\n';
        str += '        <editor-group-name>html editors</editor-group-name>\n';
        str += '        <editor-group-item>\n';
        str += '            <description>description-not-yet-impl</description>\n';
        str += '            <enabled>enabled-not-yet-impl</enabled>\n';
        str += '            <full-path>full-path-not-yet-impl</full-path>\n';
        str += '        </editor-group-item>\n';
        str += '    </editor-group>        \n';
        */
        str += '</view-source-with>\n';
        str += '\n';
        
        try {
            var os = ViewSourceWithCommon.makeOutputStream(this.configPath);
            os.write(str, str.length);
            os.flush();
            os.close();
            
            // These params doesn't go on xml file
            // and must be written ALWAYS after file to ensure
            // data consistency
            this.setString(VSW_PREF_CONFIG_PATH, this._configPath);

            ViewSourceWithCachedPrefs.setGlobalPrefs(this);
        } catch (err) {
            alert("ViewSourceWithCachedPrefs.savePrefs " + err);
        }
    }

};




/**
 * Used to import old settings
 */

const VSW_PREF_SOURCE_FOLDER    = "dafi.viewsource.destFolder";
const VSW_PREF_SHOW_WARNING     = "dafi.viewsource.showWarning";

const VSW_PREF_EDITOR_ITEM      = "dafi.viewsource.editorItem";
const VSW_PREF_EDITOR_DEFAULT_INDEX = "dafi.viewsource.editorDefault";

function ViewSourceOldPrefs() {
    this.prefBranch = Components.classes['@mozilla.org/preferences-service;1']
                      .getService()
                      .QueryInterface(Components.interfaces.nsIPrefBranch);
}

ViewSourceOldPrefs.prototype = {
    get destFolder() {
        return this.getString(VSW_PREF_SOURCE_FOLDER, "");
    },

    get showWarning() {
        return this.getBool(VSW_PREF_SHOW_WARNING, true);
    },

    getString : function(prefName, defValue) {
        var prefValue;
        try {
            prefValue = this.prefBranch.getCharPref(prefName);
        } catch (ex) {
            prefValue = null;
        }
        return prefValue == null ? defValue : prefValue;
    },

    getBool : function(prefName, defValue) {
        var prefValue;
        try {
            prefValue = this.prefBranch.getBoolPref(prefName);
        } catch (ex) {
            prefValue = null;
        }
        return prefValue == null ? defValue : prefValue;
    },

    get editorData() {
        var ar = new Array();
        
        var item = null;
        for (var i = 0; (item = this.getString(VSW_PREF_EDITOR_ITEM + i, null)) != null; i++) {
            if (item == "LAST_ITEM") {
                break;
            }
            var s = item.split("\t");
            if (s.length >= 3) {
                ar.push(new ViewSourceEditorData(s[0], s[1], s[2]));
            } else {
                alert("skipped " + item);
            }
        }
        
        return ar;
    },
    
    get editorDefaultIndex() {
        return this.getString(VSW_PREF_EDITOR_DEFAULT_INDEX, "-1");
    }
};





