// License: http://creativecommons.org/licenses/by-nc-sa/2.0/

function SyncMarksBookmarks(syncMarks) {
  this._init(syncMarks);
}

SyncMarksBookmarks.prototype = {
  _syncMarks:null,
  _settings:null,

  _rootFolder:null,
  _personalFolder:null,

////////////////////////////////////////////////////////////////////////////////
// Initialize the bookmarks iterator.

  _init:function(syncMarks) {
    this._syncMarks = syncMarks;
    this._settings = syncMarks.settings;

    initServices();
    initBMService();

    this._rootFolder = RDF.GetResource("NC:BookmarksRoot");
    this._personalFolder = RDF.GetResource("NC:PersonalToolbarFolder");
  },

////////////////////////////////////////////////////////////////////////////////
// Returns the bookmark root specified by the user's settings.

  get bookmarks() {
    BMSVC.readBookmarks();
    return this._getTree(RDF.GetResource(this._settings.bmFolder));
  },

////////////////////////////////////////////////////////////////////////////////
// Determines if the bookmarks have changed since the last sync.

  get changed() {
    return this._changed(this.bookmarks);
  },

////////////////////////////////////////////////////////////////////////////////
// Determines if a specific branch has changed.

  _changed:function(bmarks) {
    // walk through all the bookmarks and do their thang...
    for (var idx = 0; idx < bmarks.length; idx++) {
      // don't worry about live bookmarks changing...
      //TODO :jah: is this really what we should do?
      if (bmarks[idx].type == "Livemark") {
        continue;
      }

      // check the current bookmark's mod date....
      if (bmarks[idx].modDate > this._settings.lastSync) {
        return true;
      }

      // and now its children if necessary...
      if (bmarks[idx].children != null) {
        if (this._changed(bmarks[idx].children)) {
          return true;
        }
      }
    }

    return false;
  },

////////////////////////////////////////////////////////////////////////////////
// Returns all sub-nodes of a specified starting folder as a tree.

  _getTree:function(res) {
    var container = Components.classes["@mozilla.org/rdf/container;1"]
                    .createInstance(Components.interfaces.nsIRDFContainer);
    container.Init(BMDS, res);

    var bookmarks = new Array();
    var elements = container.GetElements();

    // loop over all the elements in the folder
    while (elements.hasMoreElements()) {

      // add the current element to our array
      var element = elements.getNext().QueryInterface(Components.interfaces.nsIRDFResource);
      var bmark = new SyncMarksBookmark(element, this._syncMarks);
      bookmarks.push(bmark);

      switch (bmark.type) {
        // if the bookmark is a folder, set its children...
        case "Folder":
        case "PersonalToolbarFolder":
        case "IEFavoriteFolder":
          bmark.children = this._getTree(element);
          break;

        // if the bookmark is a regular bookmark, don't do anything special
        case "BookmarkSeparator":
        case "IEFavorite":
        case "ImmutableBookmark":
        case "Bookmark":
        case "Livemark":
        default: break;
      }
    }

    return bookmarks;
  },

};

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

function SyncMarksBookmark(res, syncMarks) {
  this._init(res, syncMarks);
}

SyncMarksBookmark.prototype = {
  _res:null,
  _syncMarks:null,
  _bookmarks:null,

  children:null,

////////////////////////////////////////////////////////////////////////////////


  _init:function(res, syncMarks) {
    this._res = res;
    this._syncMarks = syncMarks;
    this._bookmarks = this._syncMarks.bookmarks;
  },

////////////////////////////////////////////////////////////////////////////////

  get id()        { return escape(this.uri); },
  get uri()       { return this._res.Value; },
  get url()       { return this._getLiteral(this._res, this._urlArc); },
  get name()      { return this._getLiteral(this._res, this._nameArc); },
  get desc()      { return this._getLiteral(this._res, this._descArc); },
  get schedule()  { return this._getLiteral(this._res, this._dateSchedArc); },
  get addDate()   { return this._getDate(this._res, this._dateAddArc); },
  get modDate()   { return this._getDate(this._res, this._dateModArc); },
  get visitDate() { return this._getDate(this._res, this._dateVisitArc); },

////////////////////////////////////////////////////////////////////////////////

  // Known types:
  //  o BookmarkSeparator
  //  o IEFavorite
  //  o ImmutableBookmark
  //  o Bookmark
  //  o Livemark
  //  o Folder
  //  o PersonalToolbarFolder
  //  o IEFavoriteFolder

  get type() {
    var type = BMDS.GetTarget(this._res, this._typeArc, true);
    if (this._res.Value == "NC:PersonalToolbarFolder") {
      return "PersonalToolbarFolder";
    } else if (type) {
      var uri = type.QueryInterface(Components.interfaces.nsIRDFResource).Value;
      return uri.split("#")[1];
    } else if (this._rdfcu.IsContainer(BMDS, this._res)) {
      return "Folder";
    } else if (this._getLiteral(this._res, this._urlArc)) {
      return "Bookmark";
    }
    return "";
  },

////////////////////////////////////////////////////////////////////////////////

  get _typeArc()      { return RDF.GetResource("http://www.w3.org/1999/02/22-rdf-syntax-ns#type"); },
  get _idArc()        { return RDF.GetResource("http://home.netscape.com/NC-rdf#ID"); },
  get _urlArc()       { return RDF.GetResource("http://home.netscape.com/NC-rdf#URL"); },
  get _nameArc()      { return RDF.GetResource("http://home.netscape.com/NC-rdf#Name"); },
  get _descArc()      { return RDF.GetResource("http://home.netscape.com/NC-rdf#Description"); },
  get _keywordArc()   { return RDF.GetResource("http://home.netscape.com/NC-rdf#ShortcutURL"); },
  get _iconArc()      { return RDF.GetResource("http://home.netscape.com/NC-rdf#Icon"); },
  get _feedArc()      { return RDF.GetResource("http://home.netscape.com/NC-rdf#FeedURL"); },
  get _sideArc()      { return RDF.GetResource("http://home.netscape.com/NC-rdf#WebPanel"); },
  get _dateAddArc()   { return RDF.GetResource("http://home.netscape.com/NC-rdf#BookmarkAddDate"); },
  get _dateModArc()   { return RDF.GetResource("http://home.netscape.com/WEB-rdf#LastModifiedDate"); },
  get _dateVisitArc() { return RDF.GetResource("http://home.netscape.com/WEB-rdf#LastVisitDate"); },
  get _dateSchedArc() { return RDF.GetResource("http://home.netscape.com/WEB-rdf#Schedule"); },

////////////////////////////////////////////////////////////////////////////////

  _getLiteral:function(res, arc) {
    try {
      var target = BMDS.GetTarget(res, arc, true);          
      if (target) {
        return target.QueryInterface(Components.interfaces.nsIRDFLiteral).Value;
      }
    } catch (e) { /* probably a bad interface */ }
    return null;
  },

////////////////////////////////////////////////////////////////////////////////

  _getDate:function(res, arc) {
    try {
      var target = BMDS.GetTarget(res, arc, true);          
      if (target) {
        return target.QueryInterface(Components.interfaces.nsIRDFDate).Value / 1000;
      }
    } catch (e) { /* probably a bad interface */ }
    return null;
  },

};
