const rssXMLHeader = '<?xml version="1.0" encoding="UTF-8"?>\n';
const ns =
{
	rdf: "http://www.w3.org/1999/02/22-rdf-syntax-ns#",
	dc: "http://purl.org/dc/elements/1.1/",
	content: "http://purl.org/rss/1.0/modules/content/",
	rss1:"http://purl.org/rss/1.0/"
}

/**
 * Convert string from given charset to Unicode which can be processed by javascript
 * @param string charset Character set of data
 * @param string data Data to be converted
 * @return string Unicode string output
 */
function toUnicode(charset, data)
{
	var data;
	try
	{
		var uniConv = Components.classes["@mozilla.org/intl/scriptableunicodeconverter"].createInstance(Components.interfaces.nsIScriptableUnicodeConverter);
		uniConv.charset = charset;
		data = uniConv.ConvertToUnicode(data);
	}
	catch(e) {}
	return data;
}


const rssType_all = 0;
const rssType_selection = 1;


/**
 * Get the text in the clipboard w/ the mime type text/unicode
 * @return string String of data if successful, FALSE if failure
 */
function getClipboardText()
{
	var clip = Components.classes["@mozilla.org/widget/clipboard;1"].createInstance(Components.interfaces.nsIClipboard);
	if (!clip) return false;

	var trans = Components.classes["@mozilla.org/widget/transferable;1"].createInstance(Components.interfaces.nsITransferable);
	if (!trans) return false;
	trans.addDataFlavor("text/unicode");

	clip.getData(trans, clip.kGlobalClipboard);

	var str=new Object();
	var strLength=new Object();

	trans.getTransferData("text/unicode", str, strLength);

	if (str) str=str.value.QueryInterface(Components.interfaces.nsISupportsString);
	if (str) pastetext=str.data.substring(0, strLength.value / 2);

	return pastetext;
}


/**
 * Parse an item from the DOM Element "item" - called by parseData
 * @param Array_of_items_elements items
 */
function parseItems(items)
{
	var src = items.length ? items : [items];
	var data = [];
	var item, itemSrc, count2, j, pDate;
	try
	{
		var count = src.length;
		for (var i=0; i<count; i++)
		{
			item = new RSSItem();
			itemSrc = src[i].childNodes;
			if (src[i].hasAttributeNS(ns.rdf, "about"))
				item.guid = src[i].getAttributeNS(ns.rdf, "about");
			count2 = itemSrc.length;
			for (j=0; j<count2; j++)
			{
				switch (itemSrc[j].localName)
				{
					case "title":
					case "link":
					case "guid":
					case "comments":
						try
						{
							item[itemSrc[j].nodeName] = itemSrc[j].firstChild.data;
						}
						catch (e) {}
						break;

					case "creator":
						if (itemSrc[j].namespaceURI != ns.dc) break;
					case "author":
						try
						{
							item.author = itemSrc[j].firstChild.data;
						}
						catch (e) {}
						break;

					case "subject":
						if (itemSrc[j].namespaceURI != ns.dc) break;
					case "category":
						try
						{
							item.category = itemSrc[j].firstChild.data;
						}
						catch (e) {}
						break;

					case "encoded":
						if (itemSrc[j].namespaceURI != ns.content) break;
					case "description":
						try
						{
							item.desc = itemSrc[j].firstChild.data;
						}
						catch (e) {}
						break;

					case "date":
						if (itemSrc[j].namespaceURI != ns.dc) break;
						pDate = itemSrc[j].firstChild.data.match(/(\d{4})-(\d\d)-(\d{2})(?:T(\d\d:\d\d(?::\d\d)?)(?:\.\d+)?(Z|[+-]\d\d:\d\d))?$/);
						if (pDate[4] == undefined)
						{
							pDate[4] = "00:00";
							pDate[5] = "+00:00";
						}
						else if (pDate[5] == "Z")
							pDate[5] = "+00:00";
						item.pubDate = time(new Date(pDate[1]+"/"+pDate[2]+"/"+pDate[3]+" "+pDate[4]+" "+pDate[5]));
						break;

					case "pubDate":
						try
						{
							item.pubDate = time(itemSrc[j].firstChild.data);
						}
						catch (e) {}
						break;

					case "source":
						try
						{
							item.source = itemSrc[j].firstChild.data;
							if (itemSrc[j].hasAttribute("url")) item.sourceurl = itemSrc[j].getAttribute("url");
						}
						catch (e) {}
						break;
				}
			}
			data.push(eval(item.toSource()));
		}
	}
	catch (e) {}
	return data;
}

/**
 * Parse a string w/ RSS content and load/import to RSS Editor
 * @param string str String to be parsed
 * @param boolean replace TRUE for loading a complete file, FALSE for import
 * @return int Number of items loaded/imported, FALSE for failure
 */
function parseData(str, replace)
{
	var objDOMParser = new DOMParser;
	var d = objDOMParser.parseFromString(str, "text/xml");

	var channelData = {title:"", link:"", desc:"", lang:"", copyright:"", managingEditor:"", webMaster:"", pubDate:"", lastBuildDate:"", image:""};
	var items = [];
	try
	{
		if (d.documentElement.nodeName != "rss" && (d.documentElement.localName != "RDF" || d.documentElement.namespaceURI != ns.rdf)) throw "notvalid";
		var channels = d.getElementsByTagName("channel");
		if (channels.length != 1) throw "notvalid";
		var channel = channels[0].childNodes;
		var count = channel.length;
		var j, image, item, count2;
		for (var i=0; i<count; i++)
		{
			switch (channel[i].localName)
			{
				case "title":
				case "link":
				case "managingEditor":
				case "webMaster":
					try
					{
						channelData[channel[i].nodeName] = channel[i].firstChild.data;
					}
					catch (e) {}
					break;

				case "rights":
					if (channel[i].namespaceURI != ns.dc) break;
				case "copyright":
					try
					{
						channelData.copyright = channel[i].firstChild.data;
					}
					catch (e) {}
					break;

				case "description":
					try
					{
						channelData.desc = channel[i].firstChild.data;
					}
					catch (e) {}
					break;
				case "language":
					try
					{
						channelData.lang = channel[i].firstChild.data;
					}
					catch (e) {}
					break;
				case "pubDate":
				case "lastBuildDate":
					try
					{
						channelData[channel[i].nodeName] = time(channel[i].firstChild.data);
					}
					catch (e) {}
					break;
				case "image":
					image = channel[i].childNodes;
					count2 = image.length;
					{
						for (j=0; j<count2; j++)
						{
							if (image[j].nodeName == "url")
								try
								{
									channelData.image = image[j].firstChild.data;
								}
								catch (e) {}
						}
					}
					break;
			}
		}
		items = parseItems(d.getElementsByTagName("item"));
	}
	catch (e)
	{
		window.alert("Not valid RSS file");
		return false;
	}

	if (replace)
	{
		setValue("channelTitle", channelData.title);
		setValue("channelLink", channelData.link);
		setValue("channelDesc", channelData.desc);
		setValue("channelLang", channelData.lang);
		setValue("channelCopyright", channelData.copyright);
		setValue("channelManagingEditor", channelData.managingEditor);
		setValue("channelWebMaster", channelData.webMaster);
		setValue("channelPubDate", channelData.pubDate);
		setValue("channelLastBuildDate", channelData.lastBuildDate);
		setValue("channelImage", channelData.image);

		list = document.getElementById("itemList");
		while (list.lastChild)
			list.removeChild(list.lastChild);
	}

	count = items.length;
	for (i=0; i<count; i++)
		addListitem(items[i], false);

	return count;
}


/**
 * Save data (called by "save" or "save-as" or "export" or "publish")
 */
function saveRSS(foStream, type)
{
	itemUpdate();

	var objXMLSerializer = new XMLSerializer;
	foStream.write(rssXMLHeader, rssXMLHeader.length);
	objXMLSerializer.serializeToStream(buildRSS(type), foStream, "UTF-8");
}

/**
 * Build DOM element "rss"
 */
function buildRSS(type)
{
	var doc = document.implementation.createDocument(null, "rss", null);
	doc.documentElement.setAttribute("version", "2.0");
	doc.documentElement.appendChild(buildChannel(type));
	return doc;
}

/**
 * Build DOM element "channel"
 */
function buildChannel(type)
{
	var d = document.implementation.createDocument(null, "rss", null);
	var e = d.createElement("channel");
	var b; // buffer
	var ranges; // ranges of items to be built

	// Required elements
	insertElement(e, "title", getValue("channelTitle"));
	insertElement(e, "link", getValue("channelLink"));
	insertElement(e, "description", getValue("channelDesc"));

	// Optional elements
	getValue("channelLang") != "" && insertElement(e, "language", getValue("channelLang"));
	getValue("channelCopyright") != "" && insertElement(e, "copyright", getValue("channelCopyright"));
	getValue("channelManagingEditor") != "" && insertElement(e, "managingEditor", getValue("channelManagingEditor"));
	getValue("channelWebMaster") != "" && insertElement(e, "webMaster", getValue("channelWebMaster"));

	// Dates are optional, validated and converted to UTC
	if ((b = new Date(getValue("channelPubDate")).toUTCString()) != "Invalid Date")
		insertElement(e, "pubDate", b);
	if ((b = new Date(getValue("channelLastBuildDate")).toUTCString()) != "Invalid Date")
		insertElement(e, "lastBuildDate", b);

	// Optional image
	if (getValue("channelImage") != "")
	{
		e.appendChild(b = d.createElement("image"));
		insertElement(b, "url", getValue("channelImage"));
		// Contents of these two elements should be same as channel/title and channel/link
		insertElement(b, "title", getValue("channelTitle"));
		insertElement(b, "link", getValue("channelLink"));
	}

	// Set range of items to be built
	var tree = document.getElementById("tree");
	switch (type)
	{
		case rssType_selection:
			ranges = treeSelection(tree);
			break;
		case rssType_all:
		default:
			ranges = [{start: 0, end: tree.view.rowCount - 1}];
	}
	buildItems(e, ranges);

	return e;
}

/**
 * Append the "item" elements to the supplied element according to the ranges given
 */
function buildItems(element, ranges)
{
	var list = document.getElementById("itemList");
	var count = ranges.length;
	var item, buf;
	for (var i=0; i<count; i++)
	{
		for (var j=ranges[i].start; j<=ranges[i].end; j++)
		{
			try
			{
				item = eval(list.childNodes[j].firstChild.lastChild.getAttribute("label"));
				buf = element.ownerDocument.createElement("item");
				(item.title != "" || item.desc == "") && insertElement(buf, "title", item.title);
				item.link != "" && insertElement(buf, "link", item.link);
				item.desc != "" && insertElement(buf, "description", item.desc);
				item.author != "" && insertElement(buf, "author", item.author);
				item.category != "" && insertElement(buf, "category", item.category);
				item.comments != "" && insertElement(buf, "comments", item.comments);
				item.guid != "" && insertElement(buf, "guid", item.guid).setAttribute("isPermaLink", "false");
				if ((b = new Date(item.pubDate).toUTCString()) != "Invalid Date")
					insertElement(buf, "pubDate", b);
				(item.source != "" || item.sourceurl != "") && insertElement(buf, "source", item.source).setAttribute("url", item.sourceurl);

				element.appendChild(buf);
			}
			catch (e) {}
		}
	}
}


// The object about the opened file
var openedFile = null;


/**
 * Start a new file (delete everything in the editing environment)
 */
function newFile()
{
	if (!window.confirm("Contents of open file will be lost")) return;
	// Reset fields in Channel Tab
	setValue("channelTitle", "");
	setValue("channelLink", "");
	setValue("channelDesc", "");
	setValue("channelLang", "");
	setValue("channelCopyright", "");
	setValue("channelManagingEditor", "");
	setValue("channelWebMaster", "");
	setValue("channelPubDate", "");
	setValue("channelLastBuildDate", "");
	setValue("channelImage", "");

	// Reset items
	var list = document.getElementById("itemList");
	while (list.lastChild)
		list.removeChild(list.lastChild);
	// Reset fields in Items Tab: done by itemSelect() called by onselect event

	openedFile = null;
}

/**
 * Open the open/import file dialog, select the file and read it
 * @param int mode: 0 - open, 1 - import
 * @param out data: Unicode string of the file being read
 * @param nsIFilePicker (success) or null (cancel)
 */
function getFile(mode, data)
{
	const nsIFilePicker = Components.interfaces.nsIFilePicker;
	var fp = Components.classes["@mozilla.org/filepicker;1"].createInstance(nsIFilePicker);
	fp.init(window, mode == 1 ? "Import" : "Open", nsIFilePicker.modeOpen);

	fp.appendFilter("RSS Files", "*.rss; *.xml");
	fp.appendFilters(nsIFilePicker.filterAll);

	var res = fp.show();
	if (res != nsIFilePicker.returnOK) return null;

	var fiStream = Components.classes["@mozilla.org/network/file-input-stream;1"].createInstance(Components.interfaces.nsIFileInputStream);
	fiStream.init(fp.file, 0x01, 00000, false);

	var siStream = Components.classes["@mozilla.org/scriptableinputstream;1"].createInstance(Components.interfaces.nsIScriptableInputStream);
	siStream.init(fiStream);

	data.value = toUnicode("UTF-8", siStream.read(-1));
	return fp;
}

// Open an RSS file
function openFile()
{
	var data = {};
	var fp = getFile(0, data);
	if (fp == null) return;

	parseData(data.value, true)
	openedFile = fp;
}


/**
 * Import a string which is a valid RSS file
 * @param string s
 */
function importString(s)
{
	// paste it before "selectedItem"
	var count = parseData(s, false);
	if (!count) return;

	var view = document.getElementById("tree").view;
	// Select the pasted items
	if (selectedItem == null)
		view.selection.rangedSelect(count - 1, 0, false);
	else
	{
		var index = view.getIndexOfItem(selectedItem.parentNode);
		view.selection.rangedSelect(index - 1, index - count, false);
	}
	itemSelect(false);
}


// Import an RSS file
function importFile()
{
	var data = {};
	var fp = getFile(1, data);
	if (fp == null) return;
	importString(data.value)
}


/**
 * Save RSS to a filepicker
 * @param nsIFilePicker fp
 * @param boolean selectionOnly
 */
function save(fp, selectionOnly)
{
	try
	{
		var nsIFileOutputStream = Components.interfaces.nsIFileOutputStream;
		var foStream = Components.classes["@mozilla.org/network/file-output-stream;1"].createInstance(nsIFileOutputStream);
		foStream.init(fp.file, 0x2a, 0420, false); // wronly | create | truncate
		saveRSS(foStream, selectionOnly ? rssType_selection : rssType_all);
		foStream.close();
		return true;
	}
	catch (e)
	{
		window.alert ("Failure saving file. Reasons:\n" + e);
		return false;
	}
}


/**
 * Save current file
 */
function saveFile()
{
	openedFile ? save(openedFile, false) : saveAs();
}


/**
 * Save current file as a new file
 */
function saveAs()
{
	var fp = saveDialog("Save As");
	if (!fp) return;
	if (!save(fp, false)) return;
	openedFile = fp;
}


/**
 * Export a file
 */
function exportFile()
{
	var fp = saveDialog("Export");
	fp && save(fp, true);
}


/**
 * Copy items to clipboard - as an RSS file
 */
function itemsCopy()
{
	itemUpdate();

	var objXMLSerializer = new XMLSerializer;
	var str = rssXMLHeader;
	str += objXMLSerializer.serializeToString(buildRSS(rssType_selection));

	const gClipboardHelper = Components.classes["@mozilla.org/widget/clipboardhelper;1"].getService(Components.interfaces.nsIClipboardHelper);
	gClipboardHelper.copyString(str);
}


/**
 * Paste from an RSS file in the clipboard
 */
function itemsPaste()
{
	importString(getClipboardText());
}