/*** -*- Mode: Javascript; tab-width: 2;

The contents of this file are subject to the Mozilla Public
License Version 1.1 (the "License"); you may not use this file
except in compliance with the License. You may obtain a copy of
the License at http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS
IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
implied. See the License for the specific language governing
rights and limitations under the License.

The Original Code is Urban Rage Software code.
The Initial Developer of the Original Code is Eric Plaster.

Portions created by Urban Rage Software are
Copyright (C) 2000 Urban Rage Software.  All
Rights Reserved.

Contributor(s): Eric Plaster <plaster@urbanrage.com)> (original author)
                Martin Kutschker <martin.t.kutschker@blackbox.net> (polishing)

---------------------

RDF Class:
==========

This is a class to make (remote) RDF file manipulation of simple maps easy and painless.
It requires the about attributes to be in a "magic format":

*) Your root sequence needs to be "something:data" 
*) Your sequences or nodes need to be "something:nodename"
*) Your need to be hierarchically named: "something:anotherthing:nodename"

When you initialize the class, you pass the "something" in as the root. 

RDFFile Class:
==============

Requires:
  file.js

This is the same as RDF class, with the exception that it takes a local file path/url
instead of a generic url. The resource is loaded synchronously by default. If the file
does not exist it will be created on the fly (optional).

---------------------

RDF(src, root, nc, flags)
  Constructor:
    [in]  src   --> A url that points to the rdf file
    [in]  root  --> Root sequence for the data (i.e. 'urn:animals' or 'urn:placea')
    [in]  nc    --> nc for the rdf file (i.e. 'http://www.urbanrage.com/rdf#')
    [in]  flags --> flags (optional bitmask)
  Flags:
    JSLIB_RDF_FLAG_SYNC --> load RDF source synchronously *)
  Example:
    var src = "http://www.foo.com/my.rdf";
    var root = "urn:foo";
    var nc = "http://www.urbanrage.com/rdf#";
    var flags = RDF_FLAG_SYNC;

    var gRDF = new RDF(src, root, nc, flags);

*) If another thread loads the resource already asynchron, you still have to poll for
   the loaded property.


RDFFile(path, root, nc, flags)
  Constructor:
    [in]  path  --> A path or file url to the rdf file (you must have permissions)
    [in]  root  --> Root sequence for the data (i.e. 'urn:animals' or 'urn:places')
    [in]  nc    --> nc for the rdf file (i.e. 'http://www.urbanrage.com/rdf#')
    [in]  flags --> flags (optional bitmask)
  Flags:
    JSLIB_RDF_FLAG_DONT_CREATE --> don't create RDF file
  Example:
    var path = "/usr/local/foo/my.rdf";
    var root = "urn:foo";
    var nc = "http://www.urbanrage.com/rdf#";
    var flags = RDF_FLAG_DONT_CREATE;

    var gRDFFile = new RDFFile(path, root, nc, flags);

---------------------

dsource
  The remote datasource. Null, if the constructor failed somehow.

loaded
  Boolean, which tells if the resource has finished loading.

---------------------

addSeq(aSeq)
  Addes a "Sequence" to the rdf file (i.e. a catagory)
  Checks if the given name, value exists.
    [in]  aSeq  --> The name of the sequence (i.e. "birds" or "cities:minneapolis").
  Example:
    gRDF.addSeq("city")

removeSeq(aSeq, deep)
  Removes the "Sequence" from the datasource and the rdf file.
    [in]  aSeq  --> The name of the sequence (i.e. "birds" or "cities:minneapolis").
    [in]  deep   --> Recursively remove sequeces and nodes.  WARNING, if you have circular references, 
                     you could run into some real problems here.
  Example:
    gRDF.addSeq("cities:minneapolis");
    gRDF.addSeq("cities:st paul");
    gRDF.removeSeq("cities", true);  // removes both minneapolis, st paul, and cities from the rdf file.

isSeq(aSeq)
  Returns true if it is a sequence.
    [in]  aSeq  --> The name of the sequence (i.e. "birds" or "cities:minneapolis").
  Example:
    if(gRDF.isSeq("cities")) {
      gRDF.removeSeq("cities", true);  // removes both minneapolis, st paul, and cities from the rdf file.
    }

doesSeqExist(aSeq)
  Returns true if the sequence exists in the rdf file.
    [in]  aSeq  --> The name of the sequence (i.e. "birds" or "cities:minneapolis").
  Example:
    if(gRDF.doesSeqExist("cities")) {
      ...
    }

getSeqSubNodes(aSeq)
  Returns an "Array" of Nodes that are contained by "aSeq".
    [in]  aSeq  --> The name of the sequence (i.e. "birds" or "cities:minneapolis").
  Example:
    var list = gRDF.getSeqSubNodes("cities:minneapolis");
    for(var i=0; i<list.length; i++) {
      // do something....
    }

getSeqSubSeqs(aSeq)
  Returns an "Array" of Sequences that are contained by "aSeq".
    [in]  aSeq  --> The name of the sequence (i.e. "birds" or "cities:minneapolis").
  Example:
    var list = gRDF.getSeqSubSeqs("cities");
    for(var i=0; i<list.length; i++) {
      // do something....
    }

addNode(aNode)
  Addes a node to a sequence.  This can be the default one that already set up.
    [in]  aNode  --> The name of the node (i.e. "birds" or "cities:minneapolis").
  Example:
    gRDF = new rdf(file, "urn:cities", "http://www.urbanrage.com/rdf#");
    gRDF.addNode("prefs");  // adds the node "urn:cities:prefs"

removeNode(aNode)
  Removes the node from the rdf file.
    [in]  aNode  --> The name of the node (i.e. "birds" or "cities:minneapolis").
  Example:
    gRDF.removeNode("prefs");  // removes the node "urn:cities:prefs"

isNode(aNode)
  Returns true if it is a node.
    [in]  aNode  --> The name of the node (i.e. "prefs" or "cities:minneapolis").
  Example:
    if(gRDF.isNode("prefs")) {
      gRDF.removeNode("prefs");  // removes the prefs node
    }

doesNodeExist(aNode)
  Returns true if the node exists in the rdf file.
    [in]  aNode  --> The name of the node (i.e. "prefs" or "cities:minneapolis").
  Example:
    if(gRDF.doesNodeExist("prefs")) {
      ...
    }

setAttribute(aNode, name, value)
  Sets an attribute to a node with the given name/value pair
    [in]  aNode  --> The name of the node (i.e. "prefs" or "cities:minneapolis").
    [in]  name   --> The attribute name.
    [in]  value  --> The value of the attribute.
  Example:
    gRDF.setAttribute("prefs", "currentCity", "minneapolis");

getAttribute(aNode, name)
  Gets an attribute to a node with the given name.
    [in]  aNode  --> The name of the node (i.e. "prefs" or "cities:minneapolis").
    [in]  name   --> The attribute name.
  Example:
    gRDF.getAttribute("prefs", "currentCity");  // returns "minneapolis"

removeAttribute(aNode, name)
  Removes an attribute that is associated with the given node.
    [in]  aNode  --> The name of the node (i.e. "prefs" or "cities:minneapolis").
    [in]  name   --> The attribute name.
  Example:
    gRDF.removeAttribute("prefs", "currentCity");  // removes the attribute "currentCity"

doesAttributeExist(aNode, name)
  Returns true if there is an attribute with the given "name" arc'd off of aNode.
    [in]  aNode  --> The name of the node (i.e. "prefs" or "cities:minneapolis").
    [in]  name   --> The attribute name.
  Example:
    if(gRDF.doesAttributeExist("prefs", "currentCity")) {
      // do something....
    }

flush()
  Writes changed information out to the rdf file.  NOTE:  If you delete nodes or sequences, it
  doesn't take affect until you call this function.  This was done to increase processing time.

  Example:
    gRDF.removeAttribute("prefs", "currentCity");  // removes the attribute "currentCity"
    gRDF.flush();

------------------
*/

const JSLIB_CONTAINER_PROGID = '@mozilla.org/rdf/container;1';
const JSLIB_CONTAINER_UTILS_PROGID = '@mozilla.org/rdf/container-utils;1';
const JSLIB_LOCATOR_PROGID = '@mozilla.org/filelocator;1';
const JSLIB_RDF_PROGID = '@mozilla.org/rdf/rdf-service;1';
const JSLIB_RDF_DS_PROGID = '@mozilla.org/rdf/datasource;1?name=xml-datasource';

const JSLIB_RDF_FLAG_SYNC = 1;        // load RDF source synchronously
const JSLIB_RDF_FLAG_DONT_CREATE = 2; // don't create RDF file (RDFFile only)

function RDF(src, root, nc, flags) {
  this.RDF = Components.classes[JSLIB_RDF_PROGID].getService();
  this.RDF = this.RDF.QueryInterface(Components.interfaces.nsIRDFService);
  this.RDFC = Components.classes[JSLIB_CONTAINER_PROGID].getService();
  this.RDFC = this.RDFC.QueryInterface(Components.interfaces.nsIRDFContainer);
  this.RDFCUtils = Components.classes[JSLIB_CONTAINER_UTILS_PROGID].getService();
  this.RDFCUtils = this.RDFCUtils.QueryInterface(Components.interfaces.nsIRDFContainerUtils);

  if(src) {
    this._init(src, root, nc, flags);
  }
}

RDF.prototype = {
  RDF        : null,
  RDFC       : null,
  RDFCUtils  : null,
  src        : null,
  root       : null,
  nc         : null,
  dsource    : null,
  loaded     : false,
  
  _init : function(src, node, nc, flags) {
    flags = flags || 0;
    this.src = src;
    this.root = node;
    this.nc = nc;

    load = true; // load source

    // Create an RDF/XML datasource using the XPCOM Component Manager
    var ds = Components
             .classes[JSLIB_RDF_DS_PROGID]
             .createInstance(Components.interfaces.nsIRDFDataSource);

    // The nsIRDFRemoteDataSource interface has the interfaces
    // that we need to setup the datasource.
    var remote = ds.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource);

    try {
      remote.Init(src); // throws an exception if URL already in use
    }
    catch(err) {
      // loading already
      load = false;
    }

    if (load) {
      try {
        remote.Refresh((flags & JSLIB_RDF_FLAG_SYNC) ? true: false);
      }
      catch(err) {
        dump(err);
        return;
      }
    }
    else {
      ds = this.RDF.GetDataSource(src);
      remote = ds.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource);
    }

    try {
      if (remote.loaded) {
        this.loaded = true;
      }
      else {
        var obs = {
          rdf: this, // backreference to ourselves

          onBeginLoad: function(aSink)
          {},

          onInterrupt: function(aSink)
          {},

          onResume: function(aSink)
          {},

          onEndLoad: function(aSink)
          {
             this.rdf.loaded = true;
          },

          onError: function(aSink, aStatus, aErrorMsg)
          {
            dump(aErrorMsg);
          }
        };

        // RDF/XML Datasources are all nsIRDFXMLSinks
        var sink = ds.QueryInterface(Components.interfaces.nsIRDFXMLSink);
  
        // Attach the observer to the datasource-as-sink
        sink.addXMLSinkObserver(obs);
      }
    }
    catch(err) {
      dump(err);
      return;
    }

    this.dsource = ds;
  },

  _getRealNode : function(aNode) {
    var node;
    if(!aNode) {
      node = this.root+":data";
    } else {
      if(aNode.indexOf(this.root+":") == -1) {
        node = this.root+":"+aNode;
      } else {
        return aNode;
      }
    }
    return node;
  },

  addSeq : function(aSeq)
  {
    if(!aSeq) throw("addSeq must supply an argument");
    if(aSeq == this.root+":data") throw("Cannot create root Seq");

    var realnode = this._getRealNode(aSeq);
    var res = this.RDF.GetResource(realnode);

    var pos = realnode.lastIndexOf(":");
    var parent = realnode.slice(0, pos);
    if(parent == this.root) {
      parent = parent+":data";
    }

    var parentres = this.RDF.GetResource(parent);
    if( parentres) {
      this.RDFC.Init(this.dsource, parentres);

      this.RDFCUtils.MakeSeq(this.dsource, res);
      this.RDFC.AppendElement(res);
    }
  },

  removeSeq : function(aSeq, deep)
  {
    if(aSeq == this.root+":data") throw("Cannot remove root Seq");
    var realnode = this._getRealNode(aSeq);
    var res = this.RDF.GetResource(realnode);

    if(this.RDFCUtils.IsSeq(this.dsource, res)) {
      if(deep) {
        this._deleteSeqRecursively(res);
      }
      this.removeNode(aSeq);
    } else {
      throw("Trying to remove a Seq when it's a node");
    }
  },

  _deleteSeqRecursively : function(res) 
  {
    this.RDFC.Init(this.dsource, res);

    var elems = this.RDFC.GetElements();
    while(elems.hasMoreElements()) {
      var elem = elems.getNext();
      if(this.RDFCUtils.IsSeq(this.dsource, elem)) {
        this._deleteSeqRecursively(elem);
        this.RDFC.Init(this.dsource, res);
      }
      var arcs = this.dsource.ArcLabelsOut(elem);
      while(arcs.hasMoreElements()) {
        var arc = arcs.getNext();
        var targets = this.dsource.GetTargets(elem, arc, true);
        while (targets.hasMoreElements()) {
          var target = targets.getNext();
          this.dsource.Unassert(elem, arc, target, true);
        }
      }
      this.RDFC.RemoveElement(elem, false);
    }
  },

  isSeq : function(aSeq)
  {
    var realnode = this._getRealNode(aSeq);
    var res = this.RDF.GetResource(realnode);
    if(res && this.RDFCUtils.IsSeq(this.dsource, res)) {
      return true;
    } else {
      return false;
    }
  },

  doesSeqExist : function(aSeq)
  {
    return this.doesNodeExist(aSeq);
  },

  getSeqSubNodes : function(aSeq)
  {
    var realnode = this._getRealNode(aSeq);
    var list = new Array;

    var res = this.RDF.GetResource(realnode);
    this.RDFC.Init(this.dsource, res);

    var elems = this.RDFC.GetElements();
    while(elems.hasMoreElements()) {
      var elem = elems.getNext();
      elem = elem.QueryInterface(Components.interfaces.nsIRDFResource);
      if(!this.RDFCUtils.IsSeq(this.dsource, elem)) {
        list.push(elem.Value);
      }
    }
    return list;
  },

  getSeqSubSeqs : function(aSeq)
  {
    var realnode = this._getRealNode(aSeq);
    var list = new Array;

    var res = this.RDF.GetResource(realnode);
    this.RDFC.Init(this.dsource, res);

    var elems = this.RDFC.GetElements();
    while(elems.hasMoreElements()) {
      var elem = elems.getNext();
      elem = elem.QueryInterface(Components.interfaces.nsIRDFResource);
      if(this.RDFCUtils.IsSeq(this.dsource, elem)) {
        list.push(elem.Value);
      }
    }
    return list;
  },

  addNode : function(aNode)
  {
    var realnode = this._getRealNode(aNode);
    if(realnode == this.root+":data") throw("Cannot add root Seq");

    var pos = realnode.lastIndexOf(":");
    var parent = realnode.slice(0, pos);
    if(parent == this.root) {
      parent = parent+":data";
    }

    var res = this.RDF.GetResource(realnode);
    var parentres = this.RDF.GetResource(parent);
    if(parentres) {
      this.RDFC.Init(this.dsource, parentres);
      this.RDFC.AppendElement(res);
    }
  },

  removeNode : function(aNode)
  {
    var realnode = this._getRealNode(aNode);
    var res = this.RDF.GetResource(realnode);
    var root = this.RDF.GetResource(this.root+":data");

    var pos = realnode.lastIndexOf(":");
    var parent = realnode.slice(0, pos);
    if(parent == this.root) {
      parent = parent+":data";
    }
    var parentres = this.RDF.GetResource(parent);

    this.RDFC.Init(this.dsource, parentres);

    var arcs = this.dsource.ArcLabelsOut(res);
    while(arcs.hasMoreElements()) {
      var arc = arcs.getNext();
      var targets = this.dsource.GetTargets(res, arc, true);
      while (targets.hasMoreElements()) {
        var target = targets.getNext();
        this.dsource.Unassert(res, arc, target, true);
      }
    }
    this.RDFC.RemoveElement(res, false);
  },

  isNode : function(aNode)
  {
    var realnode = this._getRealNode(aNode);
    var res = this.RDF.GetResource(realnode);
    if(res && !(this.RDFCUtils.IsSeq(this.dsource, res)) ) {
      return true;
    } else {
      return false;
    }
  },

  doesNodeExist : function(aNode)
  {
    var realnode = this._getRealNode(aNode);
    var rv = false;

    var res = this.RDF.GetResource(realnode);
    var pos = realnode.lastIndexOf(":");
    var parent = realnode.slice(0, pos);
    if(parent == this.root) {
      parent = parent+":data";
    }

    if(parent) {
      var parentres = this.RDF.GetResource(parent);
      this.RDFC.Init(this.dsource, parentres);

      var index = this.RDFC.IndexOf(res.QueryInterface(Components.interfaces.nsIRDFNode));
      if(index != -1) {
        rv = true;
      }
    }
    return rv;
  },

  setAttribute : function(aNode, name, value)
  {
    var realnode = this._getRealNode(aNode);
    var newnode = this.RDF.GetResource(realnode);
    var oldvalue = this.getAttribute(realnode, name);

    if(newnode) {
      // Add an assertion to the RDF datasource for each property of the resource
      if(oldvalue) { 
        this.dsource.Change(newnode,
            this.RDF.GetResource(this.nc + name),
            this.RDF.GetLiteral(oldvalue),
            this.RDF.GetLiteral(value) );
      } else {
        this.dsource.Assert(newnode,
            this.RDF.GetResource(this.nc + name),
            this.RDF.GetLiteral(value),
            true );
      }
    }
  },

  getAttribute : function(aNode, name)
  {
    var realnode = this._getRealNode(aNode);

    var itemRes = this.RDF.GetResource(this.nc + name);
    if (!itemRes) return null;

    var IDRes = this.RDF.GetResource(realnode);
    if (!IDRes) return null;
        
    var thisNode = this.dsource.GetTarget(IDRes, itemRes, true);
    if (thisNode) thisNode = thisNode.QueryInterface(Components.interfaces.nsIRDFLiteral);
    if (thisNode)
    {
      return thisNode.Value;
    }
    return null;
  },

  removeAttribute : function(aNode, name)
  {
    var realnode = this._getRealNode(aNode);

    var src = this.RDF.GetResource(realnode);
    if(src) {
      var prop = this.RDF.GetResource(this.nc + name, true);
      var target = this.dsource.GetTarget(src, prop, true);
      this.dsource.Unassert(src, prop, target);
    }
  },

  doesAttributeExist : function(aNode, name)
  {
    var value = this.getAttribute(aNode, name);
    if(value) {
      return true;
    }
    return false;
  },

  getDatasource : function() 
  {
    return this.dsource;
  },

  flush : function()
  {
    this.dsource.QueryInterface(Components.interfaces.nsIRDFRemoteDataSource).Flush();
  }

};

/*
------------------
*/

function RDFFile(path, root, nc, flags) {
  this.created = false;

  if(path) {
    this._file_init(path, root, nc);
  }
}
RDFFile.prototype = new RDF;

RDFFile.prototype._file_init = function (path, root, nc, flags) {
  flags = flags || JSLIB_RDF_FLAG_SYNC; // default to synchronous loading

  var file = path;

  if(path.substr(0,7) != "file://")
    file = "file://" + path;
  else
    path = path.substr(7);

  // Ensure we have a base RDF file to work with
  var rdf_file = new File(path);

  if (!rdf_file.exists() && !(flags & JSLIB_RDF_FLAG_DONT_CREATE)) {
    if (rdf_file.open("w") != JS_FILE_OK) {
      return;
    }

    if (rdf_file.write(
               '<?xml version="1.0" ?>\n' +
               '<RDF:RDF\n' +
               '     xmlns:SIMPLE="' + nc + '"\n' +
               '     xmlns:RDF="http://www.w3.org/1999/02/22-rdf-syntax-ns#">\n' +
               '  <RDF:Seq about="' + root + ':data">\n' +
               '  </RDF:Seq>\n' +
               '</RDF:RDF>\n') != JS_FILE_OK) {
      rdf_file.close();
      return;
    }

    this.created = true;
  }
  rdf_file.close();

  // Get a reference to the available datasources
  this._init(file, root, nc, flags);
};
