#define TRACING 0
/*
   Defines a parallel multilevel preconditioner 
*/
#include "mpi.h"
#include "petscvec.h"
#include "petscis.h"
#include "petscmat.h"
#include "src/mat/matimpl.h"
#include "petscpc.h"
#include "src/sles/pc/pcimpl.h"
#include "src/vec/impls/mpi/pvecimpl.h"
#include "src/sles/slesimpl.h"
#include "petscoptions.h"

#include "parpre_pc.h"
#include "parpre_pipeline.h"
#include "parpre_subdomains.h"
#include "parpre_is.h"
#include "./ml_impl.h"
#include "./ml_head.h"

#define CHUNCKSIZE   100
#define MAXLEVELS 100
#define BOTTOM_LEVEL 0

extern int MatMultAXBY_AIJ
(Scalar a, Mat aijin,Vec xx, Scalar b, Vec yy,Vec zz);
extern int MatMatMult_MPIAIJ(Mat A,Mat B,Mat *C);
extern int MatDiagonalScale_MPIAIJ(Mat A,Vec ll,Vec rr);
extern int MatMatSubtract_AIJ(Mat a, Mat b, Mat *c);
extern int MatMaxRowLen_MPIAIJ(Mat A, int *rowlen);
extern int MatMaxRowOffDiagElement_MPIAIJ(Mat A,Vec e);
extern int MatMaxColOffDiagElement_MPIAIJ(Mat A,Vec e);

/****************************************************************
 * Inits and such
 ****************************************************************/

#undef __FUNC__
#define __FUNC__ "InitOneLevel"
static int InitOneLevel
(MPI_Comm comm,int level,Mat mat,IS idx,
 Mat *lmat,int *llevel,IS *lindices,
 Vec *u,Vec *v,Vec *w,Vec *diag,
 int orth,int old_isize,int old_jsize,int *isize,int *jsize,int global_size,
 int trace)
{
  int ierr;

  PetscFunctionBegin;
  *llevel = level; *lindices = idx; *lmat = mat;

  if (orth) {
    if (level==0) {
      *isize = (int)sqrt(global_size);
      *jsize = global_size/(*isize);
    } else {
      if (level-2*(level/2)==0) {
	*isize = old_isize/2; *jsize = old_jsize;
      } else {
	*isize = old_isize; *jsize = old_jsize/2;
      }
    }
    if ((*isize)*(*jsize)!=global_size) SETERRQ(1,0,"Grid not divisible");
    printf("Grid is %dx%d\n",*isize,*jsize);
  }

  {
    int s,ss;
    ierr = MatGetLocalSize(mat,&s,&ss); CHKERRQ(ierr);
    ierr = VecCreateMPI(comm,s,PETSC_DECIDE,u); CHKERRQ(ierr);
    ierr = VecDuplicate(*u,v); CHKERRQ(ierr);
    ierr = VecDuplicate(*u,w); CHKERRQ(ierr);
  }
/* !!! */
  ierr = VecDuplicate(*u,diag); CHKERRQ(ierr);

  if (trace & AMLTraceFill) {
    int b;
    ierr = MatMaxRowLen_MPIAIJ(mat,&b); CHKERRQ(ierr);
    PetscPrintf(comm,"Max row length at level %d: %d\n",level,b);
  }

  PetscFunctionReturn(0);
}

#if TRACING
#undef __FUNC__
#define __FUNC__ "LambdaOneEst"
static int LambdaOneEst(Mat mat,MC_OneLevel_struct *this_level)
{
  Scalar one=1.0, xax,xx;

  PetscFunctionBegin;
  ierr = VecSet(&one,this_level->u); CHKERRQ(ierr);
  ierr = MatMult(mat,this_level->u,this_level->v); CHKERRQ(ierr);
  ierr = VecPointwiseMult(this_level->u,this_level->v,this_level->w);
  CHKERRQ(ierr);
  ierr = VecPointwiseMult(this_level->u,this_level->u,this_level->v);
  CHKERRQ(ierr);
  ierr = VecSum(this_level->w,&xax); CHKERRQ(ierr);
  ierr = VecSum(this_level->v,&xx); CHKERRQ(ierr);
  printf("lambda_1 approx: %e\n",xax/xx);
  PetscFunctionReturn(0);
}
#endif

#undef __FUNC__
#define __FUNC__ "SetupPolynomialCoffs"
static int SetupPolynomialCoffs(int level,int degree,Scalar **r_coffs)
{
  int d;
  Scalar *coffs;

  PetscFunctionBegin;
  coffs = (Scalar *) PetscMalloc((degree+1)*sizeof(Scalar)); CHKPTRQ(coffs);

  /* Pascal's triangle */
  for (d=1; d<=degree; d++) {
    int e;
    coffs[0]=1; coffs[d]=1;
    for (e=d-1; e>0; e--) coffs[e]=coffs[e]+coffs[e-1];
  }
  /* Alternating signs: note that these are minus the poly coeffs.
     This saves a few scalings in the solve. */
  for (d=0; d<=degree; d=d+2)
    coffs[d]=-coffs[d];

  *r_coffs = coffs;
  PetscFunctionReturn(0);
}
  
/****************************************************************
 * Sub solvers
 ****************************************************************/
#undef __FUNC__
#define __FUNC__ "SetSmoothers"
static int SetSmoothers
(AMLSmootherChoice smoother_choice,MC_OneLevel_struct *this_level,
 char *prefix)
{
  MPI_Comm comm = this_level->comm;
  int ierr;

  PetscFunctionBegin;
/*
  ierr = PCCreate(comm,&(this_level->global_solve)); CHKERRQ(ierr);
  ierr = PCSetOperators(this_level->global_solve,
			this_level->mat,this_level->mat,0); CHKERRQ(ierr);
  ierr = PCSetType(this_level->global_solve,PCBJACOBI); CHKERRQ(ierr);
  ierr = PCSetVector(this_level->global_solve,this_level->u); CHKERRQ(ierr);
  ierr = PCSetUp(this_level->global_solve); CHKERRQ(ierr);
*/

  /* == create the smoothers == */
  if (smoother_choice==AMLSmoothNone) {
    this_level->pre_smooth = 0;
    this_level->post_smooth = 0;
  } else {
    KSP local;
    ierr = ParPreGenerateSLES
      (comm,&(this_level->pre_smooth)); CHKERRQ(ierr);
    ierr = SLESGetKSP(this_level->pre_smooth,&local); CHKERRQ(ierr);
    ierr = SLESSetOptionsPrefix
      (this_level->pre_smooth,prefix); CHKERRQ(ierr);
    ierr = SLESAppendOptionsPrefix
      (this_level->pre_smooth,"presmoother_"); CHKERRQ(ierr);
    ierr = SLESSetFromOptions(this_level->pre_smooth); CHKERRQ(ierr);

    ierr = ParPreGenerateSLES
      (comm,&(this_level->post_smooth)); CHKERRQ(ierr);
    ierr = SLESGetKSP(this_level->post_smooth,&local); CHKERRQ(ierr);
    ierr = SLESSetOptionsPrefix
      (this_level->post_smooth,prefix); CHKERRQ(ierr);
    ierr = SLESAppendOptionsPrefix
      (this_level->post_smooth,"postsmoother_"); CHKERRQ(ierr);
    ierr = SLESSetFromOptions(this_level->post_smooth); CHKERRQ(ierr);

    if (smoother_choice && AMLPreSmooth) {
      ierr = SLESSetOperators
	(this_level->pre_smooth,this_level->mat,this_level->mat,
	 (MatStructure)0);
      CHKERRQ(ierr);
    }
    if (smoother_choice && AMLPostSmooth) {
      ierr = SLESSetOperators
	(this_level->post_smooth,this_level->mat,this_level->mat,
	 (MatStructure)0);
      CHKERRQ(ierr);
    }
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "Set11Solver"
static int Set11Solver(Mat g11,Vec g1,Vec g2,char *prefix,SLES *a11_solve)
{
  MPI_Comm comm = g11->comm;
  SLES a11; PC local_pc;
  int ierr;
  
  PetscFunctionBegin;
  /* define the (1,1) inverter */
  ierr = ParPreGenerateSLES(comm,&a11); CHKERRQ(ierr);
  ierr = SLESSetOptionsPrefix(a11,prefix); CHKERRQ(ierr);
  ierr = SLESAppendOptionsPrefix(a11,"11solver_"); CHKERRQ(ierr);
  ierr = SLESSetFromOptions(a11); CHKERRQ(ierr);

  ierr = SLESSetOperators(a11,g11,g11,(MatStructure)0); CHKERRQ(ierr);
  ierr = SLESGetPC(a11,&local_pc); CHKERRQ(ierr);
  ierr = PCSetVector(local_pc,g1); CHKERRQ(ierr);
  ierr = SLESSetUp(a11,g1,g2); CHKERRQ(ierr);
  *a11_solve = a11;

  /*ierr = ParPreSetupSLES(*a11_solve,pctype,g11,g1); CHKERRQ(ierr);*/
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "SetupLastLevel"
static int SetupLastLevel
(Mat last_mat,Vec u2,Vec v2,char *prefix,SLES *last_solve)
{
  SLES last;
  PC local_pc; int ierr;

  PetscFunctionBegin;
  ierr = ParPreGenerateSLES(last_mat->comm,&last); CHKERRQ(ierr);
  ierr = SLESSetOptionsPrefix(last,prefix); CHKERRQ(ierr);
  ierr = SLESAppendOptionsPrefix(last,"lastlevel_"); CHKERRQ(ierr);
  ierr = SLESSetFromOptions(last); CHKERRQ(ierr);

  ierr = SLESSetOperators(last,last_mat,last_mat,(MatStructure)0);
  CHKERRQ(ierr);
  ierr = SLESGetPC(last,&local_pc); CHKERRQ(ierr);
  ierr = PCSetVector(local_pc,u2); CHKERRQ(ierr);
  ierr = SLESSetUp(last,u2,v2); CHKERRQ(ierr);

  *last_solve = last;

  PetscFunctionReturn(0);
}

/****************************************************************
 * Submatrix handling                                           *
 ****************************************************************/
#undef __FUNC__
#define __FUNC__ "SetupOneLevel"
static int SetupOneLevel
(MPI_Comm comm,Mat mat,IS idx,int level,int trace,
 int mis_from_strong,int pivot_repair,
 int isize,int jsize,int global_size,
 char *prefix,
 AMLFillMethod fill_method,int modification,AMLSolveScheme solve_scheme,
 AMLCoarseGridChoice grid_choice,AMLSchurChoice schur_choice,
 AMLSmootherChoice smoothers,
 int cutoff,double weight,int it11,int transfer,int degree,
 MC_OneLevel_struct **return_level,int *back_track,int orth)
{
  MC_OneLevel_struct *this_level;
  int stop_now,ierr;

  PetscFunctionBegin;
  /*  MatFileDump(mat,"lev",level);*/

  /* initial setup of the level */
  this_level = (MC_OneLevel_struct *)
    PetscMalloc(sizeof(MC_OneLevel_struct)); CHKPTRQ(this_level);
  if (trace & AMLTraceProgress) PetscPrintf(comm,"=> Level %d; setup\n",level);
  ierr = InitOneLevel
    (comm,level,mat,idx,
     &(this_level->mat),&(this_level->level),&(this_level->indices),
     &(this_level->u),&(this_level->v),&(this_level->w),&(this_level->diag),
     orth,isize,jsize,&(this_level->isize),&(this_level->jsize),
     global_size,
     trace);
  this_level->solve_scheme = solve_scheme; this_level->comm = comm;
  this_level->transfer = transfer; this_level->it11 = it11;
  this_level->degree = degree; 
  ierr = SetupPolynomialCoffs
    (this_level->level,this_level->degree,&(this_level->coffs)); CHKERRQ(ierr);
#if TRACING
  ierr = LambdaOneEst(mat,this_level); CHKERRQ(ierr);
#endif
  ierr = MatGetDiagonal(this_level->mat,this_level->diag); CHKERRQ(ierr);
  if (trace & AMLTraceProgress) PetscPrintf(comm,"set smoothers\n");
  ierr = SetSmoothers(smoothers,this_level,prefix); CHKERRQ(ierr);

  if (trace & AMLTraceProgress) PetscPrintf(comm,"partitioning matrix\n");
  ierr = PartitionMatrix
    (mat,this_level->u,this_level->v,level,grid_choice,
     &(this_level->g1),&(this_level->u1),&(this_level->h1),&(this_level->v1),
     &(this_level->k1),&(this_level->w1),
     &(this_level->g2),&(this_level->u2),&(this_level->h2),&(this_level->v2),
     &(this_level->k2),&(this_level->w2),
     &(this_level->get_clr),&(this_level->put_clr),
     &(this_level->get_rest),&(this_level->put_rest),
     this_level->diag,&(this_level->cdiag),&(this_level->cdiag1),
     this_level->indices,
     &(this_level->indices1),&(this_level->indices2),&(this_level->g_indices2),
     &(this_level->g12),&(this_level->a12),
     &(this_level->g21),&(this_level->a21),&(this_level->l21),
     &(this_level->g11),&(this_level->g22), trace,
     mis_from_strong && /* ( schur_choice!=AMLSchurVariational ) && */
     ( (level==BOTTOM_LEVEL) || !(fill_method==AMLFillStrong) ),
     weight,cutoff,&stop_now,back_track,
     orth,this_level->isize,this_level->jsize);
  CHKERRQ(ierr);

  if (*back_track) {
    if (level==BOTTOM_LEVEL) SETERRQ(1,0,"Whole matrix independent");
    PetscFunctionReturn(0);
  } 

  ierr = Set11Solver
    (this_level->g11,this_level->g1,this_level->g2,prefix,
     &(this_level->a11_solve));
  CHKERRQ(ierr);
  if (fill_method == AMLFillNone) {
    this_level->s2 = this_level->g22;
  } else {
    Mat fill;
    if (trace & AMLTraceProgress) PetscPrintf(comm,"Making fill matrix\n");
    ierr = NextLevelFill
      (comm,level, grid_choice,schur_choice,
       this_level->g21,this_level->l21,this_level->a21,
       &this_level->b12,
       this_level->indices1,this_level->indices2,this_level->g_indices2,
       this_level->a11_solve,this_level->g11,this_level->cdiag1,
       this_level->g12,this_level->a12,this_level->g22,
       this_level->g1,this_level->h1,this_level->k1,
       this_level->g2,this_level->h2,this_level->k2,
       this_level->u1,this_level->v1,this_level->w1,this_level->u2,
       modification,trace,weight,&fill);
    CHKERRQ(ierr);
    /*printf("orig 22 block\n");MatView(this_level->g22,0);
      printf("fill matrix\n");MatView(fill,0);*/
    if (trace & AMLTraceProgress) PetscPrintf(comm,"Making Schur complement\n");
    ierr = NextMatFill
      (comm,fill_method,0,pivot_repair,trace,weight,
       fill,this_level->g22,this_level->u2,this_level->v2,
       &(this_level->s2));
    CHKERRQ(ierr);
    /*    printf("matrix filled\n");MatView(this_level->s2,0);*/
    ierr = MatDestroy(fill); CHKERRQ(ierr);
  }
  if (stop_now) {
    this_level->next_level = 0;
    if (trace & AMLTraceProgress) PetscPrintf(comm,"Setup last level\n");
    ierr = SetupLastLevel
      (this_level->s2,this_level->g2,this_level->h2,
       prefix,&(this_level->a22_solve));
    CHKERRQ(ierr);    
  } else {
    int back_track,gsize,idum;
    ierr = MatGetSize(this_level->s2,&gsize,&idum); CHKERRQ(ierr);
    ierr = SetupOneLevel
      (comm,this_level->s2,this_level->indices2,level+1,trace,
       mis_from_strong,pivot_repair,
       this_level->isize,this_level->jsize,gsize,
       prefix,
       fill_method,modification,solve_scheme,
       grid_choice,schur_choice,smoothers,
       cutoff,weight,it11,transfer,degree,
       &(this_level->next_level),&back_track,orth);
    CHKERRQ(ierr);
    if (back_track) {
      this_level->next_level = 0;
      ierr = SetupLastLevel
	(this_level->s2,this_level->u2,this_level->v2,
	 prefix,&(this_level->a22_solve));
	CHKERRQ(ierr);
    }
  }
  *return_level = this_level;

  PetscFunctionReturn(0);
}
#undef __FUNC__
#define __FUNC__ "PCSetFromOptions_MLevel"
int PCSetFromOptions_MLevel(PC pc)
{
  char string[64];
  char *f; PetscTruth flg; int ierr,lev;

  PetscFunctionBegin;
  ierr = OptionsGetString
    (pc->prefix,"-multilevel_solutionscheme",string,64,&flg);
  CHKERRQ(ierr);
  if (flg) {
    AMLSolveScheme Scheme=0; int found = 0;
    /*    printf("solution scheme <%s>\n",string);*/
    ierr = PetscStrstr(string,"mg",&f); CHKERRQ(ierr);
    if (f) {found++; Scheme = AMLSolveMG;}
    ierr = PetscStrstr(string,"poly",&f); CHKERRQ(ierr);
    if (f) {found++; Scheme = AMLSolvePolynomial;}
    ierr = PetscStrstr(string,"lu",&f); CHKERRQ(ierr);
    if (f) {found++; Scheme = AMLSolveILU;}
    if (found>0) {
      ierr = AMLSetSolutionScheme(pc,Scheme); CHKERRQ(ierr);
    } else SETERRQ(1,0,"Multilevel set from options: unknown scheme\n");
  }

  ierr = OptionsGetString
    (pc->prefix,"-multilevel_fillmethod",string,64,&flg);
  CHKERRQ(ierr);
  if (flg) {
    AMLFillMethod Method=0; int found = 0;
    /*    printf("fill method <%s>\n",string);*/
    ierr = PetscStrstr(string,"none",&f); CHKERRQ(ierr);
    if (f) {found++; Method = AMLFillNone;}
    ierr = PetscStrstr(string,"diagonal",&f); CHKERRQ(ierr);
    if (f) {found++; Method = AMLFillDiag;}
    ierr = PetscStrstr(string,"strong",&f); CHKERRQ(ierr);
    if (f) {found++; Method = AMLFillStrong;}
    ierr = PetscStrstr(string,"full",&f); CHKERRQ(ierr);
    if (f) {found++; Method = AMLFillFull;}
    if (found>0) {
      ierr = AMLSetFillMethod(pc,Method); CHKERRQ(ierr); 
    } else SETERRQ(1,0,"Multilevel fill method: unknown\n");
  }

  ierr = OptionsGetString
    (pc->prefix,"-multilevel_coarsegrid",string,64,&flg);
  CHKERRQ(ierr);
  if (flg) {
    AMLCoarseGridChoice Choice=0; int found = 0;
    ierr = PetscStrstr(string,"dependent",&f); CHKERRQ(ierr);
    if (f)
      {found++; Choice = AMLCoarseGridDependent;}
    ierr = PetscStrstr(string,"independent",&f); CHKERRQ(ierr);
    if (f)
      {found++; Choice = AMLCoarseGridIndependent;}
    /* printf("coarse grid <%s>\n",string);*/
    if (found>0) {
      ierr = AMLSetCoarseGridDependent(pc); CHKERRQ(ierr);
      } else SETERRQ(1,0,"Multilevel coarse grid: unknown");
  }

  ierr = OptionsGetString
    (pc->prefix,"-multilevel_smoothers",string,64,&flg);
  CHKERRQ(ierr);
  if (flg) {
    AMLSmootherChoice Choice; int found = 0;
    /*printf("smoother choice <%s>\n",string);*/
    ierr = PetscStrstr(string,"pre",&f); CHKERRQ(ierr);
    if (f) {found++; Choice = AMLPreSmooth;}
    ierr = PetscStrstr(string,"post",&f); CHKERRQ(ierr);
    if (f) {found++; Choice = AMLPostSmooth;}
    ierr = PetscStrstr(string,"symmetric",&f); CHKERRQ(ierr);
    if (f) {found++; Choice = AMLPrePostSmooth;}
    ierr = PetscStrstr(string,"none",&f); CHKERRQ(ierr);
    if (f) {found++; Choice = AMLSmoothNone;}
    if (found>0) {
      ierr = AMLSetSmootherChoice(pc,Choice); CHKERRQ(ierr);
    } else SETERRQ(1,0,"Multilevel smoother choice: unknown");
  }

  ierr = OptionsGetInt(pc->prefix,"-multilevel_tracelevel",&lev,&flg);
  CHKERRQ(ierr);
  if (flg) {ierr = AMLSetTraceLevel(pc,lev); CHKERRQ(ierr);}

  ierr = OptionsGetInt(pc->prefix,"-multilevel_degree",&lev,&flg);
  CHKERRQ(ierr);
  if (flg) {ierr = AMLSetCycleDegree(pc,lev); CHKERRQ(ierr);}

  {
    Scalar val;
    ierr = OptionsGetScalar
      (pc->prefix,"-multilevel_strongratio",&val,&flg); CHKERRQ(ierr);
    ierr = PetscRealPart(val); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCSetup_MLevel"
int PCSetup_MLevel(PC pc)
{
  Mat base_mat = pc->pmat;
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
  IS idx_set;
  char *prefix;
  int gsize,idum,ierr;

  PetscFunctionBegin;
  {
    int rstart,rend,loc;
    ierr = MatGetOwnershipRange(base_mat,&rstart,&rend); CHKERRQ(ierr);
    loc = rend-rstart;
    ierr = ISCreateStride(base_mat->comm,loc,rstart,1,&idx_set);
    CHKERRQ(ierr);
  }
  ierr = PCGetOptionsPrefix(pc,&prefix); CHKERRQ(ierr);

  ierr = MatGetSize(base_mat,&gsize,&idum); CHKERRQ(ierr);

  ierr = SetupOneLevel
    (base_mat->comm,base_mat,idx_set,BOTTOM_LEVEL,pc_data->trace_level,
     pc_data->mis_from_strong,pc_data->pivot_repair,
     0,0,gsize,
     prefix,
     pc_data->fill_method,pc_data->modification,pc_data->solve_scheme,
     pc_data->grid_choice,pc_data->schur_choice,
     pc_data->smoother_choice,
     pc_data->cutoff,pc_data->weight,
     pc_data->it11,pc_data->transfer,pc_data->degree,
     &(pc_data->level_stuff),&idum,pc_data->orth); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

/****************************************************************
 * Solve                                                        *
 ****************************************************************/

#undef __FUNC__
#define __FUNC__ "PCApply_MLevel"
static int PCApply_MLevel(PC pc,Vec x,Vec y)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
  int ierr;

  PetscFunctionBegin;
  /*printf("input vector\n"); VecView(x,0);*/
  ierr = SolveMultiLevel(pc_data->level_stuff,x,y,pc_data->trace_level);
  CHKERRQ(ierr);
  /*printf("output vector\n"); VecView(y,0);*/

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "AMLDestroyLevel"
int AMLDestroyLevel(MC_OneLevel_struct *level)
{
  int ierr;

  PetscFunctionBegin;
  PetscFree(level->coffs);
  if (level->level==BOTTOM_LEVEL) {/* this is the s2 of a lower level */
    ierr = MatDestroy(level->mat); CHKERRQ(ierr);}
  ierr = MatDestroy(level->a21); CHKERRQ(ierr);
  ierr = MatDestroy(level->a12); CHKERRQ(ierr);
  ierr = MatDestroy(level->g11); CHKERRQ(ierr);
  ierr = MatDestroy(level->g12); CHKERRQ(ierr);
  ierr = MatDestroy(level->g21); CHKERRQ(ierr);
  ierr = MatDestroy(level->g22); CHKERRQ(ierr);
  /*ierr = MatDestroy(level->s2); CHKERRQ(ierr);*/
  ierr = VecDestroy(level->u); CHKERRQ(ierr);
  ierr = VecDestroy(level->v); CHKERRQ(ierr);
  ierr = VecDestroy(level->w); CHKERRQ(ierr);
  ierr = VecDestroy(level->u1); CHKERRQ(ierr);
  ierr = VecDestroy(level->u2); CHKERRQ(ierr);
  ierr = VecDestroy(level->v1); CHKERRQ(ierr);
  ierr = VecDestroy(level->v2); CHKERRQ(ierr);
  ierr = VecDestroy(level->w1); CHKERRQ(ierr);
  ierr = VecDestroy(level->w2); CHKERRQ(ierr);
  ierr = VecDestroy(level->g1); CHKERRQ(ierr);
  ierr = VecDestroy(level->g2); CHKERRQ(ierr);
  ierr = VecDestroy(level->h1); CHKERRQ(ierr);
  ierr = VecDestroy(level->h2); CHKERRQ(ierr);
  ierr = VecDestroy(level->k1); CHKERRQ(ierr);
  ierr = VecDestroy(level->k2); CHKERRQ(ierr);
  ierr = SLESDestroy(level->a11_solve); CHKERRQ(ierr);
  /*
  if (level->a22_solve) {
    ierr = SLESDestroy(level->a22_solve); CHKERRQ(ierr);}
  */
  if (level->pre_smooth) {
    ierr = SLESDestroy(level->pre_smooth); CHKERRQ(ierr);}
  if (level->post_smooth) {
    ierr = SLESDestroy(level->post_smooth); CHKERRQ(ierr);}
  ierr = ISDestroy(level->indices1); CHKERRQ(ierr);
  ierr = ISDestroy(level->indices2); CHKERRQ(ierr);
  ierr = ISDestroy(level->g_indices2); CHKERRQ(ierr);
  ierr = VecScatterDestroy(level->get_clr); CHKERRQ(ierr);
  ierr = VecScatterDestroy(level->put_clr); CHKERRQ(ierr);
  ierr = VecScatterDestroy(level->get_rest); CHKERRQ(ierr);
  ierr = VecScatterDestroy(level->put_rest); CHKERRQ(ierr);
  if (level->next_level) {
    ierr = AMLDestroyLevel(level->next_level); CHKERRQ(ierr);
    PetscFree(level->next_level);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCDestroy_MLevel"
int PCDestroy_MLevel(PC pc)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
  int ierr;

  PetscFunctionBegin;
  if (pc_data->level_stuff) {
    ierr = AMLDestroyLevel(pc_data->level_stuff); CHKERRQ(ierr);
    PetscFree(pc_data->level_stuff);
  }
  PetscFunctionReturn(0);

}

/*#undef __FUNC__
#define __FUNC__ "PCDuplicate_MLevel"
int PCDuplicate_MLevel(PC old,PC *new)
{
  PetscFunctionBegin;
  *new = old;
  PetscFunctionReturn(0);
}*/

#undef __FUNC__
#define __FUNC__ "PCView_MultiLevel"
int PCView_MultiLevel(PC pc,Viewer viewer)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
  int ierr;

  PetscFunctionBegin;
  PetscPrintf(pc->comm,"Fill method: ");
  switch ((int)pc_data->fill_method) 
    {case AMLFillNone : PetscPrintf(pc->comm,"none"); break;
    case AMLFillDiag : PetscPrintf(pc->comm,"diagonal"); break;
    case AMLFillStrong : PetscPrintf(pc->comm,"strong"); break;
    case AMLFillFull : PetscPrintf(pc->comm,"full"); break;}
  PetscPrintf(pc->comm,"\n");

  PetscPrintf(pc->comm,"Coarse grid choice: ");
  switch ((int)pc_data->grid_choice)
    {case AMLCoarseGridDependent :
      PetscPrintf(pc->comm,"dependent"); break;
    case AMLCoarseGridIndependent :
      PetscPrintf(pc->comm,"independent"); break;}
  PetscPrintf(pc->comm,"\n");

  PetscPrintf(pc->comm,"Schur complement method: ");
  switch ((int)pc_data->schur_choice)
    {case AMLSchurElimination :
      PetscPrintf(pc->comm,"elimination"); break;
    case AMLSchurVariational :
      PetscPrintf(pc->comm,"variational"); break;}
  PetscPrintf(pc->comm,"\n");

  PetscPrintf(pc->comm,"Smoother choice: ");
  switch ((int)pc_data->smoother_choice)
    {case AMLSmoothNone :
      PetscPrintf(pc->comm,"none"); break;
    case AMLPreSmooth :
      PetscPrintf(pc->comm,"pre only"); break;
    case AMLPostSmooth :
      PetscPrintf(pc->comm,"post only"); break;
    case AMLPrePostSmooth :
      PetscPrintf(pc->comm,"pre and post"); break;}
  PetscPrintf(pc->comm,"\n");

  PetscPrintf(pc->comm,"Strong element ratio: %e\n",pc_data->weight);

  if (pc_data->pivot_repair>0)
    PetscPrintf(pc->comm,"Pivot repair strategy %d\n",pc_data->pivot_repair);

  ierr = ViewLevel(pc_data->level_stuff,viewer); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCCreate_MLevel"
int PCCreate_MLevel(PC pc)
{
  PC_MCol_struct *bij;
  int ierr;

  PetscFunctionBegin;
  pc->ops->apply     = PCApply_MLevel;
  pc->ops->applyrichardson = 0;
  pc->ops->destroy   = PCDestroy_MLevel;
  pc->ops->setfromoptions   = PCSetFromOptions_MLevel;
  pc->ops->setup     = PCSetup_MLevel;
  /*  pc->type      = PCMultiLevel;*/
  pc->ops->view      = PCView_MultiLevel;

  /* create a local solve, equivalent to smoother */
  /* VE this has to go !!! */
  ierr = PCParallelSubdomainsCreate(pc,sizeof(PC_MCol_struct));

  bij = (PC_MCol_struct *) pc->data;
  bij->fill_method = AMLFillNone; bij->modification = 0;
  bij->solve_scheme = AMLSolveILU;
  bij->grid_choice = AMLCoarseGridDependent;
  bij->schur_choice = AMLSchurElimination;
  bij->smoother_choice = AMLSmoothNone;
  bij->cutoff = 10; bij->it11 = 1; bij->transfer = 1; bij->degree = 1;
  bij->mis_from_strong = 0; bij->pivot_repair = 1; bij->weight = .25;
  bij->trace_level = 0;
  bij->orth = 0;

  PetscFunctionReturn(0);
}
