/*
   Defines an Domain Decomposition  preconditioner
*/
#include "parpre_pc.h"
#include "sles.h"
#include "src/sles/slesimpl.h"
#include "src/pc/pcimpl.h"
#include "vec.h"
#include "src/vec/vecimpl.h"
#include "parpre_subdomains.h"
#include "parpre_mat.h"
#include "parpre_mat.h"
#include "src/mat/parpre_matimpl.h"
#include "options.h"
#include "src/mat/impls/aij/mpi/mpiaij.h"

#define CHUNCKSIZE   100

typedef struct {
  PCParallelSubdomainStruct subdomains;
  SLES interface_method;
  Mat C11,C12,C21,/* C12_big,C21_big,*/ C22;
  Mat interface_system; 
  Vec global_vector,global_vector2; /* unused ? */
  Vec local_edge_vec,local_edge_vec2, mpi_edge_vec,mpi_edge_vec2;
  Vec local_intl_vec,local_intl_vec2, mpi_intl_vec,mpi_intl_vec2;
  Vec interface_vector,interface_vector2; /* unnecessary? */
  VecScatter get_edge,put_edge,get_intl,put_intl;
} PC_DDecomp_struct;

/****************************************************************
 * User Interface
 ****************************************************************/
#undef __FUNC__
#define __FUNC__ "PCDomainDecompGetInterfacePC"
int PCDomainDecompGetInterfacePC(PC pc,PC *intpc)
{
  PC_DDecomp_struct *data = (PC_DDecomp_struct *) pc->data;
  SLES int_solve = data->interface_method;
  int ierr;

  if (((PetscObject)pc)->cookie == PC_COOKIE) {
    ierr = SLESGetPC(int_solve,intpc); CHKERRQ(ierr);
  } else return 1;

  return 0;

}

#undef __FUNC__
#define __FUNC__ "InitialInterface"
static int InitialInterface(Mat base_mat,IS *interface)
{
  Mat_MPIAIJ *Aij = (Mat_MPIAIJ *) base_mat->data;
  VecScatter ctx = Aij->Mvctx;
  VecScatter_MPI_General *to = (VecScatter_MPI_General *) ctx->todata;
  int i,istart,ierr,*idx, n = to->starts[to->n];

  ierr = MatGetOwnershipRange(base_mat,&istart,&i); CHKERRQ(ierr);
  idx = (int *) PetscMalloc((n+1)*sizeof(int)); CHKPTRQ(idx);
  for (i=0; i<n; i++) idx[i] = to->indices[i]+istart;
  ierr = ISCreateGeneral(MPI_COMM_SELF,n,idx,interface);
  PetscFree(idx);

  return 0;
}

#undef __FUNC__
#define __FUNC__ "RestrictInterface"
static int RestrictInterface(Mat base_mat,IS *interface)
{
  Mat_MPIAIJ *Aij = (Mat_MPIAIJ *) base_mat->data;
  Mat off_mat = Aij->B;
  int istart,iend,i,n_new,n_int,ierr,*new_idx,*int_idx;

  ierr = ISGetSize(*interface,&n_int); CHKERRQ(ierr);
  new_idx = (int *) PetscMalloc((n_int+1)*sizeof(int)); CHKPTRQ(new_idx);

  /*printf("Restricting interface: ");*/
  ierr = ISGetIndices(*interface,&int_idx); CHKERRQ(ierr);
  ierr = MatGetOwnershipRange(base_mat,&istart,&iend); CHKERRQ(ierr);
  n_new = 0;
  for ( i=0; i<n_int; i++ ){
    int row = int_idx[i],accept;
    if ( (row<istart) | (row>=iend) )
      accept = 0;
    else {
      int lrow = row-istart,col,ncol,*idx; accept = 0;
      ierr = MatGetRow(off_mat,lrow,&ncol,&idx,PETSC_NULL); CHKERRQ(ierr);
      if (ncol==0)
	accept = 1;
      else 
	for (col=0; col<ncol; col++) {
	  if (Aij->garray[idx[col]]>=iend) {accept=1; break;}
	}
      ierr = MatRestoreRow(off_mat,lrow,&ncol,&idx,PETSC_NULL); CHKERRQ(ierr);
      /*    accept = 1;*/
    }
    if (accept)
      new_idx[n_new++] = int_idx[i];
    /*if (accept) printf("%d:Yes ",row); else printf("%d:No ",row);*/
  }
  /*printf("\n");*/
  ierr = ISRestoreIndices(*interface,&int_idx); CHKERRQ(ierr);
  ierr = ISDestroy(*interface); CHKERRQ(ierr);
  ierr = ISCreateGeneral(MPI_COMM_SELF,n_new,new_idx,interface); CHKERRQ(ierr);
  ierr = ISSort(*interface); CHKERRQ(ierr);
  PetscFree(new_idx);

  return 0;
}

#undef __FUNC__
#define __FUNC__ "LocalSplitSets"
static int LocalSplitSets
(Mat base_mat,IS interface,IS *edge_set,IS *intl_set)
{
  int istart,iend,loc_size, *idx,nidx, i_idx,i,n_e,n_i, *ee,*ii, ierr;

  ierr = MatGetOwnershipRange(base_mat,&istart,&iend); CHKERRQ(ierr);
  loc_size = iend-istart;
  ierr = ISGetSize(interface,&nidx); CHKERRQ(ierr);
  ierr = ISGetIndices(interface,&idx); CHKERRQ(ierr);

  n_e = 0;
  for (i=0; i<nidx; i++)
    if ( (idx[i]>=istart) & (idx[i]<iend) )
      n_e++;

  ee = (int *) PetscMalloc( (n_e+1)*sizeof(int) );
  CHKPTRQ(ee);
  ii = (int *) PetscMalloc( (loc_size-n_e+1)*sizeof(int) );
  CHKPTRQ(ii);

  n_e = n_i = 0;
  i_idx = 0;
  if (nidx>0) {
    while ( (i_idx<nidx) & (idx[i_idx]<istart) ) i_idx++;
    for ( i=istart; i<iend; i++ )
      if (i==idx[i_idx]) {
	ee[n_e++] = i; i_idx++;
      } else
	ii[n_i++] = i;
  } else
    for (i=istart; i<iend; i++) ii[n_i++] = i;

  ierr = ISRestoreIndices(interface,&idx); CHKERRQ(ierr);

  ierr = ISCreateGeneral(MPI_COMM_SELF,n_e,ee,edge_set); CHKERRQ(ierr);
  ierr = ISCreateGeneral(MPI_COMM_SELF,n_i,ii,intl_set); CHKERRQ(ierr);
  PetscFree(ee); PetscFree(ii);

  return 0;
}

#undef __FUNC__
#define __FUNC__ "DDVectors"
static int DDVectors
(PC_DDecomp_struct *pc_data,MPI_Comm comm, int nloc,int n_edge,int n_intl)
{
  Scalar *g;
  int ierr;

  /* internal vector */
  ierr = VecCreateMPI(comm,n_intl,PETSC_DECIDE,&pc_data->mpi_intl_vec);
  CHKERRQ(ierr);
  ierr = VecDuplicate(pc_data->mpi_intl_vec,&pc_data->mpi_intl_vec2);
  CHKERRQ(ierr);
  /* local vectors with the same allocated array space */
  ierr = VecGetArray(pc_data->mpi_intl_vec,&g); CHKERRQ(ierr);
  ierr = VecCreateSeqWithArray
    (MPI_COMM_SELF,n_intl,g,&pc_data->local_intl_vec);
  CHKERRQ(ierr);
  ierr = VecGetArray(pc_data->mpi_intl_vec2,&g); CHKERRQ(ierr);
  ierr = VecCreateSeqWithArray
    (MPI_COMM_SELF,n_intl,g,&pc_data->local_intl_vec2);
  CHKERRQ(ierr);

  /* edge vectors */
  ierr = VecCreateMPI(comm,n_edge,PETSC_DECIDE,&pc_data->mpi_edge_vec);
  CHKERRQ(ierr);
  ierr = VecDuplicate(pc_data->mpi_edge_vec,&pc_data->mpi_edge_vec2);
  CHKERRQ(ierr);
  /* local vectors with the same allocated array space */
  ierr = VecGetArray(pc_data->mpi_edge_vec,&g); CHKERRQ(ierr);
  ierr = VecCreateSeqWithArray
    (MPI_COMM_SELF,n_edge,g,&pc_data->local_edge_vec);
  CHKERRQ(ierr);
  ierr = VecGetArray(pc_data->mpi_edge_vec2,&g); CHKERRQ(ierr);
  ierr = VecCreateSeqWithArray
    (MPI_COMM_SELF,n_edge,g,&pc_data->local_edge_vec2);
  CHKERRQ(ierr);

  /* create global vectors */
  ierr = VecCreateMPI
    (comm,nloc,PETSC_DECIDE,&pc_data->global_vector);
  CHKERRQ(ierr);
  ierr = VecDuplicate
    (pc_data->global_vector,&pc_data->global_vector2);
  CHKERRQ(ierr);

  ierr = VecCreateMPI
    (comm,n_edge,PETSC_DECIDE,&pc_data->interface_vector);
  CHKERRQ(ierr);
  ierr = VecDuplicate
    (pc_data->interface_vector,&pc_data->interface_vector2);
  CHKERRQ(ierr);

return 0;
}

#undef __FUNC__
#define __FUNC__ "DDScatters"
static int DDScatters
(PC_DDecomp_struct *pc_data,MPI_Comm comm,
 int n_interior,IS intl_set,IS edge_set)
{
  IS edge_src=edge_set,edge_tar,intl_src=intl_set,intl_tar;
  int i_start,e_start,n_edge_vars,n_intl_vars;
  int ierr;

  /* create internal and interface vectors */
  ierr = ISGetSize(edge_set,&n_edge_vars); CHKERRQ(ierr);
  ierr = ISGetSize(intl_set,&n_intl_vars); CHKERRQ(ierr);
  ierr = DDVectors
    (pc_data,comm,n_interior,n_edge_vars,n_intl_vars);
  CHKERRQ(ierr);

  {
    int np,ip,*ps,p,s;

    /* consecutive global index sets for internal & edge vars */
    MPI_Comm_size(comm,&np); MPI_Comm_rank(comm,&ip);
    ps = (int *) PetscMalloc(np*sizeof(int)); CHKPTRQ(ps);

    MPI_Allgather((void*)&n_edge_vars,1,MPI_INT,(void*)ps,1,MPI_INT,comm);
    s = 0;
    for (p=0; p<np; p++) {
      int t = ps[p];
      ps[p] = s;
      s += t;
    }
    e_start = ps[ip];

    MPI_Allgather((void*)&n_intl_vars,1,MPI_INT,(void*)ps,1,MPI_INT,comm);
    s = 0;
    for (p=0; p<np; p++) {
      int t = ps[p];
      ps[p] = s;
      s += t;
    }
    i_start = ps[ip];
  }

  ierr = ISCreateStride(MPI_COMM_SELF,n_edge_vars,e_start,1,&edge_tar);
  CHKERRQ(ierr);
  ierr = ISCreateStride(MPI_COMM_SELF,n_intl_vars,i_start,1,&intl_tar);
  CHKERRQ(ierr);

  ierr = VecScatterCreate
    (pc_data->global_vector,intl_src,pc_data->mpi_intl_vec,intl_tar,
     &pc_data->get_intl);
  CHKERRQ(ierr);

  ierr = VecScatterCreate
    (pc_data->mpi_intl_vec,intl_tar,pc_data->global_vector,intl_src,
     &pc_data->put_intl);
  CHKERRQ(ierr);
  
  ierr = VecScatterCreate
    (pc_data->global_vector,edge_src,pc_data->mpi_edge_vec,edge_tar,
     &pc_data->get_edge);
  CHKERRQ(ierr);
  ierr = VecScatterCreate
    (pc_data->mpi_edge_vec,edge_tar,pc_data->global_vector,edge_src,
     &pc_data->put_edge);
  CHKERRQ(ierr);
  
  ierr = ISDestroy(edge_tar); CHKERRQ(ierr);
  ierr = ISDestroy(intl_tar); CHKERRQ(ierr);

  return 0;
}

#undef __FUNC__
#define __FUNC__ "DDBlocks"
static int DDBlocks
(Mat base_mat,PC pc,
 IS intl_set,IS edge_set,IS g_intl_set,IS g_edge_set,
 Mat *from_interface,Mat *to_interface,Mat *interface_mat)
{
  PC_DDecomp_struct *pc_data = (PC_DDecomp_struct *) pc->data;
  IS i_sets[7],j_sets[7];
  Mat *res_mat;
  int ierr;

  i_sets[0] = intl_set; j_sets[0] = intl_set;
  i_sets[1] = edge_set; j_sets[1] = edge_set;
  i_sets[2] = edge_set; j_sets[2] = intl_set;
  i_sets[3] = intl_set; j_sets[3] = edge_set;

  i_sets[4] = intl_set; j_sets[4] = g_edge_set;
  i_sets[5] = edge_set; j_sets[5] = g_edge_set;
  i_sets[6] = g_edge_set; j_sets[6] = intl_set;

  ierr = MatGetSubMatrices
    (base_mat,7,i_sets,j_sets,MAT_INITIAL_MATRIX,&res_mat);
  CHKERRQ(ierr);
  pc_data->C11 = res_mat[0]; pc_data->C22 = res_mat[1];
  pc_data->C21 = res_mat[2]; pc_data->C12 = res_mat[3];
  *to_interface = res_mat[4]; *interface_mat = res_mat[5];
  *from_interface = res_mat[6];
  PetscFree(res_mat);

  /* Initialise various local solution methods */
  {
    SLES local_sles;
    ierr = PCParallelGetLocalSLES(pc,&local_sles); CHKERRQ(ierr);
    ierr = SLESSetOperators(local_sles,pc_data->C11,pc_data->C11,
			    (MatStructure)0);
    CHKERRQ(ierr);
  }

  return 0;
}
  

#include "src/pc/utils/auxs.c"

#undef __FUNC__
#define __FUNC__ "PCSetup_DomainDecomp"
static int PCSetup_DomainDecomp(PC pc)
{
  Mat base_mat = pc->pmat, from_interface,to_interface,interface_mat;
  MPI_Comm    comm = base_mat->comm;
  PC_DDecomp_struct *pc_data = (PC_DDecomp_struct *) pc->data;
  IS g_edge_set,g_intl_set;
  int         interface_size,local_jsize,ierr,idum;

  /* make sure you're only called for parallel execution */
  if (!(base_mat->type==MATMPIAIJ)) {
    SETERRQ(1,0,"Domain Decomp preconditioner only implemented for AIJMPI\n");
  }

  ierr = MatGetLocalSize(base_mat,&local_jsize,&idum); CHKERRQ(ierr);

  /* set the pc and interface comm */
  pc->comm = comm;
  pc_data->interface_method->comm = comm;

  {
    IS edge_set,intl_set;

    {
      IS interface_set;

      /* set the interface to the points connecting to other procs */
      ierr = InitialInterface(base_mat,&interface_set); CHKERRQ(ierr);

      /* restrict interface to connections to higher procs */
      ierr = RestrictInterface(base_mat,&interface_set); CHKERRQ(ierr);
      /*
	ierr = MatIncreaseOverlap(base_mat,1,&interface_set,1); CHKERRQ(ierr);
	ierr = RestrictInterface(base_mat,&interface_set); CHKERRQ(ierr);
	*/
      ierr = LocalSplitSets(base_mat,interface_set,&edge_set,&intl_set);
      CHKERRQ(ierr);
      ierr = ISGetSize(edge_set,&interface_size); CHKERRQ(ierr);
      ierr = ISGetGlobalContent(base_mat->comm,edge_set,&g_edge_set);
      CHKERRQ(ierr);
      ierr = ISGetGlobalContent(base_mat->comm,intl_set,&g_intl_set);
      CHKERRQ(ierr);
      ierr = ISDestroy(interface_set); CHKERRQ(ierr);
    }

    /* setup vectors and vector scatters to/from the interface */
    ierr = DDScatters(pc_data,comm,local_jsize,intl_set,edge_set);
    CHKERRQ(ierr);
    /* extract blocks and set local solution method*/
    ierr = DDBlocks(base_mat,pc,intl_set,edge_set,g_intl_set,g_edge_set,
		    &from_interface,&to_interface,&interface_mat);
    CHKERRQ(ierr);

    /* dispose of temporary data */
    ierr = ISDestroy(intl_set); CHKERRQ(ierr);
    ierr = ISDestroy(edge_set); CHKERRQ(ierr);
    ierr = MatDestroy(pc_data->C22); CHKERRQ(ierr);
  }

  {
    PC local_pc;
    PCParallelGetLocalPC(pc,&local_pc);
    ierr = PCSetVector(local_pc,pc_data->local_intl_vec); CHKERRQ(ierr);
    ierr = PCSetUp(local_pc); CHKERRQ(ierr);
  }

  {
    Mat A11C12,filler;
    SLES local_sles;
    PC local_pc;

    ierr = PCParallelGetLocalSLES(pc,&local_sles); CHKERRQ(ierr);
    ierr = MatSolveMat_AIJ(local_sles,to_interface,&A11C12); CHKERRQ(ierr);
    ierr = MatDestroy(to_interface); CHKERRQ(ierr);

    ierr = MatMatMult_AIJ(from_interface,A11C12,&filler); CHKERRQ(ierr);
    ierr = MatDestroy(from_interface); CHKERRQ(ierr);
    ierr = MatDestroy(A11C12); CHKERRQ(ierr);

    ierr = MatrixAij2MpiAbut
      (interface_size,interface_mat,comm,&pc_data->interface_system);
    CHKERRQ(ierr);

    {
      int m,n, irow,nrows,*idx;
      
      ierr = ISGetSize(g_edge_set,&nrows); CHKERRQ(ierr);
      ierr = MatGetLocalSize(filler,&m,&n); CHKERRQ(ierr);
      if ( (m!=nrows) | (n!=nrows) ) SETERRQ(1,0,"Filler / edge mismatch");
      ierr = ISGetIndices(g_edge_set,&idx); CHKERRQ(ierr);
      for (irow=0; irow<nrows; irow++) {
	int ncols,*cols,icol, row=irow/*idx[irow]*/; Scalar *vals;
	ierr = MatGetRow(filler,irow,&ncols,&cols,&vals); CHKERRQ(ierr);
	for (icol=0; icol<ncols; icol++) {
	  vals[icol] = -vals[icol];
	  /*cols[icol] = idx[icol];*/
	}
	ierr = MatSetValues
	  (pc_data->interface_system,1,&row,ncols,cols,vals,ADD_VALUES); 
	CHKERRQ(ierr);
	ierr = MatRestoreRow(filler,irow,&ncols,&cols,&vals); CHKERRQ(ierr);
      }
    }
    ierr = MatDestroy(filler); CHKERRQ(ierr);
    ierr = MatAssemblyBegin(pc_data->interface_system,MAT_FINAL_ASSEMBLY);
    CHKERRQ(ierr);
    ierr = MatAssemblyEnd(pc_data->interface_system,MAT_FINAL_ASSEMBLY);
    CHKERRQ(ierr);

    ierr = SLESSetOperators
      (pc_data->interface_method,pc_data->interface_system,
       pc_data->interface_system,(MatStructure)0);
    CHKERRQ(ierr);
    ierr = SLESGetPC(pc_data->interface_method,&local_pc); CHKERRQ(ierr);
    ierr = PCSetVector(local_pc,pc_data->interface_vector); CHKERRQ(ierr);
    ierr = PCSetUp(local_pc); CHKERRQ(ierr);
  }

  ierr = ISDestroy(g_intl_set); CHKERRQ(ierr);
  ierr = ISDestroy(g_edge_set); CHKERRQ(ierr);

  return 0;
}

#undef __FUNC__
#define __FUNC__ "PCApply_DomainDecomp"
static int PCApply_DomainDecomp(PC pc,Vec x,Vec y)
{
  PC_DDecomp_struct *pc_data = (PC_DDecomp_struct *) pc->data;
  SLES local_sles;
  int ierr,its;
  /*internal*/
  Vec tmp_i = pc_data->mpi_intl_vec, tmp_li = pc_data->local_intl_vec;
  Vec tmp_i2 = pc_data->mpi_intl_vec2, tmp_li2 = pc_data->local_intl_vec2;
  /* edge */
  Vec tmp_e = pc_data->mpi_edge_vec/* ,tmp_le = pc_data->local_edge_vec*/;
  Vec tmp_e2 = pc_data->mpi_edge_vec2,tmp_le2 = pc_data->local_edge_vec2;
  Scalar zero = 0.0, mone = -1.0;

  ierr = VecSet(&zero,y); CHKERRQ(ierr);
  ierr = PCParallelGetLocalSLES(pc,&local_sles); CHKERRQ(ierr);

  /* pack interior in smaller vector */
  ierr = VecScatterBegin
    (x,tmp_i,INSERT_VALUES,SCATTER_FORWARD,pc_data->get_intl);
  CHKERRQ(ierr);
  ierr = VecScatterEnd
    (x,tmp_i,INSERT_VALUES,SCATTER_FORWARD,pc_data->get_intl);
  CHKERRQ(ierr);

  /* solve interior into tmp_i */
  ierr = SLESSolve(local_sles,tmp_li,tmp_li2,&its); CHKERRQ(ierr);

  /* pack edge in smaller vector */
  ierr = VecScatterBegin
    (x,tmp_e,INSERT_VALUES,SCATTER_FORWARD,pc_data->get_edge);
  CHKERRQ(ierr);
  ierr = VecScatterEnd
    (x,tmp_e,INSERT_VALUES,SCATTER_FORWARD,pc_data->get_edge);
  CHKERRQ(ierr);

  /* multiply interior vector to edge vector tmp_e */
  ierr = MatMult(pc_data->C21,tmp_li2,tmp_le2); CHKERRQ(ierr);

  /* subtract from existing edge into edge_vec */
  ierr = VecAXPY(&mone,tmp_e2,tmp_e); CHKERRQ(ierr);
  
  /* solve on the edge, data is on pc_data->edge_vec */
  ierr = SLESSolve(pc_data->interface_method,tmp_e,tmp_e2,&its); CHKERRQ(ierr);

  /* write edge into global output */
  ierr = VecScatterBegin
    (tmp_e2,y,INSERT_VALUES,SCATTER_FORWARD,pc_data->put_edge);
  CHKERRQ(ierr);
  ierr = VecScatterEnd
    (tmp_e2,y,INSERT_VALUES,SCATTER_FORWARD,pc_data->put_edge);
  CHKERRQ(ierr);

  /* multiply edge to internal vector into tmp_i */
  ierr = MatMult(pc_data->C12,tmp_le2,tmp_li2); CHKERRQ(ierr);

  /* subtract from existing internal into tmp_i */
  ierr = VecAXPY(&mone,tmp_i2,tmp_i); CHKERRQ(ierr);

  /* solve internal into intl_vec */
  ierr = SLESSolve(local_sles,tmp_li,tmp_li2,&its); CHKERRQ(ierr);

  /* write internal into global output */
  ierr = VecScatterBegin
    (tmp_i2,y,INSERT_VALUES,SCATTER_FORWARD,pc_data->put_intl);
  CHKERRQ(ierr);
  ierr = VecScatterEnd
    (tmp_i2,y,INSERT_VALUES,SCATTER_FORWARD,pc_data->put_intl);
  CHKERRQ(ierr);

  return 0;

}

#undef __FUNC__
#define __FUNC__ "PCDomainDecomp_View"
static int PCDomainDecomp_View(PC pc,Viewer viewer)
{
  PC_DDecomp_struct *pc_data = (PC_DDecomp_struct *) pc->data;
  int ierr;

  PetscPrintf(pc->comm,">> Scatter to extract interface system:\n");
  ierr = VecScatterView(pc_data->get_intl,viewer); CHKERRQ(ierr);
  PetscPrintf(pc->comm,">> Interface method\n");
  ierr = SLESView(pc_data->interface_method,viewer); CHKERRQ(ierr);
  ierr = PCSubdomainsView(pc,viewer); CHKERRQ(ierr);

  return 0;
}
#undef __FUNC__
#define __FUNC__ "PCSetFromOptions_DomainDecomp"
static int PCSetFromOptions_DomainDecomp(PC pc)
{
  PC_DDecomp_struct *data = (PC_DDecomp_struct *) pc->data;
  SLES sles = data->interface_method;
  char *prefix; int ierr;

  /* set the interface method from options */
  ierr = PCGetOptionsPrefix(pc,&prefix); CHKERRQ(ierr);
  ierr = SLESSetOptionsPrefix(sles,prefix); CHKERRQ(ierr);
  ierr = SLESAppendOptionsPrefix(sles,"interface_"); CHKERRQ(ierr);
  ierr = SLESSetFromOptions(sles); CHKERRQ(ierr);

  /* set the subdomain method from options */
  ierr = PCParallelLocalSolveSetFromOptions(pc); CHKERRQ(ierr);

  return 0;
}

#undef __FUNC__
#define __FUNC__ "PCDestroy_DDecomp"
static int PCDestroy_DDecomp(PC pc)
{
  SLES local_sles;
  PC_DDecomp_struct *pc_data = (PC_DDecomp_struct *) pc->data;
  int ierr;

  ierr = PCParallelGetLocalSLES(pc,&local_sles); CHKERRQ(ierr);
  ierr = SLESDestroy(local_sles); CHKERRQ(ierr);
  ierr = SLESDestroy(pc_data->interface_method); CHKERRQ(ierr);

  ierr = MatDestroy(pc_data->C11); CHKERRQ(ierr);
  ierr = MatDestroy(pc_data->C12); CHKERRQ(ierr);
  ierr = MatDestroy(pc_data->C21); CHKERRQ(ierr);
  /*
  ierr = MatDestroy(pc_data->C12_big); CHKERRQ(ierr);
  ierr = MatDestroy(pc_data->C21_big); CHKERRQ(ierr);
  */
  ierr = MatDestroy(pc_data->interface_system); CHKERRQ(ierr);

  ierr = VecDestroy(pc_data->mpi_edge_vec); CHKERRQ(ierr);
  ierr = VecDestroy(pc_data->local_edge_vec); CHKERRQ(ierr);
  ierr = VecDestroy(pc_data->mpi_edge_vec2); CHKERRQ(ierr);
  ierr = VecDestroy(pc_data->local_edge_vec2); CHKERRQ(ierr);

  ierr = VecDestroy(pc_data->mpi_intl_vec); CHKERRQ(ierr);
  ierr = VecDestroy(pc_data->local_intl_vec); CHKERRQ(ierr);
  ierr = VecDestroy(pc_data->mpi_intl_vec2); CHKERRQ(ierr);
  ierr = VecDestroy(pc_data->local_intl_vec2); CHKERRQ(ierr);

  ierr = VecDestroy(pc_data->interface_vector); CHKERRQ(ierr);

  ierr = VecScatterDestroy(pc_data->get_edge); CHKERRQ(ierr);
  ierr = VecScatterDestroy(pc_data->put_edge); CHKERRQ(ierr);
  ierr = VecScatterDestroy(pc_data->get_intl); CHKERRQ(ierr);
  ierr = VecScatterDestroy(pc_data->put_intl); CHKERRQ(ierr);

  return 0;
}

#undef __FUNC__
#define __FUNC__ "PCCreate_DomainDecomp"
int PCCreate_DomainDecomp(PC pc)
{
  int ierr;

  pc->apply     = PCApply_DomainDecomp;
  pc->applyrich = 0;
  pc->destroy   = PCDestroy_DDecomp;
  pc->setfromoptions   = PCSetFromOptions_DomainDecomp;
  pc->printhelp = 0;
  pc->setup     = PCSetup_DomainDecomp;
  /*  pc->type      = PCDomainDecomp;*/
  pc->view      = PCDomainDecomp_View;

  /* create subsolvers for the interface system and interior */
  ierr = PCParallelSubdomainsCreate(pc,sizeof(PC_DDecomp_struct));
  CHKERRQ(ierr);
  {
    PC_DDecomp_struct *bij = (PC_DDecomp_struct *) pc->data;
    SLES sles; KSP subksp; PC subpc;

    bij->C11 = bij->C12 = bij->C21 = 0;
    /* bij->C12_big = bij->C21_big = 0;*/
    bij->get_edge = bij->put_edge = bij->get_intl = bij->put_intl = 0;

    ierr = SLESCreate(pc->comm,&sles); CHKERRQ(ierr);
    ierr = SLESGetKSP(sles,&subksp); CHKERRQ(ierr);
    ierr = KSPSetType(subksp,KSPPREONLY); CHKERRQ(ierr);
    ierr = SLESGetPC(sles,&subpc); CHKERRQ(ierr);
    ierr = PCSetType(subpc,PCNONE); CHKERRQ(ierr);
    bij->interface_method = sles;
  }

  return 0;
}

