#define TRACING 0
#define RUGE 0
/*
   Defines a parallel multilevel preconditioner 
*/
#include "mpi.h"
#include "vec.h"
#include "is.h"
#include "mat.h"
#include "src/mat/matimpl.h"
#include "pc.h"
#include "src/pc/pcimpl.h"
#include "src/vec/impls/mpi/pvecimpl.h"
#include "src/sles/slesimpl.h"
#include "options.h"

#include "src/pc/pcextra.h"
#include "parpre_pc.h"
#include "parpre_pipeline.h"
#include "parpre_subdomains.h"
#include "parpre_is.h"
#include "./ml_impl.h"
#include "./ml_head.h"

#define CHUNCKSIZE   100
#define MAXLEVELS 100
#define BOTTOM_LEVEL 0

extern int MatMultAXBY_AIJ
(Scalar a, Mat aijin,Vec xx, Scalar b, Vec yy,Vec zz);
extern int MatMatMult_MPIAIJ(Mat A,Mat B,Mat *C);
extern int MatDiagonalScale_MPIAIJ(Mat A,Vec ll,Vec rr);
extern int MatMatSubtract_AIJ(Mat a, Mat b, Mat *c);
extern int MatMaxRowLen_MPIAIJ(Mat A, int *rowlen);
extern int MatMaxRowOffDiagElement_MPIAIJ(Mat A,Vec e);
extern int MatMaxColOffDiagElement_MPIAIJ(Mat A,Vec e);

/****************************************************************
 * Inits and such
 ****************************************************************/

#undef __FUNC__
#define __FUNC__ "InitOneLevel"
static int InitOneLevel
(MC_OneLevel_struct *this_level,int lvl,Mat mat,IS idx,
#if RUGE
 int old_isize,int old_jsize,
#endif
 AMLSolveScheme solve_scheme,AMLCoarseGridChoice grid_choice,
 AMLFillMethod fill_method,PCType local_pctype)
{
  int ierr;

  this_level->level = lvl; this_level->indices = idx;
  this_level->mat = mat; this_level->comm = mat->comm;

  ierr = MatGetOwnershipRange
    (mat,&(this_level->rstart),&(this_level->rend)); CHKERRQ(ierr);
  this_level->local_size = this_level->rend-this_level->rstart;
  MPI_Allreduce
    ((void*)&this_level->local_size,(void*)&this_level->global_size,
     1,MPI_INT,MPI_SUM,this_level->comm);
#if RUGE
  {
    int isize,jsize;
    if (lvl==0) {
      isize = (int)sqrt(this_level->global_size);
      jsize = this_level->global_size/isize;
    } else {
      if (lvl-2*(lvl/2)==0) {
	isize = old_isize/2; jsize = old_jsize;
      } else {
	isize = old_isize; jsize = old_jsize/2;
      }
    }
    if (isize*jsize!=this_level->global_size) SETERRQ(1,0,"Grid not divisible");
printf("Grid is %dx%d\n",isize,jsize);
    this_level->isize = isize; this_level->jsize = jsize;
  }
#endif

  this_level->pre_smooth = this_level->post_smooth = 0;
  this_level->solve_scheme = solve_scheme;
  this_level->grid_choice = grid_choice;
  this_level->fill_method = fill_method;
  this_level->local_pctype = local_pctype;
  {
    int s,ss;
    ierr = MatGetLocalSize(mat,&s,&ss); CHKERRQ(ierr);
    ierr = VecCreateMPI
      (mat->comm,s,PETSC_DECIDE,&(this_level->u)); CHKERRQ(ierr);
    ierr = VecDuplicate(this_level->u,&(this_level->v)); CHKERRQ(ierr);
    ierr = VecDuplicate(this_level->u,&(this_level->w)); CHKERRQ(ierr);
  }
/* !!! */
  ierr = VecDuplicate(this_level->u,&(this_level->diag)); CHKERRQ(ierr);

  return 0;
}

#if TRACING
#undef __FUNC__
#define __FUNC__ "LambdaOneEst"
static int LambdaOneEst(Mat mat,MC_OneLevel_struct *this_level)
{
  Scalar one=1.0, xax,xx;
  ierr = VecSet(&one,this_level->u); CHKERRQ(ierr);
  ierr = MatMult(mat,this_level->u,this_level->v); CHKERRQ(ierr);
  ierr = VecPointwiseMult(this_level->u,this_level->v,this_level->w);
  CHKERRQ(ierr);
  ierr = VecPointwiseMult(this_level->u,this_level->u,this_level->v);
  CHKERRQ(ierr);
  ierr = VecSum(this_level->w,&xax); CHKERRQ(ierr);
  ierr = VecSum(this_level->v,&xx); CHKERRQ(ierr);
  printf("lambda_1 approx: %e\n",xax/xx);
}
#endif

/****************************************************************
 * Sub solvers
 ****************************************************************/
#undef __FUNC__
#define __FUNC__ "SetSmoothers"
static int SetSmoothers
(AMLSmootherChoice smoother_choice,MC_OneLevel_struct *this_level,
 SLES pre,SLES post,char *prefix)
{
  MPI_Comm comm = this_level->comm;
  int ierr;

/*
  ierr = PCCreate(comm,&(this_level->global_solve)); CHKERRQ(ierr);
  ierr = PCSetOperators(this_level->global_solve,
			this_level->mat,this_level->mat,0); CHKERRQ(ierr);
  ierr = PCSetType(this_level->global_solve,PCBJACOBI); CHKERRQ(ierr);
  ierr = PCSetVector(this_level->global_solve,this_level->u); CHKERRQ(ierr);
  ierr = PCSetUp(this_level->global_solve); CHKERRQ(ierr);
*/

  /* == create the smoothers == */
  if (smoother_choice) {

    ierr = ParPreGenerateSLES
      (comm,&(this_level->pre_smooth)); CHKERRQ(ierr);
    ierr = SLESSetOptionsPrefix
      (this_level->pre_smooth,prefix); CHKERRQ(ierr);
    ierr = SLESAppendOptionsPrefix
      (this_level->pre_smooth,"presmoother_"); CHKERRQ(ierr);
    ierr = SLESSetFromOptions(this_level->pre_smooth); CHKERRQ(ierr);

    ierr = ParPreGenerateSLES
      (comm,&(this_level->post_smooth)); CHKERRQ(ierr);
    ierr = SLESSetOptionsPrefix
      (this_level->post_smooth,prefix); CHKERRQ(ierr);
    ierr = SLESAppendOptionsPrefix
      (this_level->post_smooth,"postsmoother_"); CHKERRQ(ierr);
    ierr = SLESSetFromOptions(this_level->post_smooth); CHKERRQ(ierr);

    if (smoother_choice && AMLPreSmooth) {
      ierr = SLESSetOperators
	(this_level->pre_smooth,this_level->mat,this_level->mat,0);
      CHKERRQ(ierr);
    }
    if (smoother_choice && AMLPostSmooth) {
      ierr = SLESSetOperators
	(this_level->post_smooth,this_level->mat,this_level->mat,0);
      CHKERRQ(ierr);
    }
  }

  return 0;
}

#undef __FUNC__
#define __FUNC__ "SetupLastLevel"
static int SetupLastLevel
(MC_OneLevel_struct *this_level,Mat last_mat,char *prefix,SLES *last_solve)
{
  SLES last;
  PC local_pc; int ierr;

  ierr = ParPreGenerateSLES(this_level->comm,&last); CHKERRQ(ierr);
  ierr = SLESSetOptionsPrefix(last,prefix); CHKERRQ(ierr);
  ierr = SLESAppendOptionsPrefix(last,"lastlevel_"); CHKERRQ(ierr);
  ierr = SLESSetFromOptions(last); CHKERRQ(ierr);

  ierr = SLESSetOperators(last,last_mat,last_mat,0); CHKERRQ(ierr);
  ierr = SLESGetPC(last,&local_pc); CHKERRQ(ierr);
  ierr = PCSetVector(local_pc,this_level->u2); CHKERRQ(ierr);
  ierr = SLESSetUp(last,this_level->u2,this_level->v2); CHKERRQ(ierr);

  *last_solve = last;

  return 0;
}

/****************************************************************
 * Submatrix handling                                           *
 ****************************************************************/
/* decide whether we are going to have more levels */
#undef __FUNC__
#define __FUNC__ "ThisLevelContinueDecide"
static int ThisLevelContinueDecide
(MC_OneLevel_struct *this_level,int *more_levels,int cutoff)
{
  int local_next = this_level->size2;
  int local_test = local_next, more_matrix;

  if ((cutoff>0) && (local_test<cutoff)) local_test = 0;
  MPI_Allreduce((void *)&local_test,(void *)more_levels,
		1,MPI_INT,MPI_SUM,this_level->comm);
  /*printf("local test=%d, global test=%d; cutoff=%d\n",local_test,*more_levels,cutoff);*/
  MPI_Allreduce((void *)&local_next,(void *)&more_matrix,
		1,MPI_INT,MPI_SUM,this_level->comm);
  if (*more_levels<more_matrix) more_levels = 0;
  if (!more_matrix)
    SETERRQ(1,0,"Should have been caught: no 2-block in ExtractBlocks");

  return 0;
}


#undef __FUNC__
#define __FUNC__ "ReleaseNextLevel"
static int ReleaseNextLevel(MC_OneLevel_struct *this_level)
{
  /*
  int ierr;

  ierr = VecDestroy(this_level->u); CHKERRQ(ierr);
  ierr = VecDestroy(this_level->v); CHKERRQ(ierr);
  ierr = VecDestroy(this_level->w); CHKERRQ(ierr);
  ierr = VecDestroy(this_level->diag); CHKERRQ(ierr);
  ierr = SLESDestroy(this_level->a11_solve); CHKERRQ(ierr);

  PetscFree(this_level->next_level);
  */
  this_level->next_level = 0;

  return 0;
}

#undef __FUNC__
#define __FUNC__ "SetupOneLevel"
static int SetupOneLevel
(Mat mat,IS idx,PCType local_pctype,int lvl,int make_strong,
#if RUGE
 int isize,int jsize,
#endif
 char *prefix,
 AMLFillMethod fill_method,AMLSolveScheme solve_scheme,
 AMLCoarseGridChoice grid_choice,AMLSmootherChoice smoothers,
 SLES pre,SLES post,
 int cutoff,int it11,
 MC_OneLevel_struct **return_level,int *early_return)
{
  MC_OneLevel_struct *this_level;
  IS set2g;
  int ierr;

  /* initial setup of the level */
  this_level = (MC_OneLevel_struct *)
    PetscMalloc(sizeof(MC_OneLevel_struct)); CHKPTRQ(this_level);

  printf("Colour %d\n",lvl);
  ierr = InitOneLevel(this_level,lvl,mat,idx,
#if RUGE
		      isize,jsize,
#endif
		      solve_scheme,grid_choice,fill_method,local_pctype);
#if TRACING
  ierr = LambdaOneEst(mat,this_level); CHKERRQ(ierr);
#endif
  ierr = MatGetDiagonal(this_level->mat,this_level->diag); CHKERRQ(ierr);
  ierr = SetSmoothers(smoothers,this_level,pre,post,prefix); CHKERRQ(ierr);

  {
    int local_rest;
    ierr = PartitionMatrix(this_level,make_strong,&set2g,
			   early_return,&local_rest);
    CHKERRQ(ierr);
    if (local_rest==0) {
      if (lvl == BOTTOM_LEVEL)
	SETERRQ(1,0,"One processor SOL. Proceed with fingers crossed\n");
      /* *early_return =1 ; return 0;*/
    }
  }
  ierr = Set11Solver(this_level,&(this_level->a11_solve)); CHKERRQ(ierr);
  if (*early_return) {
    return 0;
  }
  if (this_level->fill_method == AMLFillNone) {
    this_level->s2 = this_level->g22;
  } else {
    ierr = CreateNextSystem(this_level,&(this_level->s2)); CHKERRQ(ierr);
    /*MatView(this_level->s2,0);*/
  }
  {
    int more_levels;

    /* if there is enough matrix left, go to the next level */
    ierr = ThisLevelContinueDecide(this_level,&more_levels,cutoff);
    CHKERRQ(ierr);
    if (!more_levels) {
      printf("Last level too sparse.\n");
      this_level->next_level = 0;
      ierr = SetupLastLevel
	(this_level,this_level->g22,prefix,&(this_level->a22_solve));
      CHKERRQ(ierr);
    } else {
      int early_ret;
      ierr = SetupOneLevel
	(this_level->s2,set2g,local_pctype,lvl+1,make_strong,
#if RUGE
	 this_level->isize,this_level->jsize,
#endif
	 prefix,
	 fill_method,solve_scheme,grid_choice,smoothers,
	 pre,post,cutoff,it11,
	 &(this_level->next_level),&early_ret); CHKERRQ(ierr);
      if (early_ret) {
	ierr = ReleaseNextLevel(this_level); CHKERRQ(ierr);
	ierr = SetupLastLevel(this_level,this_level->s2,prefix,
			      &(this_level->a22_solve));
	CHKERRQ(ierr);
      }
    }
  }
  *return_level = this_level;
  return 0;
}

#undef __FUNC__
#define __FUNC__ "PCSetFromOptions_MLevel"
int PCSetFromOptions_MLevel(PC pc)
{
  char method[4]; char scheme[2];
  int ierr,flg,eq;

  ierr = OptionsGetString
    (pc->prefix,"-multilevel_solutionscheme",scheme,2,&flg);
  CHKERRQ(ierr);
  if (flg) {
    eq = strcmp(scheme,"mg");
    if (eq) {
      ierr = AMLSetSolutionScheme(pc,AMLSolveMG);
      CHKERRQ(ierr);
    } else {
      eq = strcmp(scheme,"lu");
      if (eq) {
	ierr = AMLSetSolutionScheme(pc,AMLSolveILU);
	CHKERRQ(ierr);
      } else SETERRQ
	       (1,0,"Multilevel set from options: unknown scheme \n");
    }}
  ierr = OptionsGetString
    (pc->prefix,"-multilevel_fillmethod",method,4,&flg);
  CHKERRQ(ierr);
  if (flg) {
    eq = strcmp(method,"none");
    if (eq) {
      ierr = AMLSetFillMethod(pc,AMLFillNone);
      CHKERRQ(ierr); 
    } else {
      eq = strcmp(method,"diag");
      if (eq) {
	ierr = AMLSetFillMethod(pc,AMLFillDiag);
	CHKERRQ(ierr); 
      } else {
	eq = strcmp(method,"stro");
	if (eq) {
	  ierr = AMLSetFillMethod(pc,AMLFillStrong);
	  CHKERRQ(ierr); 
	} else {
	  eq = strcmp(method,"full");
	  if (eq) {
	    ierr = AMLSetFillMethod(pc,AMLFillFull);
	    CHKERRQ(ierr);
	  } else SETERRQ
		   (1,0,"Multilevel set from options: unknown method \n");
	}}}}

  return 0;
}

#undef __FUNC__
#define __FUNC__ "PCSetup_MLevel"
int PCSetup_MLevel(PC pc)
{
  Mat base_mat = pc->mat;
  PC local_pc; PCType local_pctype;
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
  IS idx_set;
  char *prefix;
  int idum,ierr;

  {
    int rstart,rend,loc;
    ierr = MatGetOwnershipRange(base_mat,&rstart,&rend); CHKERRQ(ierr);
    loc = rend-rstart;
    ierr = ISCreateStride(base_mat->comm,loc,rstart,1,&idx_set);
    CHKERRQ(ierr);
  }
  ierr = PCParallelGetLocalPC(pc,&local_pc); CHKERRQ(ierr);
  ierr = PCGetType(local_pc,&local_pctype,PETSC_NULL); CHKERRQ(ierr);
  ierr = PCGetOptionsPrefix(pc,&prefix); CHKERRQ(ierr);

  ierr = SetupOneLevel
    (base_mat,idx_set,local_pctype,BOTTOM_LEVEL,pc_data->mis_from_strong,
#if RUGE
     0,0,
#endif
     prefix,
     pc_data->fill_method,pc_data->solve_scheme,pc_data->grid_choice,
     pc_data->smoother_choice,
     pc_data->pre_smoother,pc_data->post_smoother,
     pc_data->cutoff,pc_data->it11,
     &(pc_data->level_stuff),&idum); CHKERRQ(ierr);

  return 0;
}

/****************************************************************
 * Solve                                                        *
 ****************************************************************/

#undef __FUNC__
#define __FUNC__ "PCApply_MLevel"
static int PCApply_MLevel(PC pc,Vec x,Vec y)
{
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
  int ierr;

/*printf("input vector\n"); VecView(x,0);*/
  ierr = SolveMultiLevel(pc_data->level_stuff,x,y); CHKERRQ(ierr);
/*printf("output vector\n"); VecView(y,0);*/

  return 0;
}

#undef __FUNC__
#define __FUNC__ "PCDestroy_MLevel"
int PCDestroy_MLevel(PetscObject obj)
{
/*
  PC pc = (PC) obj;
  PC_MCol_struct *pc_data = (PC_MCol_struct *) pc->data;
*/
  return 0;

}

#undef __FUNC__
#define __FUNC__ "PCDuplicate_MLevel"
int PCDuplicate_MLevel(PC old,PC *new)
{
  *new = old;
  return 0;
}

#undef __FUNC__
#define __FUNC__ "PCCreate_MLevel"
int PCCreate_MLevel(PC pc)
{
  PC_MCol_struct *bij;
  int ierr;

  pc->apply     = PCApply_MLevel;
  pc->applyrich = 0;
  pc->destroy   = PCDestroy_MLevel;
  pc->setfrom   = PCSetFromOptions_MLevel;
  pc->printhelp = 0;
  pc->setup     = PCSetup_MLevel;
  pc->type      = PCMultiLevel;

  /* create a local solve, equivalent to smoother */
  ierr = PCParallelSubdomainsCreate(pc,sizeof(PC_MCol_struct));

  bij = (PC_MCol_struct *) pc->data;
  bij->fill_method = AMLFillNone;
  bij->solve_scheme = AMLSolveILU;
  bij->grid_choice = AMLCoarseGridDependent;
  bij->smoother_choice = AMLSmoothNone;
  bij->cutoff = 10; bij->it11 = 1;
  bij->mis_from_strong = 0;

  /* create a subsolver for the 11 blocks, the pre/post smoother */
  ierr = ParPreGenerateSLES(pc->comm,&(bij->pre_smoother)); CHKERRQ(ierr);
  ierr = ParPreGenerateSLES(pc->comm,&(bij->post_smoother)); CHKERRQ(ierr);

  /*
  {
    SLES local_method;
    ierr = ParPreGenerateSLES(pc->comm,&local_method); CHKERRQ(ierr);
    ierr = PCParallelSetLocalSLES(pc,local_method); CHKERRQ(ierr);
  }
  */
  {
    PC local_pc;
    ierr = PCParallelGetLocalPC(pc,&local_pc); CHKERRQ(ierr);
    ierr = PCSetType(local_pc,PCJACOBI); CHKERRQ(ierr);
  }

  return 0;
}
