/*---------------------------------------------------------------*/
/*     	CAPSS: A Cartesian Parallel Sparse Solver                */
/*     	Beta Release                                             */
/*      Author: Padma Raghavan                                   */
/*---------------------------------------------------------------*/

/* Quicksort: computes indices (in pbase) so access by this index results in 
 * access  in ascending order of  values in  key_list.
 * This implementation incorporates * four optimizations discussed in Sedgewick:
 * hairy, ugly but sort takes time ....
 *
 * 1. Non-recursive, using an explicit stack of pointer that store the 
 *    next array partition to sort.  To save time, this maximum amount 
 *    of space required to store an array of MAX_INT is allocated on the 
 *    stack.  Assuming a 32-bit integer, this needs only 32 * 
 *    sizeof(stack_node) == 136 bits.  Pretty cheap, actually.
 *
 * 2. Chose the pivot element using a median-of-three decision tree.
 *    This reduces the probability of selecting a bad pivot value and 
 *    eliminates certain extraneous comparisons.
 *
 * 3. Only quicksorts TOTAL_ELEMS / MAX_THRESH partitions, leaving
 *    insertion sort to order the MAX_THRESH items within each partition.  
 *    This is a big win, since insertion sort is faster for small, mostly
 *    sorted array segements.
 *
 * 4. The larger of the two sub-partitions is always pushed onto the
 *    stack first, with the algorithm then concentrating on the
 *    smaller partition.  This *guarantees* no more than log (n)
 *    stack size is needed (actually O(1) in this case)!
 */

typedef int mytype;
typedef int myindex;

/* Discontinue quicksort algorithm when partition gets below this size.
   This particular magic number was chosen to work best on a Sun 4/260. */
#define MAX_THRESH 4

/* Byte-wise swap two items. */
#define SWAP(a, b)						\
  do {								\
    register mytype tmp;					\
    tmp = a;							\
    a = b;							\
    b = tmp;						  	\
  } while (0)

/* Stack node declarations used to store unfulfilled partition obligations. */
typedef struct 
  {
    myindex *lo;
    myindex *hi;
  } stack_node;

/* The next 4 #defines implement a very fast in-line stack abstraction. */
#define STACK_SIZE	(8 * sizeof(long))
#define PUSH(low, high)	((void) ((top->lo = (low)), (top->hi = (high)), ++top))
#define	POP(low, high)	((void) (--top, (low = top->lo), (high = top->hi)))
#define	STACK_NOT_EMPTY	(stack < top)                


quick_sort(ntotal,pbase, key_list)
mytype pbase[], key_list[];
myindex ntotal;
{
  register mytype *base_ptr = (mytype *) pbase;
  mytype  mytmp, *mytmp2;


  /* Allocating SIZE bytes for a pivot buffer facilitates a better
     algorithm below since we can do comparisons directly on the pivot. */
  mytype pivot;
  myindex max_thresh = MAX_THRESH * sizeof(mytype);
  myindex total_elems = ntotal;
 
  for(mytmp=0, mytmp2= pbase; mytmp <ntotal; mytmp++, mytmp2++)
	*mytmp2 = mytmp;

  if (total_elems == 0)
    /* Avoid lossage with unsigned arithmetic below.  */
    return;

  if (total_elems > MAX_THRESH) {
      mytype *lo = base_ptr;
      mytype *hi = &lo[total_elems - 1];
      /* Largest size needed for 32-bit int!!! */
      stack_node stack[STACK_SIZE];
      stack_node *top = stack + 1;

      while (STACK_NOT_EMPTY) {
          mytype *left_ptr;
          mytype *right_ptr;

	  /* Select median value from among LO, MID, and HI. Rearrange
	     LO and HI so the three values are sorted. This lowers the 
	     probability of picking a pathological pivot value and 
	     skips a comparison for both the LEFT_PTR and RIGHT_PTR. */

	  mytype *mid = lo + ((hi - lo) >> 1);

#define CMP <
	  if (key_list[*mid] CMP key_list[*lo]) {
	    SWAP(*mid, *lo);
          }
	  if (key_list[*hi] CMP key_list[*mid]) {
	    SWAP(*mid, *hi);
          }
	  else {
	    goto jump_over;
          }
	  if (key_list[*mid] CMP key_list[*lo]) {
	    SWAP(*mid, *lo);
          }
	jump_over:;
          pivot = *mid;

	  left_ptr  = lo + 1;
	  right_ptr = hi - 1; 

	  /* Here's the famous ``collapse the walls'' section of quicksort.  
	     Gotta like those tight inner loops!  They are the main reason 
	     that this algorithm runs much faster than others. */
	  do {
	      while (key_list[*left_ptr] CMP key_list[pivot])
		left_ptr++;

	      while( key_list[pivot] CMP key_list[*right_ptr])
		right_ptr--;

	      if (left_ptr < right_ptr) {
		  SWAP(*left_ptr, *right_ptr);
		  left_ptr++;
		  right_ptr--;
	      }
	      else if (left_ptr == right_ptr) {
		  left_ptr++;
		  right_ptr--;
		  break;
	      }
	  } 
	  while (left_ptr <= right_ptr);

          /* Set up pointers for next iteration.  First determine whether
             left and right partitions are below the threshold size.  If so, 
             ignore one or both.  Otherwise, push the larger partition's
             bounds on the stack and continue sorting the smaller one. */

          if ((right_ptr - lo) <= max_thresh) {
              if ((hi - left_ptr) <= max_thresh)
		/* Ignore both small partitions. */
                POP(lo, hi); 
              else
		/* Ignore small left partition. */  
                lo = left_ptr;
          }
          else if ((hi - left_ptr) <= max_thresh)
	    /* Ignore small right partition. */
            hi = right_ptr;
          else if ((right_ptr - lo) > (hi - left_ptr)) {                   
	      /* Push larger left partition indices. */
              PUSH(lo, right_ptr);
              lo = left_ptr;
          }
          else {                   
	      /* Push larger right partition indices. */
              PUSH(left_ptr, hi);
              hi = right_ptr;
          }
        }
    }

  /* Once the BASE_PTR array is partially sorted by quicksort the rest
     is completely sorted using insertion sort, since this is efficient 
     for partitions below MAX_THRESH size. BASE_PTR points to the beginning 
     of the array to sort, and END_PTR points at the very last element in
     the array (*not* one beyond it!). */

#define min(x, y) ((x) < (y) ? (x) : (y))

  {
    mytype *end_ptr = &base_ptr[total_elems - 1];
    mytype *tmp_ptr = base_ptr;
    mytype *thresh = min(end_ptr, base_ptr + max_thresh);
    register mytype *run_ptr;

    /* Find smallest element in first threshold and place it at the
     * array's beginning.  This is the smallest array element,
     * and the operation speeds up insertion sort's inner loop.
     */

    for (run_ptr = tmp_ptr + 1; run_ptr <= thresh; run_ptr++)
      if (key_list[*run_ptr] CMP key_list[*tmp_ptr])
        tmp_ptr = run_ptr;

    if (tmp_ptr != base_ptr)
      SWAP(*tmp_ptr, *base_ptr);

    /* Insertion sort, running from left-hand-side up to right-hand-side.  */

    run_ptr = base_ptr + 1;
    while ((++run_ptr) <= end_ptr)
      {
	tmp_ptr = run_ptr - 1;
	while (key_list[*run_ptr] CMP key_list[*tmp_ptr])
	  tmp_ptr--;

	tmp_ptr++;
        if (tmp_ptr != run_ptr)
          {
            mytype *trav;

	    trav = run_ptr + 1;
	    while (--trav >= run_ptr)
              {
                mytype c = *trav;
                mytype *hi, *lo;

                for (hi = lo = trav; (--lo) >= tmp_ptr; hi = lo)
                  *hi = *lo;
                *hi = c;
              }
          }
      }
  }
}

/*quick sort of fieldds of size two*/

quick_sort2(ntotal,pbase, key_list)
mytype pbase[], key_list[];
myindex ntotal;
{
  register mytype *base_ptr = (mytype *) pbase;
  mytype  mytmp, *mytmp2;


  /* Allocating SIZE bytes for a pivot buffer facilitates a better
     algorithm below since we can do comparisons directly on the pivot. */
  mytype pivot;
  myindex max_thresh = MAX_THRESH * sizeof(mytype);
  myindex total_elems = ntotal;
 
  for(mytmp=0, mytmp2= pbase; mytmp <ntotal; mytmp++, mytmp2++)
	*mytmp2 = mytmp;

  if (total_elems == 0)
    /* Avoid lossage with unsigned arithmetic below.  */
    return;

  if (total_elems > MAX_THRESH) {
      mytype *lo = base_ptr;
      mytype *hi = &lo[total_elems - 1];
      /* Largest size needed for 32-bit int!!! */
      stack_node stack[STACK_SIZE];
      stack_node *top = stack + 1;

      while (STACK_NOT_EMPTY) {
          mytype *left_ptr;
          mytype *right_ptr;

	  /* Select median value from among LO, MID, and HI. Rearrange
	     LO and HI so the three values are sorted. This lowers the 
	     probability of picking a pathological pivot value and 
	     skips a comparison for both the LEFT_PTR and RIGHT_PTR. */

	  mytype *mid = lo + ((hi - lo) >> 1);

#define CMP <
	  if ((key_list[*mid+*mid] CMP key_list[*lo+*lo]) ||
			( (key_list[*mid+*mid] == key_list[*lo+*lo]) &&
			(key_list[*mid+*mid+1] CMP key_list[*lo+*lo+1]))) {
	    SWAP(*mid, *lo);
          }
	  if ((key_list[*hi+*hi] CMP key_list[*mid+*mid]) ||
	  	((key_list[*hi+*hi] == key_list[*mid+*mid]) &&
	  	(key_list[*hi+*hi+1] CMP key_list[*mid+*mid]+1))){
	    SWAP(*mid, *hi);
          } 
	  else {
	    goto jump_over;
          }
	  if ( (key_list[*mid+*mid] CMP key_list[*lo+*lo]) ||
		((key_list[*mid+*mid] == key_list[*lo+*lo]) &&
		(key_list[*mid+*mid+1] CMP key_list[*lo+*lo]))) {
	    SWAP(*mid, *lo);
          }
	jump_over:;
          pivot = *mid;

	  left_ptr  = lo + 1;
	  right_ptr = hi - 1; 

	  /* Here's the famous ``collapse the walls'' section of quicksort.  
	     Gotta like those tight inner loops!  They are the main reason 
	     that this algorithm runs much faster than others. */
	  do {
	      while ((key_list[*left_ptr+*left_ptr] CMP key_list[pivot+pivot])
			||
		((key_list[*left_ptr+*left_ptr] == key_list[pivot+pivot]) &&
		(key_list[*left_ptr+*left_ptr+1] CMP key_list[pivot+pivot+1])))
		left_ptr++;

	      while(( key_list[pivot+pivot] CMP key_list[*right_ptr+*right_ptr])
			|| 
		((key_list[pivot+pivot]== key_list[*right_ptr+*right_ptr]) &&
	       (key_list[pivot+pivot+1] CMP key_list[*right_ptr+*right_ptr+1])))
		right_ptr--;

	      if (left_ptr < right_ptr) {
		  SWAP(*left_ptr, *right_ptr);
		  left_ptr++;
		  right_ptr--;
	      }
	      else if (left_ptr == right_ptr) {
		  left_ptr++;
		  right_ptr--;
		  break;
	      }
	  } 
	  while (left_ptr <= right_ptr);

          /* Set up pointers for next iteration.  First determine whether
             left and right partitions are below the threshold size.  If so, 
             ignore one or both.  Otherwise, push the larger partition's
             bounds on the stack and continue sorting the smaller one. */

          if ((right_ptr - lo) <= max_thresh) {
              if ((hi - left_ptr) <= max_thresh)
		/* Ignore both small partitions. */
                POP(lo, hi); 
              else
		/* Ignore small left partition. */  
                lo = left_ptr;
          }
          else if ((hi - left_ptr) <= max_thresh)
	    /* Ignore small right partition. */
            hi = right_ptr;
          else if ((right_ptr - lo) > (hi - left_ptr)) {                   
	      /* Push larger left partition indices. */
              PUSH(lo, right_ptr);
              lo = left_ptr;
          }
          else {                   
	      /* Push larger right partition indices. */
              PUSH(left_ptr, hi);
              hi = right_ptr;
          }
        }
    }

  /* Once the BASE_PTR array is partially sorted by quicksort the rest
     is completely sorted using insertion sort, since this is efficient 
     for partitions below MAX_THRESH size. BASE_PTR points to the beginning 
     of the array to sort, and END_PTR points at the very last element in
     the array (*not* one beyond it!). */

#define min(x, y) ((x) < (y) ? (x) : (y))

  {
    mytype *end_ptr = &base_ptr[total_elems - 1];
    mytype *tmp_ptr = base_ptr;
    mytype *thresh = min(end_ptr, base_ptr + max_thresh);
    register mytype *run_ptr;

    /* Find smallest element in first threshold and place it at the
     * array's beginning.  This is the smallest array element,
     * and the operation speeds up insertion sort's inner loop.
     */

    for (run_ptr = tmp_ptr + 1; run_ptr <= thresh; run_ptr++)
      if ((key_list[*run_ptr+*run_ptr] CMP key_list[*tmp_ptr +*tmp_ptr])
	 ||
         ((key_list[*run_ptr+*run_ptr] == key_list[*tmp_ptr +*tmp_ptr]) &&
         (key_list[*run_ptr+*run_ptr+1] CMP key_list[*tmp_ptr +*tmp_ptr+1])))
        tmp_ptr = run_ptr;

    if (tmp_ptr != base_ptr)
      SWAP(*tmp_ptr, *base_ptr);

    /* Insertion sort, running from left-hand-side up to right-hand-side.  */

    run_ptr = base_ptr + 1;
    while ((++run_ptr) <= end_ptr)
      {
	tmp_ptr = run_ptr - 1;
	while ((key_list[*run_ptr+*run_ptr] CMP key_list[*tmp_ptr+*tmp_ptr])||
	      ((key_list[*run_ptr+*run_ptr] == key_list[*tmp_ptr+*tmp_ptr]) &&
	    (key_list[*run_ptr+*run_ptr+1] CMP key_list[*tmp_ptr+*tmp_ptr+1]))) 
	  tmp_ptr--;

	tmp_ptr++;
        if (tmp_ptr != run_ptr)
          {
            mytype *trav;

	    trav = run_ptr + 1;
	    while (--trav >= run_ptr)
              {
                mytype c = *trav;
                mytype *hi, *lo;

                for (hi = lo = trav; (--lo) >= tmp_ptr; hi = lo)
                  *hi = *lo;
                *hi = c;
              }
          }
      }
  }
}
