
static char rcsid[] = 
	"$Id: trace.c,v 3.20 1995/07/27 14:48:40 kohl Exp $";

#include "xpvm.h"

/* Trace Lookup Trie Init Routines */

init_did_trie()
{
	DID dptr;

	int i;

	DID_TRIE = create_triestack();

	for ( i=TEV_DID_FIRST ; i <= TEV_DID_MAX ; i++ )
	{
		if ( DID_LIST == NULL )
			dptr = DID_LIST = create_did();

		else
			dptr = dptr->next = create_did();

		strcpy( dptr->name, pvmtevdidlist[i].did );

		dptr->desc = pvmtevdidlist[i].desc;

		add_to_trie( DID_TRIE, dptr->name, (void *) dptr );
	}
}

init_event_trie()
{
	int i;

	EVENT_TRIE = create_triestack();

	HANDLE_TRIE = create_triestack();

	add_to_trie( HANDLE_TRIE, "newtask",	(void *) HANDLE_NEWTASK );
	add_to_trie( HANDLE_TRIE, "spntask",	(void *) HANDLE_SPNTASK );
	add_to_trie( HANDLE_TRIE, "endtask",	(void *) HANDLE_ENDTASK );

	add_to_trie( HANDLE_TRIE, "host_add",	(void *) HANDLE_HOST_ADD );
	add_to_trie( HANDLE_TRIE, "host_del",	(void *) HANDLE_HOST_DEL );
	add_to_trie( HANDLE_TRIE, "host_sync",	(void *) HANDLE_HOST_SYNC );

	add_to_trie( HANDLE_TRIE, "output",		(void *) HANDLE_OUTPUT );

	add_to_trie( HANDLE_TRIE, "exit",		(void *) HANDLE_EXIT );

	add_to_trie( HANDLE_TRIE, "send",		(void *) HANDLE_SEND );
	add_to_trie( HANDLE_TRIE, "mcast",		(void *) HANDLE_MCAST );
	add_to_trie( HANDLE_TRIE, "psend",		(void *) HANDLE_PSEND );

	add_to_trie( HANDLE_TRIE, "recv",		(void *) HANDLE_RECV );
	add_to_trie( HANDLE_TRIE, "nrecv",		(void *) HANDLE_NRECV );
	add_to_trie( HANDLE_TRIE, "precv",		(void *) HANDLE_PRECV );
	add_to_trie( HANDLE_TRIE, "trecv",		(void *) HANDLE_TRECV );
}

/* Trace File Generation Routines */

TEVDESC add_tevdesc( TD )
TEVDESC TD;
{
	TEVDESC *newlist;

	TEVDESC tdptr;
	TEVDESC last;

	int newsize;
	int index;
	int i;

	/* Verify Descriptor List is Large Enough */

	if ( TD->eid >= TEVDESC_SIZE )
	{
		newsize = 2 * ( TD->eid + 1 );

		newlist = (TEVDESC *) malloc( (unsigned) newsize
			* sizeof( TEVDESC ) );
		memcheck( newlist, "New TEVDESC Pointer List" );

		for ( i=0 ; i < TEVDESC_SIZE ; i++ )
		{
			newlist[i] = TEVDESC_LIST[i];

			TEVDESC_LIST[i] = (TEVDESC) NULL;
		}

		for ( i=TEVDESC_SIZE ; i < newsize ; i++ )
			newlist[i] = (TEVDESC) NULL;

		free( TEVDESC_LIST );

		TEVDESC_LIST = newlist;

		TEVDESC_SIZE = newsize;
	}

	/* Search for Existing Descriptor */

	tdptr = TEVDESC_LIST[ TD->eid ];

	last = (TEVDESC) NULL;

	index = 0;

	while ( tdptr != NULL )
	{
		if ( cmp_tevdesc( TD, tdptr ) )
		{
			free_tevdesc( &TD );

			return( tdptr );
		}

		if ( !strcmp( tdptr->name, TD->name )
			&& tdptr->entry_exit == TD->entry_exit )
		{
			index++;
		}

		last = tdptr;

		tdptr = tdptr->next;
	}

	/* Add New Descriptor */

	if ( last != NULL )
	{
		last->next = TD;

		TD->index = index;
	}

	else
	{
		TEVDESC_LIST[ TD->eid ] = TD;

		TD->index = 0;
	}

	TD->dump = TRUE;

	return( TD );
}

set_tevlist( TT, H, TD )
TEVTASK TT;
HOST H;
TEVDESC TD;
{
	if ( TT != NULL )
	{
		check_tevsize( &(TT->tevlist), &(TT->tevsize), TD->eid );

		TT->tevlist[ ( TD->eid * MAX_TEV ) + TD->entry_exit ] =
			TD->index;
	}

	else if ( H != NULL )
	{
		check_tevsize( &(H->tevlist), &(H->tevsize), TD->eid );

		H->tevlist[ ( TD->eid * MAX_TEV ) + TD->entry_exit ] =
			TD->index;
	}

	else
		printf( "Error: No Task or Host for TEV Index List\n" );
}

check_tevsize( list, size, eid )
int **list;
int *size;
int eid;
{
	int *newlist;
	int newsize;
	int i;

	if ( ( ( eid + 1 ) * MAX_TEV ) - 1 >= *size )
	{
		newsize = 2 * ( eid + 1 ) * MAX_TEV;

		newlist = (int *) malloc( (unsigned) newsize * sizeof(int) );
		memcheck( newlist, "New TEVDESC Index List" );

		for ( i=0 ; i < *size ; i++ )
		{
			newlist[i] = (*list)[i];

			(*list)[i] = -1;
		}

		for ( i=(*size) ; i < newsize ; i++ )
			newlist[i] = -1;

		free( *list );

		*list = newlist;
		*size = newsize;
	}
}

get_tevindex( TT, H, eid, entry_exit )
TEVTASK TT;
HOST H;
int eid;
int entry_exit;
{
	int index;

	if ( TT != NULL )
		index = TT->tevlist[ ( eid * MAX_TEV ) + entry_exit ];

	else if ( H != NULL )
		index = H->tevlist[ ( eid * MAX_TEV ) + entry_exit ];

	else
		index = -1;

	return( index );
}

TEVDESC get_tevdesc( eid, entry_exit, index )
int eid;
int entry_exit;
int index;
{
	TEVDESC tdptr;

	if ( index < 0 )
		return( (TEVDESC) NULL );

	tdptr = TEVDESC_LIST[ eid ];

	while ( tdptr != NULL &&
		( tdptr->entry_exit != entry_exit || tdptr->index != index ) )
	{
		tdptr = tdptr->next;
	}

	return( tdptr );
}

dump_tevdesc( TD )
TEVDESC TD;
{
	DATADESC DD;

	fprintf( TRACE_OUT, "\n#%d: \"%s(", TD->eid, TD->name );

	if ( TD->entry_exit == ENTRY_TEV )
		fprintf( TRACE_OUT, "0" );

	else if ( TD->entry_exit == EXIT_TEV )
		fprintf( TRACE_OUT, "1" );

	if ( TD->index )
		fprintf( TRACE_OUT, ",%d", TD->index );

	fprintf( TRACE_OUT, ")\"\n{\n" );

	DD = TD->ddesc;

	while ( DD != NULL )
	{
		if ( DD->did != NULL )
		{
			fprintf( TRACE_OUT, "\t// \"%s\" \"%s\"\n",
				DD->did->name, DD->did->desc );

			fprintf( TRACE_OUT, "\t%s \"%s\"",
				TYPE_STRS[ DD->dt ], DD->did->name );

			if ( DD->array == TEV_DATA_ARRAY )
				fprintf( TRACE_OUT, "[]" );

			if ( DD->dt == TEV_DATA_STRING )
				fprintf( TRACE_OUT, "[]" );

			fprintf( TRACE_OUT, ";\n" );
		}

		DD = DD->next;
	}

	fprintf( TRACE_OUT, "};;\n\n" );

	TD->dump = FALSE;
}

store_trace_event( TD )
TEVDESC TD;
{
	DATADESC DD;

	char	upk_byte[MAX_UNPACK_ARR_SIZE];
	float	upk_float[MAX_UNPACK_ARR_SIZE * 2];
	double	upk_double[MAX_UNPACK_ARR_SIZE * 2];
	int		upk_int[MAX_UNPACK_ARR_SIZE];
	long	upk_long[MAX_UNPACK_ARR_SIZE];
	short	upk_short[MAX_UNPACK_ARR_SIZE];
	char	upk_str[MAX_UNPACK_ARR_SIZE];

	char **strarr;

	int marker;
	int maxlen;
	int len;
	int num;
	int i;

	if ( TD->dump )
		dump_tevdesc( TD );

	fprintf( TRACE_OUT, "\"%s(", TD->name );

	if ( TD->entry_exit == ENTRY_TEV )
		fprintf( TRACE_OUT, "0" );

	else if ( TD->entry_exit == EXIT_TEV )
		fprintf( TRACE_OUT, "1" );

	if ( TD->index )
		fprintf( TRACE_OUT, ",%d", TD->index );

	fprintf( TRACE_OUT, ")\" { " );

	DD = TD->ddesc;

	while ( DD != NULL )
	{
		if ( DD->array == TEV_DATA_ARRAY )
		{
			PVMCKERR( pvm_upkint( &num, 1, 1 ),
				"Array Unpack", return( FALSE ) );

			if ( num > MAX_UNPACK_ARR_SIZE
				&& DD->dt != TEV_DATA_STRING )
			{
				printf( "Error: Unpack Size %d Too Large\n", num );

				return( FALSE );
			}

			if ( DD->dt == TEV_DATA_CPLX || DD->dt == TEV_DATA_DCPLX )
				fprintf( TRACE_OUT, "[%d] { ", num * 2 );

			/* matching } */

			else if ( DD->dt == TEV_DATA_STRING )
				fprintf( TRACE_OUT, "[%d] ", num );

			else
				fprintf( TRACE_OUT, "[%d] { ", num );

			/* matching } */
		}

		else
			num = 1;

		switch ( DD->dt )
		{
			case TEV_DATA_NULL: break;

			case TEV_DATA_BYTE:
			{
				PVMCKERR( pvm_upkbyte( upk_byte, num, 1 ),
					"Event BYTE Unpack", return( FALSE ) );

				if ( DD->array == TEV_DATA_ARRAY )
					fprintf( TRACE_OUT, "\"" );

				for ( i=0 ; i < num ; i++ )
					fprintf( TRACE_OUT, "%c", upk_byte[i] );

				if ( DD->array == TEV_DATA_ARRAY )
					fprintf( TRACE_OUT, "\"" );

				break;
			}

			case TEV_DATA_CPLX:
			{
				PVMCKERR( pvm_upkfloat( upk_float, num * 2, 1 ),
					"Event CPLX Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%f, %f",
						upk_float[ 2 * i ],
						upk_float[ (2 * i) + 1 ] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_DCPLX:
			{
				PVMCKERR( pvm_upkdouble( upk_double, num * 2, 1 ),
					"Event DCPLX Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%lf, %lf",
						upk_double[ 2 * i ],
						upk_double[ (2 * i) + 1 ] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_DOUBLE:
			{
				PVMCKERR( pvm_upkdouble( upk_double, num, 1 ),
					"Event DOUBLE Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%lf", upk_double[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_FLOAT:
			{
				PVMCKERR( pvm_upkfloat( upk_float, num, 1 ),
					"Event FLOAT Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%f", upk_float[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_INT:
			{
				PVMCKERR( pvm_upkint( upk_int, num, 1 ),
					"Event INT Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%d", upk_int[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_UINT:
			{
				PVMCKERR( pvm_upkint( upk_int, num, 1 ),
					"Event UINT Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%u", (unsigned) upk_int[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_LONG:
			{
				PVMCKERR( pvm_upklong( upk_long, num, 1 ),
					"Event LONG Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%ld", upk_long[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_ULONG:
			{
				PVMCKERR( pvm_upklong( upk_long, num, 1 ),
					"Event ULONG Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%ld", upk_long[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_SHORT:
			{
				PVMCKERR( pvm_upkshort( upk_short, num, 1 ),
					"Event SHORT Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%d", upk_short[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_USHORT:
			{
				PVMCKERR( pvm_upkshort( upk_short, num, 1 ),
					"Event USHORT Unpack", return( FALSE ) );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "%u", upk_short[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );
				}

				break;
			}

			case TEV_DATA_STRING:
			{
				if ( num < 1 )
					break;

				strarr = (char **) malloc( (unsigned) num
						* sizeof(char *) );
				memcheck( strarr, "String Array" );

				maxlen = 0;

				for ( i=0 ; i < num ; i++ )
				{
					PVMCKERR( pvm_upkstr( upk_str ),
						"Event STRING Unpack", return( FALSE ) );

					if ( (len = strlen( upk_str )) > maxlen )
						maxlen = len;

					strarr[i] = copy_str( upk_str );
				}

				fprintf( TRACE_OUT, "[%d] { ", maxlen + 1 );

				for ( i=0 ; i < num ; i++ )
				{
					fprintf( TRACE_OUT, "\"%s\"", strarr[i] );

					if ( i < num - 1 )
						fprintf( TRACE_OUT, ", " );

					free( strarr[i] );
				}

				if ( DD->array == TEV_DATA_SCALAR )
					fprintf( TRACE_OUT, " }" );

				free( strarr );

				break;
			}

			case TEV_DATA_STRUCT_START:
			case TEV_DATA_STRUCT_END:
			case TEV_DATA_DEFERRED:
				printf( "DT Not Impl\n" );
				break;

			default:
				printf( "DT Unknown\n" );
		}

		/* matching { */

		if ( DD->array == TEV_DATA_ARRAY )
			fprintf( TRACE_OUT, " }" );

		if ( DD->next != NULL )
			fprintf( TRACE_OUT, ", " );

		DD = DD->next;
	}

	PVMCKERR( pvm_upkint( &marker, 1, 1 ),
		"Data ID Message", return( FALSE ) );

	if ( marker == TEV_MARK_EVENT_RECORD_END )
		fprintf( TRACE_OUT, " };;\n" );

	else
	{
		printf( "Error: No Event End Marker\n" );

		return( FALSE );
	}

	return( TRUE );
}

store_output_event( tid, str, len )
int tid;
char *str;
int len;
{
	char tmp[4096];

	int index;
	int num;
	int i;

	index = 0;

	num = 0;

	for ( i=0 ; i < len ; i++ )
	{
		if ( str[i] != '\n' )
			tmp[index++] = str[i];

		else
		{
			tmp[index] = '\0';

			write_output_event( tid, tmp );

			index = 0;

			num++;
		}
	}

	if ( index )
	{
		tmp[index] = '\0';

		write_output_event( tid, tmp );

		num++;
	}

	return( num );
}

write_output_event( tid, str )
int tid;
char *str;
{
	if ( !( TEV33_TRACE[ TRACE33_OUTPUT ] ) )
	{
		fprintf( TRACE_OUT, "\n#%d: %s\n",
			TRACE33_OUTPUT + 1,
			TEV33_TRACE_DESCRIPTORS[ TRACE33_OUTPUT ] );

		(TEV33_TRACE[ TRACE33_OUTPUT ])++;
	}

	fprintf( TRACE_OUT, "\"%s\" { %d, %d, %d",
		TEV33_TRACE_NAMES[ TRACE33_OUTPUT ],
		TRACE_TIME.tv_sec, TRACE_TIME.tv_usec, tid );

	dump_trace_str( str, TRUE );

	fprintf( TRACE_OUT, " };;\n" );
}

save_host_status_events()
{
	HOST H;

	int sync;
	int num;

	if ( TRACE_OUT == NULL )
		return( 0 );

	if ( !( TEV33_TRACE[ TRACE33_HOST_ADD ] ) )
	{
		fprintf( TRACE_OUT, "\n#%d: %s\n",
			TRACE33_HOST_ADD + 1,
			TEV33_TRACE_DESCRIPTORS[ TRACE33_HOST_ADD ] );

		(TEV33_TRACE[ TRACE33_HOST_ADD ])++;
	}

	H = MAIN_NET->host_list;

	sync = 0;

	num = 0;

	while ( H != NULL )
	{
		if ( H->in_pvm == IN_PVM )
		{
			write_host_add_event( H );

			if ( H->delta.tv_sec != 0 || H->delta.tv_usec != 0 )
				sync++;

			num++;
		}

		H = H->next;
	}

	if ( sync )
	{
		H = MAIN_NET->host_list;

		while ( H != NULL )
		{
			if ( H->in_pvm == IN_PVM &&
				( H->delta.tv_sec != 0 || H->delta.tv_usec != 0 ) )
			{
				write_host_sync_event( H );

				num++;
			}

			H = H->next;
		}
	}

	if ( num )
		fflush( TRACE_OUT );

	return( num );
}

write_host_add_event( H )
HOST H;
{
	if ( !( TEV33_TRACE[ TRACE33_HOST_ADD ] ) )
	{
		fprintf( TRACE_OUT, "\n#%d: %s\n",
			TRACE33_HOST_ADD + 1,
			TEV33_TRACE_DESCRIPTORS[ TRACE33_HOST_ADD ] );

		(TEV33_TRACE[ TRACE33_HOST_ADD ])++;
	}

	fprintf( TRACE_OUT, "\"%s\" { %d, %d, %d",
		TEV33_TRACE_NAMES[ TRACE33_HOST_ADD ],
		TRACE_TIME.tv_sec, TRACE_TIME.tv_usec, H->pvmd_tid );

	dump_trace_str( H->refname, TRUE );

	dump_trace_str( H->alias, TRUE );

	dump_trace_str( H->arch->code, TRUE );

	fprintf( TRACE_OUT, ", %d };;\n", H->speed );
}

write_host_del_event( H, tid )
HOST H;
int tid;
{
	char tmp[1024];

	if ( !( TEV33_TRACE[ TRACE33_HOST_DEL ] ) )
	{
		fprintf( TRACE_OUT, "\n#%d: %s\n",
			TRACE33_HOST_DEL + 1,
			TEV33_TRACE_DESCRIPTORS[ TRACE33_HOST_DEL ] );

		(TEV33_TRACE[ TRACE33_HOST_DEL ])++;
	}

	fprintf( TRACE_OUT, "\"%s\" { %d, %d, %d",
		TEV33_TRACE_NAMES[ TRACE33_HOST_DEL ],
		TRACE_TIME.tv_sec, TRACE_TIME.tv_usec, tid );

	if ( strcmp( H->refname, "" ) )
	{
		sscanf( H->refname, "%s", tmp );

		dump_trace_str( tmp, TRUE );
	}

	else
		dump_trace_str( H->refname, TRUE );

	fprintf( TRACE_OUT, " };;\n" );

	fflush( TRACE_OUT );
}

write_host_sync_event( H )
HOST H;
{
	if ( !( TEV33_TRACE[ TRACE33_HOST_SYNC ] ) )
	{
		fprintf( TRACE_OUT, "\n#%d: %s\n",
			TRACE33_HOST_SYNC + 1,
			TEV33_TRACE_DESCRIPTORS[ TRACE33_HOST_SYNC ] );

		(TEV33_TRACE[ TRACE33_HOST_SYNC ])++;
	}

	fprintf( TRACE_OUT, "\"%s\" { %d, %d, %d, %d, %d };;\n",
		TEV33_TRACE_NAMES[ TRACE33_HOST_SYNC ],
		TRACE_TIME.tv_sec, TRACE_TIME.tv_usec, H->pvmd_tid,
		H->delta.tv_sec, H->delta.tv_usec );
}

/* Trace Processing Routines */

read_trace_event()
{
	char tmp[4096];
	char c;

	int entry_exit;
	int index;
	int flag;
	int eid;
	int i;

	if ( TRACE_IN == NULL )
		return( FALSE );

	while ( TRUE )
	{
		c = getc( TRACE_IN );

		if ( c != (char) EOF )
		{
			switch ( c )
			{
				/* Record Descriptor Packet */

				case '#':
				{
					/* Get event ID */

					flag = fscanf( TRACE_IN, "%d", &eid );

					if ( flag != 1 )
					{
						printf( "Error Reading Event ID\n" );

						return( FALSE );
					}

					/* Get event name */

					if ( !find_event_str( "\"" ) )
						return( FALSE );

					i = 0;

					while ( (c = getc( TRACE_IN )) != (char) EOF
						&& c != '"' && c != '(' )
					{
						tmp[i++] = c;
					}

					CKEOF( c, "EOF Reading Event Descriptor Name\n",
						return( FALSE ) );

					tmp[i] = '\0';

					/* PVM 3.4 Trace */

					if ( c == '(' )
					{
						/* Get event entry/exit & index */

						c = getc( TRACE_IN );

						CKEOF( c, "EOF Reading Event Descriptor\n",
							return( FALSE ) );

						if ( c == '0' )
						{
							entry_exit = ENTRY_TEV;

							c = getc( TRACE_IN );
						}

						else if ( c == '1' )
						{
							entry_exit = EXIT_TEV;

							c = getc( TRACE_IN );
						}

						else
							entry_exit = IGNORE_TEV;

						CKEOF( c, "EOF Reading Event Descriptor\n",
							return( FALSE ) );

						if ( c == '.' )
						{
							flag = fscanf( TRACE_IN, "%d", &index );

							if ( flag != 1 )
							{
								printf( "Error Reading Event Index\n" );

								return( FALSE );
							}

							if ( !find_event_str( ")" ) )
								return( FALSE );
						}

						else
							index = 0;

						if ( !find_event_str( "\"" ) )
							return( FALSE );

						if ( !read_descriptor( eid, tmp,
							entry_exit, index ) )
						{
							return( FALSE );
						}
					}

					/* PVM 3.3 Trace - Ignore Descriptor */

					else
					{
						if ( !find_event_end() )
							return( FALSE );
					}

					break;
				}

				/* Record Data Packet */

				case '"':
				{
					i = 0;

					while ( (c = getc( TRACE_IN )) != (char) EOF
						&& c != '"' && c != '(' )
					{
						tmp[i++] = c;
					}

					CKEOF( c, "EOF Reading Event Record Name\n",
						return( FALSE ) );

					tmp[i] = '\0';

					/* PVM 3.4 Trace */

					if ( c == '(' )
					{
						c = getc( TRACE_IN );

						CKEOF( c, "EOF Reading Event Record\n",
							return( FALSE ) );

						if ( c == '0' )
						{
							entry_exit = ENTRY_TEV;

							c = getc( TRACE_IN );
						}

						else if ( c == '1' )
						{
							entry_exit = EXIT_TEV;

							c = getc( TRACE_IN );
						}

						else
							entry_exit = IGNORE_TEV;

						CKEOF( c, "EOF Reading Event Record\n",
							return( FALSE ) );

						if ( c == '.' )
						{
							flag = fscanf( TRACE_IN, "%d", &index );

							if ( flag != 1 )
							{
								printf( "Error Reading Event Index\n" );

								return( FALSE );
							}

							if ( !find_event_str( ")" ) )
								return( FALSE );
						}

						else
							index = 0;

						if ( !find_event_str( "\"" ) )
							return( FALSE );

						return( process_trace_event( tmp,
							entry_exit, index ) );
					}

					/* PVM 3.3 Trace */

					else
						return( process_trace33_event( tmp ) );

					break;
				}

				/* Command Packet */

				case '%':
				{
					if ( !find_event_end() )
						return( FALSE );

					break;
				}

				/* Stream Attribute Packet */

				case '/':
				{
					c = getc( TRACE_IN );

					CKEOF( c, "EOF Reading Trace Comment\n",
						return( FALSE ) );

					if ( c != '*' )
					{
						printf( "\nError Reading Trace File\n" );
						printf( "\t- '*' expected (/* */)\n\n" );
					}

					if ( !find_event_end() )
						return( FALSE );

					break;
				}

				/* White Space */

				case ' ':
				case '\t':
				case '\n':
				{
					break;
				}
			}
		}

		else
		{
			if ( TRACE_FILE_STATUS != TRACE_FILE_PLAYBACK )
				printf( "EOF Reading Trace File\n" );

			return( FALSE );
		}
	}
}

read_descriptor( eid, name, entry_exit, index )
int eid;
char *name;
int entry_exit;
int index;
{
	DATADESC DD;

	TEVDESC TD;
	TEVDESC tdptr;

	char dtstr[255];
	char tmp[1024];
	char did[4];

	char last;
	char c;

	int done;
	int i;

	/* Find first bracket */

	if ( !find_event_str( "{" ) )
		return( FALSE );

	/* matching } */

	/* Create Descriptor Structure */

	TD = create_tevdesc();

	TD->name = copy_str( name );

	TD->eid = eid;

	TD->entry_exit = entry_exit;

	TD->index = index;

	/* Read in Data Descriptor Statements */

	done = 0;

	do
	{
		/* matching { */

		while ( (c = getc( TRACE_IN )) != (char) EOF
			&& c != '/' && c != '}' );

		CKEOF( c, "EOF Reading Event Descriptor\n", return( FALSE ) );

		/* Another Descriptor Statement */

		if ( c == '/' )
		{
			if ( TD->ddesc == NULL )
				DD = TD->ddesc = create_datadesc();

			else
				DD = DD->next = create_datadesc();

			/* Get Second '/' */

			c = getc( TRACE_IN );

			CKEOF( c, "EOF Reading Event Descriptor\n",
				return( FALSE ) );

			if ( c != '/' )
			{
				printf( "Error Parsing Event Descriptor\n" );

				return( FALSE );
			}

			/* Read in DID String */

			if ( !find_event_str( "\"" ) )
				return( FALSE );

			i = 0;

			while ( (c = getc( TRACE_IN )) != (char) EOF
				&& c != '"' && i < 3 )
			{
				did[i++] = c;
			}

			CKEOF( c, "EOF Reading Event Descriptor\n",
				return( FALSE ) );

			if ( c != '"' )
			{
				printf( "Error Parsing Event Descriptor\n" );

				return( FALSE );
			}

			did[i] = '\0';

			/* Get Descriptor DID struct */

			DD->did = (DID) lookup_trie( DID_TRIE, (void *) did );

			/* Skip over Description */

			if ( !find_event_str( "\"" ) )
				return( FALSE );

			if ( !find_event_str( "\"" ) )
				return( FALSE );

			/* Get Data Type */

			while ( (c = getc( TRACE_IN )) != (char) EOF
				&& ( c == ' ' || c == '\t' || c == '\n' ) );

			CKEOF( c, "EOF Reading Event Descriptor\n",
				return( FALSE ) );

			dtstr[0] = c;

			i = 1;

			while ( (c = getc( TRACE_IN )) != (char) EOF
				&& c != ' ' && c != '\t' && c != '\n' && c != '"' )
			{
				dtstr[i++] = c;
			}

			CKEOF( c, "EOF Reading Event Descriptor\n",
				return( FALSE ) );

			dtstr[i] = '\0';

			DD->dt = get_dt_str( dtstr );

			if ( DD->dt < 0 )
			{
				printf( "Error: Unknown Data Type \"%s\".\n", dtstr );

				return( FALSE );
			}

			/* Skip over var name */

			if ( c != '"' )
			{
				if ( !find_event_str( "\"" ) )
					return( FALSE );
			}

			if ( !find_event_str( "\"" ) )
				return( FALSE );

			/* Check for Data Array */

			c = getc( TRACE_IN );

			CKEOF( c, "EOF Reading Event Descriptor\n",
				return( FALSE ) );

			if ( c == '[' )
			{
				if ( DD->dt == TEV_DATA_BYTE
					|| DD->dt == TEV_DATA_STRING )
				{
					/* Get closing ']' */

					c = getc( TRACE_IN );

					CKEOF( c, "EOF Reading Event Descriptor\n",
						return( FALSE ) );

					if ( c != ']' )
					{
						printf( "Error Parsing Event Descriptor\n" );

						return( FALSE );
					}

					/* Check for another [] */

					c = getc( TRACE_IN );

					CKEOF( c, "EOF Reading Event Descriptor\n",
						return( FALSE ) );

					/* [][] -> STRING[] */
					if ( c == '[' )
					{
						if ( DD->dt == TEV_DATA_BYTE )
							DD->dt = TEV_DATA_STRING;

						DD->array = TEV_DATA_ARRAY;
					}

					/* [] -> BYTE[] */
					else
					{
						if ( DD->dt == TEV_DATA_STRING )
							DD->dt = TEV_DATA_BYTE;

						DD->array = TEV_DATA_ARRAY;
					}
				}

				else
					DD->array = TEV_DATA_ARRAY;
			}

			else
				DD->array = TEV_DATA_SCALAR;
		}

		/* End of Descriptor */

		else
		{
			if ( !find_event_end() )
				return( FALSE );

			done++;
		}
	}
	while ( !done );

	tdptr = (TEVDESC) lookup_trie( EVENT_TRIE, name );

	if ( tdptr == NULL )
		add_to_trie( EVENT_TRIE, name, (void *) TD );

	else
	{
		while ( tdptr->next != NULL )
		{
			if ( cmp_tevdesc( TD, tdptr ) )
			{
				free_tevdesc( &TD );

				return( TRUE );
			}

			tdptr = tdptr->next;
		}

		if ( cmp_tevdesc( TD, tdptr ) )
		{
			free_tevdesc( &TD );

			return( TRUE );
		}

		else
			tdptr->next = TD;
	}

	/* Set Handle ID */

	TD->hid = (int) lookup_trie( HANDLE_TRIE, TD->name );

	return( TRUE );
}

process_trace_event( name, entry_exit, index )
char *name;
int entry_exit;
int index;
{
	DATADESC DD;

	TEVDESC TD;

	TEVREC TR;
	TEVREC trptr;

	char	read_str[MAX_UNPACK_ARR_SIZE];
	double	read_double;
	float	read_float;
	short	read_short;
	long	read_long;
	int		read_int;

	char c;

	int i, j;
	int flag;
	int num;

	TD = (TEVDESC) lookup_trie( EVENT_TRIE, name );

	while ( TD != NULL &&
		( TD->entry_exit != entry_exit || TD->index != index ) )
	{
		TD = TD->next;
	}

	if ( TD == NULL )
	{
		printf( "Error: Event \"%s\" Descriptor Not Found\n", name );

		find_event_end();

		return( FALSE );
	}

	if ( !find_event_str( "{" ) )
		return( FALSE );
	
	/* matching } */

	DD = TD->ddesc;

	TR = (TEVREC) NULL;

	while ( DD != NULL )
	{
		if ( TR == NULL )
			trptr = TR = create_tevrec();
		
		else
			trptr = trptr->next = create_tevrec();

		trptr->ddesc = DD;

		if ( DD->array == TEV_DATA_ARRAY )
		{
			if ( !find_event_str( "[" ) )
				return( FALSE );

			flag = fscanf( TRACE_IN, "%d", &num );

			if ( flag != 1 )
			{
				printf( "Error Parsing Event Record\n" );

				find_event_end();

				return( FALSE );
			}

			if ( !find_event_str( "{" ) )
				return( FALSE );

			/* matching } */
		}

		else
			num = 1;

		trptr->value = make_value( DD->dt, num );

		if ( trptr->value == NULL )
		{
			printf( "Error Allocating Value dt=%d num=%d\n",
				DD->dt, num );

			return( FALSE );
		}

		trptr->num = num;

		switch ( DD->dt )
		{
			case TEV_DATA_NULL: break;

			case TEV_DATA_BYTE:
			{
				if ( DD->array == TEV_DATA_ARRAY )
				{
					if ( !find_event_str( "\"" ) )
						return( FALSE );

					i = 0;

					while ( i < num
						&& (c = getc( TRACE_IN )) != (char) EOF
						&& c != '"' )
					{
						ARR_VALUE_OF( trptr->value, char, i++ ) = c;
					}

					CKEOF( c, "EOF Parsing Event Record\n",
						return( FALSE ) );

					if ( i == num )
					{
						printf( "Error: Character Overflow\n" );

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, char, i ) = '\0';
				}

				else
				{
					while ( (c = getc( TRACE_IN )) != (char) EOF
						&& ( c == ' ' || c == '\t' || c == '\n' ) );

					CKEOF( c, "EOF Parsing Event Record\n",
						return( FALSE ) );

					VALUE_OF( trptr->value, char ) = c;
				}

				break;
			}

			case TEV_DATA_CPLX:
			case TEV_DATA_FLOAT:
			{
				for ( i=0 ; i < num ; i++ )
				{
					flag = fscanf( TRACE_IN, "%f", &read_float );

					if ( flag != 1 )
					{
						printf( "Error Parsing Event Record\n" );

						find_event_end();

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, float, i ) = read_float;

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				break;
			}

			case TEV_DATA_DCPLX:
			case TEV_DATA_DOUBLE:
			{
				for ( i=0 ; i < num ; i++ )
				{
					flag = fscanf( TRACE_IN, "%lf", &read_double );

					if ( flag != 1 )
					{
						printf( "Error Parsing Event Record\n" );

						find_event_end();

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, double, i ) =
						read_double;

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				break;
			}

			case TEV_DATA_INT:
			{
				for ( i=0 ; i < num ; i++ )
				{
					flag = fscanf( TRACE_IN, "%d", &read_int );

					if ( flag != 1 )
					{
						printf( "Error Parsing Event Record\n" );

						find_event_end();

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, int, i ) = read_int;

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				break;
			}

			case TEV_DATA_UINT:
			{
				for ( i=0 ; i < num ; i++ )
				{
					flag = fscanf( TRACE_IN, "%u", &read_int );

					if ( flag != 1 )
					{
						printf( "Error Parsing Event Record\n" );

						find_event_end();

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, int, i ) = read_int;

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				break;
			}

			case TEV_DATA_LONG:
			case TEV_DATA_ULONG:
			{
				for ( i=0 ; i < num ; i++ )
				{
					flag = fscanf( TRACE_IN, "%ld", &read_long );

					if ( flag != 1 )
					{
						printf( "Error Parsing Event Record\n" );

						find_event_end();

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, long, i ) = read_long;

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				break;
			}

			case TEV_DATA_SHORT:
			{
				for ( i=0 ; i < num ; i++ )
				{
					flag = fscanf( TRACE_IN, "%d", &read_short );

					if ( flag != 1 )
					{
						printf( "Error Parsing Event Record\n" );

						find_event_end();

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, short, i ) = read_short;

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				break;
			}

			case TEV_DATA_USHORT:
			{
				for ( i=0 ; i < num ; i++ )
				{
					flag = fscanf( TRACE_IN, "%u", &read_short );

					if ( flag != 1 )
					{
						printf( "Error Parsing Event Record\n" );

						find_event_end();

						return( FALSE );
					}

					ARR_VALUE_OF( trptr->value, short, i ) = read_short;

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				break;
			}

			case TEV_DATA_STRING:
			{
				if ( DD->array == TEV_DATA_SCALAR )
				{
					if ( !find_event_str( "{" ) )
						return( FALSE );
				}

				for ( i=0 ; i < num ; i++ )
				{
					if ( !find_event_str( "\"" ) )
						return( FALSE );

					j = 0;

					while ( (c = getc( TRACE_IN )) != (char) EOF
						&& c != '"' )
					{
						read_str[j++] = c;
					}

					CKEOF( c, "EOF Parsing Event Record\n",
						return( FALSE ) );

					read_str[j] = '\0';

					ARR_VALUE_OF( trptr->value, char *, i ) =
						copy_str( read_str );

					if ( i < num - 1 )
					{
						if ( !find_event_str( "," ) )
							return( FALSE );
					}
				}

				if ( DD->array == TEV_DATA_SCALAR )
				{
					if ( !find_event_str( "}" ) )
						return( FALSE );
				}

				break;
			}

			case TEV_DATA_STRUCT_START:
			case TEV_DATA_STRUCT_END:
			case TEV_DATA_DEFERRED:
			{
				printf( "DT Not Impl\n" );

				find_event_end();

				return( FALSE );
			}

			default:
			{
				printf( "DT Unknown\n" );

				find_event_end();

				return( FALSE );
			}
		}

		if ( DD->array == TEV_DATA_ARRAY )
		{
			/* matching { */

			if ( !find_event_str( "}" ) )
				return( FALSE );
		}

		if ( DD->next != NULL )
		{
			if ( !find_event_str( "," ) )
				return( FALSE );
		}

		DD = DD->next;
	}

	if ( vflag )
		dump_tevrec( name, entry_exit, index, TR );

	handle_event( TD, TR );

	free_tevrec( &TR );

	return( TRUE );
}

handle_event( TD, TR )
TEVDESC TD;
TEVREC TR;
{
	TEVREC tr_dtids;
	TEVREC tr_name;

	TASK T;

	char event_str[1024];
	char tmp[1024];

	char *u_usec_pad;
	char *s_usec_pad;
	char *usec_pad;

	char *str;

	int u_sec, u_usec;
	int s_sec, s_usec;

	int msgtag;
	int nbytes;
	int flags;
	int tusec;
	int tsec;
	int dtid;
	int stid;
	int tid;
	int buf;

	int do_state;
	int state;
	int i;

	str = (char *) NULL;

	tsec = GET_TEVREC_VALUE( TR, "TS", int );
	tusec = GET_TEVREC_VALUE( TR, "TU", int );

	do_state = TRUE;

	switch ( TD->hid )
	{
		case HANDLE_NEWTASK:
		{
			GET_TEVREC( TR, tr_name, "TN" );

			if ( !TRACE_GROUP_TASKS
				&& GROUPTASK( (char *) tr_name->value ) )
			{
				return;
			}

			T = create_task();

			T->tid = GET_TEVREC_VALUE( TR, "TID", int );

			T->name = copy_str( (char *) tr_name->value );

			T->ptid = GET_TEVREC_VALUE( TR, "PT", int );

			T->pvmd_tid = pvm_tidtohost( T->tid );

			T->host = get_host_tid( T->pvmd_tid, PLAYBACK_ADDED );

			T->flags = GET_TEVREC_VALUE( TR, "TF", int );

			T->status = TASK_ALIVE;

			/* Handle New Task Times */

			if ( BASE_TIME.tv_sec == -1 )
			{
				sync_time( T->host, &tsec, &tusec );

				BASE_TIME.tv_sec = tsec;
				BASE_TIME.tv_usec = tusec;

				update_time( 0, 0 );

				tsec = tusec = 0;
			}

			else
			{
				normalize_time( T, &tsec, &tusec );

				check_time( tsec, tusec );
			}

			usec_pad = pad_num( tusec, 6 );

			sprintf( event_str,
				"New Task %s @ %d.%s: tid=%x ptid=%x host=%x flags=%d.",
				T->name, tsec, usec_pad, T->tid, T->ptid,
				T->pvmd_tid, T->flags );

			free( usec_pad );

			taskAdd( T, tsec, tusec, event_str );

			str = event_str;

			break;
		}

		case HANDLE_SPNTASK: break;

		case HANDLE_ENDTASK:
		{
			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			if ( T->status != TASK_DEAD )
				check_time( tsec, tusec );

			flags = GET_TEVREC_VALUE( TR, "TF", int );

			u_sec = GET_TEVREC_VALUE( TR, "TUS", int );
			u_usec = GET_TEVREC_VALUE( TR, "TUU", int );

			s_sec = GET_TEVREC_VALUE( TR, "TSS", int );
			s_usec = GET_TEVREC_VALUE( TR, "TSU", int );

			u_usec_pad = pad_num( u_usec, 6 );
			s_usec_pad = pad_num( s_usec, 6 );
			usec_pad = pad_num( tusec, 6 );

			sprintf( event_str,
			"End Task @ %d.%s: tid=%x stat=0x%x user=%d.%s sys=%d.%s.",
				tsec, usec_pad, tid, flags, u_sec, u_usec_pad,
				s_sec, s_usec_pad );

			free( u_usec_pad );
			free( s_usec_pad );
			free( usec_pad );

			taskExit( T, tsec, tusec, event_str );

			do_state = FALSE;

			str = event_str;

			break;
		}

		case HANDLE_HOST_ADD:
		{
			break;
		}

		case HANDLE_HOST_DEL:
		{
			break;
		}

		case HANDLE_HOST_SYNC:
		{
			break;
		}

		case HANDLE_OUTPUT:
		{
			break;
		}

		case HANDLE_EXIT:
		{
			if ( TD->entry_exit != ENTRY_TEV )
				break;

			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			usec_pad = pad_num( tusec, 6 );

			sprintf( event_str, "exit{0} @ %d.%s.", tsec, usec_pad );

			free( usec_pad );

			taskState( T, tsec, tusec, STATE_SYSTEM, event_str );

			taskExit( T, tsec, tusec + 1, event_str );

			do_state = FALSE;

			str = event_str;

			break;
		}

		case HANDLE_SEND:
		{
			if ( TD->entry_exit != ENTRY_TEV )
				break;

			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			dtid = GET_TEVREC_VALUE( TR, "DST", int );

			msgtag = GET_TEVREC_VALUE( TR, "MC", int );

			nbytes = GET_TEVREC_VALUE( TR, "MNB", int );

			send_message( T, tid, dtid, msgtag, nbytes, tsec, tusec );

			usec_pad = pad_num( tusec, 6 );

			sprintf( event_str,
				"send{0} @ %d.%s: to %x code=%d nbytes=%d.",
				tsec, usec_pad, dtid, msgtag, nbytes );

			free( usec_pad );

			str = event_str;

			break;
		}

		case HANDLE_MCAST:
		{
			if ( TD->entry_exit != ENTRY_TEV )
				break;

			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			msgtag = GET_TEVREC_VALUE( TR, "MC", int );

			nbytes = GET_TEVREC_VALUE( TR, "MNB", int );

			usec_pad = pad_num( tusec, 6 );

			sprintf( event_str,
				"mcast{0} @ %d.%s: code=%d nbytes=%d to ",
				tsec, usec_pad, msgtag, nbytes );

			free( usec_pad );

			GET_TEVREC( TR, tr_dtids, "MDL" );

			for ( i=0 ; i < tr_dtids->num ; i++ )
			{
				dtid = ARR_VALUE_OF( tr_dtids->value, int, i );

				if ( dtid != tid )
				{
					send_message( T, tid, dtid, msgtag, nbytes,
						tsec, tusec );
				}

				if ( i < tr_dtids->num - 1 )
					sprintf( tmp, "%x, ", dtid );

				else
					sprintf( tmp, "%x.", dtid );

				append_cmd_str( event_str, tmp, 1024,
					(char *) NULL, FALSE );
			}

			str = event_str;

			break;
		}

		case HANDLE_PSEND:
		{
			if ( TRUE )
				break;

			if ( TD->entry_exit != ENTRY_TEV )
				break;

			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

/* XXX FIX THIS XXX */

			break;
		}

		case HANDLE_RECV:
		{
			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			if ( TD->entry_exit == ENTRY_TEV )
			{
				str = tevrec_string( TD, TR, tid, tsec, tusec );

				if ( str == NULL )
					return;

				taskState( T, tsec, tusec, STATE_IDLE, str );

				do_state = FALSE;

				break;
			}

			else if ( TD->entry_exit != EXIT_TEV )
				break;

			buf = GET_TEVREC_VALUE( TR, "MB", int );

			msgtag = GET_TEVREC_VALUE( TR, "MC", int );

			nbytes = GET_TEVREC_VALUE( TR, "MNB", int );

			stid = GET_TEVREC_VALUE( TR, "SRC", int );

			recv_message( T, tid, stid, msgtag, nbytes, tsec, tusec );

			usec_pad = pad_num( tusec, 6 );

			sprintf( event_str,
				"recv{1} @ %d.%s: from %x buf=%d code=%d nbytes=%d.",
				tsec, usec_pad, stid, buf, msgtag, nbytes );
			
			free( usec_pad );

			str = event_str;

			break;
		}

		case HANDLE_NRECV:
		{
			if ( TD->entry_exit != EXIT_TEV )
				break;

			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			buf = GET_TEVREC_VALUE( TR, "MB", int );

			if ( buf > 0 )
			{
				msgtag = GET_TEVREC_VALUE( TR, "MC", int );

				nbytes = GET_TEVREC_VALUE( TR, "MNB", int );

				stid = GET_TEVREC_VALUE( TR, "SRC", int );

				recv_message( T, tid, stid, msgtag, nbytes,
					tsec, tusec );

				usec_pad = pad_num( tusec, 6 );

				sprintf( event_str,
				"nrecv{1} @ %d.%s: from %x buf=%d code=%d nbytes=%d.",
					tsec, usec_pad, stid, buf, msgtag,
					nbytes );

				free( usec_pad );

				str = event_str;
			}

			else
			{
				usec_pad = pad_num( tusec, 6 );

				sprintf( event_str, "nrecv{1} @ %d.%s: no message.",
					tsec, usec_pad );

				free( usec_pad );

				str = event_str;
			}

			break;
		}

		case HANDLE_PRECV:
		{
			if ( TRUE )
				break;

			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			if ( TD->entry_exit == ENTRY_TEV )
			{
				str = tevrec_string( TD, TR, tid, tsec, tusec );

				if ( str == NULL )
					return;

				taskState( T, tsec, tusec, STATE_IDLE, str );

				do_state = FALSE;

				break;
			}

			else if ( TD->entry_exit != EXIT_TEV )
				break;

/* XXX FIX THIS XXX */

			break;
		}

		case HANDLE_TRECV:
		{
			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			if ( TD->entry_exit == ENTRY_TEV )
			{
				str = tevrec_string( TD, TR, tid, tsec, tusec );

				if ( str == NULL )
					return;

				taskState( T, tsec, tusec, STATE_IDLE, str );

				do_state = FALSE;

				break;
			}

			else if ( TD->entry_exit != EXIT_TEV )
				break;

			buf = GET_TEVREC_VALUE( TR, "MB", int );

			if ( buf > 0 )
			{
				msgtag = GET_TEVREC_VALUE( TR, "MC", int );

				nbytes = GET_TEVREC_VALUE( TR, "MNB", int );

				stid = GET_TEVREC_VALUE( TR, "SRC", int );

				recv_message( T, tid, stid, msgtag, nbytes,
					tsec, tusec );

				usec_pad = pad_num( tusec, 6 );

				sprintf( event_str,
				"trecv{1} @ %d.%s: from %x buf=%d code=%d nbytes=%d.",
					tsec, usec_pad, stid, buf, msgtag, nbytes );

				free( usec_pad );

				str = event_str;
			}

			else
			{
				usec_pad = pad_num( tusec, 6 );

				sprintf( event_str,
					"trecv{1} @ %d.%s: no message, timed out.",
					tsec, usec_pad );

				free( usec_pad );

				str = event_str;
			}

			break;
		}

		default: break;
	}

	if ( do_state && TD->entry_exit != IGNORE_TEV )
	{
		if ( str == NULL )
		{
			tid = GET_TEVREC_VALUE( TR, "TID", int );

			if ( (T = get_task_tid( tid )) == NULL )
				return;

			normalize_time( T, &tsec, &tusec );

			check_time( tsec, tusec );

			str = tevrec_string( TD, TR, tid, tsec, tusec );

			if ( str == NULL )
				return;
		}

		if ( TD->entry_exit == ENTRY_TEV )
			state = STATE_SYSTEM;

		else if ( TD->entry_exit == EXIT_TEV )
			state = STATE_RUNNING;

		taskState( T, tsec, tusec, state, str );
	}

	if ( eflag )
	{
		printf( "PVM Event: %d/%d %s\n", tsec, tusec, str );

		fflush( stdout );
	}

	if ( str != event_str && str != NULL )
		free( str );
}

task_output( tid, str )
int tid;
char *str;
{
	char tmp[4096];

	if ( !strcmp( str, "GOTEOF" ) )
	{
		sprintf( tmp, "task_output %x EOF", tid );

		if ( OUTPUT_FP != NULL )
		{
			fprintf( OUTPUT_FP, "[t%x] EOF\n", tid );

			fflush( OUTPUT_FP );
		}
	}

	else if ( strcmp( str, "CREATION" ) && strcmp( str, "GOTSPAWN" ) )
	{
		sprintf( tmp, "task_output %x {%s}", tid, str );

		if ( OUTPUT_FP != NULL )
		{
			fprintf( OUTPUT_FP, "[t%x] %s\n", tid, str );

			fflush( OUTPUT_FP );
		}
	}

	else
		return;

	Tcl_Eval( interp, tmp );
}

normalize_time( T, psec, pusec )
TASK T;
int *psec;
int *pusec;
{
	/* Adjust Host Delay Time */

	if ( T != NULL && T->host != NULL )
		sync_time( T->host, psec, pusec );

	/* Adjust Time Relative to Base Time */

	if ( BASE_TIME.tv_sec != -1 )
	{
		*psec -= BASE_TIME.tv_sec;
		*pusec -= BASE_TIME.tv_usec;

		if ( *pusec < 0 )
		{
			*pusec += 1000000;

			(*psec)--;
		}
	}

	else
	{
		*psec = 0;
		*pusec = 0;
	}
}

sync_time( H, psec, pusec )
HOST H;
int *psec;
int *pusec;
{
	if ( H != NULL )
	{
		*psec += H->delta.tv_sec;
		*pusec += H->delta.tv_usec;

		if ( *pusec > 1000000 )
		{
			*pusec -= 1000000;

			(*psec)++;
		}

		else if ( *pusec < 0 )
		{
			*pusec += 1000000;

			(*psec)--;
		}
	}
}

check_time( tsec, tusec )
int tsec, tusec;
{
	int flag;

	if ( BASE_TIME.tv_sec == -1 )
		return;

	flag = 0;

	if ( tsec > CURRENT_TIME.tv_sec )
		flag++;

	else if ( tsec == CURRENT_TIME.tv_sec
		&& tusec > CURRENT_TIME.tv_usec )
	{
		flag++;
	}

	if ( flag )
		update_time( tsec, tusec );
}

update_time( sec, usec )
int sec;
int usec;
{
	int new_timex;

	REFRESH_GLOBAL( SCALE );
	REFRESH_GLOBAL( TIMEX );

	CURRENT_TIME.tv_sec = sec;
	CURRENT_TIME.tv_usec = usec;

	if ( tflag )
		return;

	new_timex = X_OF_TIME( sec, usec, INT_GLOBVAL( SCALE ) );

	if ( new_timex != INT_GLOBVAL( TIMEX ) )
		set_view_time( new_timex );

	return( TCL_OK );
}

set_view_time( timex )
int timex;
{
	char value[255];
	char cmd[1024];

	char *usec_pad;

	int tmp_csize;
	int qtx;
	int wt;
	int fo;

	REFRESH_GLOBAL( ST_CANVAS_HEIGHT );
	REFRESH_GLOBAL( UT_CANVAS_HEIGHT );
	REFRESH_GLOBAL( FRAME_BORDER );
	REFRESH_GLOBAL( FRAME_OFFSET );
	REFRESH_GLOBAL( TIMEID_PRIM );
	REFRESH_GLOBAL( ST_CWIDTH );
	REFRESH_GLOBAL( UT_CWIDTH );
	REFRESH_GLOBAL( ST_TIMEID );
	REFRESH_GLOBAL( UT_TIMEID );
	REFRESH_GLOBAL( ST_XVIEW );
	REFRESH_GLOBAL( UT_XVIEW );

	/* Adjust Control Panel Time Display */

	usec_pad = pad_num( CURRENT_TIME.tv_usec, 6 );

	sprintf( cmd, "%s.time configure -text {Time: %d.%s}",
		CTRL, CURRENT_TIME.tv_sec, usec_pad );

	free( usec_pad );

	Tcl_Eval( interp, cmd );

	/* Adjust TIMEX (timex) */

	sprintf( value, "%d", timex );

	SET_TCL_GLOBAL( interp, "timex", value );

	/* Adjust Space-Time View Scrolling */

	fo = INT_GLOBVAL( FRAME_OFFSET ) * INT_GLOBVAL( FRAME_BORDER );

	tmp_csize = timex + ( 2 * INT_GLOBVAL( FRAME_BORDER ) );

	wt = tmp_csize + ( 2 * fo ) - ST_SCROLL_MARK;

	if ( wt > INT_GLOBVAL( ST_CWIDTH ) )
	{
		ST_SCROLL_MARK = tmp_csize - INT_GLOBVAL( ST_CWIDTH ) + fo;

		if ( TIMEIDX != -1
			&& !strcmp( CHAR_GLOBVAL( TIMEID_PRIM ), "ST" ) )
		{
			qtx = TIMEIDX - INT_GLOBVAL( ST_XVIEW )
				+ ST_SCROLL_MARK;

			sprintf( cmd, "%s coords %d %d 0 %d %d",
				ST_C, INT_GLOBVAL( ST_TIMEID ), qtx, qtx,
				INT_GLOBVAL( ST_CANVAS_HEIGHT ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "%s raise %d",
				ST_C, INT_GLOBVAL( ST_TIMEID ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "%s coords %d %d 0 %d %d",
				UT_C, INT_GLOBVAL( UT_TIMEID ), qtx, qtx,
				INT_GLOBVAL( UT_CANVAS_HEIGHT ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "%s raise %d",
				UT_C, INT_GLOBVAL( UT_TIMEID ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "set_query_time %d", qtx );

			Tcl_Eval( interp, cmd );
		}

		sprintf( cmd, "do_xview %s %d", ST_C, ST_SCROLL_MARK );

		Tcl_Eval( interp, cmd );

		sprintf( value, "%d", ST_SCROLL_MARK );

		SET_TCL_GLOBAL( interp, "st_xview", value );

		sprintf( cmd, "scrollSet %s %d %d 1 %d",
			ST_SBH, tmp_csize, INT_GLOBVAL( ST_CWIDTH ),
			ST_SCROLL_MARK );

		Tcl_Eval( interp, cmd );
	}

	taskSTIncr( CURRENT_TIME.tv_sec, CURRENT_TIME.tv_usec );

	/* Adjust Utilization View Scrolling */

	wt = tmp_csize + (2 * fo ) - UT_SCROLL_MARK;

	if ( wt > INT_GLOBVAL( UT_CWIDTH ) )
	{
		UT_SCROLL_MARK = tmp_csize - INT_GLOBVAL( UT_CWIDTH ) + fo;

		if ( TIMEIDX != -1
			&& !strcmp( CHAR_GLOBVAL( TIMEID_PRIM ), "UT" ) )
		{
			qtx = TIMEIDX - INT_GLOBVAL( UT_XVIEW )
				+ UT_SCROLL_MARK;

			sprintf( cmd, "%s coords %d %d 0 %d %d",
				UT_C, INT_GLOBVAL( UT_TIMEID ), qtx, qtx,
				INT_GLOBVAL( UT_CANVAS_HEIGHT ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "%s raise %d",
				UT_C, INT_GLOBVAL( UT_TIMEID ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "%s coords %d %d 0 %d %d",
				ST_C, INT_GLOBVAL( ST_TIMEID ), qtx, qtx,
				INT_GLOBVAL( ST_CANVAS_HEIGHT ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "%s raise %d",
				ST_C, INT_GLOBVAL( ST_TIMEID ) );

			Tcl_Eval( interp, cmd );

			sprintf( cmd, "set_query_time %d", qtx );

			Tcl_Eval( interp, cmd );
		}

		sprintf( cmd, "do_xview %s %d", UT_C, UT_SCROLL_MARK );

		Tcl_Eval( interp, cmd );

		sprintf( value, "%d", UT_SCROLL_MARK );

		SET_TCL_GLOBAL( interp, "ut_xview", value );

		sprintf( cmd, "scrollSet %s %d %d 1 %d",
			UT_SBH, tmp_csize, INT_GLOBVAL( UT_CWIDTH ),
			UT_SCROLL_MARK );

		Tcl_Eval( interp, cmd );
	}

	utIncr( CURRENT_TIME.tv_sec, CURRENT_TIME.tv_usec );

	Tcl_Eval( interp, "update" );
}

/* ARGSUSED */
int update_scroll_marks_proc( clientData, itp, argc, argv )
ClientData clientData;
Tcl_Interp *itp;
int argc;
char **argv;
{
	REFRESH_GLOBAL( FRAME_BORDER );
	REFRESH_GLOBAL( FRAME_OFFSET );
	REFRESH_GLOBAL( TIMEX );

	ST_SCROLL_MARK = INT_GLOBVAL( TIMEX ) - INT_GLOBVAL( ST_CWIDTH )
		+ ( INT_GLOBVAL( FRAME_BORDER )
			* ( 2 + INT_GLOBVAL( FRAME_OFFSET ) ) );

	ST_SCROLL_MARK = ST_SCROLL_MARK < 0 ? 0 : ST_SCROLL_MARK;

	UT_SCROLL_MARK = INT_GLOBVAL( TIMEX ) - INT_GLOBVAL( UT_CWIDTH )
		+ ( INT_GLOBVAL( FRAME_BORDER )
			* ( 2 + INT_GLOBVAL( FRAME_OFFSET ) ) );

	UT_SCROLL_MARK = UT_SCROLL_MARK < 0 ? 0 : UT_SCROLL_MARK;

	return( TCL_OK );
}

update_trace_time( sec, usec )
int sec;
int usec;
{
	struct timeval tm;

	if ( sec == -1 )
	{
		gettimeofday( &tm, (struct timezone *) NULL );

		sec = tm.tv_sec;
		usec = tm.tv_usec;
	}

	if ( sec > TRACE_TIME.tv_sec ||
		( sec == TRACE_TIME.tv_sec && usec > TRACE_TIME.tv_usec ) )
	{
		TRACE_TIME.tv_sec = sec;
		TRACE_TIME.tv_usec = usec;
	}
}

/* Trace Utility Routines */

dump_trace_header()
{
	char *ds;

	if ( TRACE_OUT == NULL )
		return;

	fprintf( TRACE_OUT, "/*\n" );

	ds = date_str();

	fprintf( TRACE_OUT, " * \"Creation Date\" \"%s\"\n", ds );

	free( ds );

	fprintf( TRACE_OUT, " * \"Machine\" \"XPVM %s (PVM %s)\"\n",
		XPVM_VERSION, pvm_version() );

	fprintf( TRACE_OUT, " */ ;;\n\n" );
}

dump_trace_str( str, comma_flag )
char *str;
int comma_flag;
{
	if ( comma_flag )
	{
		fprintf( TRACE_OUT, ", [%d] { \"%s\" }",
			strlen( str ) + 1, str );
	}

	else
		fprintf( TRACE_OUT, "[%d] { \"%s\" }", strlen( str ) + 1, str );
}

dump_sddf_headers()
{
	int eid;

	printf( "\n" );
	printf( "XPVM Trace Format Descriptors - SDDF:\n" );
	printf( "=====================================\n" );

	for ( eid=0 ; eid < TRACE33_MAX ; eid++ )
	{
		if ( strcmp( TEV33_TRACE_DESCRIPTORS[ eid ], "" ) )
		{
			printf( "\n#%d: %s\n", eid + 1,
				TEV33_TRACE_DESCRIPTORS[ eid ] );
		}
	}

	exit( 0 );
}

read_trace_str( str, size, end_flag )
char *str;
int size;
int end_flag;
{
	char c;

	int i;

	*str = '\0';

	if ( !find_event_str( "\"" ) )
		return( FALSE );

	i = 0;

	while ( (c = getc( TRACE_IN )) != (char) EOF && c != '"'
		&& i < size - 1 )
	{
		str[i++] = c;
	}

	str[i] = '\0';

	CKEOF( c, "EOF Reading Trace String\n", return( FALSE ) );

	if ( c != '"' )
	{
		if ( !find_event_str( "\"" ) )
			return( FALSE );
	}

	if ( end_flag )
	{
		/* Matching { */

		if ( !find_event_str( "}" ) )
			return( FALSE );
	}

	return( TRUE );
}

check_overwrite_exists( itp )
Tcl_Interp *itp;
{
	FILE *fp;

	char tmp[1024];

	if ( TRACE_OVERWRITE_FLAG )
		return( TRUE );

	fp = fopen( TRACE_FILE, "r" );

	if ( fp != NULL )
	{
		fclose( fp );

		sprintf( tmp, "prompt_trace_overwrite %s", TRACE_FILE );

		Tcl_Eval( itp, tmp );

		return( FALSE );
	}

	else
		return( TRUE );
}

find_event_str( str )
char *str;
{
	char tmp[1024];

	char c;

	int index;
	int i, j;
	int nope;
	int len;

	len = strlen( str );

	for ( i=0 ; i < len ; i++ )
		tmp[i] = '\0';

	i = 0;

	while ( (c = getc( TRACE_IN )) != (char) EOF )
	{
		tmp[i] = c;

		nope = 0;

		for ( j=0 ; j < len && !nope ; j++ )
		{
			index = i - len + j + 1;

			index = index < 0 ? index + len : index;

			if ( tmp[index] != str[j] )
				nope++;
		}

		if ( !nope )
			return( TRUE );

		i = ( i + 1 ) % len;
	}

	printf( "EOF Reading Trace File - \"%s\" not found.\n", str );

	return( FALSE );
}

find_event_end()
{
	return( find_event_str( ";;" ) );
}

