/* ---------------------------------------------------------------------
*
*  -- PBLAS routine (version 1.0) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory,
*     and University of California, Berkeley.
*     March 17, 1995
*
*  ---------------------------------------------------------------------
*/
/*
*  Include files
*/
#include "pblas.h"

void pdger_( m, n, alpha, X, ix, jx, desc_X, incx, Y, iy, jy, desc_Y,
             incy, A, ia, ja, desc_A )
/*
*  .. Scalar Arguments ..
*/
   int         * ia, * incx, * incy, * ix, * iy, * ja, * jx, * jy, * m,
               * n;
   double      * alpha;
/* ..
*  .. Array Arguments ..
*/
   int         desc_A[], desc_X[], desc_Y[];
   double      A[], X[], Y[];
{
/*
*  Purpose
*  =======
*
*  PDGER   performs the rank 1 operation
*
*     sub( A ) := alpha*sub( X )*sub( Y )' + sub( A ),
*
*  where sub( A ) denotes A(IA:IA+M-1,JA:JA+N-1),
*
*        sub( X ) denotes X(IX,JX:JX+N-1) if INCX = M_X,
*                         X(IX:IX+N-1,JX) if INCX = 1 and INCX <> M_X,
*
*        sub( Y ) denotes Y(IY,JY:JY+N-1) if INCY = M_Y,
*                         Y(IY:IY+N-1,JY) if INCY = 1 and INCY <> M_Y.
*
*  alpha is a scalar, sub( X ) is an M element distributed vector,
*  sub( Y ) is an N element distributed vector and sub( A ) is an
*  M-by-N distributed matrix.
*
*  Notes
*  =====
*
*  A description vector is associated with each 2D block-cyclicly dis-
*  tributed matrix.  This vector stores the information required to
*  establish the mapping between a matrix entry and its corresponding
*  process and memory location.
*
*  In the following comments, the character _ should be read as
*  "of the distributed matrix".  Let A be a generic term for any 2D
*  block cyclicly distributed matrix.  Its description vector is DESC_A:
*
*  NOTATION        STORED IN     EXPLANATION
*  --------------- ------------- ---------------------------------------
*  M_A    (global) desc_A[M_]    The number of rows in the distributed
*                                matrix.
*  N_A    (global) desc_A[N_]    The number of columns in the distribu-
*                                ted matrix.
*  MB_A   (global) desc_A[MB_]   The blocking factor used to distribute
*                                the rows of the matrix.
*  NB_A   (global) desc_A[NB_]   The blocking factor used to distribute
*                                the columns of the matrix.
*  RSRC_A (global) desc_A[RSRC_] The process row over which the first
*                                row of the matrix is distributed.
*  CSRC_A (global) desc_A[CSRC_] The process column over which the first
*                                column of the matrix is distributed.
*  CTXT_A (global) desc_A[CTXT_] The BLACS context handle, indicating
*                                the global context of the operation on
*                                the matrix.
*                                The context itself is global.
*  LLD_A  (local)  desc_A[LLD_]  The leading dimension of the local
*                                array storing the local blocks of the
*                                distributed matrix A.
*                                LLD_A >= MAX(1,LOCp(M_A)).
*
*  Let K be the number of rows or columns of a distributed matrix,
*  and assume that its process grid has dimension p x q.
*  LOCp( K ) denotes the number of elements of K that a process
*  would receive if K were distributed over the p processes of its
*  process column.
*  Similarly, LOCq( K ) denotes the number of elements of K that a
*  process would receive if K were distributed over the q processes of
*  its process row.
*  The values of LOCp() and LOCq() may be determined via a call to the
*  ScaLAPACK tool function, NUMROC.
*          LOCp( M ) = NUMROC( M, MB_A, MYROW, RSRC_A, NPROW ),
*          LOCq( N ) = NUMROC( N, NB_A, MYCOL, CSRC_A, NPCOL ).
*
*  Because vectors may be seen as particular matrices, a distributed
*  vector is considered to be a distributed matrix.
*
*  If INCX = 1 and INCX <> M_X, the process row having the first
*  entries of sub( X ) must also contain the first block of sub( A ),
*  similarly if INCY = M_Y, the process column having the first entries
*  of sub( Y ) must also contain the first block of sub( A ).
*
*  If INCX = M_X (resp. (INCX = 1 and INCX <> M_X)), the column (resp.
*  row) offset of sub( X ) must be equal to the row offset of sub( A ),
*  i.e MOD(JX-1,NB_X) (resp. MOD(IX-1,MB_X)) = MOD(IA-1,MB_A). Moreover,
*  the column (resp. row) blocksize of sub( X ) must be equal to the
*  row blocksize of sub( A ), i.e NB_X = MB_A (resp. MB_X = MB_A).
*
*  If INCY = M_Y (resp. (INCY = 1 and INCY <> M_Y)), the column (resp.
*  row) offset of sub( Y ) must be equal to the column offset of
*  sub( A ), i.e MOD(JY-1,NB_Y) (resp. MOD(IY-1,MB_Y)) = MOD(JA-1,NB_A).
*  Moreover, the column (resp. row) blocksize of sub( Y ) must be equal
*  to the column blocksize of sub( A ), i.e NB_Y = NB_A
*  (resp. MB_Y = NB_A).
*
*  Parameters
*  ==========
*
*  M       (global input) pointer to INTEGER
*          The number of rows to be operated on i.e the number of rows
*          of the distributed submatrix sub( A ). M >= 0.
*
*  N       (global input) pointer to INTEGER
*          The number of columns to be operated on i.e the number of
*          columns of the distributed submatrix sub( A ). N >= 0.
*
*  ALPHA   (global input) pointer to DOUBLE PRECISION
*          On entry, ALPHA specifies the scalar alpha.
*
*  X       (local input) DOUBLE PRECISION array containing the local
*          pieces of a distributed matrix of dimension of at least
*                  ( (JX-1)*M_X + IX + ( M - 1 )*abs( INCX ) )
*          This array contains the entries of the distributed vector
*          sub( X ).
*
*  IX      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix X to operate on.
*
*  JX      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix X to operate on.
*
*  DESCX   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix X.
*
*  INCX    (global input) pointer to INTEGER
*          The global increment for the elements of X. Only two values
*          of INCX are supported in this version, namely 1 and M_X.
*
*  Y       (local input) DOUBLE PRECISION array containing the local
*          pieces of a distributed matrix of dimension of at least
*                ( (JY-1)*M_Y + IY + ( N - 1 )*abs( INCY ) )
*          This array contains the entries of the distributed vector
*          sub( Y ).
*
*  IY      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix Y to operate on.
*
*  JY      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix Y to operate on.
*
*  DESCY   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix Y.
*
*  INCY    (global input) pointer to INTEGER
*          The global increment for the elements of Y. Only two values
*          of INCY are supported in this version, namely 1 and M_Y.
*
*  A       (local input/local output) DOUBLE PRECISION pointer into the
*          local memory to an array of dimension (LLD_A,LOCq(JA+N-1)).
*          Before entry, the leading M-by-N part of the distributed
*          matrix sub( A ) must contain the matrix of coefficients.
*          On exit, sub( A ) is overwritten by the updated distributed
*          matrix.
*
*  IA      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix A to operate on.
*
*  JA      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix A to operate on.
*
*  DESCA   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix A.
*
*  =====================================================================
*
*  .. Local Scalars ..
*/
   char        * ctop, * rtop, xdist, ydist;
   int         iacol, iarow, icoffa, ictxt, iia, iix, iiy, iroffa,
               ixcol, ixrow, iycol, iyrow, info, jja, jjx, jjy, lcm,
               lcmp, lcmq, locincx, locincy, mm, mp0, mycol, myrow, nca,
               nn, nprow, npcol, nra, nq0, tmp1, tmp2, wksz;
/* ..
*  .. PBLAS Buffer ..
*/
   double      * buff;
/* ..
*  .. External Functions ..
*/
   void        blacs_gridinfo_();
   void        pbchkvect();
   void        pbchkmat();
   void        pberror_();
   char        * getpbbuf();
   char        * ptop();
   F_VOID_FCT  pbdger_();
   F_INTG_FCT  ilcm_();
/* ..
*  .. Executable Statements ..
*
*  Get grid parameters
*/
   ictxt = desc_X[CTXT_];
   blacs_gridinfo_( &ictxt, &nprow, &npcol, &myrow, &mycol );
/*
*  Test the input parameters
*/
   info = 0;
   if( nprow == -1 )
      info = -707;
   else
   {
      iroffa = (*ia-1) % desc_A[MB_];
      icoffa = (*ja-1) % desc_A[NB_];
      pbchkvect( *m, 1, *ix, *jx, desc_X, *incx, 7, &iix, &jjx,
                 &ixrow, &ixcol, nprow, npcol, myrow, mycol, &info );
      pbchkvect( *n, 2, *iy, *jy, desc_Y, *incy, 12, &iiy, &jjy,
                 &iyrow, &iycol, nprow, npcol, myrow, mycol, &info );
      pbchkmat( *m, 1, *n, 2, *ia, *ja, desc_A, 17, &iia, &jja,
                &iarow, &iacol, nprow, npcol, myrow, mycol,
                &nra, &nca, &info );
      if( info == 0 )
      {
         if( *incx == desc_X[M_] )
         {
            if( ((*jx-1) % desc_X[NB_]) != iroffa )
               info = -15;
            else if( desc_A[MB_] != desc_X[NB_] )
               info = -1704;
         }
         else if( ( *incx == 1 ) && ( *incx != desc_X[M_] ) )
         {
            if( ( ((*ix-1) % desc_X[MB_]) != iroffa ) ||
                ( ixrow != iarow ) )
               info = -15;
            else if( desc_A[MB_] != desc_X[MB_] )
               info = -1703;
         }
         else
         {
            info = -8;
         }
         if( *incy == desc_Y[M_] )
         {
            if( ( ((*jy-1) % desc_Y[NB_]) != icoffa ) ||
                ( iycol != iacol ) )
               info = -16;
            else if( desc_A[NB_] != desc_Y[NB_] )
               info = -1704;
         }
         else if( ( *incy == 1 ) && ( *incy != desc_Y[M_] ) )
         {
            if( ((*iy-1) % desc_Y[MB_]) != icoffa )
               info = -16;
            else if( desc_A[NB_] != desc_Y[MB_] )
               info = -1703;
         }
         else
         {
            info = -13;
         }
         if( ictxt != desc_Y[CTXT_] )
            info = -1207;
         if( ictxt != desc_A[CTXT_] )
            info = -1707;
      }
   }
   if( info )
   {
      pberror_( &ictxt, "PDGER", &info );
      return;
   }
/*
*  Quick return if possible
*/
   if( ( *m == 0 ) || ( *n == 0 ) || ( *alpha == ZERO ) )
      return;
/*
*  Figure out the arguments to be passed to pbdger
*/
   mm = *m + iroffa;
   nn = *n + icoffa;
   lcm = ilcm_( &nprow, &npcol );
   if( *incx == desc_X[M_] )
   {
      xdist = 'R';
      locincx = desc_X[LLD_];
      tmp1 = mm / desc_A[MB_];
      mp0 = MYROC0( tmp1, *m, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, *n, desc_A[NB_], npcol );
      if( ( ixrow == iarow ) && ( (*m) <= ( desc_A[MB_]-iroffa ) ) )
         ixrow = -1;
      if( *incy == desc_Y[M_] )
      {
         lcmp = lcm / nprow;
         ydist = 'R';
         locincy = desc_Y[LLD_];
         tmp1 = mp0 / desc_A[MB_];
         wksz = mp0 + MYROC0( tmp1, mp0, desc_A[MB_], lcmp ) +
                   2*nq0;
      }
      else
      {
         lcmp = lcm / nprow;
         lcmq = lcm / npcol;
         ydist = 'C';
         locincy = 1;
         tmp1 = mp0 / desc_A[MB_];
         tmp2 = nq0 / desc_A[NB_];
         wksz = mp0 + nq0 + MYROC0( tmp1, mp0, desc_A[MB_], lcmp ) +
                   MYROC0( tmp2, nq0, desc_A[NB_], lcmq );
      }
   }
   else
   {
      xdist = 'C';
      locincx = 1;
      tmp1 = mm / desc_A[MB_];
      mp0 = MYROC0( tmp1, mm, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, nn, desc_A[NB_], npcol );
      if( ( ixcol == iacol ) && ( (*n) <= (desc_A[NB_]-icoffa ) ) )
         ixcol = -1;
      if( *incy == desc_Y[M_] )
      {
          ydist = 'R';
          locincy = desc_Y[LLD_];
          wksz = mp0 + nq0;
      }
      else
      {
          lcmq = lcm / npcol;
          ydist = 'C';
          locincy = 1;
          tmp2 = nq0 / desc_A[NB_];
          wksz = mp0 + nq0 + MYROC0( tmp2, nq0, desc_A[NB_], lcmq );
      }
   }
   buff = (double *)getpbbuf( "PDGER", wksz*sizeof(double) );
/*
*  Call PB-BLAS routine
*/
   if( xdist == 'C' && ydist == 'R' )
   {
      rtop = ptop( BROADCAST, ROW, TOPGET );
      ctop = ptop( BROADCAST, COLUMN, TOPGET );
      pbdger_( &ictxt, C2F_CHAR( &xdist ), C2F_CHAR( &ydist ), m, n,
               &desc_A[MB_], &desc_A[NB_], &iroffa, &icoffa, alpha,
               &X[iix-1+(jjx-1)*desc_X[LLD_]], &locincx,
               &Y[iiy-1+(jjy-1)*desc_Y[LLD_]], &locincy,
               &A[iia-1+(jja-1)*desc_A[LLD_]], &desc_A[LLD_],
               &ixrow, &ixcol, &iyrow, &iycol, &iarow, &iacol,
               C2F_CHAR( "A" ), C2F_CHAR( rtop ), C2F_CHAR( NO ),
               C2F_CHAR( ctop ), C2F_CHAR( NO ), buff );
   }
   else
   {
      pbdger_( &ictxt, C2F_CHAR( &xdist ), C2F_CHAR( &ydist ), m, n,
               &desc_A[MB_], &desc_A[NB_], &iroffa, &icoffa, alpha,
               &X[iix-1+(jjx-1)*desc_X[LLD_]], &locincx,
               &Y[iiy-1+(jjy-1)*desc_Y[LLD_]], &locincy,
               &A[iia-1+(jja-1)*desc_A[LLD_]], &desc_A[LLD_],
               &ixrow, &ixcol, &iyrow, &iycol, &iarow, &iacol,
               C2F_CHAR( "A" ), C2F_CHAR( TOPDEF ), C2F_CHAR( NO ),
               C2F_CHAR( TOPDEF ), C2F_CHAR( NO ), buff );
   }
}
