C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  (Stripped down PVM-only version (4/13/95), for use in ParkBench     #
C   benchmark suite)                                                   #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C swap.f                                                               C
C                                                                      C
C The following routines implement the following low level             C
C interprocessor communication commands used in PSTSWM using           C
C PVM interprocessor communication commands:                           C
C                                                                      C
C SWAP      - used when swapping data between two processors           C
C SWAP_SEND - used to send messages as part of a swap                  C
C SWAP_RECV - used to receive messages as part of a swap               C
C SWAP_RECVBEGIN - used when initiating a receive as part of a swap    C
C SWAP_RECVEND   - used when completing a receive as part of a swap    C
C SWAP1     - first of three routines that implement swap              C
C SWAP2     - second of three routines that implement swap             C
C SWAP3     - third of three routines that implement swap              C
C                                                                      C
C These routines all have the following communication options:         C
C  IF (COMMOPT .EQ. 0) simple                                          C
C  IF (COMMOPT .EQ. 1) ordered                                         C
C They also have the following communication protocol options          C
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send                 C
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive              C
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype                        C
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous                C
C PVM does not provide nonblocking communication commands. Protocol    C
C options involving nonblocking commands will cause the program to     C
C abort.                                                               C
C                                                                      C
C SWAP_SEND/SWAP_RECV, SWAP_SEND/SWAP_RECVBEGIN/SWAP_RECVEND, and      C
C SWAP1/SWAP2/SWAP3 all represent different ways of partitioning the   C
C basic swap routine.                                                  C
C                                                                      C
C Use of SWAP_SEND and SWAP_RECV allows computation to be "inserted"   C
C between the beginning of the send and the completion of the recv,    C
C but extreme care must be used when using these routines.             C
C If the simple option is used, then SWAP_SEND must precede SWAP_RECV. C
C If the ordered option is used, then                                  C
C IF (ORDER .EQ. 1) SWAP_SEND/SWAP_RECV                                C
C IF (ORDER .NE. 1) SWAP_RECV/SWAP_SEND                                C
C Note that SWAP_SEND/SWAP_RECV can be used to "match" SWAP in         C
C other processes.                                                     C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP(COMMOPT, PROTOPT, ORDER, ME, MTAG, 
     &                SWAPNODE, SNDLTH, SNDMSG, RCVLTH, RCVMSG)
C
C This subroutine sends the message in the SNDMSG buffer to SWAPNODE,
C and receives a message from SWAPNODE into RCVMSG. 
C
C Communication options (COMMOPT) for SWAP include:
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C Communication protocol options (PROTOPT) for SWAP include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH1, EXCH2, FRFFT1, HALF1, HALF2, IRFFT1, LGTRNS1, 
C            LGTRNS3
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      INTEGER SNDMSG(*)
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C data description parameters
      INTEGER DATATYPE, DATALTH
C
C---- Executable Statements --------------------------------------------
C
C     Get datatype and data length
      CALL DATADESC(DATATYPE, DATALTH)
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple swap: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         simple swap
          CALL PVMFINITSEND(PVMDEFAULT, F)
          CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
          CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP:',/,
     &             ' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered swap:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         ordered swap
          IF (ORDER .EQ. 1) THEN
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ELSE
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered swap 
          IF (ORDER .EQ. 1) THEN
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ELSE
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP_SEND(COMMOPT, PROTOPT, ORDER, ME, MTAG, 
     &                     SWAPNODE, SNDLTH, SNDMSG, RCVLTH, RCVMSG)
C
C This subroutine sends the message in the SNDMSG buffer to SWAPNODE 
C as part of a swap operation. The swap is completed after both 
C SWAP_SEND and SWAP_RECV have been called. If nonblocking sends are
C used and (COMMOPT .EQ. 0) .OR. ((COMMOPT .EQ. 1).AND.(ORDER .EQ. 1)), 
C then the send is not guaranteed to be complete until after SWAP_RECV
C is called.
C
C Communication options (COMMOPT) for SWAPSEND/SWAPRECV include:
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C Communication protocol options (PROTOPT) for SWAPSEND/SWAPRECV
C include: 
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH1, EXCH2, FRFFT2, HALF1, HALF2, IRFFT2
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      INTEGER SNDMSG(*)
C length of message to be received
C (only used for forcetype option)
      INTEGER RCVLTH
C
C     Output
C
C message received
C (only used for forcetype option)
      INTEGER RCVMSG(*)
C
C---- Local Variables -------------------------------------------------
C
C error return variable
      INTEGER F
C data description parameters
      INTEGER DATATYPE, DATALTH
C
C---- Executable Statements -------------------------------------------
C
C     Get datatype and data length
      CALL DATADESC(DATATYPE, DATALTH)
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       send of simple swap: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         send of simple swap
          CALL PVMFINITSEND(PVMDEFAULT, F)
          CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
          CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAPSEND:',/,
     &             ' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       send of ordered swap:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         send of ordered swap
          CALL PVMFINITSEND(PVMDEFAULT, F)
          CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
          CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         send of synchronous ordered swap 
          IF (ORDER .EQ. 1) THEN
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
          ELSE
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP_SEND:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP_RECV(COMMOPT, PROTOPT, ORDER, ME, MTAG, 
     &                     SWAPNODE, RCVLTH, RCVMSG)
C
C This subroutine receives a message into RCVMSG as part of a swap 
C operation. The swap is complete after both SWAP_SEND and SWAP_RECV 
C have been called.
C
C Communication options (COMMOPT) for SWAPSEND/SWAPRECV include:
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C Communication protocol options (PROTOPT) for SWAPSEND/SWAPRECV
C include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH1, EXCH2, FRFFT2, HALF1, HALF2, IRFFT2
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C data description parameters
      INTEGER DATATYPE, DATALTH
C
C---- Executable Statements --------------------------------------------
C
C     Get datatype and data length
      CALL DATADESC(DATATYPE, DATALTH)
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       receive of simple swap: send/recv
C***********************************************************************
C
        IF (PROTOPT .EQ. 0) THEN
C
C         receive of simple swap
          CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAPRECV:',/,
     &             ' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       receive of ordered swap:
C       IF (ORDER .EQ. 1) receive of send/recv
C       IF (ORDER .NE. 1) receive of recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         receive of ordered swap
          CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         receive of synchronous ordered swap 
          IF (ORDER .EQ. 1) THEN
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ELSE
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP_RECV:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP_RECVBEGIN(COMMOPT, PROTOPT, 
     &                          ORDER, ME, MTAG, SWAPNODE, RCVLTH, 
     &                          RCVMSG)
C
C This subroutine posts a request to receive a message into RCVMSG
C as part of a swap operation. The receive is completed in SWAP_RECVEND,
C and SWAP_RECVBEGIN/SWAP_RECVEND are used with SWAP_SEND to complete
C the swap.
C
C Communication options (COMMOPT) for SWAPSEND/SWAPRECVBEGIN/SWAPRECVEND 
C include: 
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C Communication protocol options (PROTOPT) for 
C SWAPSEND/SWAPRECVBEGIN/SWAPRECVEND include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH2, HALF2
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be received
      INTEGER RCVLTH
C buffer where message is to be received
      INTEGER RCVMSG(*)
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       recvbegin of simple swap: recvbegin (send/recvend)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C         SWAP_RECVEND posts the receive in this case.
        ELSE
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP_RECVBEGIN:',
     &           /,' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       recvbegin of ordered swap:
C       IF (ORDER .EQ. 1) recvbegin (send/recvend)
C       IF (ORDER .NE. 1) recvbegin (recvend/send)
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         SWAP_RECVEND posts the receive in this case.
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         no recvbegin in a "synchronous" ordered swap 
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP_RECVBEGIN:',
     &         /,' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP_RECVEND(COMMOPT, PROTOPT, ORDER, ME, 
     &                        MTAG, SWAPNODE, RCVLTH, RCVMSG)
C
C This subroutine waits until the message requested in SWAP_RECVBEGIN
C has arrived. SWAP_RECVBEGIN/SWAP_RECVEND are used with SWAP_SEND to 
C implement a swap.
C
C Communication options (COMMOPT) for SWAPSEND/SWAPRECVBEGIN/SWAPRECVEND
C include: 
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C Communication protocol options (PROTOPT) for 
C SWAPSEND/SWAPRECVBEGIN/SWAPRECVEND include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH2, HALF2
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C data description parameters
      INTEGER DATATYPE, DATALTH
C
C---- Executable Statements --------------------------------------------
C
C     Get datatype and data length
      CALL DATADESC(DATATYPE, DATALTH)
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       recvend of simple swap: (recvbegin/send) recvend
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         Complete "recvend" of blocking receive protocol.
          CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP_RECVEND:',
     &           /,' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       recvend of ordered swap:
C       IF (ORDER .EQ. 1) (recvbegin/send) recvend
C       IF (ORDER .NE. 1) (recvbegin) recvend (send)
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         Complete "recvend" of blocking receive protocol.
          CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         no recvbegin in "synchronous" ordered swap 
          IF (ORDER .EQ. 1) THEN
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ELSE
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP_RECVEND:',
     &         /,' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP1(COMMOPT, PROTOPT, SAFEFORCE, ORDER,
     &                 ME, MTAG, SWAPNODE, RCVLTH, RCVMSG)
C
C This subroutine begins a swap operation that will be completed by
C SWAP2 and SWAP3. It posts a receive and sends handshaking messages
C when forcetypes are used. 
C
C Communication options (COMMOPT) for SWAP1/SWAP2/SWAP3 include:
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C  IF (COMMOPT .EQ. 2) delayed-recv swap: send ... recv
C Communication protocol options (PROTOPT) for SWAP1/SWAP2/SWAP3
C include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH2, HALF2, LGTRNS2, LGTRNS4
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C send handshaking message to guarantee correctness of using forcetype
C protocol? (not used in PVM implementation)
      LOGICAL SAFEFORCE
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be received
      INTEGER RCVLTH
C buffers where messages are to be received
      INTEGER RCVMSG(*)
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF ((COMMOPT .EQ. 0) .OR. (COMMOPT .EQ. 2)) THEN
C***********************************************************************
C       simple swap: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         This library does not provide nonblocking receive.
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP1:',
     &           /,' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered swap:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         This library does not provide nonblocking receive.
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         Synchronous ordered swap does not use nonblocking receive.
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP1:',
     &         /,' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP2(COMMOPT, PROTOPT, SAFEFORCE, ORDER, 
     &                 ME, MTAG, SWAPNODE, SNDLTH, SNDMSG, RCVLTH, 
     &                 RCVMSG)
C
C This subroutine continues the swap operation begun in SWAP1. It
C initiates the send and waits for the receive to complete. 
C
C Communication options (COMMOPT) for SWAP1/SWAP2/SWAP3 include:
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C  IF (COMMOPT .EQ. 2) delayed-recv swap: send ... recv
C Communication protocol options (PROTOPT) for SWAP1/SWAP2/SWAP3
C include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH2, HALF2, LGTRNS2, LGTRNS4
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C send handshaking message to guarantee correctness of using forcetype
C protocol? (not necessary if user has other guarantees)
      LOGICAL SAFEFORCE
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      INTEGER SNDMSG(*)
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C data description parameters
      INTEGER DATATYPE, DATALTH
C
C---- Executable Statements --------------------------------------------
C
C     Get datatype and data length
      CALL DATADESC(DATATYPE, DATALTH)
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple swap: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         simple swap
          CALL PVMFINITSEND(PVMDEFAULT, F)
          CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
          CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP2:',
     &           /,' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered swap:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         ordered swap
          IF (ORDER .EQ. 1) THEN
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ELSE
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered swap 
          IF (ORDER .EQ. 1) THEN
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ELSE
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFINITSEND(PVMDEFAULT, F)
            CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
            CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
            CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
C     Choose communication algorithm.
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       dealyed swap: send ... recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         swap send
          CALL PVMFINITSEND(PVMDEFAULT, F)
          CALL PVMFPACK(DATATYPE, SNDMSG, SNDLTH/DATALTH, 1, F)
          CALL PVMFSEND(TIDS(SWAPNODE), MTAG, F)
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP2:',
     &         /,' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SWAP3(COMMOPT, PROTOPT, ME, MTAG, SWAPNODE,
     &                 RCVLTH, RCVMSG)
C
C This subroutine completes the swap operation begun in SWAP1 and SWAP2.
C It waits until the send and receive request made in SWAP2 have
C completed.
C
C Communication options (COMMOPT) for SWAP1/SWAP2/SWAP3 include:
C  IF (COMMOPT .EQ. 0) simple swap: send/recv
C  IF (COMMOPT .EQ. 1) ordered swap: [send/recv]|[recv/send]
C  IF (COMMOPT .EQ. 2) delayed-recv swap: send ... recv
C Communication protocol options (PROTOPT) for SWAP1/SWAP2/SWAP3 
C include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C PVM does not provide nonblocking communication commands. Protocol 
C options involving nonblocking commands will cause the program to     
C abort.                                                               
C
C called by: EXCH2, HALF2
C calls: PVM routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C processor id
      INTEGER ME
C message tag
      INTEGER MTAG
C processor swapping messages with
      INTEGER SWAPNODE
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      INTEGER RCVMSG(*)
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C data description parameters
      INTEGER DATATYPE, DATALTH
C
C---- Executable Statements --------------------------------------------
C
C     Get datatype and data length
      CALL DATADESC(DATATYPE, DATALTH)
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple swap: send/recv
C***********************************************************************
C
C       Choose communication protocol.
C
        IF (PROTOPT .EQ. 0) THEN
C         Nothing left to do when using PVM commands.
        ELSE
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP3:',
     &           /,' COMMUNICATION PROTOCOL UNSUPPORTED IN PVM',/,
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered swap:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (deadlock avoidance is the responsibility of the calling
C        program)
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 0) .OR. (PROTOPT .EQ. 6)) THEN
C         Nothing left to do when using PVM commands.
        ELSE
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv swap: recvbegin ... send ... recvend
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .EQ. 0) THEN
C
C         Receive message.
          CALL PVMFRECV(TIDS(SWAPNODE), MTAG, F)
          CALL PVMFUNPACK(DATATYPE, RCVMSG, RCVLTH/DATALTH, 1, F)
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined swap option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SWAP3:',
     &         /,' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
