C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C picl.F                                                               C
C                                                                      C
C These routines implement the following PICL 2.0 commands using       C
C PVM commands:                                                        C
C                                                                      C
C BCAST0F   - broadcast                                                C
C CHECK0F   - enable/disable PICL-level error checking          (NOP)  C
C CLOCKSYNC0- synchronize clocks                                       C
C CLOSE0F   - disable interprocess communication                (NOP)  C
C GETDATA0F - query datatype of message being communicated      (NOP)  C
C GMAX0F    - global vector maximum                                    C
C GMIN0F    - global vector minimum                                    C
C GRAY0F    - forward gray code mapping                                C
C OPEN0F    - enable interprocess communication                        C
C RECV0F    - receive a tagged message                          (NOP)  C
C SEND0F    - send a tagged message                             (NOP)  C
C SETDATA0F - set datatype of message being communicated        (NOP)  C
C TRACESTATISTICSF - enable collection of profile data          (NOP)  C
C TRACENODEF       - enable collection of trace data            (NOP)  C
C TRACEDATAF       - save data associated with specified event  (NOP)  C
C TRACEEVENTF      - mark occurrence of an event                (NOP)  C
C TRACEEXITF       - disable collection of trace data           (NOP)  C
C TRACEFILESF      - open files for saving trace data           (NOP)  C
C TRACEFLUSHF      - send trace data to disk                    (NOP)  C
C TRACELEVELSF     - specify what and how much trace data to collect   C
C                                                               (NOP)  C
C WHO0F     - number of processors, process id, host id query          C
C                                                                      C
C The PICL commands are used in the initialization, timing, and error  C
C analysis sections of the code. The intention is that the performance C
C sensitive code (in swap.F and sendrecv.F) be implemented directly    C
C using native commands. The routines in picl.F are then used to       C
C eliminate the remaining dependence on the PICL library. The reasons  C
C for this are twofold:                                                C
C a) On some machines, the swap and sendrecv routines can be           C
C    implemented much more efficiently by not using PICL message       C
C    passing.                                                          C
C b) Porting PSTSWM to a new platform need not depend on porting       C
C    the full PICL library first.                                      C
C Note that                                                            C
C  1) PICL is a superset of the PVM message passing semantics, and     C
C     no efficiency is given up by using PICL. These routines are      C
C     provided simply for consistency when comparing timings with otherC
C     platforms for which PICL is not efficient.                       C
C  2) Note that the full PICL library must be used in order to trace   C
C     or profile the performance of the code.                          C
C  3) These routines do not represent safe implementations of the PICL C
C     commands outside of PSTSWM. The contexts in which the commands   C
C     are called are used to keep the code simple.                     C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE BCAST0F(BUF, BUFSIZE, MSGTYPE, ROOT)
C
C This routine broadcasts a message buf of given length and type  from
C the root to all processors.
C
C called by: ALGINP, GLOBALMAX, GLOBALMIN, INPUT, MESH_MAP_INIT,
C            RING_MAP_INIT, PRBINP, PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF BYTES IN BUFFER
      INTEGER BUFSIZE
C COMMUNICATION BUFFER
      CHARACTER*1 BUF(BUFSIZE)
C MESSAGE TYPE
      INTEGER MSGTYPE
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C data description parameters
      INTEGER DATATYPE, DATALTH
C
C---- Executable Statements -------------------------------------------
C
C     Get datatype and data length
      CALL DATADESC(DATATYPE, DATALTH)
C
      IF (PICLME .EQ. ROOT) THEN
        CALL PVMFINITSEND(PVMDEFAULT, F)
        CALL PVMFPACK(DATATYPE, BUF, BUFSIZE/DATALTH, 1, F)
        CALL PVMFMCAST(NPROCS, TIDS, MSGTYPE, F)
      ELSE
        CALL PVMFRECV(TIDS(ROOT), MSGTYPE, F)
        CALL PVMFUNPACK(DATATYPE, BUF, BUFSIZE/DATALTH, 1, F)
      ENDIF
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE CHECK0F(CHECK)
C
C This function enables or disables PICL error checking. In the native 
C PVM implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C TURN CHECKING ON (1) OR OFF (0)
      INTEGER CHECK
C
C---- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE CLOCKSYNC0F()
C
C This function synchronizes the processors. It does not attempt to
C synchronize the clocks, because event tracing is only supported in the
C PICL implementation.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C
C---- Executable Statements --------------------------------------------
C
      CALL PVMFBARRIER('pstswm', NPROCS, F)
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE CLOSE0F()
C
C This function disables interprocess communication. 
C
C called by: PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C
C---- Executable Statements --------------------------------------------
C
      CALL PVMFLVGROUP('pstswm', F)
      CALL PVMFEXIT(F)
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GETDATA0F(SAVETYPE)
C
C This function returns the data type specified by the last call to
C SETDATA0F. 
C
C called by: MESH_MAP_INIT, RING_MAP_INIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Output
C 
C Current data type
      CHARACTER*1 SAVETYPE(1)
C
C----- Executable Statements -------------------------------------------
C
      SAVETYPE(1) = PICLTYPE(1)
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GMAX0F(DATA, NDATA, DTYPE, MSGTYPE, ROOT)
C
C This routine computes a componentwise maximum of a vector over all
C processors. This implementation of the routine is sufficient for use 
C with the current implementation of PSTSWM, but only datatypes 2, 4, 
C and 5 are needed in PSTSWM, and only these are supported here. 
C
C called by: GLOBALMAX, INPUT, TOUTPUT
C calls: GCMPI, GCMPR, GCMPD
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF ITEMS IN DATA ARRAY
      INTEGER NDATA
C DATA BUFFER
      INTEGER DATA(*)
C DATA TYPE
C  1: INTEGER*2
C  2: INTEGER
C  3: INTEGER*4
C  4: REAL*4
C  5: REAL*8
      INTEGER DTYPE
C MESSAGE TYPE
      INTEGER MSGTYPE
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C
C---- External Routines ------------------------------------------------
C
C reduction function
      EXTERNAL PvmMax
C
C----- Executable Statements -------------------------------------------
C
      IF (DTYPE .EQ. 2) THEN
        CALL PVMFREDUCE(PvmMax, DATA, NDATA, INTEGER4, MSGTYPE, 
     &                  'pstswm', ROOT, F)
      ELSEIF (DTYPE .EQ. 4) THEN
        CALL PVMFREDUCE(PvmMax, DATA, NDATA, REAL4, MSGTYPE, 
     &                  'pstswm', ROOT, F)
      ELSEIF (DTYPE .EQ. 5) THEN
        CALL PVMFREDUCE(PvmMax, DATA, NDATA, REAL8, MSGTYPE, 
     &                  'pstswm', ROOT, F)
      ELSE
        WRITE(0,901) PICLME, DTYPE
  901   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE GMAX0F: ',
     &          /,' BAD DATATYPE FOR PVM PICL EMULATION MODE ',
     &          /,' (ONLY TYPES 2, 4, AND 5 SUPPORTED) ',
     &          /,' ME = ',I4,' DATATYPE = ',I4)
        CALL PVMFEXIT()
        STOP
      ENDIF
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE GMIN0F(DATA, NDATA, DTYPE, MSGTYPE, ROOT)
C
C This routine computes a componentwise minimum of a vector over all
C processors. This implementation of the routine is sufficient for use 
C with the current implementation of PSTSWM, but only datatypes 2, 4, 
C and 5 are needed in PSTSWM, and only these are supported here. 
C
C called by: GLOBALMIN, TOUTPUT
C calls: GCMPI, GCMPR, GCMPD
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
#     include "fpvm3.h"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF ITEMS IN DATA ARRAY
      INTEGER NDATA
C DATA BUFFER
      INTEGER DATA(*)
C DATA TYPE
C  1: INTEGER*2
C  2: INTEGER
C  3: INTEGER*4
C  4: REAL*4
C  5: REAL*8
      INTEGER DTYPE
C MESSAGE TYPE
      INTEGER MSGTYPE
C ROOT 
      INTEGER ROOT
C
C---- Local Variables --------------------------------------------------
C
C error return variable
      INTEGER F
C
C---- External Routines ------------------------------------------------
C
C reduction function
      EXTERNAL PvmMin
C
C----- Executable Statements -------------------------------------------
C
      IF (DTYPE .EQ. 2) THEN
        CALL PVMFREDUCE(PvmMin, DATA, NDATA, INTEGER4, MSGTYPE, 
     &                  'pstswm', ROOT, F)
      ELSEIF (DTYPE .EQ. 4) THEN
        CALL PVMFREDUCE(PvmMin, DATA, NDATA, REAL4, MSGTYPE, 
     &                  'pstswm', ROOT, F)
      ELSEIF (DTYPE .EQ. 5) THEN
        CALL PVMFREDUCE(PvmMin, DATA, NDATA, REAL8, MSGTYPE, 
     &                  'pstswm', ROOT, F)
      ELSE
        WRITE(0,901) PICLME, DTYPE
  901   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE GMIN0F: ',
     &          /,' BAD DATATYPE FOR PVM PICL EMULATION MODE ',
     &          /,' (ONLY TYPES 2, 4, AND 5 SUPPORTED) ',
     &          /,' ME = ',I4,' DATATYPE = ',I4)
        CALL PVMFEXIT()
        STOP
      ENDIF
C
      RETURN
C
      END
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      INTEGER FUNCTION GRAY0F(INDEX)
C
C This routine returns the binary-reflected Gray code for the input.
C
C called by: MESH_MAP_INIT, RING_MAP_INIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C INDEX
      INTEGER INDEX
C
C---- External Functions ----------------------------------------------
C
C Exclusive OR
      INTEGER XOR
C
C----- Executable Statements -------------------------------------------
C
      GRAY0F = XOR(INDEX/2,INDEX)
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE OPEN0F(NUMPROCS,ME,HOST)
C
C This routine enables interprocess communication. It also returns the
C number of processors, local process id, and host process id.
C
C called by: PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Output
C 
C NUMBER OF PROCESSORS
      INTEGER NUMPROCS
C PROCESSOR ID
      INTEGER ME
C HOST ID
      INTEGER HOST
C
C---- Local Variables --------------------------------------------------
C
C HOST ID
      INTEGER HOST
C
C----- Executable Statements -------------------------------------------
C
C     Determine number of processors and various ids
      CALL OPEN0(NPROCS, PICLME, HOST)
C     Initialize TIDS array
      CALL GETTIDS(TIDS)
C     Return values
      NUMPROCS = NPROCS
      ME = PICLME
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RECV0F(BUF, BUFSIZE, TYPE)
C
C This routine blocks until a message with the indicated message tag
C is received. In the native PVM implementation of PSTSWM, it does 
C nothing. (It is used only in the handshaking when saving trace data.)
C
C called by: TOUTPUT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF BYTES IN BUFFER
      INTEGER BUFSIZE
C COMMUNICATION BUFFER
      CHARACTER*1 BUF(BUFSIZE)
C MESSAGE TYPE
      INTEGER TYPE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SEND0F(BUF, BUFSIZE, MSGTYPE, DEST)
C
C This routine sends a message with message tag TYPE to DEST.
C In the native PVM implementation of PSTSWM, it does nothing.
C (It is used only in the handshaking when saving trace data.)
C
C called by: TOUTPUT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C NUMBER OF BYTES IN BUFFER
      INTEGER BUFSIZE
C COMMUNICATION BUFFER
      CHARACTER*1 BUF(BUFSIZE)
C MESSAGE MSGTYPE
      INTEGER MSGTYPE
C MESSAGE DESTINATION
      INTEGER DEST
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SETDATA0F(NEWTYPE)
C
C This function sets the data type for subsequent interprocess 
C communication calls. 
C
C called by: ALGINP, FLTSUM, ILTCAST, MESH_MAP_INIT, PRBINP,
C            PRFINP, PSTSWM, RFTLON, RING_MAP_INIT, TIMING
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C pvm implementation-specific parameters
#     include "fpvm3.h"
C pstswm-specific parameters
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C 
C Current data type
      CHARACTER*1 NEWTYPE(1)
C
C----- Executable Statements -------------------------------------------
C
      PICLTYPE(1) = NEWTYPE(1)
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACESTATISTICSF(EVENTS, PICLTIME, PICLCNT, PICLVOL,
     &                           USERTIME, USERCNT)
C
C This routine enables the collection of profile data. In the native 
C PVM implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C NUMBER OF EVENTS
      INTEGER EVENTS
C PROFILING LEVELS
      INTEGER PICLTIME, PICLCNT, PICLVOL, USERTIME, USERCNT
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACENODEF(TRACESIZE, FLUSH, SYNC)
C
C This routine enables the collection of trace data. In the native 
C PVM implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C TRACE BUFFER SIZE
      INTEGER TRACESIZE
C FLUSH OPTION
      INTEGER FLUSH
C SYNC OPTION
      INTEGER SYNC
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEDATAF(EVENTTYPE, DATAID, DATATYPE, SIZE, DATA)
C
C This routine saves data associated with a given user event. In the
C native PVM implementation of PSTSWM, it does nothing.
C
C called by: TINIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C EVENT TYPE
      INTEGER EVENTTYPE
C DATA ID
      INTEGER DATAID
C DATA TYPE
      CHARACTER*1 DATATYPE(1)
C DATA SIZE
      INTEGER SIZE
C DATA
      INTEGER DATA(*)
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEEVENTF(RECORDSTRING, EVENT, NPARAMS, PARAMS)
C
C This routine marks the occurrence of an event. In the native PVM
C implementation of PSTSWM, it does nothing.
C
C called by: ADVECT, EXPLIC, FLTSUM, ILTCAST, SIMPLIC, TRANSPOSE,
C            TSTART, TSTOP
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C RECORD STRING TYPE
      CHARACTER*1 RECORDSTRING(1)
C EVENT ID
      INTEGER EVENT
C NUMBER OF PARAMETERS
      INTEGER NPARAMS
C PARAMETERS
      INTEGER PARAMS(NPARAMS)
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEEXITF()
C
C This routine disables the collection of trace data. In the native 
C PVM implementation of PSTSWM, it does nothing.
C
C called by: TEXIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEFILESF(TEMPFILE, PERMFILE, VERBOSE)
C
C This routine opens disk files for saving trace data.
C In the native PVM implementation of PSTSWM, it does nothing.
C
C called by: TEXIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C NAME OF TEMPORARY FILE
      CHARACTER*1 TEMPFILE(1)
C NAME OF PERMANENT FILE
      CHARACTER*1 PERMFILE(1)
C TRACE MODE
      INTEGER VERBOSE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACEFLUSHF()
C
C This routine sends trace data to temporary or permanent disk storage.
C In the native PVM implementation of PSTSWM, it does nothing.
C
C called by: TOUTPUT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TRACELEVELF(PICL, USER, TRACE)
C
C This routine specifes the types and amounts of trace data to collect.
C In the native PVM implementation of PSTSWM, it does nothing.
C
C called by: TINIT, TSTART
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C LEVEL OF PICL EVENT TRACING
      INTEGER PICL
C LEVEL OF USER EVENT TRACING
      INTEGER USER
C LEVEL OF TRACE EVENT TRACING
      INTEGER TRACE
C
C----- Executable Statements -------------------------------------------
C
      RETURN
C
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE WHO0F(NUMPROCS, ME, HOST)
C
C This routine returns the number of processors, local process id, and 
C host process id. 
C
C called by: MESH_MAP_INIT, RING_MAP_INIT
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C picl/pvm interface variables
#     include "picl.i"
C
C---- Arguments --------------------------------------------------------
C
C     Output
C 
C NUMBER OF PROCESSORS
      INTEGER NUMPROCS
C PROCESSOR ID
      INTEGER ME
C HOST ID
      INTEGER HOST
C
C----- Executable Statements -------------------------------------------
C
C     Return requested values to user
      NUMPROCS = NPROCS
      ME       = PICLME
      HOST     = 32767
C
      RETURN
C
      END
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
