C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  (Stripped down PVM-only version (4/13/95), for use in ParkBench     #
C   benchmark suite)                                                   #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C map.f                                                                C
C                                                                      C
C The following routines initialize and manipulate the                 C
C logical-to-physical processor map arrays. Four map arrays are        C
C currently supported: MESHMAP, ROWMAP, COLMAP, and RINGMAP.           C
C                                                                      C
C MESHMAP is an indirection array mapping from a logical 2-d mesh      C
C         ([0,ISIZE-1] x [0,JSIZE-1]) to the actual processor ids      C
C         ([0,NPROCS-1]). It is a 1-d array indexed by                 C
C         (IDEX + JDEX*ISIZE) for the element (IDEX,JDEX).             C
C ROWMAP  is an indirection array mapping from a specified logical row C
C         to the actual processor ids. It is a subset of MESHMAP.      C
C COLMAP  is an indirection array mapping from a specified logical     C
C         column to the actual processor ids. It is a subset of        C
C         MESHMAP.                                                     C
C RINGMAP is an indirection array mapping from a logical 1-d ring      C
C         ([0,NPROCS-1]) to the actual processor ids ([0,NPROCS-1]).   C
C                                                                      C
C To define MESHMAP, call MESH_MAP_INIT.                               C
C To define RINGMAP, call RING_MAP_INIT.                               C
C To define ROWMAP, first call MESH_MAP_INIT, then call ROW_MAP_INIT.  C
C To define COLMAP, first call MESH_MAP_INIT, then call COL_MAP_INIT.  C
C To identify the mesh coordinates of a processor, call MESH_INDEX.    C
C To identify the ring coordinates of a processor, call RING_INDEX.    C
C                                                                      C
C As a convenience, the functions MESH_MAP and RING_MAP "read" the     C
C corresponding map arrays and return the processor ids for a given    C
C logical coordinate.                                                  C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE MESH_MAP_INIT(OPTION, ISIZE, JSIZE, MESHMAP)
C                                                                              
C This routine initializes MESHMAP, which defines the mapping
C from the logical processor mesh [0,ISIZE-1] X [0.JSIZE-1] to the 
C "physical" processors numbered 0 to ISIZE*JSIZE-1. The mapping is
C indicated by OPTION. Currently 9 options are supported:
C (5):  (I,J) <- ((J*IX)/ISIZE)*IY*ISIZE + MOD(J*IX,ISIZE)
C                + (I/IX)*ISIZE + MOD(I,IX)
C       where IY = SQRT(JSIZE) or SQRT(ISIZE) and IX = ISIZE/IY
C       (mapping I coordinate to a subgrid in logical ISIZE X JSIZE 
C        grid)
C (4):  (I,J) <- GRAY0F(MOD(I+ISIZE,ISIZE)) 
C                + GRAY0F(MOD(J+JSIZE,JSIZE))*ISIZE
C (3):  (I,J) <- GRAY0F(MOD(I+ISIZE,ISIZE)) + MOD(J+JSIZE,JSIZE)*ISIZE
C (2):  (I,J) <- MOD(I+ISIZE,ISIZE) + GRAY0F(MOD(J+JSIZE,JSIZE))*ISIZE
C (1):  (I,J) <- I + J*ISIZE 
C (0):  read mapping from the file meshmap
C (-1): (I,J) <- J + I*JSIZE 
C (-2): (I,J) <- GRAY0F(MOD(J+JSIZE,JSIZE)) + MOD(I+ISIZE,ISIZE)*JSIZE
C (-3): (I,J) <- MOD(J+JSIZE,JSIZE) + GRAY0F(MOD(I+ISIZE,ISIZE))*JSIZE
C (-4): (I,J) <- GRAY0F(MOD(J+JSIZE,JSIZE)) 
C                + GRAY0F(MOD(I+ISIZE,ISIZE))*JSIZE
C (-5): (I,J) <- ((I*IX)/JSIZE)*IY*JSIZE + MOD(I*IX,JSIZE)
C                + (J/IX)*JSIZE + MOD(J,IX)
C       where IY = SQRT(ISIZE) or SQRT(JSIZE) and IX = JSIZE/JY
C       (mapping J coordinate to a subgrid in logical ISIZE X JSIZE 
C        grid)
C
C called by: ALGINP
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C initialization option
      INTEGER OPTION
C dimensions of logical processor mesh
      INTEGER ISIZE, JSIZE
C
C     Output
C
C logical-to-physical processor mapping
      INTEGER MESHMAP(0:ISIZE-1,0:JSIZE-1)
C
C---- Local Variables --------------------------------------------------
C
C processor information
      INTEGER ME, HOST, NPROCS
C loop indices
      INTEGER I, J
C temporaries used to map a coordinate to a block of processors
      INTEGER IX, IY, ITMP, OFFSET
      REAL RSIZE
C message type used to communicate a mapping read off disk
C (OPTION .EQ. 0)
      INTEGER MSGTYPE
C external data type
      CHARACTER*4 SAVETYPE
C
C---- External Functions -----------------------------------------------
C
C offset for message types
      EXTERNAL MSGBASE
      INTEGER MSGBASE
C Gray code 
      EXTERNAL GRAY0F
      INTEGER GRAY0F
C
C---- Executable Statements --------------------------------------------
C
      CALL WHO0F(NPROCS, ME, HOST)
C
      IF (OPTION .EQ. 0) THEN
C       Read in mapping from file meshmap.
C
        CALL GETDATA0F(SAVETYPE)
        CALL SETDATA0F('INTEGER')
C
        MSGTYPE = MSGBASE()
        IF (ME .EQ. 0) THEN
C
          WRITE(6,200)
  200     FORMAT(/,' READING PARAMETERS FROM FILE meshmap:',/)
C
C         Read mapping input file.
          OPEN(8, FILE='meshmap', STATUS='OLD')
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              READ(8,*) MESHMAP(I,J)
            ENDDO
          ENDDO
C
C         Close input file.
          CLOSE(8)
C
C         Send information to other processors.
          IF (NPROCS .GT. 1) 
     &      CALL BCAST0F(MESHMAP, IBYTES*ISIZE*JSIZE, MSGTYPE, 0)
C
        ELSE
C
C         Get information from node 0.
          CALL BCAST0F(MESHMAP, IBYTES*ISIZE*JSIZE, MSGTYPE, 0)
C
        ENDIF
C
        CALL SETDATA0F(SAVETYPE)
C
      ELSEIF (ABS(OPTION) .EQ. 1) THEN
C       row linear/column linear ordering
C       (Note: this preserves the topology of the logical grid when
C        mapped to a physical mesh of the same aspect ratio.)
C
        IF (OPTION .GT. 0) THEN
C         row major ordering
C
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = I + J*ISIZE
            ENDDO
          ENDDO
C
        ELSE
C         column major ordering
C
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = J + I*JSIZE
            ENDDO
          ENDDO
C
        ENDIF
C
C
      ELSEIF (ABS(OPTION) .EQ. 2) THEN
C       row linear/column Gray code ordering
C       (Note: this is useful in distributed or log P FFT/ring LT
C        algorithms on hypercube architectures. In the FFT, swaps are
C        performed between processors that are nearest neighbors when
C        using a linear ordering. In the LT ring algorithms, logical
C        neighbors are mapped to physical neighbors when using a Gray
C        code.)  
C
        IF (OPTION .GT. 0) THEN
C         row major ordering
C      
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = MOD(I+ISIZE,ISIZE)
     &                     + GRAY0F(MOD(J+JSIZE,JSIZE))*ISIZE
            ENDDO
          ENDDO
C
        ELSE
C         column major ordering
C      
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = GRAY0F(MOD(J+JSIZE,JSIZE))
     &                     + MOD(I+ISIZE,ISIZE)*JSIZE
            ENDDO
          ENDDO
C
        ENDIF
C
      ELSEIF (ABS(OPTION) .EQ. 3) THEN
C       row Gray code/column linear ordering
C
        IF (OPTION .GT. 0) THEN
C         row major ordering
C      
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = GRAY0F(MOD(I+ISIZE,ISIZE))
     &                     + MOD(J+JSIZE,JSIZE)*ISIZE
            ENDDO
          ENDDO
C
        ELSE
C         column major ordering
C      
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = MOD(J+ISIZE,JSIZE)
     &                     + GRAY0F(MOD(I+ISIZE,ISIZE))*JSIZE
            ENDDO
          ENDDO
C
        ENDIF
C
      ELSEIF (ABS(OPTION) .EQ. 4) THEN
C       row Gray code/column Gray code ordering
C
        IF (OPTION .GT. 0) THEN
C         row major ordering
C
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = GRAY0F(MOD(I+ISIZE,ISIZE))
     &                     + GRAY0F(MOD(J+JSIZE,JSIZE))*ISIZE
            ENDDO
          ENDDO
C
        ELSE
C         column major ordering
C
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              MESHMAP(I,J) = GRAY0F(MOD(J+JSIZE,JSIZE))
     &                     + GRAY0F(MOD(I+ISIZE,ISIZE))*JSIZE
            ENDDO
          ENDDO
C
        ENDIF
C
      ELSEIF (ABS(OPTION) .EQ. 5) THEN
C
        IF (OPTION .GT. 0) THEN
C         Mapping I coordinate to logical X x Y blocks.
C
          RSIZE = ISIZE
          IY = SQRT(RSIZE)
          IX = ISIZE/IY
          DO WHILE (((IX*IY .NE. ISIZE) .OR.
     &               (IY .GT. JSIZE) .OR.
     &               (MOD(JSIZE,IY) .NE. 0)) .AND.
     &               (IY .GT. 1))
            IY = IY - 1
            IX = ISIZE/IY
          ENDDO
C
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              ITMP    = (J*IX)/ISIZE
              OFFSET  = ITMP*IY*ISIZE + MOD(J*IX,ISIZE)
              ITMP    = (I/IX)
              MESHMAP(I,J) = OFFSET + ITMP*ISIZE + MOD(I,IX)
            ENDDO
          ENDDO
C
        ELSE
C         Mapping J coordinate to logical X x Y blocks.
C
          RSIZE = JSIZE
          IY = SQRT(RSIZE)
          IX = JSIZE/IY
          DO WHILE (((IX*IY .NE. JSIZE) .OR.
     &               (IY .GT. ISIZE) .OR.
     &               (MOD(ISIZE,IY) .NE. 0)) .AND.
     &               (IY .GT. 1))
            IY = IY - 1
            IX = JSIZE/IY
          ENDDO
C
          DO J=0,JSIZE-1
            DO I=0,ISIZE-1
              ITMP    = (I*IX)/JSIZE
              OFFSET  = ITMP*IY*JSIZE + MOD(I*IX,JSIZE)
              ITMP    = (J/IX)
              MESHMAP(I,J) = OFFSET + ITMP*JSIZE + MOD(J,IX)
            ENDDO
          ENDDO
C
        ENDIF
C
      ELSE
C
C       specified a bad option
C
        IF (ME .EQ. 0) WRITE(0,202) OPTION
 202    FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE MESH_MAP_INIT ',/,
     &            ' SPECIFIED MAPPING OPTION NOT DEFINED',/,
     &            ' OPTION = ',I4)
        STOP
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RING_MAP_INIT(OPTION, SIZE, RINGMAP)
C                                                                              
C This routine initializes RINGMAP, which defines the mapping from the
C logical processor ring [0,SIZE-1] to the "physical" processors
C numbered 0 to SIZE-1. The mapping is indicated by OPTION. Currently 3
C options are supported:
C (0): read mapping from the file ringmap
C (1): (i) <- i 
C (2): (i) <- GRAY0F(i)
C
C called by: ALGINP
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C initialization option
      INTEGER OPTION
C dimension of logical processor ring
      INTEGER SIZE
C
C     Output
C
C logical-to-physical processor mapping
      INTEGER RINGMAP(0:SIZE-1)
C
C---- Local Variables --------------------------------------------------
C
C processor information
      INTEGER ME, HOST, NPROCS
C loop index
      INTEGER I
C message type used to communicate a mapping read off disk
C (OPTION .EQ. 0)
      INTEGER MSGTYPE
C external data type
      CHARACTER*4 SAVETYPE
C
C---- External Functions -----------------------------------------------
C
C offset for message types
      EXTERNAL MSGBASE
      INTEGER MSGBASE
C Gray code 
      EXTERNAL GRAY0F
      INTEGER GRAY0F
C
C---- Executable Statements --------------------------------------------
C
      CALL WHO0F(NPROCS, ME, HOST)
C
      IF (OPTION .EQ. 0) THEN
C       Read in mapping from file ringmap.
C
        CALL GETDATA0F(SAVETYPE)
        CALL SETDATA0F('INTEGER')
C
        MSGTYPE = MSGBASE()
        IF (ME .EQ. 0) THEN
C
          WRITE(6,300)
  300     FORMAT(/,' READING PARAMETERS FROM FILE ringmap:',/)
C
C         Read mapping input file.
          OPEN(8, FILE='ringmap')
          DO I=0,SIZE-1
            READ(8,*) RINGMAP(I)
          ENDDO
C
C         Close input file.
          CLOSE(8)
C
C         Send information to other processors.
          IF (NPROCS .GT. 1) 
     &      CALL BCAST0F(RINGMAP, IBYTES*SIZE, MSGTYPE, 0)
C
        ELSE
C
C         Get information from node 0.
          CALL BCAST0F(RINGMAP, IBYTES*SIZE, MSGTYPE, 0)
C
        ENDIF
C
        CALL SETDATA0F(SAVETYPE)
C
      ELSEIF (OPTION .EQ. 1) THEN
C
C       Identity map ordering.
        DO I=0,SIZE-1
          RINGMAP(I) = I
        ENDDO
C
      ELSEIF (OPTION .EQ. 2) THEN
C       Gray code ordering
C
        DO I=0,SIZE-1
          RINGMAP(I) = GRAY0F(I)
        ENDDO
C
      ELSE
C       specified a bad option
C
        IF (ME .EQ. 0) WRITE(0,301) OPTION
  301   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE RING_MAP_INIT ',/,
     &            ' SPECIFIED MAPPING OPTION NOT DEFINED',/,
     &            ' OPTION = ',I4)
        STOP
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE MESH_INDEX(P, ISIZE, JSIZE, MESHMAP, IDEX, JDEX)
C                                                                              
C This routine identifies the location of a given processor in the
C logical processor mesh by searching through the map array MESHMAP.
C
C called by: ALGINP
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C processor id
      INTEGER P
C dimensions of logical processor mesh
      INTEGER ISIZE, JSIZE
C logical mesh-to-physical processor mapping
      INTEGER MESHMAP(0:ISIZE-1,0:JSIZE-1)
C
C     Output
C
C coordinates (IDEX,JDEX) of current processor in logical mesh
      INTEGER IDEX, JDEX
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
      IDEX = -1
      JDEX = -1
      DO J=0,JSIZE-1
        DO I=0,ISIZE-1
          IF (MESHMAP(I,J) .EQ. P) THEN
            IDEX = I
            JDEX = J
          ENDIF
        ENDDO
      ENDDO
C
C     check for successful completion of search
      IF ((IDEX .EQ. -1) .OR. (JDEX .EQ. -1)) THEN
        WRITE(0,401) P
  401   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE MESH_INDEX ',/,
     &            ' UNABLE TO FIND PROCESSOR IN MESHMAP ARRAY',/,
     &            ' PROCESSOR = ',I4)
        STOP
      ENDIF
C
      RETURN
      END
C
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RING_INDEX(P, SIZE, RINGMAP, INDEX)
C                                                                              
C This routine identifies the location of a given processor in the
C logical processor ring by searching through the map array RINGMAP.
C
C called by: ALGINP
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C processor id
      INTEGER P
C dimension of logical processor ring
      INTEGER SIZE
C logical ring-to-physical processor mapping
      INTEGER RINGMAP(0:SIZE-1)
C
C     Output
C
C index of current processor in logical ring
      INTEGER INDEX
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER I
C
C---- Executable Statements --------------------------------------------
C
      INDEX = -1
      DO I=0,SIZE-1
        IF (RINGMAP(I) .EQ. P) INDEX = I
      ENDDO
C
C     check for successful completion of search
      IF (INDEX .EQ. -1) THEN
        WRITE(0,401) P
  401   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE RING_INDEX ',/,
     &            ' UNABLE TO FIND PROCESSOR IN RINGMAP ARRAY',/,
     &            ' PROCESSOR = ',I4)
        STOP
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE ROW_MAP_INIT(JDEX, ISIZE, JSIZE, MESHMAP, ROWMAP)
C                                                                              
C This routine initializes ROWMAP, which defines the mapping from
C logical processor row JDEX ((0,JDEX) to (ISIZE-1,JDEX)) to a subset of
C the "physical" processors. This mapping is already defined in the
C array MESHMAP, but is extracted for convenience in row oriented
C operations. 
C
C called by: ALGINP
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C row index
      INTEGER JDEX
C dimensions of logical processor mesh
      INTEGER ISIZE, JSIZE
C logical processor mesh map array
      INTEGER MESHMAP(0:ISIZE-1,0:JSIZE-1)
C
C     Output
C
C logical row-to-physical processor mapping
      INTEGER ROWMAP(0:ISIZE-1)
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER I
C
C---- Executable Statements --------------------------------------------
C
      DO I=0,ISIZE-1
        ROWMAP(I) = MESHMAP(I,JDEX)
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE COLUMN_MAP_INIT(IDEX, ISIZE, JSIZE, MESHMAP, COLMAP)
C                                                                              
C This routine initializes COLMAP, which defines the mapping from
C logical processor column IDEX ((IDEX,0) to (IDEX,JSIZE-1)) to a subset
C of the "physical" processors. This mapping is already defined in the
C array MESHMAP, but is extracted for convenience in column oriented
C operations. 
C
C called by: ALGINP
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C column index
      INTEGER IDEX
C dimensions of logical processor mesh
      INTEGER ISIZE, JSIZE
C logical processor mesh map array
      INTEGER MESHMAP(0:ISIZE-1,0:JSIZE-1)
C
C     Output
C
C logical column-to-physical processor mapping
      INTEGER COLMAP(0:JSIZE-1)
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER J
C
C---- Executable Statements --------------------------------------------
C
      DO J=0,JSIZE-1
        COLMAP(J) = MESHMAP(IDEX,J)
      ENDDO
C
      RETURN
      END
C
Ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
C
      INTEGER FUNCTION MESH_MAP(IDEX, JDEX, ISIZE, JSIZE, MESHMAP)
C                                                                              
C This routine returns the processor id corresponding to the logical
C mesh coordinates (IDEX,JDEX). For convenience in the application code,
C the coordinates are mapped into the legal range by assuming that the
C mesh is a torus. 
C
C called by: ALGINP
C calls:
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C column index
      INTEGER IDEX
C row index
      INTEGER JDEX
C dimensions of logical processor mesh
      INTEGER ISIZE, JSIZE
C logical-to-physical processor mapping
      INTEGER MESHMAP(0:ISIZE-1,0:JSIZE-1)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
C     map column index into legal range
      IF (ISIZE .LE. 1) THEN
        I = 0
      ELSE
        IF (IDEX .GE. 0) THEN
          I = MOD(IDEX,ISIZE)
        ELSE
          I = MOD(IDEX,ISIZE) + ISIZE
        ENDIF
      ENDIF
C
C     map row index into legal range
      IF (JSIZE .LE. 1) THEN
        J = 0
      ELSE
        IF (JDEX .GE. 0) THEN
          J = MOD(JDEX,JSIZE)
        ELSE
          J = MOD(JDEX,JSIZE) + JSIZE
        ENDIF
      ENDIF
C
C     read meshmap
      MESH_MAP = MESHMAP(I,J)
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      INTEGER FUNCTION RING_MAP(INDEX, LTH, RINGMAP)
C                                                                              
C This routine returns the processor id corresponding to the logical
C ring coordinate index. For convenience in the application code, the
C coordinates are mapped into the legal range in a  "ring" fashion.
C
C called by: ALGINP, FLTSUM, ILTCAST, RFTFAX
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C ring index
      INTEGER INDEX
C ring dimension
      INTEGER LTH
C logical-to-physical processor mapping
      INTEGER RINGMAP(0:LTH-1)
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER I
C
C---- Executable Statements --------------------------------------------
C
C     Map ring index into legal range.
      IF (LTH .LE. 1) THEN
        I = 0
      ELSE
        IF (INDEX .GE. 0) THEN
          I = MOD(INDEX,LTH)
        ELSE
          I = MOD(INDEX,LTH) + LTH
        ENDIF
      ENDIF
C
C     Read RINGMAP.
      RING_MAP = RINGMAP(I)
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC



