C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
      SUBROUTINE RFTLON(ISIGN, NSPECIES, TRIGS, WS, XX)
C
C IF (ISIGN .EQ. -1), then this routine performs a block, in-place, real
C FFT of a distributed real array XX of the form:
C                (MXLLON_P,NLVER_P,NLLAT_P,NSPECIES)
C producing a complex array of the form:
C                (MXLFC_S,NLVER_F,NLLAT_F,NSPECIES)
C If ISIGN=+1, then the inverse tranformation is performed.
C
C The algorithm used is determined by LTOPT and FTOPT.  Detailed
C descriptions of the various algorithms are provided in the text.
C In these descriptions, array dimensions that are distributed are 
C annotated with a "*". Note that for the transpose-based algorithms,
C only the untruncated Fourier coefficients (M .LE. MM) are saved.
C
C called by:  ADVECT, DZSC, EXPLIC, SHTRNS, SIMPLIC
C calls: PARFFT, REORDER1, REORDER2, REORDER3, REORDER4, REORDER5,
C        REORDER6, REORDER7, REORDER8, REORDER9, TRANSPOSE 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
#     include "problem.i"
C parallel algorithm information
#     include "algorithm.i"
C domain decomposition information
#     include "physical.i"
#     include "fourier.i"
#     include "spectral.i"
C transform arrays
#     include "trnsfm.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C direction of FFT (-1 is forward, +1 is inverse)
      INTEGER ISIGN
C number of species in XX
      INTEGER NSPECIES
C trigonometric function values used by PARFFT
      COMPLEX TRIGS(NTRIGS)
C
C     Input/Output
C
C On input, XX contains longitude data to transformed. On output, it
C contains the results of the transform.
C (Also organized as COMPLEX (MXLFC_S,NLVER_S,NLLAT_S,NSPECIES))
      REAL XX(MXLLON_P,NLVER_P,NLLAT_P,NSPECIES)
C
C     Work Space
C
C work array for communication buffers
C (big enough for REAL (MXLLON_P,MXLVER_P,MXLLAT_P,NSPECIES,BUFSWS2)
C             and REAL (MXLLON_F,MXLVER_F,MXLLAT_F,NSPECIES,BUFSWS2)
C             and COMPLEX (MXLFC_S,MXLVER_S,MXLLAT_S,NSPECIES,BUFSWS2))
      REAL WS(MXLGRID*NSPECIES,2)
C
C---- Local Variables --------------------------------------------------
C
C bases for message types used in the distributed Fourier transform
      INTEGER BASE, BASE2
C temporary variables for calculating partition sizes in double
C transpose algorithm
      INTEGER I, NVECT, NTMP1, NTMP2
C
C---- External Functions -----------------------------------------------
C
C offset for message types
      EXTERNAL MSGBASE
      INTEGER MSGBASE
C
C---- Executable Statements --------------------------------------------
C
      CALL SETDATA0F(DATAREAL)
C
C     Pack first and last Fourier coefficients into the first complex
C     location before calculating the real inverse transform.
C     Both values are real, so no information is lost.
      IF (ISIGN .EQ. 1) THEN
        IF (IM0_S .NE. -1) THEN
          CALL RFTPACK(IMNFC_S, IM0_S, NLVER_S*NLLAT_S*NSPECIES, 
     &                 2*MXLFC_S, XX)
        ENDIF
      ENDIF
C
      IF ((FTOPT .EQ. 0) .AND. (LTOPT .EQ. 0)) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       distributed FFT and distributed LT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C       Set message type offsets.
        BASE  = MSGBASE()
        BASE2 = MSGBASE()
C
        IF (ISIGN .EQ. -1) THEN
C         Forward distributed FFT
          CALL PARRFT(COMMFFT, PROTFFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, -1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
        ELSE
C         Inverse distributed FFT
          CALL PARRFT(COMMIFT, PROTIFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, +1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
        ENDIF
C
      ELSEIF(FTOPT .EQ. 1 .AND. LTOPT .EQ. 1) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       transpose FFT and transpose LT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
        IF (ISIGN .EQ. -1) THEN
C         Forward Transpose FFT/Transpose LT:
C         1) The input real array (XX) of the form
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES)    
C            is reordered to give an array (1st half of WS) with
C            the form 
C             (NLLON_P*,NLLAT_P*,NSPECIES,NVER).
C         2) This array is transposed to give (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).   
C         3) This array is "FFT"ed to produce the output (XX)
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES).  
C         4) This is reordered to give a "load balanced" ordering
C            for the next transpose (1st half of WS) 
C             (NLLAT_F*,NLVER_F*,NSPECIES,NFC-1).    
C         5) This is transposed to produce the output complex array 
C            (XX) of the form
C             (MXLFC_S*,NLVER_S*,NLAT,NSPECIES).
C
          IF(NPLON_P .GT. 1) THEN
            CALL REORDER1(XX, WS, MXLLON_P, NLLON_P, NVER,
     &                    NLLAT_P*NSPECIES)
            CALL TRANSPOSE(COMMFFT, BUFSFFT-1, PROTFFT,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     -1, 1, NLON, NVER, NLLAT_P, NSPECIES,
     &                     NDLON_P, NDVER_F, MXLLON_F, WS,
     &                     WS(1,2), XX)
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMFFT, PROTFFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, -1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF(NPFC_S .GT. 1) THEN
            CALL REORDER3(XX, WS, MXLFC_F, NFC-1, NLVER_F, NLLAT_F,
     &                    NSPECIES, ORDFWD_S) 
            IF (MM .LT. NFC-1) THEN
              CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       -2, 2, NLAT, MM+1, NLVER_F, NSPECIES,
     &                       NDLAT_F, NDFC_S, MXLFC_S, WS,
     &                       WS(1,2), XX)
            ELSE
              CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       -2, 2, NLAT, NFC-1, NLVER_F, NSPECIES,
     &                       NDLAT_F, NDFC_S, MXLFC_S, WS,
     &                       WS(1,2), XX)
            ENDIF
          ENDIF
C
        ELSE
C         Inverse Transpose FFT/Transpose LT:
C         1) The input complex array (XX)
C             (MXLFC_S*,NLVER_S*,NLAT,NSPECIES)
C            is reordered to the form (1st half of WS)
C             (NLMM_S*,NLVER_S*,NSPECIES,NLAT).
C         2) This is transposed to give an array (XX) of the form
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES),
C            reordering the wavenumbers to the expected order.
C         3) This is "FFT"ed to give the real array (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).    
C         4) This is reordered to give an array (1st half of WS)
C             (NLVER_F*,NLLAT_F*,NSPECIES,NLON).
C         5) This array is transposed to give the output complex array
C            (XX) of the form
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES).
C
          IF(NPFC_S .GT. 1) THEN
            IF (MM .LT. NFC-1) THEN
              CALL REORDER4(XX, WS, MXLFC_S, NLMM_S, NLVER_S, NLAT, 
     &                      NSPECIES)
              CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       +2, 2, MM+1, NLAT, NLVER_S, NSPECIES,
     &                       NDFC_S, NDLAT_F, MXLFC_F, WS,
     &                       WS(1,2), XX)
              CALL ZEROFC(MM+1, NFC-1, MXLFC_F, 
     &                    NLVER_F*NLLAT_F*NSPECIES, ORDINV_S, XX)
            ELSE
              CALL REORDER4(XX, WS, MXLFC_S, NLFC_S-1, NLVER_S, NLAT, 
     &                      NSPECIES)
              CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       +2, 2, NFC-1, NLAT, NLVER_S, NSPECIES,
     &                       NDFC_S, NDLAT_F, MXLFC_F, WS,
     &                       WS(1,2), XX)
            ENDIF
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMIFT, PROTIFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, +1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF(NPLON_P .GT. 1) THEN
            CALL REORDER2(XX, WS, MXLLON_F, NLON,
     &                    NLVER_F*NLLAT_F*NSPECIES) 
            CALL TRANSPOSE(COMMIFT, BUFSIFT-1, PROTIFT,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     +1, 1, NVER, NLON, NLLAT_F, NSPECIES,
     &                     NDVER_F, NDLON_P, MXLLON_P, WS,
     &                     WS(1,2), XX)
          ENDIF
C
        ENDIF
C
      ELSEIF(FTOPT .EQ. 1 .AND. LTOPT .EQ. 0) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       transpose FFT and distributed LT.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
        IF (ISIGN .EQ. -1) THEN
C         Forward Transpose FFT/Distributed LT:
C         1) The input real array (XX) with form
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES)
C            is reordered to give an array (1st half of WS) with the
C            form 
C             (NLLON_P*,NLLAT_P*,NSPECIES,NVER).
C         2) This array is transposed to give (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).  
C         3) This array is "FFT"ed to produce the output (XX)
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES)
C
          IF (NPLON_P .GT. 1) THEN
            CALL REORDER1(XX, WS, MXLLON_P, NLLON_P, NVER,
     &                    NLLAT_P*NSPECIES)
            CALL TRANSPOSE(COMMFFT, BUFSFFT-1, PROTFFT,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     -1, 1, NLON, NVER, NLLAT_P, NSPECIES,
     &                     NDLON_P, NDVER_F, MXLLON_F, WS,
     &                     WS(1,2), XX)
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMFFT, PROTFFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, -1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
        ELSE
C         Inverse Transpose FFT/Distributed LT:
C         1) The input complex array (XX) with form
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES)
C            is "FFT"ed to give the real array (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).
C         2) This is reordered to give an array (1st half of WS)
C             (NLVER_F*,NLLAT_F*,NSPECIES,NLON).
C         3) This array is transposed to give the output complex array
C            (XX) of the form
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES).
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMIFT, PROTIFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, +1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF (NPLON_P .GT. 1) THEN
            CALL REORDER2(XX, WS, MXLLON_F, NLON,
     &                    NLVER_F*NLLAT_F*NSPECIES)
            CALL TRANSPOSE(COMMIFT, BUFSIFT-1, PROTIFT,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     +1, 1, NVER, NLON, NLLAT_F, NSPECIES,
     &                     NDVER_F, NDLON_P, MXLLON_P, WS,
     &                     WS(1,2), XX)
          ENDIF
C
        ENDIF
C
      ELSEIF(FTOPT .EQ. 0 .AND. LTOPT .EQ. 1) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       distributed FFT and transpose LT.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C       Set message type offsets.
        BASE  = MSGBASE()
        BASE2 = MSGBASE()
C
        IF (ISIGN .EQ. -1) THEN
C         Forward Distributed FFT/Transpose LT:
C         1) The input real array (XX) with form
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES)
C            is "FFT"ed to give the complex array (XX)
C             (MXLFC_F*,NVER,NLLAT_F*,NSPECIES).
C         2) This is reordered to give an array (1st half of WS)
C             (NLLAT_F*,NLFC_S*,NSPECIES,NVER).
C         3) This is transposed to give the output complex array (XX)
C             (MXLFC_S*,NLVER_S*,NLAT,NSPECIES)
C
          CALL PARRFT(COMMFFT, PROTFFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, -1, NLLON_F,
     &                NVER*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF (NPVER_S .GT. 1) THEN
            IF (IMNFC_S .EQ. -1) THEN
              CALL REORDER5(XX, WS, MXLFC_F, NLFC_F-1, NLFC_S, NVER, 
     &                      NSPECIES, NLLAT_F, ORDFWD_S)
              CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT,
     &                       NPVER_S, MAPVER_S, VERDEX_S, MSGBASE(),
     &                       -3, 2, NLAT, NVER, NLFC_S, NSPECIES,
     &                       NDLAT_F, NDVER_S, MXLFC_S, WS,
     &                       WS(1,2), XX)
            ELSE
              CALL REORDER5(XX, WS, MXLFC_F, NLFC_F-1, NLFC_S-1, NVER,
     &                      NSPECIES, NLLAT_F, ORDFWD_S)
              CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT,
     &                       NPVER_S, MAPVER_S, VERDEX_S, MSGBASE(),
     &                       -3, 2, NLAT, NVER, NLFC_S-1, NSPECIES,
     &                       NDLAT_F, NDVER_S, MXLFC_S, WS,
     &                       WS(1,2), XX)
            ENDIF
          ENDIF
C
        ELSE
C         Inverse Distributed FFT/Transpose LT:
C         1) The input complex array (XX) with form   
C             (MXLFC_S*,NLVER_S*,NLAT,NSPECIES)
C            is reordered to give an array (1st half of WS) 
C             (NLVER_S*,NLFC_S*,NSPECIES,NLAT).
C         2) This is transposed to give an array (XX)
C             (MXLFC_F*,NVER,NLLAT_F*,NSPECIES)
C         3) This is "FFT"ed to give the output real array (XX)
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES).
C
          IF(NPVER_S .GT. 1) THEN
            IF (IMNFC_S .EQ. -1) THEN
              CALL REORDER6(XX, WS, MXLFC_S, NLFC_S, NLVER_S, NLAT,
     &                      NSPECIES)
              CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT,
     &                       NPVER_S, MAPVER_S, VERDEX_S, MSGBASE(),
     &                       +3, 2, NVER, NLAT, NLFC_S, NSPECIES,
     &                       NDVER_S, NDLAT_F, MXLFC_F, WS,
     &                       WS(1,2), XX)
              CALL ZEROFC(NLFC_S, NLFC_F-1, MXLFC_F, 
     &                    NLVER_F*NLLAT_F*NSPECIES, ORDINV_S, XX)
            ELSE
              CALL REORDER6(XX, WS, MXLFC_S, NLFC_S-1, NLVER_S, NLAT,
     &                      NSPECIES)
              CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT,
     &                       NPVER_S, MAPVER_S, VERDEX_S, MSGBASE(),
     &                       +3, 2, NVER, NLAT, NLFC_S-1, NSPECIES,
     &                       NDVER_S, NDLAT_F, MXLFC_F, WS,
     &                       WS(1,2), XX)
            ENDIF
          ENDIF
C
          CALL PARRFT(COMMIFT, PROTIFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, +1, NLLON_F,
     &                NVER*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
        ENDIF
C
      ELSEIF(FTOPT .EQ. 2 .AND. LTOPT .EQ. 0) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       double transpose FFT and distributed LT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
        IF (ISIGN .EQ. -1) THEN
C         Forward Double Transpose FFT/Distributed LT:
C         1) The input real array (XX) of the form
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES)    
C            is reordered to give an array (1st half of WS) with
C            the form 
C             (NLLON_P*,NVER,NLLAT_P*,NSPECIES)
C         2) This array is transposed to give (XX)
C             (MXLLON_F,*),
C            decomposing over the NVER*NLLAT_P*NSPECIES independent
C            vectors.
C         3) This array is "FFT"ed to produce the output (XX)
C             (MXLFC_F,*).  
C         4) This is reordered to give a "load balanced" ordering
C            for the next transpose (1st half of WS) 
C             (*,NFC-1).  
C         5) This is transposed to produce the output complex array 
C            (XX) of the form
C             (MXLFC_S*,NVER,NLLAT_S*,NSPECIES).
C
C         determine number of independent FFTs to calculate
C         (Cannot be precalculated because number of species
C         varies.)
          NVECT = NVER*NLLAT_P*NSPECIES
          NTMP1 = NVECT/NPVER_F
          NTMP2 = MOD(NVECT,NPVER_F)
          IF (NTMP2 .GT. 0) THEN
            MXLVER_F = NTMP1 + 1
          ELSE
            MXLVER_F = NTMP1
          ENDIF
          DO I=0,NTMP2-1
            NDVER_F(I) = NTMP1+1
          ENDDO
          DO I=NTMP2,NPVER_F-1
            NDVER_F(I) = NTMP1
          ENDDO
          NLVER_F = NDVER_F(VERDEX_F)
C
          IF (NPLON_P .GT. 1) THEN
C           reorder and transpose fields
            CALL REORDER7(XX, WS, MXLLON_P, NLLON_P, NVECT)
            CALL TRANSPOSE(COMMFFT, BUFSFFT-1, PROTFFT,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     -1, 1, NLON, NVECT, 1, 1,
     &                     NDLON_P, NDVER_F, MXLLON_F, WS,
     &                     WS(1,2), XX)
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMFFT, PROTFFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, -1, NLLON_F,
     &                NLVER_F, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF (NPFC_S .GT. 1) THEN
            CALL REORDER8(XX, WS, MXLFC_F, NFC-1, NLVER_F, ORDFWD_S) 
            IF (MM .LT. NFC-1) THEN
              CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       -2, 2, NVECT, MM+1, 1, 1,
     &                       NDVER_F, NDFC_S, MXLFC_S, WS,
     &                       WS(1,2), XX)
            ELSE
              CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       -2, 2, NVECT, NFC-1, 1, 1,
     &                       NDVER_F, NDFC_S, MXLFC_S, WS,
     &                       WS(1,2), XX)
            ENDIF
          ENDIF
C
        ELSE
C         Inverse Double Transpose FFT/Distributed LT:
C         1) The input complex array (XX)
C             (MXLFC_S*,NVER,NLLAT_S*,NSPECIES)
C            is reordered to the form (1st half of WS)
C             (NLMM_S*,NVER,NLLAT_S*,NSPECIES)
C         2) This is transposed to give an array (XX) of the form
C             (MXLFC_F,*)
C            decomposing over the NLVER_S*NLAT*NSPECIES independent
C            vectors and reordering the wavenumbers to the
C            expected order.
C         3) This is "FFT"ed to give the real array (XX)
C             (MXLLON_F,*)
C         4) This is reordered to give an array (1st half of WS)
C             (*,NLON).
C         5) This array is transposed to give the output complex array
C            (XX) of the form
C             (MXLLON_P*,NVER,NLLAT_P*,NSPECIES).
C
C         determine number of independent FFTs to calculate
C         (Cannot be precalculated because number of species
C         varies.)
          NVECT = NVER*NLLAT_P*NSPECIES
          NTMP1 = NVECT/NPVER_F
          NTMP2 = MOD(NVECT,NPVER_F)
          IF (NTMP2 .GT. 0) THEN
            MXLVER_F = NTMP1 + 1
          ELSE
            MXLVER_F = NTMP1
          ENDIF
          DO I=0,NTMP2-1
            NDVER_F(I) = NTMP1+1
          ENDDO
          DO I=NTMP2,NPVER_F-1
            NDVER_F(I) = NTMP1
          ENDDO
          NLVER_F = NDVER_F(VERDEX_F)
C
          IF(NPFC_S .GT. 1) THEN
            IF (MM .LT. NFC-1) THEN
              CALL REORDER9(XX, WS, MXLFC_S, NLMM_S, NVECT)
              CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       +2, 2, MM+1, NVECT, 1, 1,
     &                       NDFC_S, NDVER_F, MXLFC_F, WS,
     &                       WS(1,2), XX)
              CALL ZEROFC(MM+1, NFC-1, MXLFC_F, NLVER_F, ORDINV_S, XX)
            ELSE
              CALL REORDER9(XX, WS, MXLFC_S, NLFC_S-1, NVECT)
              CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT,
     &                       NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                       +2, 2, NFC-1, NVECT, 1, 1,
     &                       NDFC_S, NDVER_F, MXLFC_F, WS,
     &                       WS(1,2), XX)
            ENDIF
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMIFT, PROTIFT, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, +1, NLLON_F,
     &                NLVER_F, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF(NPLON_P .GT. 1) THEN
            CALL REORDER2(XX, WS, MXLLON_F, NLON, NLVER_F)
            CALL TRANSPOSE(COMMIFT, BUFSIFT-1, PROTIFT,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     +1, 1, NVECT, NLON, 1, 1,
     &                     NDVER_F, NDLON_P, MXLLON_P, WS,
     &                     WS(1,2), XX)
          ENDIF
C
        ENDIF
C
      ENDIF
C
C     Unpack the last Fourier coefficient from the imaginary part of 
C     the first complex location after calculating the real forward 
C     transform. Both values are real, so also zero out the imaginary
C     parts of both coefficients.
      IF (ISIGN .EQ. -1) THEN
        IF (IM0_S .NE. -1) THEN
          CALL RFTUNPACK(IMNFC_S, IM0_S, NLVER_S*NLLAT_S*NSPECIES, 
     &                   2*MXLFC_S, XX)
        ENDIF
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER1(FROM, TO, MXLLON, NLLON, NVER, COUNT)
C
C This routine reorders FROM(MXLLON,NVER,COUNT) into
C TO(NLLON,COUNT,NVER), where both arrays are real.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLLON, NLLON, NVER, COUNT
C array that is to be reordered
      REAL FROM(MXLLON,NVER,COUNT)
C
C     Output
C
C destination of reordered array
      REAL TO(NLLON,COUNT,NVER)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K
C
C---- Executable Statements --------------------------------------------
C
      DO K = 1,COUNT
        DO J = 1,NVER
          DO I = 1,NLLON
            TO(I,K,J) = FROM(I,J,K)
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER2(FROM, TO, MXLON, NLON, COUNT)
C
C This routine reorders FROM(MXLON,COUNT) into TO(COUNT,NLON), where
C both arrays are real.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLON, NLON, COUNT
C array that is to be reordered
      REAL FROM(MXLON,COUNT)
C
C     Output
C
C destination of reordered array
      REAL TO(COUNT,NLON)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
      DO J = 1,COUNT
        DO I = 1,NLON
          TO(J,I) = FROM(I,J)
        ENDDO
      ENDDO
C      
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER3(FROM, TO, MXFC, NFC, NLVER, NLLAT, NSPECIES,
     &                    ORDER)
C
C This routine reorders FROM(MXFC,NLVER,NLLAT,NSPECIES) into
C TO(NLLAT,NLVER,NSPECIES,NFC), where both arrays are complex.
C The ORDER array is used to reorder the Fourier wavenumbers (for
C load balancing).
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXFC, NFC, NLVER, NLLAT, NSPECIES
C array that is to be reordered
      COMPLEX FROM(MXFC,NLVER,NLLAT,NSPECIES)
C array defining reordering
      INTEGER ORDER(NFC)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLLAT,NLVER,NSPECIES,NFC)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLLAT
          DO J = 1,NLVER
            DO I = 1,NFC
              TO(K,J,L,ORDER(I)) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER4(FROM, TO, MXLFC, NLFC, NLVER, NLAT, NSPECIES)
C
C This routine reorders FROM(MXLFC,NLVER,NLAT,NSPECIES) into
C TO(NLFC,NLVER,NSPECIES,NLAT). Both arrays are complex.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLFC, NLFC, NLVER, NLAT, NSPECIES
C array that is to be reordered
      COMPLEX FROM(MXLFC,NLVER,NLAT,NSPECIES)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLFC,NLVER,NSPECIES,NLAT)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLAT
          DO J = 1,NLVER
            DO I = 1,NLFC
              TO(I,J,L,K) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER5(FROM, TO, MXLFC, NLFC, NLMM, NLVER, NSPECIES, 
     &                    NLLAT, ORDER)
C
C This routine reorders FROM(MXLFC,NLVER,NLLAT,NSPECIES) into
C TO(NLLAT,NLMM,NSPECIES,NLVER), where both arrays are complex.
C The ORDER array is used to reorder the Fourier wavenumbers so
C that the truncated wavenumbers need not be transposed.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C total number of Fourier wavenumbers
      INTEGER NLFC
C number of untruncated Fourier wavenumbers
      INTEGER NLMM
C (other) dimensions of input and output arrays
      INTEGER MXLFC, NLVER, NSPECIES, NLLAT
C array that is to be reordered
      COMPLEX FROM(MXLFC,NLVER,NLLAT,NSPECIES)
C array defining reordering
      INTEGER ORDER(NLFC)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLLAT,NLMM,NSPECIES,NLVER)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLLAT
          DO J = 1,NLVER
            DO I = 1,NLFC
              IF (ORDER(I) .LE. NLMM)
     &          TO(K,ORDER(I),L,J) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER6(FROM, TO, MXLFC, NLFC, NLVER, NLAT, NSPECIES)
C
C This routine reorders FROM(MXLFC,NLVER,NLAT,NSPECIES) into
C TO(NLVER,NLFC,NSPECIES,NLAT). Both arrays are complex.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLFC, NLFC, NLVER, NLAT, NSPECIES
C array that is to be reordered
      COMPLEX FROM(MXLFC,NLVER,NLAT,NSPECIES)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLVER,NLFC,NSPECIES,NLAT)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLAT
          DO J = 1,NLVER
            DO I = 1,NLFC
              TO(J,I,L,K) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER7(FROM, TO, MXLON, NLON, COUNT)
C
C This routine reorders FROM(MXLON,COUNT) into TO(NLON,COUNT), where
C both arrays are real.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLON, NLON, COUNT
C array that is to be reordered
      REAL FROM(MXLON,COUNT)
C
C     Output
C
C destination of reordered array
      REAL TO(NLON,COUNT)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
      DO J = 1,COUNT
        DO I = 1,NLON
          TO(I,J) = FROM(I,J)
        ENDDO
      ENDDO
C      
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER8(FROM, TO, MXFC, NFC, NVECT, ORDER)
C
C This routine reorders FROM(MXFC,NVECT) into TO(NVECT,NFC), where 
C both arrays are complex. The ORDER array is used to reorder the 
C Fourier wavenumbers (for load balancing).
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXFC, NFC, NVECT
C array that is to be reordered
      COMPLEX FROM(MXFC,NVECT)
C array defining reordering
      INTEGER ORDER(NFC)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NVECT,NFC)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
      DO J = 1,NVECT
        DO I = 1,NFC
          TO(J,ORDER(I)) = FROM(I,J)
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER9(FROM, TO, MXLFC, NLFC, NVECT)
C
C This routine reorders FROM(MXLFC,NVECT) into TO(NLFC,NVECT). 
C Both arrays are complex.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLFC, NLFC, NVECT
C array that is to be reordered
      COMPLEX FROM(MXLFC,NVECT)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLFC,NVECT)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
      DO J = 1,NVECT
        DO I = 1,NLFC
          TO(I,J) = FROM(I,J)
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE ZEROFC(NLMM, NLFC, MXLFC, NLVECT, ORDER, FC)
C
C This routine zeroes the truncated Fourier coefficients, needed before
C calculating the inverse real FFT.
C
C called by: DZPUV, ILTRNS, ITRNPE, RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C number of untruncated Fourier coefficients
      INTEGER NLMM
C total number of Fourier coefficients
      INTEGER NLFC
C dimensions of array of Fourier coefficients
      INTEGER MXLFC, NLVECT
C array defining ordering of wavenumbers
      INTEGER ORDER(NLFC)
C
C     Output
C
C array whose truncated Fourier coefficients are to be zeroed
      COMPLEX FC(MXLFC,NLVECT)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
C     Zero the tail of the complex coefficient spectrum.
      DO J = 1,NLVECT
        DO I = NLMM+1,NLFC
          FC(ORDER(I),J) = (0.0,0.0)
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RFTPACK(NFCLOC, ZEROLOC, MVECS, JUMP, DATA)
C
C If (NFCLOC .NE. -1) then this routine copies element 2*NFCLOC-1 into
C element 2*ZEROLOC for a sequence of MVEC vectors. IF (NFCLOC .EQ. -1)
C then it does nothing. RFTPACK is used to pack the first and last 
C Fourier coefficients into the first complex location before 
C calculating the real inverse Fourier transform. Both values are real, 
C so no information is lost. 
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C location of wavenumber NFC in complex vector
      INTEGER NFCLOC
C location of wavenumber 0 in complex vector
      INTEGER ZEROLOC
C number of vectors being modified
      INTEGER MVECS
C the offset between the start of successive vectors
      INTEGER JUMP
C
C     Input/Output
C
C data array to be modified
      REAL DATA(JUMP,MVECS)
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER I
C indices for real part of Fourier coefficient NFC and imaginary part of 
C Fourier coefficient 0
      INTEGER RDEXNFC, IDEX0
C
C---- Executable Statements --------------------------------------------
C
      IF (NFCLOC .NE. -1) THEN
C       NFC-1 is not truncated, pack value into ZEROLOC
        RDEXNFC = 2*NFCLOC-1
        IDEX0   = 2*ZEROLOC
        DO I=1,MVECS
          DATA(IDEX0,I) = DATA(RDEXNFC,I)
        ENDDO
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RFTUNPACK(NFCLOC, ZEROLOC, MVECS, JUMP, DATA)
C
C If (NFCLOC .NE. -1) then this routine copies element 2*ZEROLOC into
C element 2*NFCLOC-1 and zeroes out elements 2*ZEROLOC and 2*NFCLOC
C for a sequence of MVEC vectors. IF (NFCLOC .EQ. -1), then it just
C zeroes out element 2*ZEROLOC in the MVEC vectors. RFTUNPACK is used
C to unpack the last Fourier coefficient from the imaginary part of the 
C first coefficient after calculating the real forward Fourier
C transform. Both values are real, so no information is lost. 
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C location of wavenumber NFC in complex vector
      INTEGER NFCLOC
C location of wavenumber 0 in complex vector
      INTEGER ZEROLOC
C number of vectors being modified
      INTEGER MVECS
C the offset between the start of successive vectors
      INTEGER JUMP
C
C     Input/Output
C
C data array to be modified
      REAL DATA(JUMP,MVECS)
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER I
C indices for real and imaginary parts of Fourier coefficient NFC and 
C imaginary part of Fourier coefficient 0
      INTEGER RDEXNFC, IDEXNFC, IDEX0
C
C---- Executable Statements --------------------------------------------
C
      IF (NFCLOC .NE. -1) THEN
C       NFC-1 is not truncated, so unpack value from ZEROLOC
        RDEXNFC = 2*NFCLOC-1
        IDEXNFC = 2*NFCLOC
        IDEX0   = 2*ZEROLOC
        DO I=1,MVECS
          DATA(RDEXNFC,I) = DATA(IDEX0,I)
          DATA(IDEXNFC,I) = 0.0
          DATA(IDEX0,I)   = 0.0
        ENDDO
      ELSE
C       NFC-1 is truncated, so zero corresponding value in ZEROLOC
        IDEX0   = 2*ZEROLOC
        DO I=1,MVECS
          DATA(IDEX0,I)   = 0.0
        ENDDO
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
