C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
      SUBROUTINE GLATS(NLAT, F, WT)
C
C This routine calculates the roots of the ordinary Legendre polynomials
C of order NLAT that correspond to the latitude points used in a 
C Gaussian quadrature procedure on the sphere. The routine uses a 
C Newton-Raphson iteration procedure to determine the roots to a 
C precision given by DEPS. A routine to calculate the ordinary Legendre 
C polynomials, in this case DORDLEG, is required.
C
C called by: INPUT
C calls: DEPSLON, DORDLEG
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Common Blocks ----------------------------------------------------
C
C machine architecture information
#     include "machine.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C number of zeros between poles
      INTEGER NLAT
C
C     Output
C
C resulting roots expressed in colatitude
      REAL F(NLAT)
C corresponding Gaussian weights
      REAL WT(NLAT)
C
C---- Local Variables --------------------------------------------------
C
C "constant" variables: pi/2
      DOUBLE PRECISION PIHALF
C double precision machine accuracy
      DOUBLE PRECISION DEPS
C NLAT+1, NLAT-1
      INTEGER NLATP, NLATM
C latitude index
      INTEGER NL
C Newton iteration counter
      INTEGER NITER
C DBLE(NLAT), DBLE(NLAT+1), DBLE(NLAT/2)
      DOUBLE PRECISION DNLAT, DNLATP, DNLATH
C constant factors in Newton iteration formula
      DOUBLE PRECISION DN, DN1, A, B
C Legendre polynomial values at F(NL) for wavenumbers NLAT, NLAT-1, and
C NLAT+1, respectively.   
      DOUBLE PRECISION LG, LGM, LGP
C denominator of Newton update term and size of update term (used to 
C check for convergence)
      DOUBLE PRECISION GT, GTEMP
C previous and current Newton iterates
      DOUBLE PRECISION OLDF, NEWF
C
C---- External Functions -----------------------------------------------
C
C double precision machine epsilon function
      EXTERNAL DEPSLON
      DOUBLE PRECISION DEPSLON
C
C---- Executable Statements --------------------------------------------
C
C     Compute single precision machine accuracy.
      DEPS = DEPSLON(1.0D0)
C     Compute pi
      PIHALF = 2.0D0*ATAN(1.0D0)

C     Precompute latitude and iteration independent values.
      NLATP  = NLAT + 1
      NLATM  = NLAT - 1
      DNLAT  = NLAT
      DNLATP = NLATP
      DNLATH = NLAT/2
      DN     = DNLAT/SQRT(4.D0*DNLAT*DNLAT-1.D0)
      DN1    = DNLATP/SQRT(4.D0*DNLATP*DNLATP-1.D0)
      A      = DN1*DNLAT
      B      = DN*DNLATP
C
      DO NL = 1, NLAT/2
C
C       First guess at root; F represents cosine of colatitude.
        OLDF = -PIHALF*((NL-1)+.5D0)/DNLATH + PIHALF
        OLDF = SIN(OLDF)
C
C       Newton iteration loop
        NITER = 0
        GTEMP = 1.0D0 + 2.0D0*DEPS
        DO WHILE ((ABS(GTEMP) .GT. 2.0D0*DEPS) .AND. (NITER .LT. 100))
          NITER = NITER + 1
          CALL DORDLEG(OLDF, NLAT, LG)
          CALL DORDLEG(OLDF, NLATM, LGM)
          CALL DORDLEG(OLDF, NLATP, LGP)
          GT   = (A*LGP-B*LGM)/(OLDF*OLDF-1.0D0)
          NEWF = OLDF - LG/GT
          GTEMP= OLDF - NEWF
          OLDF = NEWF
        ENDDO
C
C       convergence check
        IF (NITER .GE. 100) THEN
          WRITE (0,90) ME, NL
   90     FORMAT(/,' PSTSWM: FATAL ERROR IN GLATS:',/,
     &             ' NO CONVERGENCE IN NEWTON ITERATION FOR ROOT',/,
     &             ' PROCID = ', I3 ,' LATITUDE  NL = ', I3)
          STOP
        ENDIF
C
C       Calculate weights and express roots in terms of colatitude.
        CALL DORDLEG(NEWF, NLATM, LGM)
        WT(NL) = 2.0D0*(1.0D0 - NEWF**2)*(DNLAT - 0.5D0)
     &         / (LGM*LGM*DNLAT*DNLAT)
        F(NL)  = PIHALF - ACOS(NEWF)
C
      ENDDO
C
C     Fill in the information for the other half of the sphere.
      DO NL = 1,NLAT/2
        F (NLAT-NL+1) = -F(NL)
        WT(NLAT-NL+1) = WT(NL)
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      DOUBLE PRECISION FUNCTION DEPSLON(X)
C
C This function estimates the double precision machine arithmetic
C roundoff in quantities of size x. It should function properly on all
C systems satisfying the following two assumptions:
C  1. The base used in representing floating point numbers is not a 
C     power of three.
C  2. The quantity A in statement 10 is represented to the accuracy used 
C     in floating point variables that are stored in memory.
C The statement number 10 and the GO TO 10 are intended to force
C optimizing compilers to generate code satisfying assumption 2.
C Under these assumptions, it should be true that
C  A   is not exactly equal to four-thirds,
C  B   has a zero for its last bit or digit,
C  C   is not exactly equal to one, and
C  DEPS measures the separation of 1.0 from the next larger floating 
C      point number.
C
C called by: GLATS
C calls:
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C value with respect to which roundoff is being calculated
C (X + DEPSLON > X)
      DOUBLE PRECISION X
C
C---- Local Variables --------------------------------------------------
C
C temporaries used in calculating DEPSLON
      DOUBLE PRECISION A, B, C, DEPS
C
C---- Executable Statements --------------------------------------------
C
      A = 4.0D0/3.0D0
   10 B = A - 1.0D0
      C = B + B + B
      DEPS = ABS(C-1.0D0)
      IF (DEPS .EQ. 0.0D0) GO TO 10
      DEPSLON = DEPS*ABS(X)
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE DORDLEG(COA, IR, SX)
C
C This routine is used to evaluate ordinary Legendre polynomials.
C
C called by: GLATS
C calls:
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C cosine of colatitude
      DOUBLE PRECISION COA
C wavenumber
      INTEGER IR
C
C     Output
C
C Legendre polynomial evaluated at COA
      DOUBLE PRECISION SX
C
C---- Local Variables --------------------------------------------------
C
C wavenumber (loop) index
      INTEGER K
C colatitude
      DOUBLE PRECISION THETA
C factors in polynomial evaluation
      DOUBLE PRECISION C1, S1
C temporaries used in computing S1
      DOUBLE PRECISION ANG, C4
C
C---- Executable Statements --------------------------------------------
C
      THETA = ACOS(COA)
C
      C1  = SQRT(2.0D0)
      DO K = 1, IR
        C1 = C1*SQRT(1.0D0-1.0D0/(4*K*K))
      ENDDO
C
      ANG = IR*THETA
      S1  = 0.0D0
      C4  = 1.0D0
C
      DO K = 0, IR, 2
        IF (K .EQ. IR) C4 = 0.5D0*C4
        S1  = S1 + C4*COS(ANG)
        ANG = THETA*(IR-K-2)
        C4  = (C4*(K+1)*(2*IR-K))/((K+2)*(2*IR-K-1))
      ENDDO
C
      SX = S1*C1
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

