!Copyright (C) 1996 Claus Bendtsen (Claus.Bendtsen@uni-c.dk)
!All rights reserved.

!This code is provided "as is", without any warranty of any kind,
!either expressed or implied, including but not limited to, any implied
!warranty of merchantibility or fitness for any purpose. In no event
!will any party who distributed the code be liable for damages or for
!any claim(s) by any other party, including but not limited to, any
!lost profits, lost monies, lost data or data rendered inaccurate,
!losses sustained by third parties, or any other special, incidental or
!consequential damages arising out of the use or inability to use the
!program, even if the possibility of such damages has been advised
!against. The entire risk as to the quality, the performance, and the
!fitness of the program for any particular purpose lies with the party
!using the code.

!This code, and any derivative of this code, may not be used in a
!commercial package without the prior explicit written permission of
!the author. Verbatim copies of this code may be made and distributed
!in any medium, provided that this copyright notice is not removed or
!altered in any way. No fees may be charged for distribution of the
!codes, other than a fee to cover the cost of the media and a
!reasonable handling fee.

!***************************************************************
!ANY USE OF THIS CODE CONSTITUTES ACCEPTANCE OF THE TERMS OF THE
!                        COPYRIGHT NOTICE
!***************************************************************

!References:
!C. Bendtsen. A Parallel Stiff ODE Solver based on MIRKS. 
!Technical report 96-06, 1996. UNI-C DTU, Bldg. 304, DK-2800, Denmark.

!C. Bendtsen. ParSODES -- A Parallel Stiff ODE Solver. User's Guide.
!Technical report 96-07, 1996. UNI-C DTU, Bldg. 304, DK-2800, Denmark.

MODULE sodes
  USE genmatrix, msolve=>solve
  IMPLICIT NONE

  ! General type for iteration matrix.
  TYPE iterationm
     TYPE(rmatrix), DIMENSION(:), POINTER :: m
     TYPE(cmatrix), DIMENSION(:), POINTER :: c
     TYPE(iterative_mtx_r), DIMENSION(:), POINTER :: im
     TYPE(iterative_mtx_c), DIMENSION(:), POINTER :: ic
  END TYPE iterationm

  INTERFACE calcnorm
     MODULE PROCEDURE calcnorm_dim1, calcnorm_dim2
  END INTERFACE

  TYPE statistics
     INTEGER :: fevals ! no. of function evaluations
     INTEGER :: nfailed ! no. of failed steps
     INTEGER :: nnewt ! no. of newton iterations
     INTEGER :: nnewtfailed ! no. of failed simpl. newton iterations.
     INTEGER :: nf1, nf2, nf3, nf4 ! type of newton failures.
     INTEGER :: nerrfailed ! no. of failed error estimations iterations.
     INTEGER :: njac ! no. of Jacobian evaluations.
     INTEGER :: njacfevals ! no. of function evaluations for numerical jac.
     INTEGER :: rfact ! no. of real factorizations of size n.
     INTEGER :: cfact ! no. of complex factorizations of size n.
     INTEGER :: rsolv ! no. of real linear system solves.
     INTEGER :: csolv ! no. of complex linear system solves.
     INTEGER :: nliniter ! no. of iterations in iterative solver
     INTEGER :: nprecond ! no. of preconditioner computations
     INTEGER :: singulariterm ! no. of times iterm. has been singular.
     INTEGER :: steps ! no. of (successful) integration steps
     INTEGER :: timer1, timer2, clicks, maxtimer
#ifdef DTIMEDEFINED
     REAL, DIMENSION(2) :: tarray
#endif
  END TYPE statistics

  INTEGER, PARAMETER :: NORM_INF=3
  INTEGER, PARAMETER :: NORM_2=2

  ! variables related to time stepping and newton iteration
  REAL(wp), PRIVATE :: iatol = 1.d-6
  REAL(wp), PRIVATE :: irtol = 1.d-3
  REAL(wp), PRIVATE :: ikappa = 3.d-2
  REAL(wp), PRIVATE :: ikappasafe = 1.0_wp
  REAL(wp), PRIVATE :: iconskappa = 1d-3
  REAL(wp), PRIVATE :: inrmdiv = 1.0_wp
  REAL(wp), PRIVATE :: ihmin = 1d-10
  INTEGER, PRIVATE :: imaxit = 10
  INTEGER, PRIVATE :: ioptit = 3
  INTEGER, PRIVATE :: ierrunit = 6
  REAL(wp), PRIVATE :: iquotl = 2.d-1
  REAL(wp), PRIVATE :: iquotm = 8.d0
  REAL(wp), PRIVATE :: isafe = 9.d-1
  REAL(wp), PRIVATE :: ierrscale = 1.d0
  REAL(wp), PRIVATE :: ijactol = 1.d-3
  REAL(wp), PRIVATE :: iincrate = 5.d-1
  REAL(wp), PRIVATE :: ihconstl = 1.0_wp
  REAL(wp), PRIVATE :: ihconstm = 1.2_wp
  INTEGER, PRIVATE :: imaxitermfail = 3
  INTEGER, PRIVATE :: ipolfitorder = 3 ! order of fitting polynomial.
  LOGICAL, PRIVATE :: ipolfit = .TRUE.
  LOGICAL, PRIVATE :: igustafsson = .TRUE.
  REAL(wp), PRIVATE :: relmax = 1.d0
  REAL(wp), PRIVATE :: relmin = 1.d0
  REAL(wp), PRIVATE :: h0 = 1.d-6
  INTEGER, PRIVATE :: inormtype = NORM_INF


  ! *** METHOD RELATED VARIABLES ***
  INTEGER, PRIVATE :: method=2 ! method id.
  INTEGER, PRIVATE :: rkorder=5 ! method order.
  CHARACTER (LEN=30), PRIVATE :: rkmethod ! method name.
  REAL(wp), PRIVATE :: errorder ! order of error estimate
  INTEGER, PRIVATE :: stages ! no. of stages in method
  ! real eigenvalues
  REAL(wp), DIMENSION(:), ALLOCATABLE, PRIVATE :: reigenvals
  ! complex eigenvalues
  COMPLEX(wp), DIMENSION(:), ALLOCATABLE, PRIVATE :: ceigenvals
  ! transformation matrix and inverse transformation matrix
  REAL(wp), DIMENSION(:,:), ALLOCATABLE, PRIVATE :: rkt, rkti, rktt, rktit
  ! quadrature points.
  REAL(wp), DIMENSION(:), ALLOCATABLE, PRIVATE :: rkc
  ! error coefficient
  REAL(wp), DIMENSION(:), ALLOCATABLE, PRIVATE :: errcoef
  ! Fitting matrix, used for normal eq.-fit for initial newton guess.
  REAL(wp), DIMENSION(:,:), ALLOCATABLE, PRIVATE :: fitA, fitG, fitAnew

  ! *** PROBLEM RELATED VARIABLES ***
  ! Variables related to numerical jacobian
  REAL(wp), PRIVATE :: br, bl, bu, facmin, facmax
  INTEGER, DIMENSION(:), ALLOCATABLE :: rowmax
  REAL(wp), DIMENSION(:), ALLOCATABLE :: difmax, absydotrm

  ! Other module variables
  INTEGER, PRIVATE :: iverboseunit=6
  INTEGER, PRIVATE :: iverboselevel=0
  LOGICAL, PRIVATE :: methodsetup=.FALSE.
  INTEGER, PRIVATE :: ilinsolver = SLV_DIRECT
  INTEGER, PRIVATE :: inlimit = 100

#ifdef MPI
  ! MPI related parameters.
  INTEGER :: mpi_realwp
  INTEGER, PRIVATE :: nprocs
  INTEGER, PRIVATE :: myrank
  INTEGER, PRIVATE :: myid
  INTEGER, PRIVATE :: communicator
  INTEGER, DIMENSION(:), ALLOCATABLE, PRIVATE :: slice
  INTEGER, DIMENSION(:), ALLOCATABLE, PRIVATE :: slicesize
  INTEGER, DIMENSION(:), ALLOCATABLE, PRIVATE :: sliceoffs
  INTEGER, PRIVATE :: jaccoltype
#endif

CONTAINS
  SUBROUTINE integ(fnk,jtype,tbeg,tend,y,info,&  ! Required parameters
       & verboseunit,verboselevel,errunit,&      ! Output units
       & calcjac,forcenumjac,&                   ! Jacobian parameters
       & resultfnk,&                             ! Func. to call with results
       & stats,&                                 ! Statistics
       & mass,&                                  ! Mass-matrix
       & name,order,&                            ! Name and order of method
       & atol,rtol,normtype,&                    ! Tolerances and norm to use
       & kappa,conskappa,kappasafe,maxit,nrmdiv,&! Variables for Newton itera.
       & jactol,minjrefactquot,maxjrefactquot,&  ! Var. for Jacobian recalc.
       & hinit,hmin,optit,safe,gustafsson,&      ! Step-size related variables
       & polfitorder,&                           ! Order of LS fit
       & linsolver,nlimit)                       ! Type of linear solver
    ! integrates mass*ydot=fnk(t,y), y(0) = y0 from tbeg to tend
    ! ******************* Required parameters *******************
    INTERFACE
       ! function to integerate.
       SUBROUTINE fnk(t,y,ydot)
         USE defines
         REAL(wp), INTENT(IN) :: t
         REAL(wp), DIMENSION(:), INTENT(IN) :: y
         REAL(wp), DIMENSION(:), INTENT(OUT) :: ydot
       END SUBROUTINE fnk
       ! jacobian function of problem
       SUBROUTINE calcjac(t,y,jac)
         USE defines
         USE genmatrix
         REAL(wp), INTENT(IN) :: t
         REAL(wp), DIMENSION(:), INTENT(IN) :: y
         TYPE(rmatrix), INTENT(OUT) :: jac
       END SUBROUTINE calcjac
       SUBROUTINE resultfnk(t,y)
         USE defines
         REAL(wp), INTENT(IN) :: t
         REAL(wp), DIMENSION(:), INTENT(IN) :: y
       END SUBROUTINE resultfnk
    END INTERFACE
#ifndef NOOPTIONALFNK
    OPTIONAL :: resultfnk, calcjac
#endif
    TYPE(mtx_type), INTENT(IN) :: jtype
    ! starting time
    REAL(wp), INTENT(IN) :: tbeg
    ! finish time. Unchanged if successful
    REAL(wp), INTENT(INOUT) :: tend
    ! in: space values at tbeg.
    ! out: space values at tend.
    REAL(wp), INTENT(INOUT), DIMENSION(:) :: y
    INTEGER, INTENT(OUT) :: info
    ! ******************* Optional parameters *******************
    TYPE(statistics), OPTIONAL, INTENT(INOUT) :: stats
    INTEGER, OPTIONAL, INTENT(IN) :: verboseunit
    INTEGER, OPTIONAL, INTENT(IN) :: verboselevel
    REAL(wp), OPTIONAL, INTENT(IN) :: atol
    REAL(wp), OPTIONAL, INTENT(IN) :: rtol
    REAL(wp), OPTIONAL, INTENT(IN) :: nrmdiv
    REAL(wp), OPTIONAL, INTENT(IN) :: conskappa
    REAL(wp), OPTIONAL, INTENT(IN) :: kappa
    REAL(wp), OPTIONAL, INTENT(IN) :: kappasafe
    REAL(wp), OPTIONAL, INTENT(IN) :: hmin
    INTEGER, OPTIONAL, INTENT(IN) :: maxit
    REAL(wp), OPTIONAL, INTENT(IN) :: jactol
    REAL(wp), OPTIONAL, INTENT(IN) :: minjrefactquot
    REAL(wp), OPTIONAL, INTENT(IN) :: maxjrefactquot
    INTEGER, OPTIONAL, INTENT(IN) :: errunit
    CHARACTER (LEN=*), OPTIONAL, INTENT(IN) :: name
    INTEGER, OPTIONAL, INTENT(IN) :: optit
    INTEGER, OPTIONAL, INTENT(IN) :: order
    INTEGER, OPTIONAL, INTENT(IN) :: polfitorder
    REAL(wp), OPTIONAL, INTENT(IN) :: hinit
    LOGICAL, OPTIONAL, INTENT(IN) :: gustafsson
    REAL(wp), OPTIONAL, INTENT(IN) :: safe
    TYPE(rmatrix), OPTIONAL, INTENT(IN) :: mass
    LOGICAL, OPTIONAL, INTENT(IN) :: forcenumjac
    INTEGER, OPTIONAL, INTENT(IN) :: normtype
    INTEGER, OPTIONAL, INTENT(IN) :: linsolver
    INTEGER, OPTIONAL, INTENT(IN) :: nlimit
    ! ******************* Internal variables *******************
    INTEGER :: tdir, iter, itermfail, i
    REAL(wp) :: habs, h, hold, erracc, hfact, rel, minnrm
    REAL(wp) :: t
    LOGICAL :: gotznew, tooslow, haverate, jaccurrent, newtfail
    LOGICAL :: rejected, first, usenumjac
    REAL(wp), DIMENSION(:,:), ALLOCATABLE :: zk, zk0, dz, fitx
    REAL(wp), DIMENSION(SIZE(y)) :: scal
    REAL(wp), DIMENSION(SIZE(y)) :: y1my0
    REAL(wp), DIMENSION(:), ALLOCATABLE :: fac
    REAL(wp) :: newnrm, oldnrm, errit, rate, err, accuisafe, thissafe
    TYPE(rmatrix) :: jac
    TYPE(iterationm) :: iterm
    TYPE(mtx_type) :: itermtype
    REAL(wp) :: quot, newtscal
    CHARACTER (LEN=*), PARAMETER :: error2 = 'ERROR2: Step SIZE too small'
    INTEGER :: vbstep, vbjac, vbfail, vbtiter
    REAL(wp) :: vbhinit
    REAL(KIND(1.D0)) :: timer
#ifdef MPI
    INCLUDE MPIINCLUDEFILE
    INTEGER :: worldgroup, stagegroup
    INTEGER, DIMENSION(3,1) :: RANGE
#endif


    ! ******************* Code start *******************
    first=.TRUE.
    haverate=.FALSE.
    jaccurrent=.TRUE.
    itermfail=0

    ! Set optional parameters
    IF (PRESENT(stats)) THEN
       stats%fevals = 0
       stats%nfailed = 0
       stats%nnewt = 0
       stats%nnewtfailed = 0
       stats%nf1 = 0
       stats%nf2 = 0
       stats%nf3 = 0
       stats%nf4 = 0
       stats%nerrfailed = 0
       stats%njac = 0
       stats%njacfevals = 0
       stats%rfact = 0
       stats%cfact = 0
       stats%rsolv = 0
       stats%csolv = 0
       stats%nliniter = 0
       stats%nprecond = 0
       stats%singulariterm = 0
       stats%steps = 0
       CALL SYSTEM_CLOCK(stats%timer1)
#ifdef DTIMEDEFINED
       CALL DTIME(stats%tarray)
#endif
    END IF

    ! setup method.
    CALL setup_method(info,name,order)
    IF (info/=0) RETURN

    IF (PRESENT(verboseunit)) iverboseunit = verboseunit
    IF (PRESENT(verboselevel)) iverboselevel = verboselevel
    IF (PRESENT(atol)) iatol = atol
    IF (PRESENT(rtol)) irtol = rtol
    IF (PRESENT(nrmdiv)) inrmdiv = nrmdiv
    IF (PRESENT(conskappa)) iconskappa = conskappa
    IF (PRESENT(kappa)) ikappa = kappa
    IF (PRESENT(kappasafe)) ikappasafe = kappasafe
    IF (PRESENT(hmin)) ihmin = hmin
    IF (PRESENT(minjrefactquot)) ihconstl = minjrefactquot
    IF (PRESENT(maxjrefactquot)) ihconstm = maxjrefactquot
    IF (PRESENT(maxit)) imaxit = maxit
    IF (PRESENT(jactol)) ijactol = jactol
    IF (PRESENT(errunit)) ierrunit = errunit
    IF (PRESENT(optit)) ioptit = optit
    IF (PRESENT(polfitorder)) ipolfitorder = polfitorder
    IF (PRESENT(gustafsson)) igustafsson = gustafsson
    IF (PRESENT(safe)) isafe = safe
    IF (PRESENT(normtype)) inormtype = normtype
    IF (PRESENT(linsolver)) ilinsolver = linsolver
    IF (PRESENT(nlimit)) inlimit = nlimit
    IF (ipolfitorder>0) THEN
       ipolfit=.TRUE.
    ELSE
       ipolfit=.FALSE.
    END IF
    usenumjac = .FALSE.
#ifndef NOOPTIONALFNK
    IF (.NOT.PRESENT(calcjac)) usenumjac = .TRUE.
#endif
    IF (PRESENT(forcenumjac)) THEN
       IF (forcenumjac) usenumjac = .TRUE.
    END IF

#ifdef MPI
    CALL MPI_COMM_SIZE(MPI_COMM_WORLD, nprocs,info)
    IF (info/=MPI_SUCCESS) THEN
       info = ERRMPIFAILED
       RETURN
    ELSE IF (nprocs<stages) THEN
       info = ERRMPIPROCS
       RETURN
    ELSE IF (nprocs>stages) THEN
       CALL MPI_COMM_GROUP(MPI_COMM_WORLD,worldgroup,info)
       IF (info/=MPI_SUCCESS) THEN
          info = ERRMPIFAILED
          RETURN
       END IF
       RANGE(1,1)=0
       RANGE(2,1)=stages-1
       RANGE(3,1)=1
       CALL MPI_GROUP_RANGE_INCL(worldgroup,1,RANGE,stagegroup,info)
       IF (info/=MPI_SUCCESS) THEN
          info = ERRMPIFAILED
          RETURN
       END IF
       CALL MPI_COMM_CREATE(MPI_COMM_WORLD,stagegroup,communicator,info)
       IF (info/=MPI_SUCCESS) THEN
          info = ERRMPIFAILED
          RETURN
       END IF
       IF (communicator==MPI_COMM_NULL) THEN
          info = 0
          RETURN
       END IF
    ELSE
       CALL MPI_COMM_DUP(MPI_COMM_WORLD,communicator,info)
       IF (info/=MPI_SUCCESS) THEN
          info = ERRMPIFAILED
          RETURN
       END IF
    END IF
    CALL MPI_COMM_RANK(communicator, myrank,info)
    IF (info/=MPI_SUCCESS) THEN
       info = ERRMPIFAILED
info=-25
       RETURN
    END IF
    myid = myrank+1
    SELECT CASE (wp)
    CASE (KIND(1.d0))
       mpi_realwp = MPI_DOUBLE_PRECISION
    CASE (KIND(1.e0))
       mpi_realwp = MPI_REAL
    CASE default
       info = ERRMPIKINDUNAVAILABLE
       RETURN
    END SELECT
#endif

#ifdef MPI
    IF ((iverboselevel>0).and.(myrank==0)) THEN
#else
    IF (iverboselevel>0) THEN
#endif
       WRITE (iverboseunit,*) '------- Stiff ODE Solver v. 1.0 --------'
       WRITE (iverboseunit,*) 'Method name:        ',rkmethod 
       WRITE (iverboseunit,'(A,I17)') ' Method order:',rkorder
       WRITE (iverboseunit,*) 'Optional parameters:'
       IF (PRESENT(stats)) THEN
          WRITE (iverboseunit,*) '   Statistics:      on'
       ELSE
          WRITE (iverboseunit,*) '   Statistics:      off'
       END IF
       IF (ipolfit) THEN
          WRITE (iverboseunit,'(A,I2,A)')&
               & '    LS inital guess: on(',ipolfitorder,')'
       ELSE
          WRITE (iverboseunit,*) '   LS inital guess: off'
       END IF
       IF (usenumjac) THEN
          WRITE (iverboseunit,*) '   Jacobian:        numerical'
       ELSE
          WRITE (iverboseunit,*) '   Jacobian:        user supplied'
       END IF
       WRITE (iverboseunit,'(A,E16.4)') '    atol:      ', iatol
       WRITE (iverboseunit,'(A,E16.4)') '    rtol:      ', irtol
       WRITE (iverboseunit,'(A,E16.4)') '    kappa:     ', ikappa
       WRITE (iverboseunit,'(A,E16.4)') '    conskappa: ',iconskappa
       WRITE (iverboseunit,'(A,E16.4)') '    kappasafe: ', ikappasafe
       WRITE (iverboseunit,'(A,E16.4)') '    hmin:      ', ihmin
       WRITE (iverboseunit,'(A,E16.4)') '    safe:      ', isafe
       WRITE (iverboseunit,'(A,I16)') '    maxit:     ', imaxit
       WRITE (iverboseunit,'(A,I16)') '    optit:     ', ioptit
       WRITE (iverboseunit,'(A,E16.4)') '    jfactminq: ', ihconstl
       WRITE (iverboseunit,'(A,E16.4)') '    jfactmaxq: ', ihconstm
       WRITE (iverboseunit,'(A,E16.4)') '    jactol:    ', ijactol
       WRITE (iverboseunit,'(A,I16)') '    errunit:   ', ierrunit
       WRITE (iverboseunit,*) '------------- Integrating..... --------------'
    END IF
    
    ! initialize method dependend parameters
    ALLOCATE(zk(SIZE(y),stages))
    ALLOCATE(dz(SIZE(y),stages))
    IF (ipolfit) THEN
       ALLOCATE(zk0(SIZE(y),stages))
       ALLOCATE(fitx(ipolfitorder+1,SIZE(y)))
       ! Form matrix to use for normal eq.-fit for initial newton guess.
       !    Form A^T, A(i,:)=[c(i)^q,...,c(i),1]
       ALLOCATE(fitA(stages,ipolfitorder+1))
       fitA(:,1)=1.0_wp
       DO i=1,ipolfitorder
          fitA(:,i+1) = rkc**i
       END DO
       !    Form G, G*G^T=A^T*A, G lower triangular.
       ALLOCATE(fitG(ipolfitorder+1,ipolfitorder+1))
       ! fitG = MATMUL(TRANSPOSE(fitA),fitA)
       CALL gemm('T','N',SIZE(fitA,DIM=2),SIZE(fitA,DIM=2),SIZE(fitA,DIM=1),&
            & 1.0_wp,fitA,SIZE(fitA,DIM=1),fitA,SIZE(fitA,DIM=1),0.0_wp,fitG,&
            & SIZE(fitG,DIM=1))
       CALL potrf('L',ipolfitorder+1,fitG,ipolfitorder+1,info)
       IF (info/=0) THEN
          info = ERRCHOLFACTFAILED
          CALL free_vars
          RETURN
       END IF
       ALLOCATE(fitAnew(stages,ipolfitorder+1))
       fitAnew(:,1) = 1.0_wp
    END IF

    IF (PRESENT(mass)) THEN
       CALL mtx_conform(jtype,mass%mtype,itermtype,info)
       IF (info/=0) THEN
          CALL free_vars
          RETURN
       END IF
    ELSE
       itermtype = jtype
    END IF
    CALL allocate_iterm(iterm,itermtype,info)
    IF (info/=0) THEN
       CALL free_vars
       RETURN
    END IF
    
    ! determine starting step size.
    tdir = SIGN(1.0_wp,tend-tbeg) ! direction of integeration
    IF (PRESENT(hinit)) THEN
       h0 = ABS(hinit)
    END IF
    habs = h0
    h = tdir*habs

    ! initialize 
    accuisafe=isafe
    thissafe=isafe
    t = tbeg
    CALL allocate_mtx(jac,jtype,info)
    IF (info/=0) THEN
       CALL free_vars
       info = ERRJACALLOFAILED
       RETURN
    END IF

    IF (usenumjac) THEN
       ALLOCATE(fac(SIZE(y)))
#ifdef MPI
       ALLOCATE(slice(stages+1))
       ALLOCATE(slicesize(stages))
       ALLOCATE(sliceoffs(stages))

       slice(1) = 1
       DO i=2,stages
          slice(i) = slice(i-1)+(CEILING(REAL(SIZE(y))/nprocs))
       END DO
       slice(stages+1) = SIZE(y)+1
       DO i=1,stages
          slicesize(i) = slice(i+1)-slice(i)
          IF (i>1) THEN
             sliceoffs(i) = sliceoffs(i-1)+slicesize(i-1)
          ELSE
             sliceoffs(i) = 0
          END IF
       END DO
       CALL MPI_TYPE_CONTIGUOUS(SIZE(jac%a(:,1)),mpi_realwp,&
            & jaccoltype,info)
       IF (info/=MPI_SUCCESS) THEN
          info = ERRMPIFAILED
info=-26
          RETURN
       END IF
       CALL MPI_TYPE_COMMIT(jaccoltype,info)
       IF (info/=MPI_SUCCESS) THEN
          info = ERRMPIFAILED
info=-27
          RETURN
       END IF
#endif
       CALL numjac(t,y,fnk,jac,fac,stats,.TRUE.)
    ELSE
       CALL calcjac(t,y,jac)
    END IF
    IF (PRESENT(stats)) stats%njac = stats%njac+1
    ! form iteration matrix
    CALL form_iterm
    IF (info/=0) RETURN

#ifdef MPI
    IF ((iverboselevel>1).and.(myrank==0)) THEN
#else
    IF (iverboselevel>1) THEN
#endif
       WRITE(unit=iverboseunit,fmt=*)&
            & '                    successfull             initial  total'
       WRITE(unit=iverboseunit,fmt=*)&
            & 'step       t         h    iter   jac fail      h     iter'
       vbstep = 1
       vbjac = 1
       vbfail = 0
       vbtiter = 0
    END IF

#ifdef NOOPTIONALFNK
    CALL resultfnk(t,y)
#else
    IF (PRESENT(resultfnk)) CALL resultfnk(t,y)
#endif

    main: DO WHILE (t < tend)
       IF (1.1_wp*habs >= ABS(tend-t)) THEN
          habs = ABS(tend-t)
          h = tdir*habs
          rel = habs/hfact
          IF ((rel<relmin).or.(rel>relmax)) THEN
             CALL form_iterm
             IF (info/=0) RETURN
          END IF
       END IF
       IF (iverboselevel>1) vbhinit = h
       rejected = .FALSE.
       onestep: DO
          gotznew = .FALSE.
          DO WHILE (.NOT.gotznew)
             IF (habs < ihmin) THEN
                ! failure
#ifdef MPI
                IF (myrank==0) THEN
#endif
                WRITE (UNIT=ierrunit,FMT='(A)') error2
#ifdef MPI
                END IF
#endif
                info = ERRSTEPSIZETOOSMALL
                tend = t
                EXIT main
             END IF
             ! compute starting value for newton iteration.
             ! possibly by extrapolation
             IF (first.OR.(.NOT.ipolfit)) THEN
                zk = 0
                CALL norm_updatescal(scal,minnrm,y)
             ELSE
                ! evaluate LSfitted polynomial at new points.
                DO i=1,ipolfitorder
                   fitAnew(:,i+1)=(1.0_wp+(h/hold)*rkc)**i
                END DO
                ! zk = transpose(matmul(fitAnew,fitx))-&
                !      & spread(y1my0,DIM=2,NCOPIES=stages)
                zk = SPREAD(y1my0,DIM=2,NCOPIES=stages)
                CALL gemm('T','T',SIZE(fitx,DIM=2),SIZE(fitAnew,DIM=1),&
                     & SIZE(fitx,DIM=1),1.0_wp,fitx,SIZE(fitx,DIM=1),&
                     & fitAnew,SIZE(fitAnew,DIM=1),-1.0_wp,zk,SIZE(zk,DIM=1))
                ! scaling array for norm
                CALL formy1(zk,dz(:,1))
                CALL norm_updatescal(scal,minnrm,y+dz(:,1),y)

                CALL rkencode(zk)
             END IF
             tooslow = .FALSE.
             iter = 1
             simplnewt: DO
                ! solve linear system at (t, y+somefunction(zk)).
                ! method dependend.
                IF (PRESENT(stats)) stats%nnewt = stats%nnewt + 1
                CALL solve(fnk,t,y,zk,jac,iterm,dz,h,minnrm,info,stats,mass)
                IF (info/=0) THEN
                   IF (PRESENT(stats)) stats%nf4=stats%nf4+1
                   info = 0
                   tooslow = .TRUE.
                   haverate = .FALSE.
                   newtfail = .FALSE.
                   EXIT simplnewt
                END IF
                IF (habs/=hfact) THEN
                   rel = 2 / (1+habs/hfact)
                   dz = rel*dz
                END IF
                CALL calcnorm(newnrm,scal,dz)

#ifdef MPI
                IF (myrank==0) THEN
#endif
                IF (iverboselevel>10) WRITE(unit=iverboseunit,&
                     & fmt='(A,E11.4)') 'newton nrm=',newnrm
#ifdef MPI
                END IF
#endif
                zk = zk+dz
                IF (newnrm <= minnrm) THEN
                   gotznew = .TRUE.
                   EXIT simplnewt
                ELSE IF (iter == 1) THEN
                   IF (haverate) THEN
                      errit = newnrm*rate/(1-rate)
                      IF (errit <= iconskappa) THEN
                         gotznew = .TRUE.
                         rate = rate**iincrate
                         EXIT simplnewt
                      END IF
                   END IF
                ELSE IF (newnrm > inrmdiv*oldnrm) THEN
                   IF (PRESENT(stats)) stats%nf1=stats%nf1+1
                   tooslow = .TRUE.
                   haverate = .FALSE.
                   newtfail = .TRUE.
                   EXIT simplnewt
                ELSE
                   IF (iter==2) THEN
                      rate = MIN(newnrm/oldnrm,0.999_wp)
                   ELSE
                      rate = MIN(SQRT(rate*newnrm/oldnrm),0.999_wp)
                   END IF
                   haverate = .TRUE.
                   errit = newnrm*rate/(1-rate)
                   IF (errit <= ikappa) THEN
                      gotznew = .TRUE.
                      EXIT simplnewt
                   ELSE IF (iter == imaxit) THEN
                      IF (PRESENT(stats)) stats%nf2=stats%nf2+1
                      tooslow = .TRUE.
                      newtfail = .TRUE.
                      EXIT simplnewt
                   ELSE IF (ikappa*ikappasafe < &
                        & errit*rate**(imaxit-iter)) THEN
                      IF (PRESENT(stats)) stats%nf3=stats%nf3+1
                      tooslow = .TRUE.
                      newtfail = .TRUE.
                      EXIT simplnewt
                   END IF
                END IF
                oldnrm = newnrm
                iter = iter+1
             END DO simplnewt
             IF (iverboselevel>1) vbtiter = vbtiter+iter
             IF (tooslow) THEN
                IF (iverboselevel>1) vbfail = vbfail+1
                IF (PRESENT(stats)) THEN
                   stats%nfailed = stats%nfailed+1
                   stats%nnewtfailed = stats%nnewtfailed+1
                END IF
#ifdef MPI
                IF (myrank==0) THEN
#endif
                IF (iverboselevel>10) WRITE(unit=iverboseunit,&
                     & fmt='(A,I4,A,E11.4,A,E11.4)') 'Convergence too slow,',&
                     & iter,' iterations, t=',t,' h=',h
#ifdef MPI
                END IF
#endif
                rejected = .TRUE.
                ! Recalculate jacobian or reduce step size.
                IF ((.NOT.jaccurrent).and.newtfail) THEN
                   IF (usenumjac) THEN
                      CALL numjac(t,y,fnk,jac,fac,stats)
                   ELSE
                      CALL calcjac(t,y,jac)
                   END IF
                   IF (iverboselevel>1) vbjac = vbjac+1
                   IF (PRESENT(stats)) stats%njac = stats%njac+1
                   CALL form_iterm
                   IF (info/=0) RETURN
                   jaccurrent = .TRUE.
                   haverate = .FALSE.
                   thissafe=thissafe*0.3d0
                ELSE
                   IF (haverate) THEN
                      habs = habs*0.8_wp*MAX(0.0001_wp,MIN(20.0_wp,&
                           & errit*rate**(imaxit-iter)/ikappa)&
                           & )**(-1.0_wp/(4.0_wp+imaxit-iter))
                   ELSE
                      habs = MAX(0.3_wp*habs,ihmin)
                   END IF
                   h = tdir*habs
                   rel = habs/hfact
                   IF ((rel<relmin).or.(rel>relmax)) THEN
                      CALL form_iterm
                      IF (info/=0) RETURN
                   END IF
                   thissafe=thissafe*0.1d0
                END IF
                haverate = .FALSE.
             END IF
          END DO
          ! decode zk dependent on method parameters
          CALL rkdecode(zk)
          ! estimate the local error
          CALL esterr(fnk,t,y,zk,h,jac,minnrm,iterm,scal,rejected.OR.first,&
               & err,stats,mass)
          ! determine step size scaling
          IF (iter>ioptit) THEN
             IF (jaccurrent) THEN
                newtscal = (iter/REAL(ioptit))**(-0.5_wp)
             ELSE
                newtscal = (iter/REAL(ioptit))**(-0.75_wp)
             END IF
             thissafe=thissafe*newtscal
          END IF
          accuisafe=SQRT(accuisafe*thissafe)
          quot = MAX(iquotl,MIN(iquotm,&
               & accuisafe/(err**(1.0_wp/(errorder+1)))))
          thissafe=isafe

#ifdef MPI
          IF (myrank==0) THEN
#endif
          IF (iverboselevel>10) WRITE(unit=iverboseunit,&
               & fmt='(A,E11.4,A,E11.4,A,E11.4)') 'Error estimate is ',&
               & err,' h=',h,' quot=',quot
#ifdef MPI
          END IF
#endif
          IF (err > 1) THEN
             IF (iverboselevel>1) THEN
                vbfail = vbfail+1
             END IF
             IF (PRESENT(stats)) THEN
                stats%nfailed = stats%nfailed+1
                stats%nerrfailed = stats%nerrfailed+1
             END IF
             IF (first) THEN
                habs = 0.1_wp*habs
             ELSE
                habs = quot*habs
             END IF
             h = tdir*habs
             rel = habs/hfact
             IF ((rel<relmin).or.(rel>relmax)) THEN
                CALL form_iterm
                IF (info/=0) RETURN
             END IF
             rejected = .TRUE.
             haverate = .FALSE.
          ELSE
             ! successful step
             ! Predictive controller of Gustafsson
             IF (igustafsson) THEN
                IF (.NOT.first) THEN
                   quot = MIN(quot,MAX(iquotl,MIN(iquotm,(h/hold)*&
                        & (erracc/err**2)**(1.0_wp/((errorder+1)*isafe)))))
                END IF
                erracc=MAX(0.01_wp,err)
             END IF
             EXIT onestep
          END IF
       END DO onestep
       ! form solution at t+h
       CALL formy1(zk,y1my0)
       y = y+y1my0
       first = .FALSE.
       jaccurrent = .FALSE.

#ifdef MPI
       IF ((iverboselevel>1).and.(myrank==0)) THEN
#else
       IF (iverboselevel>1) THEN
#endif
          IF (vbfail>0) THEN
             WRITE(unit=iverboseunit,&
                  & fmt='(I5,ES11.3,ES10.3,3I5.0,ES11.2,I6)')&
                  & vbstep, t, h, iter, vbjac, vbfail, vbhinit, vbtiter
          ELSE
             WRITE(unit=iverboseunit,fmt='(I5,ES11.3,ES10.3,2I5.0)')&
                  & vbstep, t, h, iter, vbjac
          END IF
          vbstep = vbstep+1
          vbjac = 0
          vbfail = 0
          vbtiter = 0
       END IF
       t = t+h
       hold = h

       IF (ipolfit) THEN ! if polynomial fitting for newton guess is used...
          ! zk must NOT be transformed.
          ! Compute pol. fit using normal equations.
          ! fitx = TRANSPOSE(MATMUL(zk,fitA))
          CALL gemm('T','T',SIZE(fitA,DIM=2),SIZE(zk,DIM=1),&
               & SIZE(fitA,DIM=1),1.0_wp,fitA,SIZE(fitA,DIM=1),&
               & zk,SIZE(zk,DIM=1),0.0_wp,fitx,SIZE(fitx,DIM=1))
          CALL trtrs('L','N','N',SIZE(fitG,DIM=1),SIZE(fitx,DIM=2),&
               & fitG,SIZE(fitG,DIM=1),fitx,SIZE(fitx,DIM=1),info)
          IF (info/=0) THEN
             CALL free_vars
             info = ERRSOLVEFAILED
             RETURN
          END IF
          CALL trtrs('L','T','N',SIZE(fitG,DIM=1),SIZE(fitx,DIM=2),&
               & fitG,SIZE(fitG,DIM=1),fitx,SIZE(fitx,DIM=1),info)
          IF (info/=0) THEN
             CALL free_vars
             info = ERRSOLVEFAILED
             RETURN
          END IF
       END IF

       ! Call result function
#ifndef NOOPTIONALFNK
       IF (PRESENT(resultfnk)) THEN
#endif
       CALL resultfnk(t,y)
#ifndef NOOPTIONALFNK
       END IF
#endif

       IF (PRESENT(stats)) stats%steps = stats%steps+1
       IF (((quot<ihconstl).OR.(quot>ihconstm).OR.&
          & (ilinsolver/=SLV_DIRECT)).AND.(t<tend)) THEN
          habs = quot*habs
          h = tdir*habs

          IF (haverate.AND.(rate>ijactol)) THEN
             haverate = .FALSE.
             IF (iverboselevel>1) vbjac = vbjac+1
             IF (usenumjac) THEN
                CALL numjac(t,y,fnk,jac,fac,stats)
             ELSE
                CALL calcjac(t,y,jac)
             END IF
             jaccurrent = .true.
             IF (PRESENT(stats)) stats%njac = stats%njac+1
             CALL form_iterm
             IF (info/=0) RETURN
          ELSE
             rel = habs/hfact
             IF ((rel<relmin).or.(rel>relmax)) THEN
                CALL form_iterm
                IF (info/=0) RETURN
             END IF
          END IF
       END IF
    END DO main
#ifdef MPI
    IF (PRESENT(stats).AND.(iverboselevel>0).and.(myrank==0)) THEN
#else
    IF (PRESENT(stats).AND.(iverboselevel>0)) THEN
#endif
       CALL SYSTEM_CLOCK(stats%timer2,stats%clicks,stats%maxtimer)
#ifdef DTIMEDEFINED
       CALL DTIME(stats%tarray)
#endif
       IF (stats%timer2<stats%timer1) THEN
          timer = (stats%maxtimer+stats%timer2-stats%timer1)*1.d0/stats%clicks
       ELSE
          timer = (stats%timer2-stats%timer1)*1.d0/stats%clicks
       END IF
       WRITE (iverboseunit,*) '---------------- STATISTICS -----------------'
       WRITE (iverboseunit,*) 'Total no. of. succesfull steps:  ',&
            & stats%steps
       WRITE (iverboseunit,*) 'Function evaluations:            ',&
            & stats%fevals
       WRITE (iverboseunit,*) 'Jacobian evaluations:            ',&
            & stats%njac
       IF (usenumjac) THEN
#ifndef MPI
          WRITE (iverboseunit,*) 'Jacobian (function evaluations): ',&
               & stats%njacfevals
#else
          WRITE (iverboseunit,*) 'Jacobian (function evaluations): ',&
               & stats%njacfevals, ' (approximately)'
#endif
       END IF
       IF (ilinsolver == SLV_DIRECT) THEN
          WRITE (iverboseunit,*) 'No. of. real factorizations:     ',&
               & stats%rfact
          WRITE (iverboseunit,*) 'No. of. complex factorizations:  ',&
               & stats%cfact
       ELSE
          WRITE (iverboseunit,*) 'No. of. iterations in lin. solv.:',&
              & stats%nliniter
          WRITE (iverboseunit,*) 'No. of. preconditioner recalc.:  ',&
              & stats%nprecond
       END IF
       WRITE (iverboseunit,*) 'No. of. real lin. sys. solves:   ',&
            & stats%rsolv
       WRITE (iverboseunit,*) 'No. of. complex lin. sys. solves ',&
            & stats%csolv
       WRITE (iverboseunit,*) 'Total no. of Newton iterations:  ',&
            & stats%nnewt
       WRITE (iverboseunit,*) 'No. of singular iterationmtx.:   ',&
            & stats%singulariterm
       WRITE (iverboseunit,*) 'Total no. of failed steps:       ',&
            & stats%nfailed
       WRITE (iverboseunit,*) 'No. of failed Newton steps:      ',&
            & stats%nnewtfailed
       WRITE (iverboseunit,*) '                  - type 1:      ',stats%nf1
       WRITE (iverboseunit,*) '                  - type 2:      ',stats%nf2
       WRITE (iverboseunit,*) '                  - type 3:      ',stats%nf3
       WRITE (iverboseunit,*) '                  - type 4:      ',stats%nf4
       WRITE (iverboseunit,*) 'No. of failed error estimations: ',&
            & stats%nerrfailed
       WRITE (iverboseunit,'(A,F20.3)') ' Timing (wall-clock):     ',timer
#ifdef DTIMEDEFINED
       WRITE (iverboseunit,'(A,F20.6)') ' Timing (execution):      ',&
            & stats%tarray(1)
#endif
       WRITE (iverboseunit,*) '---------------------------------------------'
    END IF

    CALL free_vars
    RETURN
  CONTAINS
    SUBROUTINE free_vars
      CALL deallocate_mtx(jac)
      CALL deallocate_iterm(iterm)
      IF (ALLOCATED(zk)) DEALLOCATE(zk)
      IF (ALLOCATED(zk0)) DEALLOCATE(zk0)
      IF (ALLOCATED(fitx)) DEALLOCATE(fitx)
      IF (ALLOCATED(dz)) DEALLOCATE(dz)
      IF (usenumjac) THEN
         IF (ALLOCATED(fac)) DEALLOCATE(fac)
#ifdef MPI
         IF (ALLOCATED(slice)) DEALLOCATE(slice)
         IF (ALLOCATED(slicesize)) DEALLOCATE(slicesize)
         IF (ALLOCATED(sliceoffs)) THEN
            DEALLOCATE(sliceoffs)
            CALL MPI_TYPE_FREE(jaccoltype,info)
         END IF
#endif
      END IF
      CALL free_method()
      RETURN
    END SUBROUTINE free_vars
    SUBROUTINE form_iterm
      IF (ilinsolver == SLV_DIRECT) THEN
         DO
            CALL calciterm(iterm,t,y,h,jac,info,mass,stats) 
            IF (info/=0) THEN
               itermfail = itermfail+1
               IF (itermfail>imaxitermfail) THEN
                  ! failure
#ifdef MPI
                  IF (myrank==0) THEN
#endif
                  WRITE (UNIT=ierrunit,FMT='(A)')&
                       & 'ERROR1: Iteration matrix repeatedly singular'
#ifdef MPI
                  END IF
#endif
                  tend = t
                  CALL free_vars
                  info = ERRITERSINGULAR
                  EXIT
               ELSE
                  habs = habs/2
                  h = tdir*habs
               END IF
            ELSE
               itermfail = 0
               hfact = habs
               EXIT
            END IF
         END DO
      ELSE
         hfact = habs
      END IF
      RETURN
    END SUBROUTINE form_iterm
  END SUBROUTINE integ

  ! ****************** Norm related routines ******************
  SUBROUTINE norm_updatescal(scal,minnrm,y1,y2)
    REAL(wp), DIMENSION(:), INTENT(in) :: y1
    REAL(wp), DIMENSION(SIZE(y1)), INTENT(out) :: scal
    REAL(wp), INTENT(out) :: minnrm
    REAL(wp), DIMENSION(:), INTENT(in), OPTIONAL :: y2
    REAL(wp), DIMENSION(SIZE(y1)) :: w1
    SELECT CASE (inormtype)
    CASE (NORM_2)
       w1 = ABS(y1)
       IF (PRESENT(y2)) THEN
          scal = 1/(iatol + MAX(ABS(y2),w1)*irtol)
       ELSE
          scal = 1/(iatol + w1*irtol)
       END IF
       minnrm = 100*eps*SQRT(SUM(w1*scal))/SIZE(w1)
    CASE default
       w1 = ABS(y1)
       IF (PRESENT(y2)) THEN
          scal = 1/(iatol + MAX(ABS(y2),w1)*irtol)
       ELSE
          scal = 1/(iatol + w1*irtol)
       END IF
       minnrm = 100*eps*MAXVAL(w1*scal)
    END SELECT
    RETURN
  END SUBROUTINE norm_updatescal

  SUBROUTINE calcnorm_dim2(nrm,scal,y)
    REAL(wp), INTENT(out) :: nrm
    REAL(wp), DIMENSION(:), INTENT(in) :: scal
    REAL(wp), DIMENSION(:,:), INTENT(in) :: y
    SELECT CASE (inormtype)
    CASE (NORM_2)
       nrm = SQRT(SUM((y*SPREAD(scal,&
            & DIM=2,ncopies=SIZE(y,DIM=2)))**2)/SIZE(y))
    CASE default
       nrm = MAXVAL(ABS(y*SPREAD(scal,DIM=2,ncopies=SIZE(y,DIM=2))))
    END SELECT
    RETURN
  END SUBROUTINE calcnorm_dim2

  SUBROUTINE calcnorm_dim1(nrm,scal,y)
    REAL(wp), INTENT(out) :: nrm
    REAL(wp), DIMENSION(:), INTENT(in) :: scal
    REAL(wp), DIMENSION(:), INTENT(in) :: y
    SELECT CASE (inormtype)
    CASE (NORM_2)
       nrm = SQRT(SUM((y*scal)**2)/SIZE(y))
    CASE default
       nrm = MAXVAL(ABS(y*scal))
    END SELECT
    RETURN
  END SUBROUTINE calcnorm_dim1

  ! ***************** Method related routines *****************

  SUBROUTINE allocate_iterm(m,itermtype,info)
    ! allocates the iteration matrix m of appropriate size and
    ! type according to method.
    TYPE(iterationm), INTENT(INOUT) :: m
    TYPE(mtx_type), INTENT(INOUT) :: itermtype
    INTEGER, INTENT(out) :: info
    INTEGER :: i
    ! Determine block size according to problem and jacobian type.
    IF (ilinsolver==SLV_DIRECT) THEN
       NULLIFY(m%im)
       NULLIFY(m%ic)
       IF (ALLOCATED(reigenvals)) THEN
          ALLOCATE(m%m(SIZE(reigenvals)))
#ifndef MPI
          DO i=1,SIZE(reigenvals)
             CALL allocate_mtx(m%m(i),itermtype,info,factorizable=.TRUE.)
             IF (info/=0) THEN
                info = ERRITERALLOFAILED
                RETURN
             END IF
          END DO
#else
          CALL allocate_mtx(m%m(myid),itermtype,info,factorizable=.TRUE.)
#endif
       ELSE
          NULLIFY(m%m)
       END IF
       IF (ALLOCATED(ceigenvals)) THEN
          ALLOCATE(m%c(SIZE(ceigenvals)))
          DO i=1,SIZE(ceigenvals)
             CALL allocate_mtx(m%c(i),itermtype,info,factorizable=.TRUE.)
             IF (info/=0) THEN
                info = ERRITERALLOFAILED
                RETURN
             END IF
          END DO
       ELSE
          NULLIFY(m%c)
       END IF
    ELSE
       NULLIFY(m%m)
       NULLIFY(m%c)
       IF (ALLOCATED(reigenvals)) THEN
          ALLOCATE(m%im(SIZE(reigenvals)))
#ifndef MPI
          DO i=1,SIZE(reigenvals)
             CALL allocate_precond(ilinsolver,m%im(i),itermtype,info)
             IF (info/=0) RETURN
          END DO
#else
          CALL allocate_precond(ilinsolver,m%im(myid),itermtype,info)
#endif
       ELSE
          NULLIFY(m%im)
       END IF
       IF (ALLOCATED(ceigenvals)) THEN
          ALLOCATE(m%ic(SIZE(ceigenvals)))
          DO i=1,SIZE(ceigenvals)
             CALL allocate_precond(ilinsolver,m%ic(i),itermtype,info)
             IF (info/=0) RETURN
          END DO
       ELSE
          NULLIFY(m%ic)
       END IF
    END IF
    RETURN
  END SUBROUTINE allocate_iterm

  SUBROUTINE deallocate_iterm(m)
    TYPE(iterationm), INTENT(INOUT) :: m
    INTEGER :: i
    IF (ASSOCIATED(m%m)) THEN
#ifndef MPI
       DO i=1,SIZE(m%m)
          CALL deallocate_mtx(m%m(i))
       END DO
#else
       CALL deallocate_mtx(m%m(myid))
#endif
       DEALLOCATE(m%m)
    END IF
    IF (ASSOCIATED(m%c)) THEN
       DO i=1,SIZE(m%c)
          CALL deallocate_mtx(m%c(i))
       END DO
       DEALLOCATE(m%c)
    END IF
    IF (ASSOCIATED(m%im)) THEN
#ifndef MPI
       DO i=1,SIZE(m%im)
          CALL deallocate_iterative_mtx(m%im(i))
       END DO
#else
       CALL deallocate_iterative_mtx(m%im(myid))
#endif
       DEALLOCATE(m%im)
    END IF
    IF (ASSOCIATED(m%ic)) THEN
       DO i=1,SIZE(m%ic)
          CALL deallocate_iterative_mtx(m%ic(i))
       END DO
       DEALLOCATE(m%ic)
    END IF
    RETURN
  END SUBROUTINE deallocate_iterm

  SUBROUTINE calciterm(m,t,y,h,jac,info,mass,stats)
    ! calculates and factorizes the iteration matrix.
    TYPE(iterationm), INTENT(INOUT) :: m
    REAL(wp), INTENT(IN) :: t
    REAL(wp), DIMENSION(:), INTENT(IN) :: y
    REAL(wp), INTENT(IN) :: h
    TYPE(rmatrix), INTENT(IN) :: jac
    INTEGER, INTENT(out) :: info
    TYPE(rmatrix), INTENT(IN), OPTIONAL :: mass
    TYPE(statistics), INTENT(inout), OPTIONAL :: stats
    INTEGER :: i
    IF (ALLOCATED(reigenvals)) THEN
#ifndef MPI
       DO i=1,SIZE(reigenvals)
          CALL fac_aimj(reigenvals(i)/h,jac,m%m(i),info,mass)
          IF (info/=0) THEN
             IF (PRESENT(stats)) THEN
                stats%rfact = stats%rfact+i
                stats%singulariterm = stats%singulariterm+1
             END IF
             RETURN
          END IF
       END DO
#else
       CALL fac_aimj(reigenvals(myid)/h,jac,m%m(myid),info,mass)
       IF (info/=0) THEN
          RETURN
       END IF
#endif
       IF (PRESENT(stats)) THEN
          stats%rfact = stats%rfact+SIZE(reigenvals)
       END IF
    END IF
    IF (ALLOCATED(ceigenvals)) THEN
       DO i=1,SIZE(ceigenvals)
          CALL fac_aimj(ceigenvals(i)/h,jac,m%c(i),info,mass)
          IF (info/=0) THEN
             IF (PRESENT(stats)) THEN
                stats%rfact = stats%rfact+SIZE(reigenvals)
                stats%cfact = stats%cfact+i
                stats%singulariterm = stats%singulariterm+1
             END IF
             RETURN
          END IF
       END DO
       IF (PRESENT(stats)) THEN
          stats%cfact = stats%cfact+SIZE(ceigenvals)
       END IF
    END IF
    RETURN
  END SUBROUTINE calciterm

SUBROUTINE solve(fnk,t,y,z,jac,m,dz,h,minnrm,info,stats,mass)
  INTERFACE
     SUBROUTINE fnk(t,y,ydot)
       USE defines
       REAL(wp), INTENT(IN) :: t
       REAL(wp), DIMENSION(:), INTENT(IN) :: y
       REAL(wp), DIMENSION(:), INTENT(OUT) :: ydot
     END SUBROUTINE fnk
  END INTERFACE
  REAL(wp), INTENT(IN) :: t
  REAL(wp), DIMENSION(:), INTENT(IN) :: y
  REAL(wp), DIMENSION(:,:), INTENT(IN) :: z
  TYPE(rmatrix), INTENT(IN) :: jac
  TYPE(iterationm), INTENT(INOUT) :: m
  REAL(wp), DIMENSION(:,:), INTENT(OUT) :: dz
  REAL(wp), INTENT(IN) :: h
  REAL(wp), INTENT(IN) :: minnrm
  INTEGER, INTENT(OUT) :: info
  TYPE(statistics), INTENT(inout), OPTIONAL :: stats
  TYPE(rmatrix), OPTIONAL, INTENT(IN) :: mass
  REAL(wp), DIMENSION(SIZE(dz,DIM=1),SIZE(dz,DIM=2)) :: z1,z2
#ifdef MPI
  REAL(wp), DIMENSION(SIZE(dz,DIM=1)) :: z3
  INCLUDE MPIINCLUDEFILE
#endif
  COMPLEX(wp), DIMENSION(SIZE(dz,DIM=1)) :: zc
  INTEGER :: offs, i
  REAL(wp) :: fac
  COMPLEX(wp) :: facc
  ! generate right hand side
  SELECT CASE (method)
  CASE (1:2)
     ! dz = SPREAD(y,DIM=2,ncopies=stages)+MATMUL(z,rktt)
     dz = SPREAD(y,DIM=2,ncopies=stages)
     CALL gemm('N','T',SIZE(z,DIM=1),SIZE(rkt,DIM=1),SIZE(z,DIM=2),1.0_wp,&
          & z,SIZE(z,DIM=1),rkt,SIZE(rkt,DIM=1),1.0_wp,dz,SIZE(dz,DIM=1))
     DO i=1,stages
        CALL fnk(t+rkc(i)*h,dz(:,i),z1(:,i))
     END DO
     ! dz = MATMUL(z1,rktit)
     CALL gemm('N','T',SIZE(z1,DIM=1),SIZE(rkti,DIM=1),SIZE(z1,DIM=2),1.0_wp,&
          & z1,SIZE(z1,DIM=1),rkti,SIZE(rkti,DIM=1),0.0_wp,dz,SIZE(dz,DIM=1))
     IF (PRESENT(stats)) stats%fevals = stats%fevals+stages
     IF (ALLOCATED(reigenvals))THEN
        offs = SIZE(reigenvals)
        DO i=1, offs
           fac = reigenvals(i)/h
           IF (PRESENT(mass)) THEN
              CALL genmv(-fac,mass,z(:,i),1.0_wp,dz(:,i),info)
           ELSE
              dz(:,i) = dz(:,i)-z(:,i)*fac
           END IF
           IF (ilinsolver==SLV_DIRECT) THEN
              CALL msolve(m%m(i),dz(:,i),info)
           ELSE
              IF (PRESENT(stats)) THEN
                 CALL itersolve(ilinsolver,fac,jac,dz(:,i),m%im(i),minnrm,&
                               & inlimit,info,mass,stats%nliniter,&
                               & stats%nprecond)
              ELSE
                 CALL itersolve(ilinsolver,fac,jac,dz(:,i),m%im(i),minnrm,&
                               & inlimit,info,mass)
              END IF
           END IF
           IF (info/=0) THEN
              RETURN
           END IF
        END DO
     ELSE
        offs = 0
     END IF
     IF (PRESENT(stats)) stats%rsolv = stats%rsolv + SIZE(reigenvals)
     IF (ALLOCATED(ceigenvals)) THEN
        DO i=1,SIZE(ceigenvals)
           IF (PRESENT(mass)) THEN
              CALL genmv(1.0_wp,mass,z(:,2*i-1+offs),0.0_wp,&
                   & z1(:,2*i-1+offs),info)
              CALL genmv(1.0_wp,mass,z(:,2*i+offs),0.0_wp,&
                   & z1(:,2*i+offs),info)
              facc = ceigenvals(i)/h
              zc = dz(:,2*i-1+offs)-z1(:,2*i-1+offs)*facc+CMPLX(0,1,KIND=wp)&
                   & *(dz(:,2*i+offs)-z1(:,2*i+offs)*facc)
           ELSE
              facc = ceigenvals(i)/h
              zc = dz(:,2*i-1+offs)-z(:,2*i-1+offs)*facc+CMPLX(0,1,KIND=wp)&
                   & *(dz(:,2*i+offs)-z(:,2*i+offs)*facc)
           END IF
           IF (ilinsolver==SLV_DIRECT) THEN
              CALL msolve(m%c(i),zc,info)
           ELSE
              IF (PRESENT(stats)) THEN
                 CALL itersolve(ilinsolver,facc,jac,zc,m%ic(i),minnrm,&
                               & inlimit,info,&
                               & mass,stats%nliniter,stats%nprecond)
              ELSE
                 CALL itersolve(ilinsolver,facc,jac,zc,m%ic(i),minnrm,&
                               & inlimit,info,mass)
              END IF
           END IF
           IF (info/=0) THEN
              RETURN
           END IF
           dz(:,2*i-1+offs) = REAL(zc,KIND=wp)
           dz(:,2*i+offs) = AIMAG(zc)
        END DO
     END IF
     IF (PRESENT(stats)) stats%csolv = stats%csolv + SIZE(ceigenvals)
  CASE(3)
#ifndef MPI
     z1 = SPREAD(y,DIM=2,ncopies=stages)+z
     DO i=1,stages
        CALL fnk(t+rkc(i)*h,z1(:,i),dz(:,i))
     END DO
#else
     z1(:,myid) = y+z(:,myid)
     CALL fnk(t+rkc(myid)*h,z1(:,myid),z3)
     CALL MPI_ALLGATHER(z3,SIZE(z3),mpi_realwp,dz,SIZE(z3),&
          & mpi_realwp,communicator,info)
     IF (info/=MPI_SUCCESS) THEN
        info = ERRMPIGATHERFAILED
        RETURN
     END IF
#endif
     IF (PRESENT(stats)) stats%fevals = stats%fevals+stages
#ifndef MPI
     IF (PRESENT(mass)) THEN
        CALL gemm('N','T',SIZE(z,DIM=1),SIZE(rkti,DIM=1),SIZE(z,DIM=2),&
          & 1.0_wp,z,SIZE(z,DIM=1),rkti,SIZE(rkti,DIM=1),0.0_wp,z2,&
          & SIZE(z2,DIM=1))
        CALL gemm('N','T',SIZE(dz,DIM=1),SIZE(rkti,DIM=1),SIZE(dz,DIM=2),&
          & 1.0_wp,dz,SIZE(dz,DIM=1),rkti,SIZE(rkti,DIM=1),0.0_wp,z1,&
          & SIZE(z1,DIM=1))
     END IF
     DO i=1, stages
        fac = reigenvals(i)/h
        IF (PRESENT(mass)) THEN
           CALL genmv(-fac,mass,z2(:,i),1.0_wp,z1(:,i),info)
        ELSE
           z2 = dz-z*fac
           ! z1(:,i) = MATMUL(z2,rkti(i,:))
           CALL gemv('N',SIZE(z2,DIM=1),SIZE(z2,DIM=2),1.0_wp,&
                & z2,SIZE(z2,DIM=1),rktit(:,i),1,0.0_wp,z1(:,i),1)
        END IF
        IF (ilinsolver==SLV_DIRECT) THEN
           CALL msolve(m%m(i),z1(:,i),info)
        ELSE
           IF (PRESENT(stats)) THEN
              CALL itersolve(ilinsolver,fac,jac,z1(:,i),m%im(i),minnrm,&
                            & inlimit,info,&
                            & mass,stats%nliniter,stats%nprecond)
           ELSE
              CALL itersolve(ilinsolver,fac,jac,z1(:,i),m%im(i),minnrm,&
                            & inlimit,info,mass)
           END IF
        END IF
        IF (info/=0) THEN
           RETURN
        END IF
     END DO
#else
     fac = reigenvals(myid)/h
     IF (PRESENT(mass)) THEN
        CALL gemv('N',SIZE(z,DIM=1),SIZE(z,DIM=2),1.0_wp,&
             & z,SIZE(z,DIM=1),rktit(:,myid),1,0.0_wp,z2(:,myid),1)
        CALL gemv('N',SIZE(dz,DIM=1),SIZE(dz,DIM=2),1.0_wp,&
             & dz,SIZE(dz,DIM=1),rktit(:,myid),1,0.0_wp,z3,1)
        CALL genmv(-fac,mass,z2(:,myid),1.0_wp,z3,info)
     ELSE
        z2 = dz-z*fac
        ! z1(:,myid) = MATMUL(z2,rkti(myid,:))
        CALL gemv('N',SIZE(z2,DIM=1),SIZE(z2,DIM=2),1.0_wp,&
             & z2,SIZE(z2,DIM=1),rktit(:,myid),1,0.0_wp,z3,1)
     END IF
     IF (ilinsolver==SLV_DIRECT) THEN
        CALL msolve(m%m(myid),z3,info)
     ELSE
        IF (PRESENT(stats)) THEN
           CALL itersolve(ilinsolver,fac,jac,z3,m%im(myid),minnrm,inlimit,&
                         & info,mass,stats%nliniter,stats%nprecond)
        ELSE
           CALL itersolve(ilinsolver,fac,jac,z3,m%im(myid),minnrm,inlimit,&
                         & info,mass)
        END IF
     END IF
     IF (info/=0) THEN
        RETURN
     END IF
     CALL MPI_ALLGATHER(z3,SIZE(z3),mpi_realwp,z1,SIZE(z3),&
          & mpi_realwp,communicator,info)
     IF (info/=MPI_SUCCESS) THEN
        info = ERRMPIGATHERFAILED
        RETURN
     END IF
#endif
     IF (PRESENT(stats)) stats%rsolv = stats%rsolv + SIZE(reigenvals)
     ! dz = MATMUL(z1,rktt)
     CALL gemm('N','T',SIZE(z1,DIM=1),SIZE(rkt,DIM=1),SIZE(z1,DIM=2),1.0_wp,&
          & z1,SIZE(z1,DIM=1),rkt,SIZE(rkt,DIM=1),0.0_wp,dz,SIZE(dz,DIM=1))
  END SELECT
END SUBROUTINE solve

  SUBROUTINE rkdecode(z)
    REAL(wp), DIMENSION(:,:), INTENT(inout) :: z
    REAL(wp), DIMENSION(SIZE(z,DIM=1),SIZE(z,DIM=2)) :: z1
    SELECT CASE (method)
    CASE (1:2)
       z1 = z
       ! z = MATMUL(z1,rktt)
       CALL gemm('N','T',SIZE(z1,DIM=1),SIZE(rkt,DIM=1),&
            & SIZE(z1,DIM=2),1.0_wp,&
            & z1,SIZE(z1,DIM=1),rkt,SIZE(rkt,DIM=1),0.0_wp,z,SIZE(z,DIM=1))
    CASE (3)
    END SELECT
    RETURN
  END SUBROUTINE rkdecode

  SUBROUTINE rkencode(z)
    REAL(wp), DIMENSION(:,:), INTENT(inout) :: z
    REAL(wp), DIMENSION(SIZE(z,DIM=1),SIZE(z,DIM=2)) :: z1
    SELECT CASE (method)
    CASE (1:2)
       z1 = z
       ! z = MATMUL(z1,rktit)
       CALL gemm('N','T',SIZE(z1,DIM=1),SIZE(rkti,DIM=1),SIZE(z1,DIM=2),&
            & 1.0_wp,z1,SIZE(z1,DIM=1),rkti,SIZE(rkti,DIM=1),0.0_wp,&
            & z,SIZE(z,DIM=1))
    CASE (3)
    END SELECT
    RETURN
  END SUBROUTINE rkencode

SUBROUTINE esterr(fnk,t,y,z,h,jac,minnrm,m,scal,rej,err,stats,mass)
  INTERFACE
     SUBROUTINE fnk(t,y,ydot)
       USE defines
       REAL(wp), INTENT(IN) :: t
       REAL(wp), DIMENSION(:), INTENT(IN) :: y
       REAL(wp), DIMENSION(:), INTENT(OUT) :: ydot
     END SUBROUTINE fnk
  END INTERFACE
  REAL(wp), INTENT(IN) :: t
  REAL(wp), DIMENSION(:), INTENT(IN) :: y
  REAL(wp), DIMENSION(:,:), INTENT(IN) :: z
  REAL(wp), INTENT(IN) :: h
  TYPE(rmatrix), INTENT(in) :: jac
  REAL(wp), INTENT(IN) :: minnrm
  TYPE(iterationm), INTENT(INOUT) :: m
  REAL(wp), DIMENSION(:), INTENT(IN) :: scal
  LOGICAL, INTENT(IN) :: rej
  REAL(wp), INTENT(OUT) :: err
  INTEGER :: info
  TYPE(statistics), OPTIONAL, INTENT(INOUT) :: stats
  TYPE(rmatrix), OPTIONAL, INTENT(in) :: mass
  REAL(wp), DIMENSION(SIZE(y)) :: yhat, dyhat
  REAL(wp), DIMENSION(stages) :: scerrcoef
  SELECT CASE (method)
  CASE (1:2)
     scerrcoef = errcoef/h
     CALL fnk(t,y,yhat)
     IF (PRESENT(stats)) stats%fevals = stats%fevals+1
     IF (PRESENT(mass)) THEN
        CALL gemv('N',SIZE(z,DIM=1),SIZE(z,DIM=2),1.0_wp,z,SIZE(z,DIM=1),&
             & scerrcoef,1,0.0_wp,dyhat,1)
        CALL genmv(1.0_wp,mass,dyhat,1.0_wp,yhat,info)
     ELSE
        ! yhat = yhat+MATMUL(z,scerrcoef)
        CALL gemv('N',SIZE(z,DIM=1),SIZE(z,DIM=2),1.0_wp,z,SIZE(z,DIM=1),&
             & scerrcoef,1,1.0_wp,yhat,1)
     END IF
     IF (ilinsolver==SLV_DIRECT) THEN
        CALL msolve(m%m(1),yhat,info)
     ELSE
        IF (PRESENT(stats)) THEN
           CALL itersolve(ilinsolver,reigenvals(1)/h,jac,yhat,m%im(1),minnrm,&
                         & inlimit,info,mass,stats%nliniter,stats%nprecond)
        ELSE
           CALL itersolve(ilinsolver,reigenvals(1)/h,jac,yhat,m%im(1),minnrm,&
                         & inlimit,info,mass)
        END IF
     END IF
     IF (PRESENT(stats)) stats%rsolv = stats%rsolv + 1
     IF (info/=0) THEN
        err = 2
        RETURN
     END IF
     CALL calcnorm(err,scal,yhat)
     IF (rej.AND.(err>=1.0_wp)) THEN
        CALL fnk(t,y+yhat,yhat)
        IF (PRESENT(stats)) stats%fevals = stats%fevals+1
        IF (PRESENT(mass)) THEN
           CALL genmv(1.0_wp,mass,dyhat,1.0_wp,yhat,info)
        ELSE
           ! yhat = yhat+MATMUL(z,scerrcoef)
           CALL gemv('N',SIZE(z,DIM=1),SIZE(z,DIM=2),1.0_wp,z,SIZE(z,DIM=1),&
                & scerrcoef,1,1.0_wp,yhat,1)
        END IF
        IF (ilinsolver==SLV_DIRECT) THEN
           CALL msolve(m%m(1),yhat,info)
        ELSE
           IF (PRESENT(stats)) THEN
              CALL itersolve(ilinsolver,reigenvals(1)/h,jac,yhat,m%im(1),&
                         & minnrm,inlimit,info,mass,&
                         & stats%nliniter,stats%nprecond)
           ELSE
              CALL itersolve(ilinsolver,reigenvals(1)/h,jac,yhat,m%im(1),&
                            & minnrm,inlimit,info,mass)
           END IF
        END IF
        IF (PRESENT(stats)) stats%rsolv = stats%rsolv + 1
        IF (info/=0) THEN
           err = 2
           RETURN
        END IF
        CALL calcnorm(err,scal,yhat)
     END IF
  CASE (3)
     ! yhat = errscale*MATMUL(z,errcoef)
     CALL gemv('N',SIZE(z,DIM=1),SIZE(z,DIM=2),ierrscale,z,SIZE(z,DIM=1),&
             & errcoef,1,0.0_wp,yhat,1)
     CALL calcnorm(err,scal,yhat)
  END SELECT
    RETURN
  END SUBROUTINE esterr

  SUBROUTINE formy1(z,dy)
    ! returns the increment in y from the previous step, e.i. y1-y0.
    REAL(wp), DIMENSION(:,:), INTENT(IN) :: z
    REAL(wp), DIMENSION(:), INTENT(OUT) :: dy
    SELECT CASE(method)
    CASE (1:2)
       dy = z(:,stages)
    CASE (3)
       dy = z(:,stages-1)
    END SELECT
  END SUBROUTINE formy1

  SUBROUTINE numjac(t,y,fnk,jac,fac,stats,first)
    REAL(wp), INTENT(in) :: t
    REAL(wp), DIMENSION(:), INTENT(in) :: y
    INTERFACE
       SUBROUTINE fnk(t,y,ydot)
         USE defines
         REAL(wp), INTENT(IN) :: t
         REAL(wp), DIMENSION(:), INTENT(IN) :: y
         REAL(wp), DIMENSION(:), INTENT(OUT) :: ydot
       END SUBROUTINE fnk
    END INTERFACE
    TYPE(rmatrix), INTENT(inout) :: jac
    REAL(wp), DIMENSION(SIZE(y)), INTENT(inout) :: fac
    TYPE(statistics), OPTIONAL, INTENT(INOUT) :: stats
    LOGICAL, INTENT(in), OPTIONAL :: first
    REAL(wp), DIMENSION(SIZE(y)) :: yscale, del, ydot0, ydot, yw, fdiff,&
         & absydot0, absydot0rm, ydotscale, tmpj, difmax, absydotrm
    INTEGER, DIMENSION(SIZE(y)) :: rowmax
    LOGICAL, DIMENSION(SIZE(y)) :: msk1, msk2
    INTEGER :: i, tmprowmax
    REAL(wp) :: tmpfac, tmpdifmax, fscale
#ifdef MPI
    INCLUDE MPIINCLUDEFILE
    INTEGER :: il, iu
#endif
#ifdef MPI
    IF (SIZE(y)>=stages) THEN
       il = slice(myid)
       iu = slice(myid+1)-1
       IF (PRESENT(first)) THEN
          IF (first) THEN
             br = eps**(0.875_wp)
             bl = eps**(0.75_wp)
             bu = eps**(0.25_wp)
             facmin = eps**(0.78_wp)
             facmax = 0.1_wp
             fac(il:iu) = SQRT(eps)
          END IF
       END IF
       yscale(il:iu) = MAX(ABS(y(il:iu)),iatol)
       del(il:iu) = (y(il:iu)+fac(il:iu)*yscale(il:iu))-y(il:iu)
       msk1(il:iu) = (del(il:iu)==0)
       msk2(il:iu) = .false.
       DO WHILE (ANY(msk1(il:iu)))
          WHERE (msk1(il:iu))
             fac(il:iu) = 100*fac(il:iu)
             msk2(il:iu) = (fac(il:iu)>=facmax)
             del(il:iu) = (y(il:iu)+fac(il:iu)*yscale(il:iu))-y(il:iu)
             msk1(il:iu) = (del(il:iu)==0)
          END WHERE
          WHERE (msk2(il:iu))
             fac(il:iu) = facmax
             del(il:iu) = iatol
             msk1(il:iu) = .false.
          END WHERE
       END DO
       CALL fnk(t,y,ydot0)
       yw = y
       WHERE (ydot(il:iu)>=0.0_wp)
          del(il:iu) = ABS(del(il:iu))
       ELSEWHERE
          del(il:iu) = -ABS(del(il:iu))
       END WHERE
       DO i=il,iu
          yw(i)=y(i)+del(i)
          CALL fnk(t,yw,ydot)
          CALL coldiv(i,jac,ydot,ydot0,fdiff,del(i),tmprowmax,difmax(i))
          rowmax(i) = tmprowmax
          CALL setcol(i,jac,fdiff)
          absydotrm(i) = ABS(ydot(tmprowmax))
          yw(i)=y(i)
       END DO
       IF (myid==1) THEN
          IF (PRESENT(stats)) stats%njacfevals = stats%njacfevals+SIZE(y)+1
       END IF
       ! Adjust fac for next call
       absydot0(il:iu) = ABS(ydot0(il:iu))
       absydot0rm(il:iu) = absydot0(rowmax(il:iu))
       msk1(il:iu) = (absydotrm(il:iu)/=0.0_wp).and.(absydot0rm(il:iu)/=0)
       IF (ANY(msk1(il:iu))) THEN
          ydotscale(il:iu) = MAX(absydotrm(il:iu),absydot0rm(il:iu))
          msk2(il:iu) = (difmax(il:iu) < br*ydotscale(il:iu))
          DO i=il,iu
             IF (msk1(i).and.msk2(i)) THEN
                tmpfac = MIN(SQRT(fac(i)),facmax)
                del(i) = (y(i)+tmpfac*yscale(i))-y(i)
                IF ((tmpfac/=fac(i)).and.(del(i)/=0.0_wp)) THEN
                   IF (ydot0(i)>=0.0_wp) THEN
                      del(i) = ABS(del(i))
                   ELSE
                      del(i) = -ABS(del(i))
                   END IF
                   yw(i) = y(i)+del(i)
                   CALL fnk(t,yw,ydot)
                   IF (PRESENT(stats)) stats%njacfevals = stats%njacfevals+1
                   yw(i)=y(i)
                   CALL coldiv(i,jac,ydot,ydot0,fdiff,del(i),&
                        & tmprowmax,tmpdifmax)
                   IF (coltest(i,jac,fdiff,tmpfac)) THEN
                      CALL setcol(i,jac,tmpj)
                      fscale = MAX(ABS(ydot(tmprowmax)),absydot0(tmprowmax))
                      IF (tmpdifmax <= bl*fscale) THEN
                         fac(i) = MIN(10*tmpfac,facmax)
                      ELSE IF (tmpdifmax > bu*fscale) THEN
                         fac(i) = MAX(0.1_wp*tmpfac, facmin)
                      ELSE
                         fac(i) = tmpfac
                      END IF
                   END IF
                END IF
             END IF
          END DO
          WHERE (msk1(il:iu).and.(.NOT.msk2(il:iu)).and.&
               & (difmax(il:iu) <= bl*ydotscale(il:iu)))
             fac(il:iu) = MIN(10*fac(il:iu),facmax)
          END WHERE
          WHERE (msk1(il:iu).and.(difmax(il:iu) > bu*ydotscale(il:iu)))
             fac(il:iu) = MAX(0.1_wp*fac(il:iu), facmin)
          END WHERE
       END IF
       CALL MPI_ALLGATHERV(jac%a(1,slice(myid)),slicesize(myid),&
            & jaccoltype, jac%a(1,1),slicesize,sliceoffs,&
            & jaccoltype,communicator,i)
       IF (i/=MPI_SUCCESS) THEN
          RETURN
       END IF
    ELSE
#endif
    IF (PRESENT(first)) THEN
       IF (first) THEN
          br = eps**(0.875_wp)
          bl = eps**(0.75_wp)
          bu = eps**(0.25_wp)
          facmin = eps**(0.78_wp)
          facmax = 0.1_wp
          fac = SQRT(eps)
       END IF
    END IF
    yscale = MAX(ABS(y),iatol)
    del = (y+fac*yscale)-y
    msk1 = (del==0)
    msk2 = .false.
    DO WHILE (ANY(msk1))
       WHERE (msk1)
          fac = 100*fac
          msk2 = (fac>=facmax)
          del = (y+fac*yscale)-y
          msk1 = (del==0)
       END WHERE
       WHERE (msk2)
          fac = facmax
          del = iatol
          msk1 = .false.
       END WHERE
    END DO
    CALL fnk(t,y,ydot0)
    WHERE (ydot>=0.0_wp)
       del = ABS(del)
    ELSEWHERE
       del = -ABS(del)
    END WHERE
    yw = y

    DO i=1,SIZE(y)
       yw(i)=y(i)+del(i)
       CALL fnk(t,yw,ydot)
       CALL coldiv(i,jac,ydot,ydot0,fdiff,del(i),tmprowmax,difmax(i))
       rowmax(i) = tmprowmax
       CALL setcol(i,jac,fdiff)
       absydotrm(i) = ABS(ydot(tmprowmax))
       yw(i)=y(i)
    END DO
    IF (PRESENT(stats)) stats%njacfevals = stats%njacfevals+SIZE(y)+1
    ! Adjust fac for next call
    absydot0 = ABS(ydot0)
    absydot0rm = absydot0(rowmax)
    msk1 = (absydotrm/=0.0_wp).and.(absydot0rm/=0)
    IF (ANY(msk1)) THEN
       ydotscale = MAX(absydotrm,absydot0rm)
       msk2 = (difmax < br*ydotscale)
       DO i=1,SIZE(y)
          IF (msk1(i).and.msk2(i)) THEN
             tmpfac = MIN(SQRT(fac(i)),facmax)
             del(i) = (y(i)+tmpfac*yscale(i))-y(i)
             IF ((tmpfac/=fac(i)).and.(del(i)/=0.0_wp)) THEN
                IF (ydot0(i)>=0.0_wp) THEN
                   del(i) = ABS(del(i))
                ELSE
                   del(i) = -ABS(del(i))
                END IF
                yw(i) = y(i)+del(i)
                CALL fnk(t,yw,ydot)
                IF (PRESENT(stats)) stats%njacfevals = stats%njacfevals+1
                yw(i)=y(i)
                CALL coldiv(i,jac,ydot,ydot0,fdiff,del(i),tmprowmax,tmpdifmax)
                IF (coltest(i,jac,fdiff,tmpfac)) THEN
                   CALL setcol(i,jac,tmpj)
                   fscale = MAX(ABS(ydot(tmprowmax)),absydot0(tmprowmax))
                   IF (tmpdifmax <= bl*fscale) THEN
                      fac(i) = MIN(10*tmpfac,facmax)
                   ELSE IF (tmpdifmax > bu*fscale) THEN
                      fac(i) = MAX(0.1_wp*tmpfac, facmin)
                   ELSE
                      fac(i) = tmpfac
                   END IF
                END IF
             END IF
          END IF
       END DO
       WHERE (msk1.and.(.NOT.msk2).and.(difmax <= bl*ydotscale))
          fac = MIN(10*fac,facmax)
       END WHERE
       WHERE (msk1.and.(difmax > bu*ydotscale))
          fac = MAX(0.1_wp*fac, facmin)
       END WHERE
    END IF
#ifdef MPI
    END IF
#endif
  END SUBROUTINE numjac

  SUBROUTINE setup_method(ierr,name,order)
    CHARACTER (LEN=*), OPTIONAL, INTENT(IN) :: name
    INTEGER, OPTIONAL, INTENT(IN) :: order
    INTEGER, INTENT(out) :: ierr
    INTEGER :: i,j
    IF (.NOT.(PRESENT(name).or.PRESENT(order))) THEN
       ierr = 0
       RETURN
    END IF
    IF (PRESENT(name)) THEN
       SELECT CASE(name)
       CASE ('hairer')
          method = 1
       CASE ('radauIIa')
          method = 2
       CASE ('mirk')
          method = 3
       CASE default
          ierr = ERRILLEGALMETHODNAME
          RETURN
       END SELECT
    END IF
    IF (PRESENT(order)) THEN 
       rkorder = order
    END IF

    CALL free_method()

    SELECT CASE(method)
    CASE (1) ! Hairer coefficients for Radau IIA, order 5
       IF (PRESENT(order).AND.(order/=5)) THEN
          ierr = ERRILLEGALMETHODORDER
          RETURN
       END IF
       stages = 3
       rkorder = 5
       rkmethod = 'Hairer RadauIIA'
       ALLOCATE(reigenvals(1))
       reigenvals = (/ 0.3637834252744496D1 /)
       ALLOCATE(ceigenvals(1))
       ceigenvals = (/ CMPLX(0.2681082873627752D1,0.3050430199247411D1,&
            & KIND=wp) /)
       ALLOCATE(rkt(stages,stages))
       rkt(1,1)=9.1232394870892942792D-02
       rkt(1,2)=-0.14125529502095420843D0
       rkt(1,3)=-3.0029194105147424492D-02
       rkt(2,1)=0.24171793270710701896D0
       rkt(2,2)=0.20412935229379993199D0
       rkt(2,3)=0.38294211275726193779D0
       rkt(3,1)=0.96604818261509293619D0
       rkt(3,2)=1.D0
       rkt(3,3)=0.D0
       ALLOCATE(rkti(stages,stages))
       rkti(1,1)=4.3255798900631553510D0
       rkti(1,2)=0.33919925181580986954D0
       rkti(1,3)=0.54177053993587487119D0
       rkti(2,1)=-4.1787185915519047273D0
       rkti(2,2)=-0.32768282076106238708D0
       rkti(2,3)=0.47662355450055045196D0
       rkti(3,1)=-0.50287263494578687595D0
       rkti(3,2)=2.5719269498556054292D0
       rkti(3,3)=-0.59603920482822492497D0
       ALLOCATE(rkc(stages))
       rkc(1) = 0.1550510257216822D0
       rkc(2) = 0.6449489742783178D0
       rkc(3) = 1.D0
       errorder = 3 ! order of embedded method
       ALLOCATE(errcoef(stages))
       errcoef(1) = -10.048809399827414d0
       errcoef(2) = 1.3821427331607483d0
       errcoef(3) = -0.33333333333333331d0 
       ikappa = 3.d-2
       iconskappa = 3.d-2
       inrmdiv = 99.d-2
       ihmin = 1d-10
       imaxit = 6
       ioptit = 6
       ierrunit = 6
       iquotl = 2.d-1
       iquotm = 8.d0
       isafe = 9.d-1
       ijactol = 1.d-3
       iincrate = 5.d-1
       ihconstl = 1.0_wp
       ihconstm = 1.2_wp
       imaxitermfail = 3
       ipolfitorder = 2
       igustafsson = .TRUE.
       ikappasafe=1.0_wp
       inrmdiv=0.99_wp
       ilinsolver=SLV_DIRECT
    CASE (2) ! CBE coefficients for Radau IIA, order 5
       rkmethod = 'RadauIIA'
       SELECT CASE (rkorder)
       CASE (5)
          stages = 3
          ALLOCATE(reigenvals(1))
          reigenvals = (/ 0.3637834252744496D1 /)
          ALLOCATE(ceigenvals(1))
          ceigenvals = (/ CMPLX(0.2681082873627752D1,-0.3050430199247411D1,&
               & KIND=wp) /)
          ALLOCATE(rkt(stages,stages))
          rkt(1,1) = 0.3989184098843109D0
          rkt(1,2) = 0.7194324508029206D0
          rkt(1,3) = -0.6470982193428867D-1
          rkt(2,1) = 0.1056924281035244D1
          rkt(2,2) = -0.1242580398020061D1
          rkt(2,3) = 0.1779731292440472D1
          rkt(3,1) = 0.4224096116580102D1
          rkt(3,2) = -0.4966088042441398D1
          rkt(3,3) = -0.5976257373071423D0
          ALLOCATE(rkti(stages,stages))
          rkti(1,1) = 0.9892574591638469D0
          rkti(1,2) = 0.7757466016809291D-1
          rkti(1,3) = 0.1239025891113443D0
          rkti(2,1) = 0.8414507668489882D0
          rkti(2,2) = 0.3607311392885671D-2
          rkti(2,3) = -0.8036816576574003D-1
          rkti(3,1) = 0
          rkti(3,2) = 0.5183320855344764D0
          rkti(3,3) = -0.1296934898547156D0
          ALLOCATE(rkc(stages))
          rkc(1) = 0.1550510257216822D0
          rkc(2) = 0.6449489742783178D0
          rkc(3) = 1.D0
          errorder = 3 ! order of embedded method
          ALLOCATE(errcoef(stages))
          errcoef(1) = -0.1004880939982742D2
          errcoef(2) = 0.1382142733160749D1
          errcoef(3) = -0.3333333333333333D0
       CASE (9)
          stages = 5
          ALLOCATE(reigenvals(1))
          reigenvals = (/ 0.6286704751729277D1 /)
          ALLOCATE(ceigenvals(2))
          ceigenvals = (/ CMPLX(0.3655694325463572D1,-0.6543736899360077D1,&
               & KIND=wp), CMPLX(0.5700953298671789D1,0.321026560030855D1,&
               & KIND=wp) /)
          ALLOCATE(rkt(stages,stages))
          rkt(1,1) = 0.2983060861850495D-1
          rkt(1,2) = -0.420828201009708D-1
          rkt(1,3) = -0.8580131000753114D-1
          rkt(1,4) = -0.1280315295795519D-1
          rkt(1,5) = 0.2462691157368414D-1
          rkt(2,1) = 0.3554793049175887D-2
          rkt(2,2) = 0.1401408683000243D0
          rkt(2,3) = 0.1555927045527496D0
          rkt(2,4) = 0.3941529856816857D-1
          rkt(2,5) = -0.4128547759931657D-2
          rkt(3,1) = 0.1739228116688772D0
          rkt(3,2) = -0.4871005447206277D0
          rkt(3,3) = -0.8598438814330406D-1
          rkt(3,4) = 0.1034366820042673D0
          rkt(3,5) = -0.7671632467677877D-1
          rkt(4,1) = 0.9057943589332474D0
          rkt(4,2) = 0.497289996976849D0
          rkt(4,3) = -0.1066765152923572D1
          rkt(4,4) = 0.3490506327897404D-1
          rkt(4,5) = -0.694406656242727D0
          rkt(5,1) = 0.2197164328161838D1
          rkt(5,2) = 0.1901627184524032D1
          rkt(5,3) = -0.4741855462601939D0
          rkt(5,4) = -0.6040113737879891D0
          rkt(5,5) = -0.1407739327272568D1
          ALLOCATE(rkti(stages,stages))
          rkti(1,1) = 0.1260610934770462D2
          rkti(1,2) = 0.581810734293098D1
          rkti(1,3) = 0.1460286490905153D1
          rkti(1,4) = -0.4330538230815622D0
          rkti(1,5) = 0.3375034295438318D0
          rkti(2,1) = 0.2183101203760791D1
          rkti(2,2) = 0.2766168343565682D1
          rkti(2,3) = -0.1374310458481839D1
          rkti(2,4) = 0.3739826571648056D0
          rkti(2,5) = -0.7950421982515843D-1
          rkti(3,1) = -0.2515348372811376D1
          rkti(3,2) = 0.140579011020588D1
          rkti(3,3) = 0.8919170965253373D0
          rkti(3,4) = -0.7159319922418132D0
          rkti(3,5) = 0.2564211986623507D0
          rkti(4,1) = 0.3338931853107858D1
          rkti(4,2) = 0.1029206134275898D2
          rkti(4,3) = 0.1192983351600468D1
          rkti(4,4) = 0.1476539230086023D1
          rkti(4,5) = -0.765130045135658D0
          rkti(5,1) = 0.220389718443461D2
          rkti(5,2) = 0.7927910481790898D1
          rkti(5,3) = -0.3895937544879015D0
          rkti(5,4) = -0.5630848662652277D0
          rkti(5,5) = -0.4907220023702316D-1
          ALLOCATE(rkc(stages))
          rkc(1) = 0.5710419611451768D-1
          rkc(2) = 0.2768430136381238D0
          rkc(3) = 0.5835904323689168D0
          rkc(4) = 0.8602401356562194D0
          rkc(5) = 0.1D1
          errorder = 5 ! order of embedded method
          ALLOCATE(errcoef(stages))
          errcoef(1) = -0.2778093394406464D2
          errcoef(2) = 0.3641478498049213D1
          errcoef(3) = -0.1252547721169119D1
          errcoef(4) = 0.5920031671845429D0
          errcoef(5) = -0.2D0
       CASE (13)
          stages = 7
          ALLOCATE(reigenvals(1))
          reigenvals = (/ 0.8936832788405216D1 /)
          ALLOCATE(ceigenvals(3))
          ceigenvals = (/ CMPLX(0.4378693561506806D1,0.1016969328379501D2,&
               & KIND=wp), CMPLX(0.714105521918764D1,0.6623045922639276D1,&
               & KIND=wp), CMPLX(0.8511834825102946D1,-0.3281013624325059D1,&
               & KIND=wp) /)
          ALLOCATE(rkt(stages,stages))
          rkt(1,1) = -0.339511860830311D-3
          rkt(1,2) = 0.1081592085676755D-1
          rkt(1,3) = 0.1560241623366841D-1
          rkt(1,4) = 0.3699501046483119D-1
          rkt(1,5) = 0.2222225520828629D-1
          rkt(1,6) = 0.7631167189644698D-1
          rkt(1,7) = 0.3590390308068146D-1
          rkt(2,1) = 0.2522234820429363D-3
          rkt(2,2) = -0.1645244214518818D-1
          rkt(2,3) = -0.3167969564113734D-1
          rkt(2,4) = -0.4045733674358957D-1
          rkt(2,5) = -0.5350340259247703D-1
          rkt(2,6) = -0.8880660089129839D-1
          rkt(2,7) = 0.2595068538551482D-3
          rkt(3,1) = -0.6398663319948682D-3
          rkt(3,2) = 0.1298580919808101D-1
          rkt(3,3) = 0.6574622815898812D-1
          rkt(3,4) = -0.365036960310355D-1
          rkt(3,5) = 0.7021817833888559D-1
          rkt(3,6) = -0.1279867088616755D-1
          rkt(3,7) = 0.6534163944416583D-1
          rkt(4,1) = -0.2482862926447874D-2
          rkt(4,2) = 0.4771150911301166D-1
          rkt(4,3) = -0.1301916487824429D0
          rkt(4,4) = 0.1115738814549869D0
          rkt(4,5) = 0.2514412008101068D0
          rkt(4,6) = -0.6983891759580982D0
          rkt(4,7) = -0.9941375350477378D-1
          rkt(5,1) = -0.1780948277032256D-1
          rkt(5,2) = -0.2919733304154363D0
          rkt(5,3) = 0.7584017492967887D-1
          rkt(5,4) = 0.1353297743567942D1
          rkt(5,5) = -0.3145977596321501D0
          rkt(5,6) = -0.2597686355790941D1
          rkt(5,7) = -0.288237076680934D1
          rkt(6,1) = -0.7225790671198764D-1
          rkt(6,2) = 0.1464227635184752D0
          rkt(6,3) = 0.5688137601314937D0
          rkt(6,4) = 0.1186504514430566D1
          rkt(6,5) = -0.410393845472507D1
          rkt(6,6) = -0.3307325930109212D1
          rkt(6,7) = -0.1460450046961945D2
          rkt(7,1) = -0.1389401053827313D0
          rkt(7,2) = 0.6828522086977547D0
          rkt(7,3) = 0.445323500998587D0
          rkt(7,4) = -0.1432599231141019D1
          rkt(7,5) = -0.7044036540217396D1
          rkt(7,6) = 0.501764489368417D0
          rkt(7,7) = -0.2786810599826281D2
          ALLOCATE(rkti(stages,stages))
          rkti(1,1) = -0.1637506358124285D4
          rkti(1,2) = -0.119942346386274D4
          rkti(1,3) = -0.3113942210231229D3
          rkti(1,4) = -0.2607663245511922D2
          rkti(1,5) = -0.2571377661689064D2
          rkti(1,6) = 0.197463242024024D2
          rkti(1,7) = -0.1044661320974606D2
          rkti(2,1) = 0.256665840115306D1
          rkti(2,2) = -0.1088250459628873D2
          rkti(2,3) = -0.3122022489248376D1
          rkti(2,4) = 0.4336476908799653D1
          rkti(2,5) = -0.2583843094704573D1
          rkti(2,6) = 0.1403249905564398D1
          rkti(2,7) = -0.4877240185547793D0
          rkti(3,1) = -0.1068894526558213D2
          rkti(3,2) = -0.8049707170154313D1
          rkti(3,3) = 0.8128202083209746D1
          rkti(3,4) = -0.1865575640446626D1
          rkti(3,5) = -0.1162258954536298D0
          rkti(3,6) = 0.3954727072132557D0
          rkti(3,7) = -0.183362482454046D0
          rkti(4,1) = 0.5884976645067275D1
          rkti(4,2) = -0.3796076738257115D1
          rkti(4,3) = -0.4251279177863518D1
          rkti(4,4) = -0.3605974530293404D-1
          rkti(4,5) = 0.7977576731409124D-1
          rkti(4,6) = 0.1829774399848508D0
          rkti(4,7) = -0.1064345614804092D0
          rkti(5,1) = -0.1179851137590266D2
          rkti(5,2) = -0.1163684087885092D2
          rkti(5,3) = 0.2933290496714734D0
          rkti(5,4) = 0.5347027172630547D0
          rkti(5,5) = 0.4707254518248327D0
          rkti(5,6) = -0.4200344692579417D0
          rkti(5,7) = 0.1549070349537874D0
          rkti(6,1) = 0.3145575975579519D1
          rkti(6,2) = -0.1013646765718914D1
          rkti(6,3) = -0.1440503250865481D1
          rkti(6,4) = -0.5174713685143369D0
          rkti(6,5) = 0.1270062414452069D0
          rkti(6,6) = -0.1751565254134347D0
          rkti(6,7) = 0.8116765502365052D-1
          rkti(7,1) = 0.1079243011240724D2
          rkti(7,2) = 0.8702856714843214D1
          rkti(7,3) = 0.1724347923139424D1
          rkti(7,4) = 0.6383724638005396D-1
          rkti(7,5) = -0.5776614378013633D-1
          rkti(7,6) = 0.3586489543228349D-1
          rkti(7,7) = -0.3090311205123198D-1
          ALLOCATE(rkc(stages))
          rkc(1) = 0.2931642715978489D-1
          rkc(2) = 0.1480785996684843D0
          rkc(3) = 0.3369846902811543D0
          rkc(4) = 0.5586715187715501D0
          rkc(5) = 0.7692338620300545D0
          rkc(6) = 0.9269456713197411D0
          rkc(7) = 0.1D1
          errorder = 7 ! order of embedded method
          ALLOCATE(errcoef(stages))
          errcoef(1) = -0.5437443689412861D2
          errcoef(2) = 0.7000024004259187D1
          errcoef(3) = -0.2355661091987557D1
          errcoef(4) = 0.1132289066106134D1
          errcoef(5) = -0.6468913267673587D0
          errcoef(6) = 0.3875333853753524D0
          errcoef(7) = -0.1428571428571429D0
       CASE default
          ierr = ERRILLEGALMETHODORDER
          RETURN
       END SELECT
    CASE (3) ! CBE coefficients for embedded MIRK, order 5(5)
       rkmethod = 'MIRK'
       SELECT CASE (rkorder)
       CASE (5)
          stages = 6
          ALLOCATE(reigenvals(stages))
          reigenvals = (/ 0.6594978074058386D1, 0.5428165572622701D1,&
               & 0.2425449862130748D1, 0.9714320889048801D0,&
               & 0.3952749062627953D1, 0.1D0 /)
          ALLOCATE(rkt(stages,stages))
          rkt(1,1) = 0.1195919975039185D0
          rkt(1,2) = 0.8296475125790142D-1
          rkt(1,3) = -0.5190427175576677D-1
          rkt(1,4) = 0.5894953672566455D-1
          rkt(1,5) = 0.5047583367875842D-1
          rkt(1,6) = 0
          rkt(2,1) = -0.2428400023934142D0
          rkt(2,2) = -0.2133296234009096D0
          rkt(2,3) = 0.1827948703858417D0
          rkt(2,4) = -0.234313529190169D0
          rkt(2,5) = -0.1556490616132116D0
          rkt(2,6) = 0
          rkt(3,1) = 0.4077955438013259D-1
          rkt(3,2) = 0.9358114992288634D-1
          rkt(3,3) = -0.3441835405497177D0
          rkt(3,4) = 0.5672135734876615D0
          rkt(3,5) = 0.1852410951625998D0
          rkt(3,6) = 0
          rkt(4,1) = 0.5625514862529609D0
          rkt(4,2) = 0.5844057270373152D0
          rkt(4,3) = -0.703661006758798D0
          rkt(4,4) = 0.7335112455967716D0
          rkt(4,5) = 0.6176347281434105D0
          rkt(4,6) = 0
          rkt(5,1) = 0.780127771989776D0
          rkt(5,2) = 0.7728646947645091D0
          rkt(5,3) = -0.5918537487555371D0
          rkt(5,4) = 0.2860981244131238D0
          rkt(5,5) = 0.7466181347164941D0
          rkt(5,6) = 0
          rkt(6,1) = 0.459064680481915D0
          rkt(6,2) = 0.4909415922281393D0
          rkt(6,3) = -0.6955078653829664D0
          rkt(6,4) = 0.8773317314155693D0
          rkt(6,5) = 0.5441865262200845D0
          rkt(6,6) = 1
          ALLOCATE(rkti(stages,stages))
          rkti(1,1) = 0.5147980935240328D2
          rkti(1,2) = 0.2214671714971131D2
          rkti(1,3) = 0.514171622477053D1
          rkti(1,4) = -0.14533911762969D1
          rkti(1,5) = 0.1063243886176818D1
          rkti(1,6) = 0
          rkti(2,1) = -0.8038717676851748D2
          rkti(2,2) = -0.4341493698203635D2
          rkti(2,3) = -0.1653474218578534D2
          rkti(2,4) = 0.7659832545587189D1
          rkti(2,5) = -0.5850312328173367D1
          rkti(2,6) = 0
          rkti(3,1) = 0.7704563965695948D1
          rkti(3,2) = -0.1993035872730801D2
          rkti(3,3) = 0.3916801429403164D2
          rkti(3,4) = -0.4674107169693949D2
          rkti(3,5) = 0.2427257855391032D2
          rkti(3,6) = 0
          rkti(4,1) = 0.3009720323050261D1
          rkti(4,2) = -0.9696536498349537D1
          rkti(4,3) = 0.1590332800443443D2
          rkti(4,4) = -0.1752002035587559D2
          rkti(4,5) = 0.8322659034184937D1
          rkti(4,6) = 0
          rkti(5,1) = 0.3437698656967314D2
          rkti(5,2) = 0.9717023942079884D1
          rkti(5,3) = 0.3669847456549504D2
          rkti(5,4) = -0.3674919692877032D2
          rkti(5,5) = 0.2233638919499753D2
          rkti(5,6) = 0
          rkti(6,1) = -0.1565848820455831D0
          rkti(6,2) = 0.5049070522363317D0
          rkti(6,3) = -0.9244353319233074D0
          rkti(6,4) = -0.2328252205184271D0
          rkti(6,5) = -0.1910616728546644D0
          rkti(6,6) = 1.D0
          ALLOCATE(rkc(stages))
          rkc(1) = 0.5710419611451768D-1
          rkc(2) = 0.2768430136381238D0
          rkc(3) = 0.5835904323689168D0
          rkc(4) = 0.8602401356562194D0
          rkc(5) = 0.1D1
          rkc(6) = 0.8D0
          errorder = 5 ! order of embedded method
          ierrscale = 1.199d0
          ALLOCATE(errcoef(stages))
          errcoef(1) = -0.7350934151517402D-1
          errcoef(2) = 0.4488920208610506D-1
          errcoef(3) = -0.2621105197793673D-1
          errcoef(4) = 0.5284395855913746D-1
          errcoef(5) = -0.1332888841246831D-1
          errcoef(6) = -0.3132832080200501D-1
          ijactol=1.d-1
          ihconstl=0.9_wp
          ihconstm=2.5_wp
          isafe=0.3_wp
          imaxit=12
          ikappasafe=1.0d0
          ioptit=3
          inrmdiv=1.d0
          ilinsolver=SLV_DIRECT
       CASE (8)
          stages = 9
          ALLOCATE(reigenvals(stages))
          reigenvals = (/ 0.6947024769858366D1, 0.1642463915066632D1,&
               & 0.2309095272202018D1, 0.4955773984324289D1,&
               & 0.5789727814545306D1, 0.6289622414000465D1,&
               & 0.3143493330803451D1, 0.4239083288497049D1, 0.1D1 /)
          ALLOCATE(rkt(stages,stages))
          rkt(1,1) = -0.2518378813726635D0
          rkt(1,2) = -0.3264527575687719D0
          rkt(1,3) = -0.2901348544263262D0
          rkt(1,4) = -0.2244886794899709D0
          rkt(1,5) = 0.2313015893195607D0
          rkt(1,6) = -0.2390442561466622D0
          rkt(1,7) = 0.2546807492282914D0
          rkt(1,8) = 0.2285396070707993D0
          rkt(1,9) = 0
          rkt(2,1) = -0.418414887436688D-1
          rkt(2,2) = -0.1787045726605837D-1
          rkt(2,3) = -0.1950331622762807D-1
          rkt(2,4) = -0.2715451347578466D-1
          rkt(2,5) = 0.3223511379222664D-1
          rkt(2,6) = -0.3603623352916012D-1
          rkt(2,7) = 0.2124701281993019D-1
          rkt(2,8) = 0.2416219059277309D-1
          rkt(2,9) = 0
          rkt(3,1) = -0.1563866999784442D-1
          rkt(3,2) = -0.1195048457863767D0
          rkt(3,3) = -0.978597626530787D-1
          rkt(3,4) = -0.424207949170482D-1
          rkt(3,5) = 0.3119872050575937D-1
          rkt(3,6) = -0.2452298589124248D-1
          rkt(3,7) = 0.7538886203962218D-1
          rkt(3,8) = 0.5340695506780418D-1
          rkt(3,9) = 0
          rkt(4,1) = -0.2111539625207136D0
          rkt(4,2) = -0.1581803773535073D0
          rkt(4,3) = -0.1626705393531735D0
          rkt(4,4) = -0.1805863216403625D0
          rkt(4,5) = 0.1936310544533635D0
          rkt(4,6) = -0.2015512088702788D0
          rkt(4,7) = 0.16610221541759D0
          rkt(4,8) = 0.1725923162908749D0
          rkt(4,9) = 0
          rkt(5,1) = -0.4550394108975334D-1
          rkt(5,2) = 0.2563324590467502D0
          rkt(5,3) = 0.199293045740586D0
          rkt(5,4) = 0.3392267289592046D-1
          rkt(5,5) = 0.2989212211721914D-2
          rkt(5,6) = -0.2253354788967003D-1
          rkt(5,7) = -0.1367068455854049D0
          rkt(5,8) = -0.7007522231756522D-1
          rkt(5,9) = 0
          rkt(6,1) = 0.4091008195852015D0
          rkt(6,2) = 0.690617482941665D0
          rkt(6,3) = 0.6489380937601783D0
          rkt(6,4) = 0.4835634033016616D0
          rkt(6,5) = -0.448222137719001D0
          rkt(6,6) = 0.430191264986528D0
          rkt(6,7) = -0.5913674455689551D0
          rkt(6,8) = -0.5213503288231456D0
          rkt(6,9) = 0
          rkt(7,1) = 0.5902579658251623D0
          rkt(7,2) = 0.4786100196399271D0
          rkt(7,3) = 0.5256916436809628D0
          rkt(7,4) = 0.5927970991263438D0
          rkt(7,5) = -0.5939296638170551D0
          rkt(7,6) = 0.5928905984338683D0
          rkt(7,7) = -0.5630989095522064D0
          rkt(7,8) = -0.5869442562436492D0
          rkt(7,9) = 0
          rkt(8,1) = 0.6100474178110251D0
          rkt(8,2) = 0.2864967987279382D0
          rkt(8,3) = 0.3771114921790699D0
          rkt(8,4) = 0.5727685468774054D0
          rkt(8,5) = -0.594415394412234D0
          rkt(8,6) = 0.6026967893391769D0
          rkt(8,7) = -0.4646808094648825D0
          rkt(8,8) = -0.5415950184124209D0
          rkt(8,9) = 0
          rkt(9,1) = -0.3037101303387791D-1
          rkt(9,2) = 0.6478864108719253D0
          rkt(9,3) = 0.3525792800417179D0
          rkt(9,4) = 0.5826054849879275D-1
          rkt(9,5) = -0.1571036564384171D-1
          rkt(9,6) = -0.5764105742712024D-2
          rkt(9,7) = -0.2072691880545187D0
          rkt(9,8) = -0.1041372588373984D0
          rkt(9,9) = 1
          ALLOCATE(rkti(stages,stages))
          rkti(1,1) = 0.9528314398091855D3
          rkti(1,2) = -0.5809125482940107D4
          rkti(1,3) = -0.3007927628120162D4
          rkti(1,4) = -0.812392005650624D2
          rkti(1,5) = 0.2520110808558409D3
          rkti(1,6) = -0.1048466825013736D4
          rkti(1,7) = 0.1582532813436829D4
          rkti(1,8) = -0.9179677222223897D3
          rkti(1,9) = 0
          rkti(2,1) = -0.2831200620510524D4
          rkti(2,2) = 0.4318016264831664D4
          rkti(2,3) = -0.512714867862505D4
          rkti(2,4) = 0.123617721926207D5
          rkti(2,5) = -0.1479021848015775D5
          rkti(2,6) = 0.1327544111258305D5
          rkti(2,7) = -0.1465699709433902D5
          rkti(2,8) = 0.7450447532988735D4
          rkti(2,9) = 0
          rkti(3,1) = 0.1007310410262985D5
          rkti(3,2) = -0.1540181931229275D5
          rkti(3,3) = 0.185787557147065D5
          rkti(3,4) = -0.4558816859349225D5
          rkti(3,5) = 0.552710785395657D5
          rkti(3,6) = -0.5071541318511941D5
          rkti(3,7) = 0.5685053130680516D5
          rkti(3,8) = -0.2907467634849165D5
          rkti(3,9) = 0
          rkti(4,1) = -0.1766639645422805D5
          rkti(4,2) = 0.3266011881522156D5
          rkti(4,3) = -0.2628092054205959D5
          rkti(4,4) = 0.8251091924548653D5
          rkti(4,5) = -0.1039974467535516D6
          rkti(4,6) = 0.1055835472537147D6
          rkti(4,7) = -0.1269384640446091D6
          rkti(4,8) = 0.6709120965973831D5
          rkti(4,9) = 0
          rkti(5,1) = -0.1103169114687853D5
          rkti(5,2) = 0.3259339273553109D5
          rkti(5,3) = -0.9474187065722788D3
          rkti(5,4) = 0.377515929055554D5
          rkti(5,5) = -0.4768496322409393D5
          rkti(5,6) = 0.5237996543861392D5
          rkti(5,7) = -0.650505831337022D5
          rkti(5,8) = 0.3498125574969944D5
          rkti(5,9) = 0
          rkti(6,1) = -0.5822002375199871D4
          rkti(6,2) = 0.2458869666103405D5
          rkti(6,3) = 0.7875629265757052D4
          rkti(6,4) = 0.1161777359989553D5
          rkti(6,5) = -0.1486432635365203D5
          rkti(6,6) = 0.185885445253106D5
          rkti(6,7) = -0.2395742501115504D5
          rkti(6,8) = 0.1311212328494977D5
          rkti(6,9) = 0
          rkti(7,1) = 0.1554352387420475D5
          rkti(7,2) = -0.2387144429220642D5
          rkti(7,3) = 0.2918402423810396D5
          rkti(7,4) = -0.7322288947372697D5
          rkti(7,5) = 0.901805806954291D5
          rkti(7,6) = -0.8510575997895156D5
          rkti(7,7) = 0.973722121263577D5
          rkti(7,8) = -0.5023148619949531D5
          rkti(7,9) = 0
          rkti(8,1) = -0.198011474398138D5
          rkti(8,2) = 0.3162853640743606D5
          rkti(8,3) = -0.3619307997297351D5
          rkti(8,4) = 0.9628416449464767D5
          rkti(8,5) = -0.1206176343076683D6
          rkti(8,6) = 0.11840607139754D6
          rkti(8,7) = -0.1394402172826093D6
          rkti(8,8) = 0.7291003352780167D5
          rkti(8,9) = 0
          rkti(9,1) = 0.2937068374711136D3
          rkti(9,2) = -0.4467669859627889D3
          rkti(9,3) = 0.5215252143810873D3
          rkti(9,4) = -0.1235201770575277D4
          rkti(9,5) = 0.1457584788582958D4
          rkti(9,6) = -0.1282235721258621D4
          rkti(9,7) = 0.1396593278543072D4
          rkti(9,8) = -0.7062058833992985D3
          rkti(9,9) = 0.1D1
          ALLOCATE(rkc(stages))
          rkc(1) = 0
          rkc(2) = 0.36797342069044D-1
          rkc(3) = 0.26228434024234D0
          rkc(4) = 0.47680563437907D0
          rkc(5) = 0.62134978740976D0
          rkc(6) = 0.83024972665914D0
          rkc(7) = 0.95454032671653D0
          rkc(8) = 1
          rkc(9) = 0.61D0
          errorder = 8 ! order of embedded method
          ierrscale = 0.845d0
          ALLOCATE(errcoef(stages))
          errcoef(1) = 0.516806719843737D0
          errcoef(2) = 0.3904711828396375D-2
          errcoef(3) = -0.7291877513896598D-2
          errcoef(4) = 0.4376727428779696D-1
          errcoef(5) = 0.5941394780989422D0
          errcoef(6) = -0.2609783198663058D-1
          errcoef(7) = 0.1782842279756358D-1
          errcoef(8) = -0.7908056422296778D-2
          errcoef(9) = -0.6159184153291202D0
          ijactol=1.d-1
          ihconstl=1.0d0	
          ihconstm=2.0d0
          isafe=0.5d0
          imaxit=12
          ikappasafe=1.0d0
          ioptit=3
          inrmdiv=1.d0
          ilinsolver=SLV_DIRECT
       CASE default
          ierr = ERRILLEGALMETHODORDER
          RETURN
       END SELECT
    END SELECT
    ! -------- General method setup --------
    ! Since the transpose of rkt and rkti is frequently used we form them.
    ALLOCATE(rktt(stages,stages))
    rktt = TRANSPOSE(rkt)
    ALLOCATE(rktit(stages,stages))
    rktit = TRANSPOSE(rkti)
    ierr = 0
    methodsetup = .TRUE.
    RETURN
  END SUBROUTINE setup_method

  SUBROUTINE free_method()
    IF (ALLOCATED(reigenvals)) DEALLOCATE(reigenvals)
    IF (ALLOCATED(ceigenvals)) DEALLOCATE(ceigenvals)
    IF (ALLOCATED(rkt)) DEALLOCATE(rkt)
    IF (ALLOCATED(rkti)) DEALLOCATE(rkti)
    IF (ALLOCATED(rktt)) DEALLOCATE(rktt)
    IF (ALLOCATED(rktit)) DEALLOCATE(rktit)
    IF (ALLOCATED(rkc)) DEALLOCATE(rkc)
    IF (ALLOCATED(errcoef)) DEALLOCATE(errcoef)
    IF (ALLOCATED(fitA)) DEALLOCATE(fitA)
    IF (ALLOCATED(fitAnew)) DEALLOCATE(fitAnew)
    IF (ALLOCATED(fitG)) DEALLOCATE(fitG)
    methodsetup = .FALSE.
    RETURN
  END SUBROUTINE free_method

  SUBROUTINE setup(info,&
       & verboseunit,verboselevel,errunit,&      ! Output units
       & name,order,&                            ! Name and order of method
       & atol,rtol,normtype,&                    ! Tolerances and norm to use
       & kappa,conskappa,kappasafe,maxit,nrmdiv,&! Variables for Newton itera.
       & jactol,minjrefactquot,maxjrefactquot,&  ! Var. for Jacobian recalc.
       & hinit,hmin,optit,safe,gustafsson,&      ! Step-size related variables
       & polfitorder,&                           ! Order of LS fit
       & linsolver,nlimit)                       ! Type of linear solver
    INTEGER, INTENT(OUT) :: info
    ! ******************* Optional parameters *******************
    INTEGER, OPTIONAL, INTENT(IN) :: verboseunit
    INTEGER, OPTIONAL, INTENT(IN) :: verboselevel
    REAL(wp), OPTIONAL, INTENT(IN) :: atol
    REAL(wp), OPTIONAL, INTENT(IN) :: rtol
    REAL(wp), OPTIONAL, INTENT(IN) :: nrmdiv
    REAL(wp), OPTIONAL, INTENT(IN) :: conskappa
    REAL(wp), OPTIONAL, INTENT(IN) :: kappa
    REAL(wp), OPTIONAL, INTENT(IN) :: kappasafe
    REAL(wp), OPTIONAL, INTENT(IN) :: hmin
    INTEGER, OPTIONAL, INTENT(IN) :: maxit
    REAL(wp), OPTIONAL, INTENT(IN) :: jactol
    REAL(wp), OPTIONAL, INTENT(IN) :: minjrefactquot
    REAL(wp), OPTIONAL, INTENT(IN) :: maxjrefactquot
    INTEGER, OPTIONAL, INTENT(IN) :: errunit
    CHARACTER (LEN=*), OPTIONAL, INTENT(IN) :: name
    INTEGER, OPTIONAL, INTENT(IN) :: optit
    INTEGER, OPTIONAL, INTENT(IN) :: order
    INTEGER, OPTIONAL, INTENT(IN) :: polfitorder
    REAL(wp), OPTIONAL, INTENT(IN) :: hinit
    LOGICAL, OPTIONAL, INTENT(IN) :: gustafsson
    REAL(wp), OPTIONAL, INTENT(IN) :: safe
    INTEGER, OPTIONAL, INTENT(IN) :: normtype
    INTEGER, OPTIONAL, INTENT(IN) :: linsolver
    INTEGER, OPTIONAL, INTENT(IN) :: nlimit

    info=0

    ! setup method.
    CALL setup_method(info,name,order)
    IF (info/=0) RETURN

    IF (PRESENT(verboseunit)) iverboseunit = verboseunit
    IF (PRESENT(verboselevel)) iverboselevel = verboselevel
    IF (PRESENT(atol)) iatol = atol
    IF (PRESENT(rtol)) irtol = rtol
    IF (PRESENT(nrmdiv)) inrmdiv = nrmdiv
    IF (PRESENT(conskappa)) iconskappa = conskappa
    IF (PRESENT(kappa)) ikappa = kappa
    IF (PRESENT(kappasafe)) ikappasafe = kappasafe
    IF (PRESENT(hmin)) ihmin = hmin
    IF (PRESENT(maxit)) imaxit = maxit
    IF (PRESENT(jactol)) ijactol = jactol
    IF (PRESENT(minjrefactquot)) ihconstl = minjrefactquot
    IF (PRESENT(maxjrefactquot)) ihconstm = maxjrefactquot
    IF (PRESENT(errunit)) ierrunit = errunit
    IF (PRESENT(optit)) ioptit = optit
    IF (PRESENT(polfitorder)) ipolfitorder = polfitorder
    IF (PRESENT(gustafsson)) igustafsson = gustafsson
    IF (PRESENT(safe)) isafe = safe
    IF (PRESENT(hinit)) h0 = ABS(hinit)
    IF (PRESENT(normtype)) inormtype = normtype
    IF (PRESENT(linsolver)) ilinsolver = linsolver
    IF (PRESENT(nlimit)) inlimit = nlimit
    RETURN
  END SUBROUTINE setup

  ! Intermediate result output functions
  SUBROUTINE printresult(t,y)
    REAL(wp), INTENT(IN) :: t
    REAL(wp), DIMENSION(:), INTENT(IN) :: y
    INTEGER :: i
    WRITE (iverboseunit,'(A,ES24.16)') ' ----- Result at: ',t
    DO i=1,SIZE(y)
       WRITE (iverboseunit,'(ES24.16)') y(i)
    END DO
    WRITE (iverboseunit,'(A)') ' -------------------- '
    RETURN
  END SUBROUTINE printresult

  SUBROUTINE noresult(t,y)
    REAL(wp), INTENT(IN) :: t
    REAL(wp), DIMENSION(:), INTENT(IN) :: y
    RETURN
  END SUBROUTINE noresult
END MODULE sodes
