MODULE genmatrix
  USE blas
  USE lapack 
#ifdef QMR
  USE qmrpack
#endif
  USE defines
  IMPLICIT NONE

  ! matrix type definition
  TYPE mtx_type
     INTEGER :: id
     INTEGER :: n
     ! For banded matrices
     INTEGER :: kl
     INTEGER :: ku
  END TYPE mtx_type
  INTEGER, PARAMETER :: MTX_TYPE_DENSE=1
  INTEGER, PARAMETER :: MTX_TYPE_BANDED=2

  INTEGER, PARAMETER :: SLV_DIRECT=Z'00'
  INTEGER, PARAMETER :: SLV_QMR=Z'01'
  INTEGER, PARAMETER :: ITER_SOLVER=Z'00FF'
  INTEGER, PARAMETER :: ITER_PRECOND=Z'FF00'
  INTEGER, PARAMETER :: ITER_PRECOND_NONE=Z'0100'
  INTEGER, PARAMETER :: ITER_PRECOND_JACOBI=Z'0200'
  INTEGER, PARAMETER :: ITER_PRECOND_FULL=Z'0300'
  INTEGER, PARAMETER :: ITER_QMR_MAXVW=5
  INTEGER, PARAMETER :: ITER_QMR_M=2*ITER_QMR_MAXVW+2

  ! General type for real and complex matrix
  TYPE rmatrix
     TYPE(mtx_type) :: mtype
     REAL(wp), DIMENSION(:,:), POINTER :: a
     INTEGER, DIMENSION(:), POINTER :: i
     ! indicates whether the matrix might be factorized
     LOGICAL :: factorizable 
  END TYPE rmatrix
  TYPE cmatrix
     TYPE(mtx_type) :: mtype
     COMPLEX(wp), DIMENSION(:,:), POINTER :: a
     INTEGER, DIMENSION(:), POINTER :: i
     ! indicates whether the matrix might be factorized
     LOGICAL :: factorizable 
  END TYPE cmatrix

  TYPE iterative_mtx_r
     TYPE(rmatrix) :: m
     REAL(wp) :: norm
     REAL(wp), DIMENSION(:,:), POINTER :: vecs
     LOGICAL :: first
     LOGICAL :: recalcprecond
     LOGICAL :: hotstart
  END TYPE iterative_mtx_r

  TYPE iterative_mtx_c
     TYPE(cmatrix) :: m
     REAL(wp) :: norm
     COMPLEX(wp), DIMENSION(:,:), POINTER :: vecs
     LOGICAL :: first
     LOGICAL :: recalcprecond
     LOGICAL :: hotstart
  END TYPE iterative_mtx_c

  INTERFACE allocate_mtx
     MODULE PROCEDURE allocate_mtx_r, allocate_mtx_c
  END INTERFACE

  INTERFACE deallocate_mtx
     MODULE PROCEDURE deallocate_mtx_r, deallocate_mtx_c
  END INTERFACE

  INTERFACE deallocate_iterative_mtx
     MODULE PROCEDURE deallocate_iterative_mtx_r, deallocate_iterative_mtx_c
  END INTERFACE

  INTERFACE factorize
     MODULE PROCEDURE factorize_r, factorize_c
  END INTERFACE

  INTERFACE fac_aimj
     MODULE PROCEDURE fac_aimj_r, fac_aimj_c
  END INTERFACE

  INTERFACE solve
     MODULE PROCEDURE solve_r, solve_c
  END INTERFACE

  INTERFACE itersolve
     MODULE PROCEDURE itersolve_r, itersolve_c
  END INTERFACE

  INTERFACE allocate_precond
     MODULE PROCEDURE allocate_precond_r, allocate_precond_c
  END INTERFACE

  INTERFACE genmv
     MODULE PROCEDURE genmv_r, genmv_c, genmv_rc
  END INTERFACE

CONTAINS
  SUBROUTINE mtx_conform(tin1,tin2,tout,ierr)
    TYPE(mtx_type), INTENT(IN) :: tin1, tin2
    TYPE(mtx_type), INTENT(OUT) :: tout
    INTEGER, INTENT(OUT) :: ierr
    ierr = 0
    IF (tin1%n==tin2%n) THEN
       tout%n=tin1%n
    ELSE
       ierr = ERRUNCONFORMABLE
       RETURN
    END IF
    SELECT CASE(tin1%id)
    CASE (MTX_TYPE_DENSE)
       SELECT CASE(tin2%id)
       CASE (MTX_TYPE_DENSE)
          tout%id = MTX_TYPE_DENSE
       CASE (MTX_TYPE_BANDED)
          tout%id = MTX_TYPE_DENSE
       CASE default
          ierr = ERRMTYPEUNDEFINED
       END SELECT
    CASE (MTX_TYPE_BANDED)
       SELECT CASE(tin2%id)
       CASE (MTX_TYPE_DENSE)
          tout%id = MTX_TYPE_DENSE
       CASE (MTX_TYPE_BANDED)
          tout%id = MTX_TYPE_BANDED
          tout%kl = MAX(tin1%kl,tin2%kl)
          tout%ku = MAX(tin1%ku,tin2%ku)
       CASE default
          ierr = ERRMTYPEUNDEFINED
       END SELECT
    CASE default
       ierr = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE mtx_conform

  SUBROUTINE allocate_mtx_r(m,mtype,ierr,factorizable)
    TYPE(rmatrix), INTENT(INOUT) :: m
    TYPE(mtx_type), INTENT(IN) :: mtype
    INTEGER, INTENT(out) :: ierr
    LOGICAL, INTENT(IN), OPTIONAL :: factorizable
    ierr = 0
    m%mtype = mtype
    IF (PRESENT(factorizable)) THEN
       m%factorizable = factorizable
    ELSE
       m%factorizable = .FALSE.
    END IF
    IF (m%factorizable) THEN
       SELECT CASE(mtype%id)
       CASE (MTX_TYPE_DENSE) ! dense matrix
          ALLOCATE(m%a(mtype%n,mtype%n))
          ALLOCATE(m%i(mtype%n))
       CASE (MTX_TYPE_BANDED) ! banded matrix
          ALLOCATE(m%a(2*mtype%kl+mtype%ku+1,mtype%n))
          ALLOCATE(m%i(mtype%n))
       CASE default
          ierr = ERRMTYPEUNDEFINED
       END SELECT
    ELSE
       SELECT CASE(mtype%id)
       CASE (MTX_TYPE_DENSE) ! dense matrix
          ALLOCATE(m%a(mtype%n,mtype%n))
          NULLIFY(m%i)
       CASE (MTX_TYPE_BANDED) ! banded matrix
          ALLOCATE(m%a(mtype%kl+mtype%ku+1,mtype%n))
          NULLIFY(m%i)
       CASE default
          ierr = ERRMTYPEUNDEFINED
       END SELECT
    END IF
    RETURN
  END SUBROUTINE allocate_mtx_r

  SUBROUTINE allocate_mtx_c(m,mtype,ierr,factorizable)
    TYPE(cmatrix), INTENT(INOUT) :: m
    TYPE(mtx_type), INTENT(IN) :: mtype
    INTEGER, INTENT(out) :: ierr
    LOGICAL, INTENT(IN), OPTIONAL :: factorizable
    ierr = 0
    m%mtype = mtype
    IF (PRESENT(factorizable)) THEN
       m%factorizable = factorizable
    ELSE
       m%factorizable = .FALSE.
    END IF
    IF (m%factorizable) THEN
       SELECT CASE(mtype%id)
       CASE (MTX_TYPE_DENSE) ! dense matrix
          ALLOCATE(m%a(mtype%n,mtype%n))
          ALLOCATE(m%i(mtype%n))
       CASE (MTX_TYPE_BANDED) ! banded matrix
          ALLOCATE(m%a(2*mtype%kl+mtype%ku+1,mtype%n))
          ALLOCATE(m%i(mtype%n))
       CASE default
          ierr = ERRMTYPEUNDEFINED
       END SELECT
    ELSE
       SELECT CASE(mtype%id)
       CASE (MTX_TYPE_DENSE) ! dense matrix
          ALLOCATE(m%a(mtype%n,mtype%n))
          NULLIFY(m%i)
       CASE (MTX_TYPE_BANDED) ! banded matrix
          ALLOCATE(m%a(mtype%kl+mtype%ku+1,mtype%n))
          NULLIFY(m%i)
       CASE default
          ierr = ERRMTYPEUNDEFINED
       END SELECT
    END IF
    RETURN
  END SUBROUTINE allocate_mtx_c

  SUBROUTINE deallocate_mtx_r(m)
    TYPE(rmatrix), INTENT(INOUT) :: m
    IF (ASSOCIATED(m%a)) DEALLOCATE(m%a)
    IF (ASSOCIATED(m%i)) DEALLOCATE(m%i)
    RETURN
  END SUBROUTINE deallocate_mtx_r

  SUBROUTINE deallocate_mtx_c(m)
    TYPE(cmatrix), INTENT(INOUT) :: m
    IF (ASSOCIATED(m%a)) DEALLOCATE(m%a)
    IF (ASSOCIATED(m%i)) DEALLOCATE(m%i)
    RETURN
  END SUBROUTINE deallocate_mtx_c

  SUBROUTINE deallocate_iterative_mtx_r(iter)
    TYPE(iterative_mtx_r), INTENT(inout) :: iter
    CALL deallocate_mtx(iter%m)
    IF (ASSOCIATED(iter%vecs)) DEALLOCATE(iter%vecs)
    RETURN
  END SUBROUTINE deallocate_iterative_mtx_r

  SUBROUTINE deallocate_iterative_mtx_c(iter)
    TYPE(iterative_mtx_c), INTENT(inout) :: iter
    CALL deallocate_mtx(iter%m)
    IF (ASSOCIATED(iter%vecs)) DEALLOCATE(iter%vecs)
    RETURN
  END SUBROUTINE deallocate_iterative_mtx_c

  SUBROUTINE factorize_r(m,info)
    ! Factorizes the matrix, m.
    ! Approx. 2/3 n^3 flops(dense), 2n*kl*(ku+1) to 2nkl(kl+ku+1) (banded)
    TYPE(rmatrix), INTENT(INOUT) :: m
    INTEGER, INTENT(out) :: info
    IF (.NOT.m%factorizable) THEN
       info = ERRUNFACTORIZABLE
       RETURN
    ELSE
       info = 0
    END IF
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE) ! full matrix
       CALL getrf(SIZE(m%a,DIM=1),SIZE(m%a,DIM=2),m%a,&
            & SIZE(m%a,DIM=1),m%i,info)
    CASE (MTX_TYPE_BANDED) ! banded matrix
       CALL gbtrf(m%mtype%n,m%mtype%n,m%mtype%kl,m%mtype%ku,m%a,&
            & SIZE(m%a,DIM=1),m%i,info)
    CASE default
       info = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE factorize_r

  SUBROUTINE factorize_c(m,info)
    ! Factorizes the matrix, m.
    ! Approx. 8/3 n^3 flops(dense), 8n*kl*(ku+1) to 8nkl(kl+ku+1) (banded)
    TYPE(cmatrix), INTENT(INOUT) :: m
    INTEGER, INTENT(out) :: info
    IF (.NOT.m%factorizable) THEN
       info = ERRUNFACTORIZABLE
       RETURN
    ELSE
       info = 0
    END IF
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE) ! full matrix
       CALL getrf(SIZE(m%a,DIM=1),SIZE(m%a,DIM=2),&
            & m%a,SIZE(m%a,DIM=1),m%i,info)
    CASE (MTX_TYPE_BANDED) ! banded matrix
       CALL gbtrf(m%mtype%n,m%mtype%n,m%mtype%kl,m%mtype%ku,m%a,&
            & SIZE(m%a,DIM=1),m%i,info)
    CASE default
       info = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE factorize_c

  SUBROUTINE fac_aimj_r(alpha,j,m,info,mass)
    ! calculates and factorizes M=alpha*I-J
    REAL(wp), INTENT(in) :: alpha
    TYPE(rmatrix), INTENT(in) :: j
    TYPE(rmatrix), INTENT(inout) :: m
    INTEGER, INTENT(out) :: info
    TYPE(rmatrix), OPTIONAL, INTENT(in) :: mass
    INTEGER :: i, l, offs, ku, kl
    info = 0
    IF (PRESENT(mass)) THEN
       SELECT CASE(mass%mtype%id)
       CASE (MTX_TYPE_DENSE)
          SELECT CASE(j%mtype%id)
          CASE (MTX_TYPE_DENSE)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n)&
                  & .and.(j%mtype%n==mass%mtype%n)) THEN
                m%a = alpha*mass%a-j%a
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
          CASE (MTX_TYPE_BANDED)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n).and.&
                  & (mass%mtype%n==j%mtype%n)) THEN
                m%a = alpha*mass%a
                IF (j%factorizable) THEN
                   offs = 1+j%mtype%ku+j%mtype%kl
                ELSE
                   offs = 1+j%mtype%ku
                END IF
                DO i=-j%mtype%ku,j%mtype%kl
                   DO l=MAX(1,1-i),MIN(j%mtype%n,j%mtype%n-i)
                      m%a(l+i,l) = m%a(l+i,l)-j%a(offs+i,l)
                   END DO
                END DO
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
          CASE default
             info = ERRMTYPEUNDEFINED
          END SELECT
       CASE (MTX_TYPE_BANDED)
          IF (mass%factorizable) THEN
             offs = 1+mass%mtype%ku+mass%mtype%kl
          ELSE
             offs = 1+mass%mtype%ku
          END IF
          SELECT CASE(j%mtype%id)
          CASE (MTX_TYPE_DENSE)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n)&
                  & .and.(mass%mtype%n==j%mtype%n)) THEN
                m%a = -j%a
                DO i=-mass%mtype%ku,mass%mtype%kl
                   DO l=MAX(1,1-i),MIN(mass%mtype%n,mass%mtype%n-i)
                      m%a(l+i,l) = m%a(l+i,l)+alpha*mass%a(offs+i,l)
                   END DO
                END DO
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
          CASE (MTX_TYPE_BANDED)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_BANDED).and.(m%mtype%n==j%mtype%n)&
                  & .and.(m%mtype%kl==MAX(j%mtype%kl,mass%mtype%kl))&
                  & .and.(m%mtype%ku==MAX(j%mtype%ku,mass%mtype%ku)).and.&
                  & m%factorizable) THEN
                IF (mass%factorizable) THEN
                   i = mass%mtype%kl+1
                ELSE
                   i = 1
                END IF
                IF (j%factorizable) THEN
                   l = j%mtype%kl+1
                ELSE
                   l = 1
                END IF
                ! Non-overlapping upperband
                IF (mass%mtype%ku>j%mtype%ku) THEN
                   m%a((m%mtype%kl+1):(m%mtype%kl+1+mass%mtype%ku-&
                        & j%mtype%ku),:) = alpha*mass%a(i:(offs-&
                        & j%mtype%ku),:)
                ELSE IF (mass%mtype%ku<j%mtype%ku) THEN
                   m%a((m%mtype%kl+1):(m%mtype%kl+1+j%mtype%ku-&
                        & mass%mtype%ku),:) = -j%a(l:(l+j%mtype%ku-&
                        & mass%mtype%ku),:)
                END IF
                ! Overlapping band
                kl = MIN(mass%mtype%kl,j%mtype%kl)
                ku = MIN(mass%mtype%ku,j%mtype%ku)
                m%a((m%mtype%kl+1+m%mtype%ku-ku):(m%mtype%kl+1+&
                     & m%mtype%ku+kl),:) = alpha*mass%a((i+mass%mtype%ku-ku):&
                     & (offs+kl),:)-j%a((l+j%mtype%ku-ku):(l+j%mtype%ku+kl),:)
                ! Non-overlapping lowerband
                IF (mass%mtype%kl>j%mtype%kl) THEN
                   m%a((m%mtype%kl+1+m%mtype%ku+kl):(2*m%mtype%kl+1+&
                        & m%mtype%ku),:) = alpha*mass%a((offs+kl):(offs+&
                        & mass%mtype%kl),:)
                ELSE IF (mass%mtype%kl<j%mtype%kl) THEN
                   m%a((m%mtype%kl+1+m%mtype%ku+kl):(2*m%mtype%kl+1+&
                        & m%mtype%ku),:) = -j%a((l+j%mtype%ku+kl):(l+&
                        & j%mtype%ku+j%mtype%kl),:)
                END IF
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
           CASE default
             info = ERRMTYPEUNDEFINED
           END SELECT
       CASE default
          info = ERRMTYPEUNDEFINED
       END SELECT
    ELSE
       SELECT CASE(j%mtype%id)
       CASE (MTX_TYPE_DENSE)
          ! check that m is of conform size and type
          IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n)) THEN
             m%a = -j%a
             DO i=1,SIZE(m%a,DIM=1)
                m%a(i,i) = m%a(i,i)+alpha
             END DO
             CALL factorize(m,info)
          ELSE
             info = ERRUNCONFORMABLE
          END IF
       CASE (MTX_TYPE_BANDED)
          ! check that m is of conform size and type
          IF ((m%mtype%id==MTX_TYPE_BANDED).and.(m%mtype%n==j%mtype%n).and.&
               & (m%mtype%kl==j%mtype%kl).and.(m%mtype%ku==j%mtype%ku).and.&
               & m%factorizable) THEN
             IF (j%factorizable) THEN
                m%a(m%mtype%kl+1:,:) = -j%a(m%mtype%kl+1:,:)
             ELSE
                m%a(m%mtype%kl+1:,:) = -j%a
             END IF
             offs = m%mtype%kl+m%mtype%ku+1
             DO i=1,SIZE(m%a,DIM=2)
                m%a(offs,i) = m%a(offs,i)+alpha
             END DO
             CALL factorize(m,info)
          ELSE
             info = ERRUNCONFORMABLE
          END IF
       CASE default
          info = ERRMTYPEUNDEFINED
       END SELECT
    END IF
    RETURN
  END SUBROUTINE fac_aimj_r

  SUBROUTINE fac_aimj_c(alpha,j,m,info,mass)
    ! calculates and factorizes M=alpha*I-J
    COMPLEX(wp), INTENT(in) :: alpha
    TYPE(rmatrix), INTENT(in) :: j
    TYPE(cmatrix), INTENT(inout) :: m
    INTEGER, INTENT(out) :: info
    TYPE(rmatrix), OPTIONAL, INTENT(in) :: mass
    INTEGER :: i, l, offs, kl, ku
    info = 0
    IF (PRESENT(mass)) THEN
       SELECT CASE(mass%mtype%id)
       CASE (MTX_TYPE_DENSE)
          SELECT CASE(j%mtype%id)
          CASE (MTX_TYPE_DENSE)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n)&
                  & .and.(j%mtype%n==mass%mtype%n)) THEN
                m%a = alpha*mass%a-j%a
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
          CASE (MTX_TYPE_BANDED)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n).and.&
                  & (mass%mtype%n==j%mtype%n)) THEN
                m%a = alpha*mass%a
                IF (j%factorizable) THEN
                   offs = 1+j%mtype%ku+j%mtype%kl
                ELSE
                   offs = 1+j%mtype%ku
                END IF
                DO i=-j%mtype%ku,j%mtype%kl
                   DO l=MAX(1,1-i),MIN(j%mtype%n,j%mtype%n-i)
                      m%a(l+i,l) = m%a(l+i,l)-j%a(offs+i,l)
                   END DO
                END DO
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
          CASE default
             info = ERRMTYPEUNDEFINED
          END SELECT
       CASE (MTX_TYPE_BANDED)
          IF (mass%factorizable) THEN
             offs = 1+mass%mtype%ku+mass%mtype%kl
          ELSE
             offs = 1+mass%mtype%ku
          END IF
          SELECT CASE(j%mtype%id)
          CASE (MTX_TYPE_DENSE)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n)&
                  & .and.(mass%mtype%n==j%mtype%n)) THEN
                m%a = -j%a
                DO i=-mass%mtype%ku,mass%mtype%kl
                   DO l=MAX(1,1-i),MIN(mass%mtype%n,mass%mtype%n-i)
                      m%a(l+i,l) = m%a(l+i,l)+alpha*mass%a(offs+i,l)
                   END DO
                END DO
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
          CASE (MTX_TYPE_BANDED)
             ! check that m, jac and mass are of conform size and type
             IF ((m%mtype%id==MTX_TYPE_BANDED).and.(m%mtype%n==j%mtype%n)&
                  & .and.(m%mtype%kl==MAX(j%mtype%kl,mass%mtype%kl))&
                  & .and.(m%mtype%ku==MAX(j%mtype%ku,mass%mtype%ku)).and.&
                  & m%factorizable) THEN
                IF (mass%factorizable) THEN
                   i = mass%mtype%kl+1
                ELSE
                   i = 1
                END IF
                IF (j%factorizable) THEN
                   l = j%mtype%kl+1
                ELSE
                   l = 1
                END IF
                ! Non-overlapping upperband
                IF (mass%mtype%ku>j%mtype%ku) THEN
                   m%a((m%mtype%kl+1):(m%mtype%kl+1+mass%mtype%ku-&
                        & j%mtype%ku),:) = alpha*mass%a(i:(offs-&
                        & j%mtype%ku),:)
                ELSE IF (mass%mtype%ku<j%mtype%ku) THEN
                   m%a((m%mtype%kl+1):(m%mtype%kl+1+j%mtype%ku-&
                        & mass%mtype%ku),:) = -j%a(l:(l+j%mtype%ku-&
                        & mass%mtype%ku),:)
                END IF
                ! Overlapping band
                kl = MIN(mass%mtype%kl,j%mtype%kl)
                ku = MIN(mass%mtype%ku,j%mtype%ku)
                m%a((m%mtype%kl+1+m%mtype%ku-ku):(m%mtype%kl+1+&
                     & m%mtype%ku+kl),:) = alpha*mass%a((i+mass%mtype%ku-ku):&
                     & (offs+kl),:)-j%a((l+j%mtype%ku-ku):(l+j%mtype%ku+kl),:)
                ! Non-overlapping lowerband
                IF (mass%mtype%kl>j%mtype%kl) THEN
                   m%a((m%mtype%kl+1+m%mtype%ku+kl):(2*m%mtype%kl+1+&
                        & m%mtype%ku),:) = alpha*mass%a((offs+kl):(offs+&
                        & mass%mtype%kl),:)
                ELSE IF (mass%mtype%kl<j%mtype%kl) THEN
                   m%a((m%mtype%kl+1+m%mtype%ku+kl):(2*m%mtype%kl+1+&
                        & m%mtype%ku),:) = -j%a((l+j%mtype%ku+kl):(l+&
                        & j%mtype%ku+j%mtype%kl),:)
                END IF
                CALL factorize(m,info)
             ELSE
                info = ERRUNCONFORMABLE
             END IF
           CASE default
             info = ERRMTYPEUNDEFINED
           END SELECT
       CASE default
          info = ERRMTYPEUNDEFINED
       END SELECT
    ELSE
       SELECT CASE(j%mtype%id)
       CASE (MTX_TYPE_DENSE)
          ! check that m is of conform size and type
          IF ((m%mtype%id==MTX_TYPE_DENSE).and.(m%mtype%n==j%mtype%n)) THEN
             m%a = -j%a
             DO i=1,SIZE(m%a,DIM=1)
                m%a(i,i) = m%a(i,i)+alpha
             END DO
             CALL factorize(m,info)
          ELSE
             info = ERRUNCONFORMABLE
          END IF
       CASE (MTX_TYPE_BANDED)
          ! check that m is of conform size and type
          IF ((m%mtype%id==MTX_TYPE_BANDED).and.(m%mtype%n==j%mtype%n).and.&
               & (m%mtype%kl==j%mtype%kl).and.(m%mtype%ku==j%mtype%ku).and.&
               & m%factorizable) THEN
             IF (j%factorizable) THEN
                m%a(m%mtype%kl+1:,:) = -j%a(m%mtype%kl+1:,:)
             ELSE
                m%a(m%mtype%kl+1:,:) = -j%a
             END IF
             offs = m%mtype%kl+m%mtype%ku+1
             DO i=1,SIZE(m%a,DIM=2)
                m%a(offs,i) = m%a(offs,i)+alpha
             END DO
             CALL factorize(m,info)
          ELSE
             info = ERRUNCONFORMABLE
          END IF
       CASE default
          info = ERRMTYPEUNDEFINED
       END SELECT
    END IF
    RETURN
  END SUBROUTINE fac_aimj_c

  SUBROUTINE solve_r(m,z,info,TRANSPOSE)
    ! Solves m*z_out = z_in
    ! Approx. 2 n^2 flops(dense), 2n*(2kl+ku) flops(banded)
    TYPE(rmatrix), INTENT(in) :: m
    REAL(wp), DIMENSION(:), INTENT(inout) :: z
    INTEGER, INTENT(out) :: info
    LOGICAL, OPTIONAL :: TRANSPOSE
    CHARACTER*1 :: trans
    trans = 'N'
    IF (PRESENT(TRANSPOSE)) THEN
       IF (TRANSPOSE) trans = 'T'
    END IF
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE) ! dense matrix
       CALL getrs(trans,SIZE(m%a,DIM=1),1,m%a,SIZE(m%a,DIM=1),&
            & m%i,z,SIZE(z,DIM=1),info)
    CASE (MTX_TYPE_BANDED) ! banded matrix
       CALL gbtrs(trans,m%mtype%n,m%mtype%kl,m%mtype%ku,1,m%a,&
            & SIZE(m%a,DIM=1),m%i,z,SIZE(z,DIM=1),info)
    CASE default
       info = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE solve_r

  SUBROUTINE solve_c(m,z,info,TRANSPOSE)
    ! Solves m*z_out = z_in
    ! Approx. 8 n^2 flops(dense), 8n*(2kl+ku) flops(banded)
    TYPE(cmatrix), INTENT(in) :: m
    COMPLEX(wp), DIMENSION(:), INTENT(inout) :: z
    INTEGER, INTENT(out) :: info
    LOGICAL, OPTIONAL :: TRANSPOSE
    CHARACTER*1 :: trans
    trans = 'N'
    IF (PRESENT(TRANSPOSE)) THEN
       IF (TRANSPOSE) trans = 'T'
    END IF
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE) ! dense matrix
       CALL getrs(trans,SIZE(m%a,DIM=1),1,m%a,SIZE(m%a,DIM=1),&
            & m%i,z,SIZE(z,DIM=1),info)
    CASE (MTX_TYPE_BANDED) ! banded matrix
       CALL gbtrs(trans,m%mtype%n,m%mtype%kl,m%mtype%ku,1,m%a,&
            & SIZE(m%a,DIM=1),m%i,z,SIZE(z,DIM=1),info)
    CASE default
       info = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE solve_c

  SUBROUTINE itersolve_r(TYPE,fac,jac,w,precond,tol,nlimit,ierr,mass,&
                        & niter,nprecond)
    ! solves (fac*mass-jac)*w_out = w_in using QMRpack.
    INTEGER, INTENT(in) :: TYPE
    REAL(wp), INTENT(in) :: fac
    TYPE(rmatrix), INTENT(in) :: jac
    REAL(wp), DIMENSION(:), INTENT(inout) :: w
    TYPE(iterative_mtx_r), INTENT(inout) :: precond
    REAL(wp), INTENT(in) :: tol
    INTEGER, INTENT(in) :: nlimit
    INTEGER, INTENT(out) :: ierr
    TYPE(rmatrix), INTENT(in), OPTIONAL :: mass
    INTEGER, INTENT(inout), OPTIONAL :: niter
    INTEGER, INTENT(inout), OPTIONAL :: nprecond
    REAL(wp), DIMENSION(ITER_QMR_M,5*ITER_QMR_M+13) :: wk
    INTEGER, DIMENSION(4,nlimit+2) :: idx
    INTEGER, DIMENSION(ITER_QMR_M,4) :: iwk
    REAL(wp), DIMENSION(SIZE(w)) :: x
    REAL(wp) :: wtol
    INTEGER, DIMENSION(4) :: info
    INTEGER :: i, nlim, maxvw
    LOGICAL :: precondcurrent
    precondcurrent = .false.
    DO
       IF (precond%first) THEN
          precond%first = .false.
          precond%recalcprecond = .true.
       END IF
       IF (precond%recalcprecond) THEN
          precond%hotstart = .false.
          precond%recalcprecond = .false.
          precond%norm = 1.0_wp
          precondcurrent = .true.
          ! compute preconditioner
          SELECT CASE (IAND(TYPE,ITER_PRECOND))
          CASE (ITER_PRECOND_JACOBI)
             IF (PRESENT(nprecond)) nprecond = nprecond+1
             IF (PRESENT(mass)) THEN
                SELECT CASE (mass%mtype%id)
                CASE (MTX_TYPE_DENSE)
                   DO i=1,mass%mtype%n
                      precond%m%a(i,1) = fac*mass%a(i,i)
                   END DO
                CASE (MTX_TYPE_BANDED)
                   precond%m%a(:,1) = fac*mass%a(mass%mtype%ku+1,:)
                END SELECT
             ELSE
                precond%m%a(:,1) = fac
             END IF
             SELECT CASE (jac%mtype%id)
             CASE (MTX_TYPE_DENSE)
                DO i=1,jac%mtype%n
                   precond%m%a(i,1) = 1/(precond%m%a(i,1)-jac%a(i,i))
                END DO
             CASE (MTX_TYPE_BANDED)
                precond%m%a(:,1) = 1/(precond%m%a(:,1)-&
                     & jac%a(jac%mtype%ku+1,:))
             END SELECT
          CASE (ITER_PRECOND_FULL)
             IF (PRESENT(nprecond)) nprecond = nprecond+1
             CALL fac_aimj(fac,jac,precond%m,ierr,mass)
             IF (ierr/=0) RETURN
             ! Since the full factorization is available just solve
             ! system and return.
             CALL solve(precond%m,w,ierr)
             RETURN
          END SELECT
       END IF
       ! Form (preconditioned) right hand side.
       SELECT CASE (IAND(TYPE,ITER_PRECOND))
       CASE (ITER_PRECOND_NONE)
          precondcurrent = .true.
          precond%vecs(:,2) = w
       CASE (ITER_PRECOND_JACOBI)
          precond%vecs(:,2) = w*precond%m%a(:,1)
       CASE (ITER_PRECOND_FULL)
          precond%vecs(:,2) = w
          CALL solve(precond%m,precond%vecs(:,2),ierr)
          IF (ierr/=0) RETURN
       END SELECT
       IF (precond%hotstart) THEN
          wtol = tol
          info(1) = 100000
          info(2) = 0
          nlim = nlimit
          maxvw = ITER_QMR_MAXVW
       ELSE
          wtol = tol
          info(1) = 0
          info(2) = 0
          nlim = nlimit
          maxvw = ITER_QMR_MAXVW
       END IF
       revcom: DO
#ifdef QMR
          CALL uqmr(jac%mtype%n,jac%mtype%n,nlim,maxvw,&
                    & ITER_QMR_M,precond%norm,wk,idx,iwk,&
                    & precond%vecs,wtol,info)
#endif
          IF (info(2)==1) THEN 
             ! Do b=(fac*mass-jac)*x
             IF (PRESENT(mass)) THEN
                CALL genmv(1.0_wp,mass,precond%vecs(:,info(3)),0.0_wp,&
                          & precond%vecs(:,info(4)),ierr)
             ELSE
                precond%vecs(:,info(4)) = precond%vecs(:,info(3))
             END IF
             CALL genmv(-1.0_wp,jac,precond%vecs(:,info(3)),fac,&
                       & precond%vecs(:,info(4)),ierr)
             ! Apply inverse preconditioner
             SELECT CASE (IAND(TYPE,ITER_PRECOND))
             CASE (ITER_PRECOND_JACOBI)
                precond%vecs(:,info(4)) = precond%vecs(:,info(4))&
                                             & *precond%m%a(:,1)
             CASE (ITER_PRECOND_FULL)
                CALL solve(precond%m,precond%vecs(:,info(4)),ierr)
                IF (ierr/=0) RETURN
             END SELECT
          ELSE IF (info(2)==2) THEN 
             ! Apply inverse transpose preconditioner.
             SELECT CASE (IAND(TYPE,ITER_PRECOND))
             CASE (ITER_PRECOND_NONE)
                x = precond%vecs(:,info(3))
             CASE (ITER_PRECOND_JACOBI)
                x = precond%vecs(:,info(3))*precond%m%a(:,1)
             CASE (ITER_PRECOND_FULL)
                x = precond%vecs(:,info(3))
                CALL solve(precond%m,x,ierr,.true.)
                IF (ierr/=0) RETURN
             END SELECT
             ! Do b=(fac*mass-jac)^T*x
             IF (PRESENT(mass)) THEN
                CALL genmv(1.0_wp,mass,x,0.0_wp,&
                          & precond%vecs(:,info(4)),ierr,.true.)
             ELSE
                precond%vecs(:,info(4)) = x
             END IF
             CALL genmv(-1.0_wp,jac,x,fac,&
                       & precond%vecs(:,info(4)),ierr,.true.)
          ELSE
             EXIT revcom
          END IF
       END DO revcom
       IF (PRESENT(niter)) niter = niter+nlim
       IF ((info(1)==0).or.(info(1)==16).or.(info(1)==48)) THEN
          ! result computed.
          ierr = 0
          w = precond%vecs(:,1)
          precond%vecs(:,3) = precond%vecs(:,ITER_QMR_MAXVW+7)
          precond%hotstart = .true.
          IF (nlim>nlimit/2) precond%recalcprecond = .TRUE.
          RETURN
       ELSE IF (info(1)==32.and.precond%hotstart) THEN
          precond%hotstart = .false.
       ELSE
          IF (precondcurrent) THEN
             ierr = ERRCOULDNOTSOLVESYSTEM
             RETURN
          ELSE
             precond%recalcprecond = .TRUE.
          END IF
       END IF
    END DO
    RETURN
  END SUBROUTINE itersolve_r

  SUBROUTINE itersolve_c(TYPE,fac,jac,w,precond,tol,nlimit,ierr,mass,&
                        & niter,nprecond)
    ! solves (fac*mass-jac)*w_out = w_in using QMRpack.
    INTEGER, INTENT(in) :: TYPE
    COMPLEX(wp), INTENT(in) :: fac
    TYPE(rmatrix), INTENT(in) :: jac
    COMPLEX(wp), DIMENSION(:), INTENT(inout) :: w
    TYPE(iterative_mtx_c), INTENT(inout) :: precond
    REAL(wp), INTENT(in) :: tol
    INTEGER, INTENT(in) :: nlimit
    INTEGER, INTENT(out) :: ierr
    TYPE(rmatrix), INTENT(in), OPTIONAL :: mass
    INTEGER, INTENT(inout), OPTIONAL :: niter
    INTEGER, INTENT(inout), OPTIONAL :: nprecond
    COMPLEX(wp), DIMENSION(ITER_QMR_M,5*ITER_QMR_M+7) :: cwk
    REAL(wp), DIMENSION(ITER_QMR_M,6) :: wk
    INTEGER, DIMENSION(3,nlimit+2) :: idx
    INTEGER, DIMENSION(ITER_QMR_M,4) :: iwk
    COMPLEX(wp), DIMENSION(SIZE(w)) :: x
    REAL(wp) :: wtol
    INTEGER, DIMENSION(4) :: info
    INTEGER :: i, nlim, maxvw
    LOGICAL :: precondcurrent
    precondcurrent = .false.
    DO
       IF (precond%first) THEN
          precond%first = .false.
          precond%recalcprecond = .true.
       END IF
       IF (precond%recalcprecond) THEN
          precond%hotstart = .false.
          precond%recalcprecond = .false.
          precond%norm = 1.0_wp
          precondcurrent = .true.
          ! compute preconditioner
          SELECT CASE (IAND(TYPE,ITER_PRECOND))
          CASE (ITER_PRECOND_JACOBI)
             IF (PRESENT(nprecond)) nprecond = nprecond+1
             IF (PRESENT(mass)) THEN
                SELECT CASE (mass%mtype%id)
                CASE (MTX_TYPE_DENSE)
                   DO i=1,mass%mtype%n
                      precond%m%a(i,1) = fac*mass%a(i,i)
                   END DO
                CASE (MTX_TYPE_BANDED)
                   precond%m%a(:,1) = fac*mass%a(mass%mtype%ku+1,:)
                END SELECT
             ELSE
                precond%m%a(:,1) = fac
             END IF
             SELECT CASE (jac%mtype%id)
             CASE (MTX_TYPE_DENSE)
                DO i=1,jac%mtype%n
                   precond%m%a(i,1) = 1/(precond%m%a(i,1)-jac%a(i,i))
                END DO
             CASE (MTX_TYPE_BANDED)
                precond%m%a(:,1) = 1/(precond%m%a(:,1)-&
                     & jac%a(jac%mtype%ku+1,:))
             END SELECT
          CASE (ITER_PRECOND_FULL)
             IF (PRESENT(nprecond)) nprecond = nprecond+1
             CALL fac_aimj(fac,jac,precond%m,ierr,mass)
             IF (ierr/=0) RETURN
             ! Since the full factorization is available just solve
             ! system and return.
             CALL solve(precond%m,w,ierr)
             RETURN
          END SELECT
       END IF
       ! Form (preconditioned) right hand side.
       SELECT CASE (IAND(TYPE,ITER_PRECOND))
       CASE (ITER_PRECOND_NONE)
          precondcurrent = .true.
          precond%vecs(:,2) = w
       CASE (ITER_PRECOND_JACOBI)
          precond%vecs(:,2) = w*precond%m%a(:,1)
       CASE (ITER_PRECOND_FULL)
          precond%vecs(:,2) = w
          CALL solve(precond%m,precond%vecs(:,2),ierr)
          IF (ierr/=0) RETURN
       END SELECT
       IF (precond%hotstart) THEN
          wtol = tol
          info(1) = 100000
          info(2) = 0
          nlim = nlimit
          maxvw = ITER_QMR_MAXVW
       ELSE
          wtol = tol
          info(1) = 0
          info(2) = 0
          nlim = nlimit
          maxvw = ITER_QMR_MAXVW
       END IF
       revcom: DO
#ifdef QMR
          CALL uqmr(jac%mtype%n,jac%mtype%n,nlim,maxvw,&
                    & ITER_QMR_M,precond%norm,cwk,wk,idx,iwk,&
                    & precond%vecs,wtol,info)
#endif
          IF (info(2)==1) THEN 
             ! Do b=(fac*mass-jac)*x
             IF (PRESENT(mass)) THEN
                CALL genmv(1.0_wp,mass,precond%vecs(:,info(3)),&
                          & CMPLX(0.0_wp,KIND=wp),&
                          & precond%vecs(:,info(4)),ierr)
             ELSE
                precond%vecs(:,info(4)) = precond%vecs(:,info(3))
             END IF
             CALL genmv(-1.0_wp,jac,precond%vecs(:,info(3)),fac,&
                       & precond%vecs(:,info(4)),ierr)
             ! Apply inverse preconditioner
             SELECT CASE (IAND(TYPE,ITER_PRECOND))
             CASE (ITER_PRECOND_JACOBI)
                precond%vecs(:,info(4)) = precond%vecs(:,info(4))&
                                             & *precond%m%a(:,1)
             CASE (ITER_PRECOND_FULL)
                CALL solve(precond%m,precond%vecs(:,info(4)),ierr)
                IF (ierr/=0) RETURN
             END SELECT
          ELSE IF (info(2)==2) THEN 
             ! Apply inverse transpose preconditioner.
             SELECT CASE (IAND(TYPE,ITER_PRECOND))
             CASE (ITER_PRECOND_NONE)
                x = precond%vecs(:,info(3))
             CASE (ITER_PRECOND_JACOBI)
                x = precond%vecs(:,info(3))*precond%m%a(:,1)
             CASE (ITER_PRECOND_FULL)
                x = precond%vecs(:,info(3))
                CALL solve(precond%m,x,ierr,.true.)
                IF (ierr/=0) RETURN
             END SELECT
             ! Do b=(fac*mass-jac)^T*x
             IF (PRESENT(mass)) THEN
                CALL genmv(1.0_wp,mass,x,CMPLX(0.0_wp,KIND=wp),&
                          & precond%vecs(:,info(4)),ierr,.true.)
             ELSE
                precond%vecs(:,info(4)) = x
             END IF
             CALL genmv(-1.0_wp,jac,x,fac,&
                       & precond%vecs(:,info(4)),ierr,.true.)
          ELSE
             EXIT revcom
          END IF
       END DO revcom
       IF (PRESENT(niter)) niter = niter+nlim
       IF ((info(1)==0).or.(info(1)==16).or.(info(1)==48)) THEN
          ! result computed.
          ierr = 0
          w = precond%vecs(:,1)
          precond%vecs(:,3) = precond%vecs(:,ITER_QMR_MAXVW+7)
          precond%hotstart = .true.
          IF (nlim>nlimit/2) precond%recalcprecond = .TRUE.
          RETURN
       ELSE IF (info(1)==32.and.precond%hotstart) THEN
          precond%hotstart = .false.
       ELSE
          IF (precondcurrent) THEN
             ierr = ERRCOULDNOTSOLVESYSTEM
             RETURN
          ELSE
             precond%recalcprecond = .TRUE.
          END IF
       END IF
    END DO
    RETURN
  END SUBROUTINE itersolve_c

  SUBROUTINE allocate_precond_r(TYPE,m,mtype,ierr)
    ! Allocates memory for preconditioner and iterative solver workspace.
    INTEGER, INTENT(in) :: TYPE
    TYPE(iterative_mtx_r), INTENT(INOUT) :: m
    TYPE(mtx_type), INTENT(IN) :: mtype
    INTEGER, INTENT(out) :: ierr
    INTEGER :: ext, exti
    ! Determine space for preconditioner
    SELECT CASE (IAND(TYPE,ITER_PRECOND))
    CASE (ITER_PRECOND_NONE)
       NULLIFY(m%m%a)
       NULLIFY(m%m%i)
    CASE (ITER_PRECOND_JACOBI)
       ALLOCATE(m%m%a(mtype%n,1))
       NULLIFY(m%m%i)
    CASE (ITER_PRECOND_FULL)
       CALL allocate_mtx(m%m,mtype,ierr,.TRUE.)
       IF (ierr/=0) RETURN
    CASE default
       ierr = ERRPRECONDUNDEFINED
       RETURN
    END SELECT
    ! Determine workspace for solver
    SELECT CASE (IAND(TYPE,ITER_SOLVER))
    CASE (SLV_QMR)
       ALLOCATE(m%vecs(mtype%n,4*ITER_QMR_MAXVW+8))
       m%first = .true.
    CASE default
       ierr = ERRSOLVERUNDEFINED
       RETURN
    END SELECT
    RETURN
  END SUBROUTINE allocate_precond_r

  SUBROUTINE allocate_precond_c(TYPE,m,mtype,ierr)
    ! Allocates memory for preconditioner and iterative solver workspace.
    INTEGER, INTENT(in) :: TYPE
    TYPE(iterative_mtx_c), INTENT(INOUT) :: m
    TYPE(mtx_type), INTENT(IN) :: mtype
    INTEGER, INTENT(out) :: ierr
    INTEGER :: ext, exti
    ! Determine space for preconditioner
    SELECT CASE (IAND(TYPE,ITER_PRECOND))
    CASE (ITER_PRECOND_NONE)
       NULLIFY(m%m%a)
       NULLIFY(m%m%i)
    CASE (ITER_PRECOND_JACOBI)
       ALLOCATE(m%m%a(mtype%n,1))
       NULLIFY(m%m%i)
    CASE (ITER_PRECOND_FULL)
       CALL allocate_mtx(m%m,mtype,ierr,.TRUE.)
       IF (ierr/=0) RETURN
    CASE default
       ierr = ERRPRECONDUNDEFINED
       RETURN
    END SELECT
    ! Determine workspace for solver
    SELECT CASE (IAND(TYPE,ITER_SOLVER))
    CASE (SLV_QMR)
       ALLOCATE(m%vecs(mtype%n,4*ITER_QMR_MAXVW+8))
       m%first = .true.
    CASE default
       ierr = ERRSOLVERUNDEFINED
       RETURN
    END SELECT
    RETURN
  END SUBROUTINE allocate_precond_c

  SUBROUTINE genmv_r(alpha,m,y,beta,z,info,TRANSPOSE)
    ! generates the matrix vector product, z=m*y+alpha*z.
    REAL(wp), INTENT(in) :: alpha
    TYPE(rmatrix), INTENT(in) :: m
    REAL(wp), DIMENSION(:), INTENT(in) :: y
    REAL(wp), INTENT(in) :: beta
    REAL(wp), DIMENSION(:), INTENT(inout) :: z
    INTEGER, INTENT(out) :: info
    LOGICAL, INTENT(in), OPTIONAL :: TRANSPOSE
    CHARACTER*1 :: trans
    trans = 'N'
    IF (PRESENT(TRANSPOSE)) THEN
       IF (TRANSPOSE) trans='T'
    END IF
    info = 0 
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE)
       CALL gemv(trans,m%mtype%n,m%mtype%n,alpha,m%a,SIZE(m%a,DIM=1),y,1,&
            & beta,z,1)
    CASE (MTX_TYPE_BANDED)
       CALL gbmv(trans,m%mtype%n,m%mtype%n,m%mtype%kl,m%mtype%ku,alpha,m%a,&
            & SIZE(m%a,DIM=1),y,1,beta,z,1)
    CASE default
       info = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE genmv_r

  SUBROUTINE genmv_c(alpha,m,y,beta,z,info,TRANSPOSE)
    ! generates the matrix vector product, z=m*y+alpha*z.
    COMPLEX(wp), INTENT(in) :: alpha
    TYPE(cmatrix), INTENT(in) :: m
    COMPLEX(wp), DIMENSION(:), INTENT(in) :: y
    COMPLEX(wp), INTENT(in) :: beta
    COMPLEX(wp), DIMENSION(:), INTENT(inout) :: z
    INTEGER, INTENT(out) :: info
    LOGICAL, INTENT(in), OPTIONAL :: TRANSPOSE
    CHARACTER*1 :: trans
    trans = 'N'
    IF (PRESENT(TRANSPOSE)) THEN
       IF (TRANSPOSE) trans='T'
    END IF
    info = 0 
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE)
       CALL gemv(trans,m%mtype%n,m%mtype%n,alpha,m%a,SIZE(m%a,DIM=1),y,1,&
            & beta,z,1)
    CASE (MTX_TYPE_BANDED)
       CALL gbmv(trans,m%mtype%n,m%mtype%n,m%mtype%kl,m%mtype%ku,alpha,&
            & m%a,SIZE(m%a,DIM=1),y,1,beta,z,1)
    CASE default
       info = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE genmv_c

  SUBROUTINE genmv_rc(alpha,m,y,beta,z,info,TRANSPOSE)
    ! generates the matrix vector product, z=m*y+alpha*z.
    REAL(wp), INTENT(in) :: alpha
    TYPE(rmatrix), INTENT(in) :: m
    COMPLEX(wp), DIMENSION(:), INTENT(in) :: y
    COMPLEX(wp), INTENT(in) :: beta
    COMPLEX(wp), DIMENSION(:), INTENT(inout) :: z
    INTEGER, INTENT(out) :: info
    LOGICAL, INTENT(in), OPTIONAL :: TRANSPOSE
    REAL(wp), DIMENSION(SIZE(y)) :: yw, zwr, zwi
    CHARACTER*1 :: trans
    trans = 'N'
    IF (PRESENT(TRANSPOSE)) THEN
       IF (TRANSPOSE) trans='T'
    END IF
    info = 0 
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE)
       ! Seperate real and complex part.       
       yw = REAL(y,KIND=wp)
       CALL gemv(trans,m%mtype%n,m%mtype%n,alpha,m%a,SIZE(m%a,DIM=1),yw,1,&
            & 0.0_wp,zwr,1)
       yw = AIMAG(y)
       CALL gemv(trans,m%mtype%n,m%mtype%n,alpha,m%a,SIZE(m%a,DIM=1),yw,1,&
            & 0.0_wp,zwi,1)
       IF (beta/=CMPLX(0.0_wp,KIND=wp)) THEN
          z = beta*z+CMPLX(zwr,zwi,KIND=wp)
       ELSE
          z = CMPLX(zwr,zwi,KIND=wp)
       END IF
    CASE (MTX_TYPE_BANDED)
       yw = REAL(y,KIND=wp)
       CALL gbmv(trans,m%mtype%n,m%mtype%n,m%mtype%kl,m%mtype%ku,alpha,&
            & m%a,SIZE(m%a,DIM=1),yw,1,0.0_wp,zwr,1)
       yw = AIMAG(y)
       CALL gbmv(trans,m%mtype%n,m%mtype%n,m%mtype%kl,m%mtype%ku,alpha,&
            & m%a,SIZE(m%a,DIM=1),yw,1,0.0_wp,zwi,1)
       IF (beta/=CMPLX(0.0_wp,KIND=wp)) THEN
          z = beta*z+CMPLX(zwr,zwi,KIND=wp)
       ELSE
          z = CMPLX(zwr,zwi,KIND=wp)
       END IF
    CASE default
       info = ERRMTYPEUNDEFINED
    END SELECT
    RETURN
  END SUBROUTINE genmv_rc

  SUBROUTINE coldiv(i,jac,f1,f2,fdiff,del,rowmax,difmax)
    ! matrix shape related routine for 
    ! numerical jacobian calculation (numjac).
    INTEGER, INTENT(in) :: i
    TYPE(rmatrix), INTENT(in) :: jac
    REAL(wp), DIMENSION(:), INTENT(in) :: f1, f2
    REAL(wp), DIMENSION(:), INTENT(out) :: fdiff
    REAL(wp), INTENT(in) :: del
    INTEGER, INTENT(out) :: rowmax
    REAL(wp), INTENT(out) :: difmax
    INTEGER :: mlarray(1),k,j
    SELECT CASE (jac%mtype%id)
    CASE (MTX_TYPE_DENSE)
       fdiff = f1-f2
       mlarray = MAXLOC(ABS(fdiff))
       rowmax = mlarray(1)
       difmax = ABS(fdiff(rowmax))
       fdiff = fdiff/del
    CASE (MTX_TYPE_BANDED)
       k = MAX(i-jac%mtype%ku,1)
       j = MIN(i+jac%mtype%kl,jac%mtype%n)
       fdiff(k:j) = f1(k:j)-f2(k:j)
       mlarray = MAXLOC(ABS(fdiff(k:j)))
       rowmax = mlarray(1)
       difmax = ABS(fdiff(rowmax))
       fdiff(k:j) = fdiff(k:j)/del
    END SELECT
    RETURN
  END SUBROUTINE coldiv

  SUBROUTINE setcol(i,m,v)
    ! Sets column i of the matrix, m equal to v.
    ! Used for numerical jacobian calculation
    INTEGER, INTENT(in) :: i
    TYPE(rmatrix), INTENT(inout) :: m
    REAL(wp), DIMENSION(:), INTENT(in) :: v
    SELECT CASE (m%mtype%id)
    CASE (MTX_TYPE_DENSE)
       m%a(:,i) = v
    CASE (MTX_TYPE_BANDED)
       m%a(MAX(1,m%mtype%ku+2-i):MIN(SIZE(m%a,DIM=1),m%mtype%ku+1+&
            & SIZE(v)-i),i) = v(MAX(i-m%mtype%ku,1):MIN(i+m%mtype%kl,SIZE(v)))
    END SELECT
    RETURN
  END SUBROUTINE setcol

  FUNCTION coltest(i,jac,v,fac)
    ! Used for numerical jacobian calculation
    INTEGER, INTENT(in) :: i
    TYPE(rmatrix), INTENT(in) :: jac
    REAL(wp), DIMENSION(:), INTENT(in) :: v
    REAL(wp), INTENT(in) :: fac
    LOGICAL :: coltest
    SELECT CASE (jac%mtype%id)
    CASE (MTX_TYPE_DENSE)
       coltest = fac*MAXVAL(ABS(v)) > MAXVAL(ABS(jac%a(:,i)))
    CASE (MTX_TYPE_BANDED)
       coltest = fac*MAXVAL(ABS(v(MAX(i-jac%mtype%ku,1):MIN(i+&
            & jac%mtype%kl,SIZE(v))))) > MAXVAL(ABS(jac%a(MAX(1,&
            & jac%mtype%ku+2-i):MIN(SIZE(jac%a,DIM=1),jac%mtype%ku+1+&
            & SIZE(v)-i),i)))
    END SELECT
    RETURN
  END FUNCTION coltest
END MODULE genmatrix
