
      SUBROUTINE GAUSS2 (N,T,TF,Y,YP,ATOL,RTOL,IER,NSTEPMAX,
     # ILIN,IWORK,WORK,LDIWORK,LDWORK,IBAND,FNC,JACOB)

* NUMERICAL SOLUTION OF SPECIAL SECOND ORDER PROBLEMS
* y''=f(t,y), y(t_0)=y_0, y'(t_0)=yp_0, t \in [t_0,t_f], t_f > t_0.
* BY USING THE  2-STAGE RK-NYSTROM GAUSS METHOD.

* THIS CODE IS DESCRIBED IN THE PAPERS: 
* 1) A CODE BASED ON THE TWO STAGE GAUSS METHOD FOR SECOND 
* ORDER PROBLEMS,
* BY S. GONZALEZ-PINTO AND R. ROJAS BELLO, SUBMITTED FOR 
* PUBLICATION TO
* ACM TRANS. ON MATHEM. SOFTW. IN SEPTEMBER 2005.

* 2) EFFICIENT ITERATIONS FOR GAUSS METHODS ON SECOND ORDER 
*     PROBLEMS,
* BY S. GONZALEZ-PINTO, S. PEREZ-RODRIGUEZ AND R. ROJAS-BELL0,
* J. COMPUT. APPL. MATH., 189 (2006), 80-97.

* ALSO PRESENTED IN THE 21TH BIENNIAL CONFERENCE ON NUMERICAL 
* ANALYSIS   28 JUNE - 1 JULY (2005), DUNDEE (SCOTLAND).
*
* AUTHOR OF THE CODE: SEVERIANO GONZALEZ PINTO (JUNE 2005)
*      DPTO. ANALISIS MATEMATICO (UNIVERSITY OF LA LAGUNA)
*      TENERIFE, CANARY ISLANDS, SPAIN
*      EMAIL: SPINTO@ULL.ES
*--------------------------------------------
* INPUT VARIABLES
*
* N: ACTUAL DIMENSION OF THE PROBLEM
* T: INITIAL POINT OF INTEGRATION
* TF: END-POINT OF INTEGRATION
* Y(N): INITIAL VALUES FOR Y
* YP(N): INITIAL VALUES FOR Y'
* ATOL,RTOL: ABSOLUTE AND RELATIVE TOLEREANCES RESPECTIVELY 
*       (SCALARS)
* NSTEPMAX: NUMBER MAXIMUM OF STEPS TO BE TRIED 
* ILIN (=0,1): ILIN=1 FOR LINEAR PROBLEMS F(T,Y)=JY + G(T)
*         ILIN=0 FOR NONLINEAR PROBLEMS
*     IWORK(LDIWORK):WORK ARRAY FOR INTEGERS(IN INPUT, 
*     IWORK(I)=0,I=1,LDIWORK)
* WORK(LDWORK): WORK ARRAY FOR REAL*8 (INPUT, WORK(I)=0, 
*         I=1,LDWORK)
* LDIWORK: DIMENSION OF ARRAY IWORK
* LDWORK: DIMENSION OF ARRAY WORK
* IBAND(3): ARRAY INDICATING HOW IS STORED THE JACOBIAN MATRIX
*              FOR FULL JACOBIANS (IBAND(1)=0).  FOR BANDED JACOBIANS
*             IBAND(1)=1. FOR BANDED JACOBIANS,IBAND(2) IS THE NUMBER 
*             OF LOWER DIAGONALS AND IBAND(3) THE NUMBER OF UPPER 
*             DIAGONALS (THE MAIN DIAGONAL IS NOT COUNTED)
* FNC: NAME(EXTERNAL) OF SUBROUTINE COMPUTING THE VALUES OF 
*           F(T,Y)
*        SUBROUTINE FNC(M,T,Y,F)
*        REAL*8 T,Y(M),F(M)
*        F(1)=... ETC.
*        RETURN
*        END
*   JACOB: NAME(EXTERNAL) OF SUBROUTINE COMPUTING THE
*  JACOBIAN MATRIX,  D(F(T,Y))/DY. IT HAS THE FOLLOWING FORM  IN 
*     ALL CASES:
*      SUBROUTINE JACOB(MD,M,T,Y,YJAC,IBAND)
*      REAL*8 T,Y(M),YJAC(MD,M)
*      INTEGER IBAND(3)
*      YJAC(I,J)=..., (I=1,MD), (J=1,M)
*        RETURN
*        END
* IF IBAND(1)=0, THE JACOBIAN IS SUPPOSED TO BE FULL, THEN
* YJAC(I,J) STORES PARTIAL F(I) / PARTIAL Y(J) AND MD=M
* IF IBAND(1)=1, THE JACOBIAN IS SUPPOSED TO BE BANDED WITH 
* IBAND(2) LOWER DIAGONALS AND IBAND(3) UPPER DIAGONALS
* (THE MAIN DIAGONAL IS NOT COUNTED). 
* IT IS NOT NECESSARY TO SPECIFY THE VALUES FOR IBAND IN THIS 
* SUBROUTINE, BUT IN THE CALLING PROGRAM.
* FOR BANDED MATRICES THE DATA ARE ENTERED IN THE FOLLOWING 
* FORM
*   YJAC(I-J+IBAND(3)+1,J)= PARTIAL F(I) / PARTIAL Y(J), FOR 
*                 I=1,..,MD; J=1,...,M, WITH MD=IBAND(2)+IBAND(3)+1
*-------------------------------------------------------
* OUTPUT VARIABLES
* T:END-POINT  FOR A SUCCESSFUL INTEGRATION
* Y(N): STORES THE NUMERICAL Y-SOLUTION AT THE END POINT
* YP(N): STORES THE NUMERICAL Y'-SOLUTION AT THE END POINT
* IER: ERROR FLAG (IER=0 MEANS SUCCESSFUL INTEGRATION)
* IWORK(I), I=1,14 RETURNS  STATISTICS DATA FROM THE INTEGRATION 
* IWORK(1): NUMBER OF SUCCESSFUL STEPS
* IWORK(2): NUMBER OF REJECTIONS BY THE LOCAL ERROR ESTIMATOR
* IWORK(3): NUMBER OF REJECTIONS BY DIVERGENCE OF THE ITERATION
* IWORK(4): NUMBER OF JACOBIANS COMPUTED
* IWORK(5): NUMBER OF (REAL) LU-FACTORIZATIONS CARRIED OUT
* IWORK(6): NUMBER OF FORWARD-BACKWARD TRIANGULAR SYSTEMS 
*                     SOLVED
* IWORK(7): NUMBER OF FUNCTIONS (FNC) EVALUATED
* IWORK(8,9,10,11): NUMBER OF TIMES EACH PREDICTOR WAS CHOSEN ON 
*   SUCCESSFUL STEPS. THUS IWORK(8-11)=12,14,7,3 INDICATES THAT THE  
*   PREDICTOR OF ORDER ONE  WAS CHOSEN 12 TIMES, 14 TIMES THE 
*   PREDICTOR OF ORDER 2, 7 TIMES THE PREDICTOR OF ORDER 3 AND SO ON.
* IWORK(12): GIVES THE NUMBER OF ITERATIONS OVER SUCCESSFUL 
*    STEPS 
* IWORK(13): ACCOUNTS THE NUMBER OF ITERATIONS FAILED BY 
*    DIVERGENCE 
* IWORK(14): STANDS FOR THE NUMBER OF ITERATIONS REJECTED BY 
*    THE ESTIMATOR
* IWORK(15). GIVES THE  TOTAL NUMBER OF STEPS TRIED.
* IWORK(I), I=21,LDIWORK: ARE USED FOR STORING PIVOTING INDEXES
*                     IN LU-FACTORIZATIONS
* 
* WORK(1): RETURNS THE INITIAL STEP-SIZE PROPOSED
* WORK(2): RETURNS THE FIRST STEP-SIZE ACCEPTED
* WORK(LDWORK) IS USED AS A WORK VECTOR IN THIS SUBROUTINE
*---------------------------------------------------------------
* THIS SUBROUTINE ALSO MAKES USE OF THE FOLLOWING SUBROUTINES, 
* WHICH ARE INCLUDED BELOW:

*1)SUBROUTINE 
* STARTING(M,NSTEP,T0,H,R,Y0,YP0,F0,Y1,Y2,Y10,Y20,AUX,INIC)
*      THIS IS FOR SELECTING THE BEST PREDICTOR AT THE CURRENT STEP
* 2) SUBROUTINE ESTIM(M,T,Y,YP,H,RG,U,IP,Y1,Y2,F,FH,AUX,EST,IBAND)
*      THIS IS TO COMPUTE THE LOCAL ERROR ESTIMATOR 
* 3) SUBROUTINE GAU2SN(M,T,Y,YP,H,U,IP,Y1,Y2,YH,YPH,
*        LS,IDER,IER,NITER,TETA,ATOL,RTOL,RATE,ILIN,AUX,CTE,IBAND)
*         THIS IS TO SOLVE THE STAGE VALUES OF THE 2-STAGE GAUSS 
*         RK-NYSTROM  BY USING  A SINGLE-NEWTON METHOD
* 4) SUBROUTINE DEC (N, NDIM, A, IP, IER)
*           FOR LU-FACTORIZATIONS WHEN THE JACOBIAN MATRIX IS FULL
* 5) SUBROUTINE SOL (N, NDIM, A, B, IP) 
*          SOLVES  LINER SYSTEMS AX=B. THIS IS USED TOGETHER WITH 
*         SUBROUTINE DEC, SEE BELOW.
* 6) SUBROUTINE DECB (N, NDIM, A, ML, MU, IP, IER)
*        FOR LU-FACTORIZATIONS WHEN THE JACOBIAN MATRIX IS BANDED
* 7) SUBROUTINE SOLB (N, NDIM, A, ML, MU, B, IP) 
*        SOLVES  BANDED LINER SYSTEMS AX=B. THIS IS USED TOGETHER 
*        WITH SUBROUTINE DECB, SEE BELOW.
*---------------------------------------------------------------
      IMPLICIT REAL*8 (A-H,O-Z)

      DIMENSION IWORK(LDIWORK), INICI(5), NRCONV(5), NREST(5)
      DIMENSION NMAXIT(3),WORK(LDWORK),Y(N),YP(N),TET(8),IBAND(3)

      parameter(gamma= 0.83333333333333333333333D-1,hmin=1.d-15)
      external fnc,jacob


* IWORK(1:20) USED FOR STATISTICS
* IWORK(21:N+20) USED FOR THE INDEX VECTOR OF PIVOTING IN LU-FACTORIZATION

* A few parameters for the code
      DATA TET/0.8D0,1.5D0,0.9D0,2.D0,1.D-2,6.D-1,7.D-1,2.D-1/
      DATA NMAXIT/10,6,4/

* ADJUSTING POSITIONS IN VECTOR WORK() FOR STORAGE OF JACOBIAN 
* AND OF THE MATRIX U.  

      N1=10*N
      N2=N**2
      IF (IBAND(1).EQ.0) THEN !Full Jacobian
         N3=N2+N1
         N4=2*N2+N1
         NN=N
      ELSE ! Banded Jacobians. 
* IBAND(1) indicates a banded Jacobian
* IBAND(2) gives the number of   lower bands and 
* IBAND(3) the number of  upper bands. Diagonal is not counted
         NDIMJAC=N*(1+IBAND(2)+IBAND(3))
         NDIM_U=NDIMJAC+N*IBAND(2)
         MLB=IBAND(2)
         MUB=IBAND(3)
         MBJAC=MLB+MUB+1
         LDE1=MBJAC+MLB

         N3=N1+NDIMJAC
         N4=N3+NDIM_U
         NN=MBJAC
      ENDIF


      NP2=2*N
      NP3=3*N
      NP4=4*N
      NP5=5*N
      NP6=6*N
      NP7=7*N
      NP8=8*N
      NP9=9*N

* global accounters 
        NTRAIL=0 ! TOTAL NUMBER OF STEPS TRIED
        nstep=0 ! num. successful steps given
        njac=0 !jacobian evaluations
        nlu=0 !LU-factorizations
        nls=0 !linear sistems solved
        iterfailcon=0 !total of iterates failed by divergence
        iterfailest=0 !total of iterates rejected by estimator
        itergood=0 ! number of good iterates
        do k=1,5
           nrconv(k)=0 !total failures of convergece at k-attempt
           nrest(k)=0 !total rejected steps (k-times) for estimator
           inici(k)=0 !number of times that predictor order k is chosen
        enddo

      do k=1,n 
         WORK(NP2+k)=y(k) 
         WORK(NP3+k)=yp(k)
      enddo

      call FNC(N,T,Y,WORK(NP8+1))
      nder=1 !number of derivative evaluations

      call JACOB (NN,N,T,Y,WORK(N1+1),IBAND) 
      njac=1 !number of Jacobian evaluations
      ijac=1 !Jacobian computed at present step (otherwise ijac=0)


* Computing the initial step-size h0 and the tolerance 
      EPS=1.D-6
      DO J=1,N 
        WORK(J)=Y(J)+EPS*YP(J)
      ENDDO
      CALL FNC(N,T,WORK(1),WORK(NP9+1))
      NDER=NDER+1
      DO J=1,N
         WORK(J)=(WORK(NP9+J)-WORK(NP8+J))/EPS
         WORK(J)=Y(J)+EPS*WORK(J)
      ENDDO
      CALL FNC(N,T,WORK(1),WORK(NP9+1))
      NDER=NDER+1
      DO J=1,N
         WORK(J)=(WORK(NP9+J)-WORK(NP8+J))/EPS
      ENDDO

      ynorm=0.d0
      do J=1,n
          ynorm=ynorm+WORK(J)**2
      enddo	

      ynorm=1.d0+sqrt(ynorm/n) !ynorm=(1+|J^2y'_0|)

* The program comes to LINE LABELLED 3  after each successful step
3     toler=0.d0
      do k=1,n
          toler=toler+y(k)**2
      enddo

      toler=atol+sqrt(toler/n)*rtol !toler=atol+|y_0|rtol 

      nfailest=0 !local failures by estimator (at the current step)
      nfailcon=0 !local failures of convergence (at the current step)


      if (ntrail.eq.0) then !the initial step-size
          h=0.8d0*(720.d0*toler/ynorm)**0.2d0
          if ((tf-t).lt.(1.2d0*h0)) h=(tf-t)
          h0=h !Saving the proposed initial step-size 
          hn=h !this is the step-size proposed at the current step
          hg=h !this is step-size of the latest LU-factorization carried out
      elseif (nstep.eq.1) then
          h1=h !Saving the accepted  initial step-size
      endif

5     NTRAIL=NTRAIL+1

      if (NTRAIL.GT.NSTEPMAX) then 
           ier=-3
           return
      else if (hn.lt.hmin*max(1.d0,Abs(t))) then
           ier=-2
           return
      endif


* By selecting predictors

      HCOC=HN/H
      call starting(N,NSTEP,T,H,HCOC,WORK(NP2+1),WORK(NP3+1),WORK(NP8+1)
     # ,WORK(1),WORK(N+1),WORK(NP4+1),WORK(NP5+1),WORK(N4+1),INIC)

* By updating LU if necessary

      if((ijac.eq.1).or.(abs(hg/hn-1.d0).gt.0.08d0)) then 
          sum=1.d0/(gamma*hn**2)
          IF (IBAND(1).EQ.0) THEN
              DO I=1,N2
                  WORK(N3+I)=-WORK(N1+I)
              ENDDO

              DO I=1,N
                  II=N3+I+N*(I-1)
                 WORK(II)=SUM+WORK(II)
              ENDDO
              CALL DEC (N,N,WORK(N3+1),IWORK(21),IERR)

          ELSE 
               DO I=1,NDIMJAC
                   II=I+(1+(I-1)/MBJAC)*MLB
                  WORK(N3+II)=-WORK(N1+I)
                ENDDO
                DO I=1,N
                   II=N3+MBJAC+(I-1)*LDE1
                  WORK(II)=SUM+WORK(II)
                ENDDO
                CALL DECB (N,LDE1,WORK(N3+1),MLB,MUB,IWORK(21),IERR)
          ENDIF

          NLU=NLU+1
          HG=HN

          if (IERR.NE.0) then
              HN=0.5*HN
              GOTO 5
          endif
      endif

* solving the internal stages

      SUM=1.D0/(GAMMA*HG**2)
      CALL GAU2SN(N,T,Y,YP,HN,WORK(N3+1),IWORK(21),WORK(NP4+1),
     $ WORK(NP5+1),WORK(NP6+1),WORK(NP7+1),LS,IDER,IERR,NITER,TETA,
     $ ATOL,RTOL,RATE,ILIN,WORK(N4+1),SUM,IBAND)


      if (ierr.eq.0) then !convergence is achieved
          nls=nls+ls !global  linear systems
          nder=nder+ider !global number of derivatives
          if (nfailcon.ge.5) then !local failures of convergence 
              nrconv(5)=nrconv(5)+1 !failure of converg (5 times)
          elseif (nfailcon.gt.0) then  
              nrconv(nfailcon)=nrconv(nfailcon)+1
          endif
      else !divergence of the iteration
          iterfailcon=iterfailcon+niter !global iterates failed
          nfailcon=nfailcon+1 !local failures of convergence
          r=min(teta,rate)
          hn=max(r,tet(8))*hn
          if((ijac.eq.0).and.(ilin.eq.0)) then 
               call JACOB (NN,N,T,Y,WORK(N1+1),IBAND) 
               njac=njac+1 !number of Jacobian evaluations
               ijac=1 !Jacobian is computed at present step 
          endif
          goto 5
      endif

* By computing the local error estimator
      rg=hg/hn
      tn=t+hn
      CALL FNC(N,TN,WORK(NP6+1),WORK(NP9+1))
       nder=nder+1
       call ESTIM(N,T,Y,YP,HN,Rg,WORK(N3+1),IWORK(21),WORK(NP4+1),
     # WORK(NP5+1),WORK(NP8+1),WORK(NP9+1),WORK(N4+1),EST,IBAND)
      nls=nls+2

* If the step is not accepted
      if(est.gt.toler) then
          iterfailest=iterfailest+niter !iterates rejected by estimator
          nfailest=nfailest+1 !local failures by estimator (per step)
          r=max(tet(8),tet(1)*(toler/est)**0.2d0)
          hn=r*hn
          if ((nfailest.eq.2).and.(ilin.eq.0)) then 
               call JACOB (NN,N,T,Y,WORK(N1+1),IBAND) 
               njac=njac+1 
               ijac=1
          endif
          goto 5
      endif

* If the step is accepted, a few variables are updated

      if (nfailest.ge.5) then
           nrest(5)=nrest(5)+1 !number of rejections by estimator(5 times or more)
      else if (nfailest.gt.0) then
           nrest(nfailest)=nrest(nfailest)+1
      endif

      nstep=nstep+1
      itergood=itergood+niter
      h=hn
      t=tn

      if (inic.gt.0) then
           inici(inic)=inici(inic)+1 !the predictor selected
      endif

      do k=1,N
           WORK(NP2+K)=Y(K)
           WORK(NP3+K)=YP(K)
           WORK(NP8+K)=WORK(NP9+K)
            Y(K)=WORK(NP6+K)
            YP(K)=WORK(NP7+K)
           WORK(K)=WORK(NP4+K)
           WORK(N+K)=WORK(NP5+K)
      enddo

* To check that the integration is accomplished
      if (abs(t/tf-1.d0).lt.1.d-10) then !successful integration
          ier=0
* a few statistics
          IWORK(1)=NSTEP
          IWORK(2)=NREST(1)+2*NREST(2)+3*NREST(3)+4*NREST(4)+5*NREST(5)
      IWORK(3)=NRCONV(1)+2*NRCONV(2)+3*NRCONV(3)+4*NRCONV(4)+5*NRCONV(5)
          IWORK(4)=NJAC
          IWORK(5)=NLU
          IWORK(6)=NLS
          IWORK(7)=NDER
          IWORK(8)=INICI(1)
          IWORK(9)=INICI(2)
          IWORK(10)=INICI(3)
          IWORK(11)=INICI(4)
          IWORK(12)=ITERGOOD
          IWORK(13)=ITERFAILCON
          IWORK(14)=ITERFAILEST
          IWORK(15)=NTRAIL

         WORK(1)=H0 !Saving the proposed initial step-size
         WORK(2)=H1 !Saving the actual initial step-size
          return
      endif

* We decide if the Jacobian must be updated for the next step
      if ((niter.gt.nmaxit(2)).and.(ilin.eq.0))then
           call JACOB (NN,N,T,Y,WORK(N1+1),IBAND) 
           njac=njac+1 
           ijac=1
      else
           ijac=0
      endif


* By computing next step-size, hn=r*h

      r=min(tet(4),tet(1)*(toler/(hmin+est))**0.2d0)
      sum=(tf-t)/h !adjusting to reach the end-point
      if(sum.le.(1.2d0*r)) then
          r=sum
          hn=r*h
         goto 3
      endif

*If convergence failed or the estimator failed, do not increase hn

      if (nfailcon.gt.0 .or. nfailest.gt. 0) then
          r=min(1.d0,r)
      endif

      if (niter.gt.nmaxit(2)) then !restricting hn by low convergence
          r=min(rate,r)
          r=max(r,tet(8))
      endif


* saving LU
      if((tet(3).lt.r).and.(r.lt.tet(2)).and.(ijac.eq.0)) then 
           hn=h
      else 
           hn=r*h
      endif
      goto 3
      end
***************************************************************
***************************************************************
      SUBROUTINE STARTING(M,NSTEP,T0,H,R,Y0,YP0,F0,Y1,Y2,Y10,Y20,AUX,
     # INIC)
* This subroutine selects the best starting algorithm for the next
* step of size hn=r*h, the latest step-size accepted was h.

      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION AUX(4*M) !work vector, h*yp0=AUX(1:M), 	
      DIMENSION y0(M),yp0(M),f0(M),y1(M),y2(M),Y10(M),Y20(M)

      parameter(C1=0.21132486540518712D0,C2=0.78867513459481288D0)

      DATA xkappa/0.5D0/XMU/0.2d0/cte/0.1d0/ !cte=xmu*xkappa

* inic says what predictor is to be chosen

      inic=0

*by computing H*YP0() 
      do k=1,m
           AUX(k)=h*yp0(k)
      enddo

* By selecting the predictor for the first step

      if (nstep.eq.0) then
          e1=0.d0
          e2=0.d0
          c2h2=0.5d0*(h*c2)**2
         do k=1,m
             e1=e1+(c2*aux(k))**2
             e2=e2+(c2h2*f0(k))**2
         enddo

         e1=Sqrt(e1)
         e2=sqrt(e2)
         if (e2.ge.(xkappa*e1)) then
                do k=1,m
                   y10(k)=y0(k)
                   y20(k)=y0(k)
                enddo
         elseif(e2.ge.(cte*e1)) then
                do k=1,m
                   y10(k)=y0(k)+c1*aux(k)
                   y20(k)=y0(k)+c2*aux(k)
                enddo
         else
                c1h2=0.5d0*(c1*h)**2
                do k=1,m
                    y10(k)=y0(k)+c1*aux(k)+c1h2*f0(k)
                    y20(k)=y0(k)+c2*aux(k)+c2h2*f0(k)
                enddo
         endif
         return
      endif
***********
* By selecting predictors for steps different from the first one

      call ord2(m,c1,y1,c2,y2,1.d0+c2*r,aux(m+1))
      call ord3(m,0.d0,y0,c1,y1,c2,y2,1.d0+c2*r,aux(2*m+1))
      call ord4(m,y0,AUX(1),y1,y2,r,c2,aux(3*m+1))

      e1=0.d0
      e2=0.d0
      e3=0.d0

      do k=1,m
          e1=e1+ (aux(m+k)-y2(k))**2
          e2=e2+ (aux(2*m+k)-aux(m+k))**2
          e3=e3+ (aux(3*m+k)-aux(2*m+k))**2
      enddo

      e1=sqrt(e1)
      e2=sqrt(e2)
      e3=sqrt(e3)

      if (e2.ge.(xkappa*e1)) then
            do k=1,m
                y10(k)=y2(k)
               y20(k)=y2(k)
            enddo
            inic=1

      elseif(e3.ge.(xkappa*e2)) then
           call ord2(m,c1,y1,c2,y2,1.d0+c1*r,y10)
           do k=1,m
                y20(k)=aux(m+k)
           enddo
           inic=2

      elseif(e3.ge.(cte*e2)) then
          call ord3(m,0.d0,y0,c1,y1,c2,y2,1.d0+r*c1,y10)
          do k=1,m
               y20(k)=aux(2*m+k)
          enddo
          inic=3
      else 
          call ord4(m,y0,aux(1),y1,y2,r,c1,y10)
          do k=1,m
              y20(k)=aux(3*m+k)
          enddo
          inic=4

      endif
      return
      end
*************************************************************
      subroutine ord2(m,t1,y1,t2,y2,t,y)
* This makes the two point Lagrange interpolation
      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION y1(m),y2(m),y(m)

      xl1=(t-t2)/(t1-t2)
      xl2=(t-t1)/(t2-t1)

      do k=1,m
          y(k)=xl1*y1(k)+xl2*y2(k)
      enddo
      return
      end
*************************************************************
      subroutine ord3(m,t0,y0,t1,y1,t2,y2,t,y)
* This makes the three point Lagrange interpolation
      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION y0(m),y1(m),y2(m),y(m)

      xl0=(t-t1)*(t-t2)/((t0-t1)*(t0-t2))
      xl1=(t-t0)*(t-t2)/((t1-t0)*(t1-t2))
      xl2=(t-t0)*(t-t1)/((t2-t0)*(t2-t1))

      do k=1,m
          y(k)=xl0*y0(k)+xl1*y1(k)+xl2*y2(k)
      enddo
      return
      end
*************************************************************
      subroutine ord4(m,y0,v0,y1,y2,r,t,y)
* For predictors of  order four 
      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION y0(m),v0(m),y1(m),y2(m),y(m)

      parameter(C1=0.21132486540518712D0,C2=0.78867513459481288D0)

      r3=Sqrt(3.d0)

      a1= -(1+r)*(-1+(-5+2*r3)*r +(-3+2*r3)*r**2)
      a2= (1+r)*(1+(5+2*r3)*r +(3+2*r3)*r**2)
      d1= -1.d0/6*r*(1+r)*(-3+r3 +(-3+2*r3)*r)
      d2= 1.d0/6*r*(1+r)*(3+r3 +(3+2*r3)*r)

      b11= 0.5d0*(1+r)*(-2*r3*(1+r) +r**2)
      b22= 0.5d0*(1+r)*(2*r3*(1+r) +r**2)

      b12=r3+(-6+4*r3)*r +(-8.5d0+5*r3)*r**2+(-3.5d0+2*r3)*r**3
      b21=-r3-(6+4*r3)*r -(8.5d0+5*r3)*r**2 - (3.5d0+2*r3)*r**3

      if (abs(c1-t).lt.1.d-8) then
          do k=1,m
              y(k)=a1*y0(k)+b11*y1(k)+b12*y2(k)+d1*v0(k)
          enddo

      elseif(abs(c2-t).lt.1.d-8) then
         do k=1,m
              y(k)=a2*y0(k)+b21*y1(k)+b22*y2(k)+d2*v0(k)
          enddo
      endif

      return
      end

*************************************************************
*************************************************************
      SUBROUTINE ESTIM(M,T,Y,YP,H,RG,U,IP,Y1,Y2,F,FH,AUX,EST,IBAND)

*this subroutine compute a local error estimator asymptotically correct
* for the Gauss-2 (2 order problems) at the point T+H

      IMPLICIT REAL*8 (A-H,O-Z)
      DIMENSION IP(M)
      DIMENSION Y(M),YP(M),IBAND(3)
      DIMENSION Y1(M),Y2(M),F(M),FH(M),U(M,M),AUX(2*M)

      PARAMETER(U1= -2.58564064605510183D0,U2= 0.18564064605510183D0)
      PARAMETER(W1=  3.23205080756887729D0,W2=-0.23205080756887729D0)
      PARAMETER(GAMMA= 0.833333333333333333333333D-1)

* FOR BANDED JACOBIANS
      IF(IBAND(1).EQ.1) THEN
          MLE=IBAND(2)
          MUE=IBAND(3)
          LDE1=2*MLE+MUE+1
      ENDIF

      CTE=1.D0/(3.D1*GAMMA*Rg**2)
      h2=(h**2)/3.d1
      CT=1.D0/(GAMMA*(RG*H)**2)

      DO K=1,m
          AA1=h*yp(k) !H*YP()
          AA2 = 2.4D0*Y(K)+ U1*Y1(K)+ U2*Y2(K)+ 0.4D0*AA1 !W()
          AUX(M+K)=-3.0D0*Y(K)+ W1*Y1(K)+ W2*Y2(K)- 0.5D0*AA1 !Wg()
          AUX(K)=AA2-CTE*AUX(M+K)+H2*(F(K)-FH(K)) !Y3()
      ENDDO

* SOLVING ONE LINEAR SYSTEM

      IF(IBAND(1).EQ.0) THEN
             CALL SOL (M, M, U, AUX(1), IP)
      ELSE
             CALL SOLB(M,LDE1,U,MLE,MUE,AUX(1),IP)
      ENDIF

      DO K=1,m
           AUX(K)=CTE*AUX(M+K)+ CT*AUX(K) !Estimator E_1
       ENDDO

* SOLVING THE SECOND LINEAR SYSTEM

      IF(IBAND(1).EQ.0) THEN
           CALL SOL (M, M, U, AUX(1), IP)
      ELSE
           CALL SOLB(M,LDE1,U,MLE,MUE,AUX(1),IP)
      ENDIF

      EST=0.D0

      DO K=1,M
           EST=EST+ AUX(K)**2
      ENDDO
      EST=CT*SQRT(EST/M) ! Estimator E_2

      RETURN
      END
*************************************************************
      SUBROUTINE GAU2SN(M,T,Y,YP,H,U,IP,Y1,Y2,YH,YPH,
     $   LS,IDER,IER,NITER,TETA,ATOL,RTOL,RATE,ILIN,AUX,CTE,IBAND)

* THIS ROUTINE SOLVES THE STAGE VALUES FOR THE 2-STAGE GAUSS METHOD
* INPUT VARIABLES:
*       M: dimension of IVP
*       T: current point. It is intended to advance from T to T+H 
*       Y(M)=Y(T),YP(M)=Y'(T)
*       H: stepsize to be tried
*       U(MD,M): matrix LU-factored.
*       IP(M): vector of index for pivoting (for solving linear systems)
*       Y1(M),Y2(M): predictors for the internal stages
*       ATOL,RTOL: absolute and relative tolerances respectively
*        ILIN=0,1: index  for  nonlinear or linear problems respectively
*       AUX(7*M): vector to store some arrays in the computations 
*       CTE: is an specific constant depending on some step-size ratios
*       IBAND(3): Array which indicates if the Jacobian is full or banded
* 
* OUTPUT VARIABLES:
*       Y1(M),Y2(M):  stage values computed
*       YH(M)= Y(T+H)
*       YPH(M)=Y'(T+H):
*       LS: number of linear systems  solved
*       IDER:  number of evaluations of function F
*       IER:  error flag (IER = -1 means divergence, IER = 0 converg)
*       NITER: number of iterations carried out
*       TETA: gives the  step-size ratio. Thus, TETA*H is the 
*                    next step-size to be tried, 
*                   whenever divergence for the step-size H is detected 
*                   RATE: says how much the step-size should be reduced 
*                   to reach convergence in 4 iterations 


      IMPLICIT REAL*8 (A-H,O-Z)

      DIMENSION NMAXIT(3),IP(M),IBAND(3)
      DIMENSION Y(M),YP(M),Y1(M),Y2(M),YH(M),YPH(M),AUX(7*M)
      DIMENSION U(M,M),TET(8),Q1(2,2)

*  2-stage Gauss coefficients for AG=A.A

      PARAMETER (AG11= 0.41666666666666667D-1,
     $   AG12=-0.19337567297406441D-1,
     $   AG21= 0.26933756729740644D0,
     $   AG22= 0.41666666666666667D-1,
     $   C1  = 0.21132486540518712D0,
     $   C2  = 0.78867513459481288D0)

* constants of the  Single-Newton  iteration

      PARAMETER (VEC11=-1.73205080756887729D0,
     $   VEC12= 1.73205080756887729D0,
     $   VEC21=-1.63923048454132638D1,
     $   VEC22= 4.39230484541326376D0)

      parameter (GAMMA= 0.8333333333333333333333333D-1)
      parameter (p12=0.071796769724490826d0) !p11=1, s12=-p12
      parameter (p21=-4.0207259421636902d0) !l21=-p21
      parameter (p22=0.71132486540518712d0)
      PARAMETER (BET=1.300111070804447842444981d0)
      parameter(q11=0.5773502691896257645091487805d0) !Q=A^2.T^{-1}-I
      parameter(q12=-0.154700538379251529018297561004d0)
      parameter(q21=2.154700538379251529018297561d0)
      parameter(q22=-0.5773502691896257645091487805d0)


      DATA TET/0.8D0,1.5D0,0.9D0,2.D0,1.D-2,6.D-1,7.D-1,2.D-1/
      DATA NMAXIT/10,6,4/

* some constants needed for the iteration

      PS11=CTE !CTE=1/(GAMMA*HG^2)
      PS21=CTE*P21
      PS12=CTE*P12
      PS22=CTE*P22
      q1(1,1)= bet*q11
      q1(1,2)= bet*q12
      q1(2,1)= bet*q21
      q1(2,2)= bet*q22
      beta=(1.d0-bet)/bet
***************
      M2=2*M
      M3=3*M
      M4=4*M
      M5=5*M
      M6=6*M

* we restart the counters IER,LS,IDER and TETA
      IER = 0
      LS = 0
      IDER=0
      teta=0 !teta=max_k |\Delta Y^{k}|/|\Delta Y^{k-1}|, k=1,NMAXIT(1)
      rate=2 !Adjust coming step-size to reach conv. in NMAXIT(3) iterates

      t1=t+c1*h
      t2=t+c2*h
      h2=h**2
      h11=h2*ag11
      h12=h2*ag12
      h21=h2*ag21
      h22=h2*ag22

* By computing the tolerance TOLSN to stop the iterations
      tolsn=0.d0
      do k=1,m
         tolsn = y(k)**2+tolsn
         AUX(k)=h*yp(k)
      enddo
      tolsn=sqrt(tolsn/m)
      tolsn=tet(5)*(atol+rtol*tolsn)+1.d-15

* FOR BANDED JACOBIANS
      IF(IBAND(1).EQ.1) THEN
          MLE=IBAND(2)
          MUE=IBAND(3)
          LDE1=2*MLE+MUE+1
      ENDIF

**************************
* Loop for the iterations 
**************************

      DO  NITER = 1,NMAXIT(1)

           IF ((ILIN.EQ.0).OR.(NITER.EQ.1)) THEN !The Largest IF
               call FNC(M,T1,Y1,AUX(M+1))
               call FNC(M,T2,Y2,AUX(M2+1))
               IDER = IDER + 2

             DO  K =1,M
             AUX(M3+K)= Y(K)+C1*AUX(K)-Y1(K)+H11*AUX(M+K)+H12*AUX(M2+K)
             AUX(M4+K)= Y(K)+C2*AUX(K)-Y2(K)+H21*AUX(M+K)+H22*AUX(M2+K)
             AUX(M5+K)= PS11*AUX(M3+K)+PS12*AUX(M4+K)
             enddo
* Solving the first linear system
             IF(IBAND(1).EQ.0) THEN
                 CALL SOL (M, M, U, AUX(M5+1), IP)
             ELSE
                 CALL SOLB(M,LDE1,U,MLE,MUE,AUX(M5+1),IP)
             ENDIF
             LS = LS + 1

* Solving the second linear system
             DO  K=1,M
               AUX(M6+K)=-PS21*(AUX(M5+K)-AUX(M3+K))+PS22*AUX(M4+K)
             ENDDO

             IF(IBAND(1).EQ.0) THEN
                CALL SOL (M, M, U, AUX(M6+1), IP)
             ELSE
                 CALL SOLB(M,LDE1,U,MLE,MUE,AUX(M6+1),IP)
             ENDIF
             LS = LS + 1

* Computing the true values for INCRE = (S*I)INCRE
             DO  K=1,M
                 AUX(M5+K)= AUX(M5+K)-P12*AUX(M6+K)
             ENDDO

* SPEEDING UP CONVERGENCE AT THE FOURTH ITERATION
           if (niter.eq.4) then
              do k=1,m
                  AUX(M5+K)=BET*AUX(M5+K)
                  AUX(M6+K)=BET*AUX(M6+K)
               enddo
           endif

* updating the internal stages and computing the INCRE-norm
           ynorm=0.d0
          DO K = 1,M
              Y1(K) = Y1(K) + AUX(M5+K)
              Y2(K) = Y2(K) + AUX(M6+K)
              ynorm=ynorm+ AUX(M5+k)**2+AUX(M6+k)**2
           ENDDO
           ynorm=sqrt(ynorm/(2*m))

           If (ynorm.lt.tolsn) then 
              DO K=1,M
                 YPH(K)=(AUX(K)+12.D0*Y(K)+VEC21*Y1(K)+VEC22*Y2(K))/H
                 YH(K)=Y(K)+VEC11*Y1(K)+VEC12*Y2(K)
              ENDDO
              RETURN
           endif

* By computing teta
           if (niter.eq.1) then
            razon=max(tet(6),(tet(1)*tolsn/ynorm)**(1.d0/(nmaxit(1)-1)))
           else
             teta=ynorm/ynorma
               if (teta.gt.razon) then
                   teta=max(tet(7)*sqrt(razon/teta),tet(8))
                   ier=-1
                   return
               endif
           endif

* Te reach convergence in 4 iterates at most for the next step
*          rate= h'/h= (0.9*tolsn/|Y^{5}(h)-Y^{4}(h)|)^(1/6)

           if ((niter-1).eq.nmaxit(3)) then
              rate=(tet(3)*tolsn/ynorm)**(1.d0/(2*nmaxit(3)-2))
           endif
           ynorma=ynorm

        ELSE !This is only for the linear case (ILIN=1) and niter > 1 
             do k=1,m !computing F(k)=Y(k)-Y(k-1)-G(k-1)
                 AUX(M+K)=AUX(M5+K)-AUX(M3+K)
                 AUX(M2+K)=AUX(M6+K)-AUX(M4+K)
             enddo

              if ((niter.lt.4).or.(niter.gt.5)) then !updating the residual
                 do k=1,m
                     AUX(M3+k)=q11*AUX(M+k)+q12*AUX(M2+k)
                     AUX(M4+k)=q21*AUX(M+k)+q22*AUX(M2+k)
                  enddo
              elseif (niter.eq.4) then
                     do k=1,m
                        AUX(M3+k)=q1(1,1)*AUX(M+k)+q1(1,2)*AUX(M2+k)
                        AUX(M4+k)=q1(2,1)*AUX(M+k)+q1(2,2)*AUX(M2+k)
                     enddo
               elseif (niter.eq.5) then
                  do k=1,m
                    AUX(M3+k)=q11*AUX(M+k)+q12*AUX(M2+k)+BETA*AUX(M3+K)
                    AUX(M4+k)=q21*AUX(M+k)+q22*AUX(M2+k)+BETA*AUX(M4+K)
                  enddo
                endif
* By solving the linear systems
             do k=1,m
                AUX(M5+k)= PS11*AUX(M3+k)+PS12*AUX(M4+k)
              enddo
              IF(IBAND(1).EQ.0) THEN
                  CALL SOL (M, M, U, AUX(M5+1), IP)
              ELSE
                 CALL SOLB(M,LDE1,U,MLE,MUE,AUX(M5+1),IP)
              ENDIF
              LS = LS + 1

              DO  K=1,M
               AUX(M6+K) = -PS21*(AUX(M5+K)- AUX(M3+K))+ PS22*AUX(M4+K)
              enddo

               IF(IBAND(1).EQ.0) THEN
                   CALL SOL (M, M, U, AUX(M6+1), IP)
               ELSE
                  CALL SOLB(M,LDE1,U,MLE,MUE,AUX(M6+1),IP)
               ENDIF
               LS = LS + 1       

* Computing the true values for INCRE=(S*I)INCRE
               DO  K=1,M
                  AUX(M5+K) = AUX(M5+K) -P12*AUX(M6+K)
               enddo

* updating the internal stages and computing the INCRE-norm
               ynorm=0.d0
                  DO K = 1,M
                       Y1(K) = Y1(K) + AUX(M5+K)
                      Y2(K) = Y2(K) + AUX(M6+K)
                       ynorm=ynorm+ AUX(M5+k)**2+AUX(M6+k)**2
                   ENDDO
               ynorm=sqrt(ynorm/(2*m))

* By checking convergence
               if (ynorm.lt.tolsn) then 
                  DO K=1,M
                    YPH(K)=(AUX(K)+12.D0*Y(K)+VEC21*Y1(K)+VEC22*Y2(K))/H
                    YH(K)=Y(K)+VEC11*Y1(K)+VEC12*Y2(K)
                   ENDDO
                   RETURN
                endif

* Reduction next stepsize h=rate*h, to reach convergence in 4 iterates
                if ((niter-1).eq.nmaxit(3)) then
                    rate=(tet(3)*tolsn/ynorm)**(1.d0/(2*nmaxit(3)-2))
                endif
* By computing teta to detect divergence
                teta=ynorm/ynorma
                if (teta.gt.razon) then
                  teta=max(tet(7)*sqrt(razon/teta),tet(8))
                   ier=-1
                   return
                endif
               YNORMA=YNORM
        ENDIF !End of Large IF
      ENDDO !Loop for the iterates
*******
      ier=-1
      niter=nmaxit(1)
      return
      end
**************************************************************
**************************************************************

* ROUTINES FOR LU-FACTORIZATIONS AND FORWARD-BACKWARD 
* SOLUTION OF LINEAR SYSTEMS

      SUBROUTINE DEC (N, NDIM, A, IP, IER)
C VERSION REAL DOUBLE PRECISION
      INTEGER N,NDIM,IP,IER,NM1,K,KP1,M,I,J
      DOUBLE PRECISION A,T
      DIMENSION A(NDIM,N), IP(N)
C-----------------------------------------------------------------------
C  MATRIX TRIANGULARIZATION BY GAUSSIAN ELIMINATION.
C  INPUT..
C     N = ORDER OF MATRIX.
C     NDIM = DECLARED DIMENSION OF ARRAY  A .
C     A = MATRIX TO BE TRIANGULARIZED.
C  OUTPUT..
C     A(I,J), I.LE.J = UPPER TRIANGULAR FACTOR, U .
C     A(I,J), I.GT.J = MULTIPLIERS = LOWER TRIANGULAR FACTOR, I - L.
C     IP(K), K.LT.N = INDEX OF K-TH PIVOT ROW.
C     IP(N) = (-1)**(NUMBER OF INTERCHANGES) OR O .
C     IER = 0 IF MATRIX A IS NONSINGULAR, OR K IF FOUND TO BE
C           SINGULAR AT STAGE K.
C  USE  SOL  TO OBTAIN SOLUTION OF LINEAR SYSTEM.
C  DETERM(A) = IP(N)*A(1,1)*A(2,2)*...*A(N,N).
C  IF IP(N)=O, A IS SINGULAR, SOL WILL DIVIDE BY ZERO.
C
C  REFERENCE..
C     C. B. MOLER, ALGORITHM 423, LINEAR EQUATION SOLVER,
C     C.A.C.M. 15 (1972), P. 274.
C-----------------------------------------------------------------------
      IER = 0
      IP(N) = 1
      IF (N .EQ. 1) GO TO 70
      NM1 = N - 1
      DO 60 K = 1,NM1
        KP1 = K + 1
        M = K
        DO 10 I = KP1,N
          IF (DABS(A(I,K)) .GT. DABS(A(M,K))) M = I  
 10     CONTINUE
        IP(K) = M
        T = A(M,K)
        IF (M .EQ. K) GO TO 20
        IP(N) = -IP(N)
        A(M,K) = A(K,K)
        A(K,K) = T
 20     CONTINUE
        IF (T .EQ. 0.D0) GO TO 80
        T = 1.D0/T
        DO 30 I = KP1,N
 30       A(I,K) = -A(I,K)*T
        DO 50 J = KP1,N
          T = A(M,J)
          A(M,J) = A(K,J)
          A(K,J) = T
          IF (T .EQ. 0.D0) GO TO 45
          DO 40 I = KP1,N
 40         A(I,J) = A(I,J) + A(I,K)*T
 45       CONTINUE
 50       CONTINUE
 60     CONTINUE
 70   K = N
      IF (A(N,N) .EQ. 0.D0) GO TO 80
      RETURN
 80   IER = K
      IP(N) = 0
      RETURN
C----------------------- END OF SUBROUTINE DEC -------------------------
      END
C
C
      SUBROUTINE SOL (N, NDIM, A, B, IP)
C VERSION REAL DOUBLE PRECISION
      INTEGER N,NDIM,IP,NM1,K,KP1,M,I,KB,KM1
      DOUBLE PRECISION A,B,T
      DIMENSION A(NDIM,N), B(N), IP(N)
C-----------------------------------------------------------------------
C  SOLUTION OF LINEAR SYSTEM, A*X = B .
C  INPUT..
C    N = ORDER OF MATRIX.
C    NDIM = DECLARED DIMENSION OF ARRAY  A .
C    A = TRIANGULARIZED MATRIX OBTAINED FROM DEC.
C    B = RIGHT HAND SIDE VECTOR.
C    IP = PIVOT VECTOR OBTAINED FROM DEC.
C  DO NOT USE IF DEC HAS SET IER .NE. 0.
C  OUTPUT..
C    B = SOLUTION VECTOR, X .
C-----------------------------------------------------------------------
      IF (N .EQ. 1) GO TO 50
      NM1 = N - 1
      DO 20 K = 1,NM1
        KP1 = K + 1
        M = IP(K)
        T = B(M)
        B(M) = B(K)
        B(K) = T
        DO 10 I = KP1,N
 10       B(I) = B(I) + A(I,K)*T
 20     CONTINUE
      DO 40 KB = 1,NM1
        KM1 = N - KB
        K = KM1 + 1
        B(K) = B(K)/A(K,K)
        T = -B(K)
        DO 30 I = 1,KM1
 30       B(I) = B(I) + A(I,K)*T
 40     CONTINUE
 50   B(1) = B(1)/A(1,1)
      RETURN
C----------------------- END OF SUBROUTINE SOL -------------------------
      END


* LU-FACTORIZATION FOR BANDED MATRICES

C
      SUBROUTINE DECB (N, NDIM, A, ML, MU, IP, IER)
      REAL*8 A,T
      DIMENSION A(NDIM,N), IP(N)

C-----------------------------------------------------------------------
C  MATRIX TRIANGULARIZATION BY GAUSSIAN ELIMINATION OF A BANDED
C  MATRIX WITH LOWER BANDWIDTH ML AND UPPER BANDWIDTH MU
C  INPUT..
C     N       ORDER OF THE ORIGINAL MATRIX A.
C     NDIM    DECLARED DIMENSION OF ARRAY  A.
C     A       CONTAINS THE MATRIX IN BAND STORAGE.   THE COLUMNS  
C                OF THE MATRIX ARE STORED IN THE COLUMNS OF  A  AND
C                THE DIAGONALS OF THE MATRIX ARE STORED IN ROWS 
C                ML+1 THROUGH 2*ML+MU+1 OF  A.
C     ML      LOWER BANDWIDTH OF A (DIAGONAL IS NOT COUNTED).
C     MU      UPPER BANDWIDTH OF A (DIAGONAL IS NOT COUNTED).
C  OUTPUT..
C     A       AN UPPER TRIANGULAR MATRIX IN BAND STORAGE AND 
C                THE MULTIPLIERS WHICH WERE USED TO OBTAIN IT.  
C     IP      INDEX VECTOR OF PIVOT INDICES.
C     IP(N)   (-1)**(NUMBER OF INTERCHANGES) OR O .
C     IER     = 0 IF MATRIX A IS NONSINGULAR, OR  = K IF FOUND TO BE
C                SINGULAR AT STAGE K.
C  USE  SOLB  TO OBTAIN SOLUTION OF LINEAR SYSTEM.
C  DETERM(A) = IP(N)*A(MD,1)*A(MD,2)*...*A(MD,N)  WITH MD=ML+MU+1.
C  IF IP(N)=O, A IS SINGULAR, SOLB WILL DIVIDE BY ZERO.
C
C  REFERENCE..
C     THIS IS A MODIFICATION (TAKING FROM THE E. HAIRER HOMEPAGE) OF
C     C. B. MOLER, ALGORITHM 423, LINEAR EQUATION SOLVER,
C     C.A.C.M. 15 (1972), P. 274.
C-----------------------------------------------------------------------
      IER = 0
      IP(N) = 1 
      MD = ML + MU + 1
      MD1 = MD + 1
      JU = 0
      IF (ML .EQ. 0) GO TO 70
      IF (N .EQ. 1) GO TO 70
      IF (N .LT. MU+2) GO TO 7
      DO 5 J = MU+2,N
      DO 5 I = 1,ML
  5   A(I,J) = 0.D0
  7   NM1 = N - 1
      DO 60 K = 1,NM1
        KP1 = K + 1
        M = MD
        MDL = MIN(ML,N-K) + MD
        DO 10 I = MD1,MDL
          IF (DABS(A(I,K)) .GT. DABS(A(M,K))) M = I
 10     CONTINUE
        IP(K) = M + K - MD
        T = A(M,K)
        IF (M .EQ. MD) GO TO 20
        IP(N) = -IP(N)
        A(M,K) = A(MD,K)
        A(MD,K) = T
 20     CONTINUE
        IF (T .EQ. 0.D0) GO TO 80
        T = 1.D0/T
        DO 30 I = MD1,MDL
 30       A(I,K) = -A(I,K)*T 
        JU = MIN0(MAX0(JU,MU+IP(K)),N)
        MM = MD
        IF (JU .LT. KP1) GO TO 55
        DO 50 J = KP1,JU
          M = M - 1
          MM = MM - 1
          T = A(M,J) 
          IF (M .EQ. MM) GO TO 35
          A(M,J) = A(MM,J)
          A(MM,J) = T
 35       CONTINUE
          IF (T .EQ. 0.D0) GO TO 45
          JK = J - K
          DO 40 I = MD1,MDL
            IJK = I - JK
 40         A(IJK,J) = A(IJK,J) + A(I,K)*T
 45       CONTINUE
 50       CONTINUE
 55     CONTINUE
 60     CONTINUE
 70   K = N
      IF (A(MD,N) .EQ. 0.D0) GO TO 80
      RETURN
 80   IER = K
      IP(N) = 0
      RETURN
C----------------------- END OF SUBROUTINE DECB ------------------------
      END
C
C
      SUBROUTINE SOLB (N, NDIM, A, ML, MU, B, IP)
      REAL*8 A,B,T
      DIMENSION A(NDIM,N), B(N), IP(N)
C-----------------------------------------------------------------------
C  SOLUTION OF LINEAR SYSTEM, A*X = B .
C  INPUT..
C    N      ORDER OF MATRIX A.
C    NDIM   DECLARED DIMENSION OF ARRAY  A .
C    A      TRIANGULARIZED MATRIX OBTAINED FROM DECB.
C    ML     LOWER BANDWIDTH OF A (DIAGONAL IS NOT COUNTED).
C    MU     UPPER BANDWIDTH OF A (DIAGONAL IS NOT COUNTED).
C    B      RIGHT HAND SIDE VECTOR.
C    IP     PIVOT VECTOR OBTAINED FROM DECB.
C  DO NOT USE IF DECB HAS SET IER .NE. 0.
C  OUTPUT..
C    B      SOLUTION VECTOR, X .
C-----------------------------------------------------------------------
      MD = ML + MU + 1
      MD1 = MD + 1
      MDM = MD - 1
      NM1 = N - 1
      IF (ML .EQ. 0) GO TO 25
      IF (N .EQ. 1) GO TO 50
      DO 20 K = 1,NM1
        M = IP(K)
        T = B(M)
        B(M) = B(K)
        B(K) = T
        MDL = MIN(ML,N-K) + MD
        DO 10 I = MD1,MDL
          IMD = I + K - MD
 10       B(IMD) = B(IMD) + A(I,K)*T
 20     CONTINUE
 25   CONTINUE
      DO 40 KB = 1,NM1
        K = N + 1 - KB
        B(K) = B(K)/A(MD,K)
        T = -B(K) 
        KMD = MD - K
        LM = MAX0(1,KMD+1)
        DO 30 I = LM,MDM
          IMD = I - KMD
 30       B(IMD) = B(IMD) + A(I,K)*T
 40     CONTINUE
 50   B(1) = B(1)/A(MD,1)
      RETURN
C----------------------- END OF SUBROUTINE SOLB ------------------------
      END
