(*:Name: `StabInt` *)

(*:Title:    Computation of the intervals of stability *)

(*:Authors:  Massimo Cafaro, 
             Facolta' di Ingegneria,
             Universita' degli Studi di Lecce, Italy
             E-mail: cafaro@sara.unile.it   

             Beatrice Paternoster,  
             Dipartimento di Informatica e Applicazioni,
             Universita' degli Studi di Salerno, Italy
             E-mail: beapat@dia.unisa.it    *)
             
(*:Keywords: stability *)

(*:Requirements: none *)

(*:Mathematica version: v3.0 *)

(*:Warnings: filenames can't contain spaces*)

(*:Sources: Analysis of stability of rational approximations
            through computer algebra, 
            M.Cafaro, B.Paternoster, to appear in
            Proceedings of the Second Workshop on Computer Algebra
            in Scientific Computing, May 31 - June 4, 1999,
            Munich, Germany, Lecture Notes in Computational Science and
            Engineering, Springer *)

BeginPackage["`StabInt`"]



StabIntervals::usage="StabIntervals[filename] return the stability
intervals of the rational functions rf saved in the file filename;" 

prec::usage="prec is the number of decimal digits (precision) to be used;"
 
pairs::usage="pairs is the number of (rational function, constant) pairs;"

rf::usage="rf is an array containing the rational functions to be examined;"

cv::usage="cv is an array containing the constants values;"

eps::usage="eps is the precision value to be used in computing if two zeros
are to be considered the same;"

z::usage="z is the variable used to solve equations;"
      
intlist::usage="intlist is a list of intervals of stability;"

printing::usage="printing = 1 enable plotting the intervals, printing = 0
disable plotting the intervals;"


height::usage="height is the value of the y axe to be reached when plotting;"


stepx::usage="stepx is the step along the x axe to be used when plotting;
an higher value
will produce a faster but less accurate plot, a lower value
will produce a slower but more accurate plot"


stepy::usage="stepy is the step along the y axe to be used when plotting;
an higher value
will produce a faster but less accurate plot, a lower value
will produce a slower but more accurate plot"


stability::usage="stability represents the intervals of stability for the
input rational function;"
                  

Begin["`Private`"]


(* ReadRationalFunction reads the rational functions and the other input
parameters from the file filename *)

ReadRationalFunction[filename_]:=Module[{},
	Get[ToString[filename]]];

(* BuildExpr build the expressions needed to solve Abs[rf[i]] < cv[i] *)

BuildExpr[f_,c_]:=Module[{},
         For[i=0,i<pairs,i++,
         If[c[i+1]!=0,
         exp[1+2*i]=Together[f[i+1]+c[i+1]];
         exp[2+2*i]=Together[c[i+1]-f[i+1]],
         exp[1+2*i]=Together[c[i+1]-f[i+1]];
         exp[2+2*i]=Together[c[i+1]-f[i+1]]]]];
         
         
         


         
(* MakeRoots returns a root list in wich all elements are positive and
distinct *)

MakeRoots[eqn_]:=Module[{zero,root,t,u},
         zero=Flatten[z /. NSolve[Numerator[eqn]==0,z,prec]];
         If[!NumberQ[Denominator[eqn]],
         root=Flatten[z /. NSolve[Denominator[eqn]==0,z,prec]],
         root=List[]];
         t=N[Select[N[Map[Chop,N[Union[zero,root],prec]],prec]
,Positive],prec];
         u=N[RotateLeft[t],prec];
         If[u!=t,rl=N[Sort[Delete[u,Position[Abs[#-#2]& @ Sequence[u,t],k_
/; k <eps]]],prec],
         If[Length[u]>=1,rl=N[List[First[u]],prec],rl=N[u,prec]]]];





(* Bound computes the intervals of the positive real axis, in which
   |rf[i]| < cv[i], using the list of positive roots
   constructed by MakeRoots. If this list is empty, Bound decides
   if the interval of stability is empty or not limited *)

Bound[expr_,roots_List,int_]:=Module[{},

If[Length[roots]==0,EmptyList[expr,int],NotEmptyList[expr,roots,int]]];
         
         EmptyList[ex_,int_]:=Module[{},
         Clear[z];
         If[Sign[N[ex /. {z->10}]] == 1 && OddQ[int] == True,
         intlist1=Append[intlist1,Interval[{0,Infinity}]],
         intlist1=Append[intlist1,Interval[{0,0}]]];
         If[Sign[N[ex /. {z->10}]] == 1 && OddQ[int] == False,
         intlist2=Append[intlist2,Interval[{0,Infinity}]],
         intlist2=Append[intlist2,Interval[{0,0}]]]];
   
        

NotEmptyList[ex_,roots_List,int_]:=Module[{p,t,u,points,values,intervals},
         t=N[Prepend[roots,0],prec];
         u=N[RotateLeft[t],prec];
         points=N[Drop[t+u,-1] / 2,prec];
         values=N[Table[Clear[z];
         N[ex /. z->points[[i]],prec] ,{i,Length[points]}],prec];
         p=Position[Map[Sign,values],1];
         intervals=List[];
         For[i=1,i<=Length[p],i++,intervals=Append[intervals,
         N[Interval[{t[[p[[i,1]]]],t[[p[[i,1]]+1]]}],prec]]];
         If[Sign[N[ex /. z->Last[t]+10,prec]] == 1,
         Clear[z];intervals=Append[intervals,Interval[{Last[t],Infinity}]]];
         If[Length[intervals]==0,intervals={Interval[{0,0}]}];
         If[OddQ[int] == True,intlist1=intervals,intlist2=intervals]];
         
         
IntersectIntervals[a_,b_]:=Module[{l1,l2,f,s,},
	l1=Length[a];
	l2=Length[b];
	result=List[];
	Do[r=IntervalIntersection[a[[i]],b[[j]]];
	f=Min[r];s=Max[r];
	If[MemberQ[result,r]==False && f!= s && f!=Infinity && s!= -Infinity,
	result=Append[result,r]]  ,{i,l1},{j,l2}];
	result=Sort[result]];
         

(* StabIntervals determines the intervals of stability 
	of the rational function, *)

StabIntervals[filename_]:=Module[{x,k,stability},
         ReadRationalFunction[filename];
         BuildExpr[rf,cv];
	 For[x = 1, x < pairs, x++,intcomp[x]=List[]];
         For[k = 1, k < 2*pairs, k+=2,
         intlist1=List[];
         intlist2=List[];
         Do[MakeRoots[exp[i]];
         Bound[exp[i],rl,i],{i,k,k+1}];
         IntersectIntervals[intlist1,intlist2];
         intcomp[Floor[k/2]+1]=result];
         temp=intcomp[1];
         cvz = 1;
         For[x = 1, x <= pairs, x++,
         If[cv[x]==0, cvz=0]];
         For[x = 1, x < pairs, x++,
         IntersectIntervals[temp,intcomp[x+1]];
         temp=result];
         stability=temp;
         Print[StringForm["Intervals of stability = ``",stability]];
         If[printing == 1 && Length[stability] != 0 && cvz==1,
         points=List[];
         If[Last[stability][[1,2]]!=0 && Last[stability][[1,2]]!=Infinity,
         For[x=0,x<Last[stability][[1,2]],x+=stepx,
         For[y=0,y < height, y+=stepy,
         k=x+ I y; 
         include=True;
         Do[Clear[z];If[(Abs[rf[i]] /. {z->k}) < cv[i],
         include=include && True, include=include && False],{i,pairs}];
         If[include == True,points=Append[points,{x,y}]]]];
         ListPlot[points]]];
         If[printing == 1 && (Length[stability] == 0 || cvz==0),
         Print["Sorry, the package cannot do the plot, either because there
are no intervals"];
         Print["or at least one of the constant values associated to your
rational functions"];
         Print["is zero"]];

ClearAll[prec,pairs,eps,rf,cv,rl,stability,intlist1,intlist2,printing,height];
         ClearAll[stepx,stepy,temp,result,points,cvz,include,z,x,k]];

End[]

Protect[StabIntervals]

EndPackage[]



