/*
   This is a Java test class that demonstrates the functionality of
   the ErkTriple routine of the odeToJava package.  It also serves as
   a help file to show you how to use the odeToJava package in
   general, as it includes most of the modules and implementations,
   and several ways to call the ErkTriple routine.  To test this
   routine, we will be solving several problems, including some from
   the CWI testset.  All are good examples that test the functionality
   of the ErkTriple routine and show you how to use different routines
   and subroutines in this package.  Once you get acquianted with the
   ways to call the ErkTriple routine, it will be easy to use the
   other five solvers as they are all called in a very similar fashion
   to that of the ErkTriple routine.  Also take notice of the two
   other java files in this directory.  These two files are functions
   that are called in this program.  These function files can teach
   you how to properly implement your own functions.  PlFTest.java is
   actually the Pleiades problem renamed and implemented outside the
   package, and Poly.java is a 4th-order polynomial ODE.  Here is a
   heavily commented walkthrough of the usage of the major components
   of this package. Enjoy!  
*/

/*
   Here we import all of the package components required to run the
   routine and solve the desired ODEs
*/

import odeToJava.*;   // import for use of the ErkTriple routine
import odeToJava.modules.*;   // import for use of the function module (Fv),
   // the Butcher tableau module (Btableau), the interpolant module (Btheta)
   // and the span module (Span)
import odeToJava.functions.*;   // import for use of some ODE functions
   // included in this package
import odeToJava.plotter.*;   // import to use the plotter to plot the result
   // of an integration
import odeToJava.interpolants.*; // import to use interpolants (the
   // interpolants folder contains a pre-defined interpolant to for
   // the Dormand-Prince scheme)

/*
   class containing the main method that tests out the ErkTriple
   routine, and several other modules of the odeToJava package
*/
public class ErkTripleTest
{
     /*
        This is the main method, a java method that has to be called
        when any program is run; it is here that we will call routines
        in the odeToJava package 
     */
     public static void main(String[] args)
     {

          // the array of initial values

          /*
             Since the Pleiades problem (and all the ODEs this package
             deals with) are initial value problems (IVPs), we need an
             initial value.  The Pleiades problem is an IVP of
             dimension 28, so the initial value will be an array of
             size 28 (i.e., 28 initial values).  Here we intialize an
             array that will contain the 28 initial values.  All
             solvers have this in common, because all solvers solve
             IVPs 
           */

          double[] x = new double[28];   // initialize array object

          x[0] = 3.0;   // fill the array with the 28 initial values
          x[1] = 3.0;
          x[2] = -1.0;
          x[3] = -3.0;
          x[4] = 2.0;
          x[5] = -2.0;
          x[6] = 2.0;

          x[7] = 3.0;
          x[8] = -3.0;
          x[9] = 2.0;
          x[10] = 0.0;
          x[11] = 0.0;
          x[12] = -4.0;
          x[13] = 4.0;

          x[14] = 0.0;
          x[15] = 0.0;
          x[16] = 0.0;
          x[17] = 0.0;
          x[18] = 0.0;
          x[19] = 1.75;
          x[20] = -1.5;

          x[21] = 0.0;
          x[22] = 0.0;
          x[23] = 0.0;
          x[24] = -1.25;
          x[25] = 1.0;
          x[26] = 0.0;
          x[27] = 0.0;

          // the Butcher tableau
          
          /*
             This is the Butcher tableau, a structure of arrays that
             define what Runge-Kutta scheme we are using to integrate
             the ODE.  The tableau below is the Butcher tableau for
             the Dormand-Prince scheme, an embedded method with the
             interpolant defined as well (it is an ERK triple).  There
             is an easier way to initialize Butcher tableaus for
             common schemes such as this one, which will be shown
             later, but for now we will explicitly initialize this
             Dormand-Prince Butcher tableau just to show you how to
             define any Butcher tableau in general 
           */

          // Butcher tableau component initializations

          double[][] a = new double[7][7]; // the matrix a (to
                                           // calculate the k's)
          double[] b = new double[7];   // the array b (the quadrature weights)
          double[] bEmb = new double[7];   // the array of embedded weights
          Btheta btheta;   // the interpolant
          double[] c = new double[7];   // the array c (the time abscissae)
         
          // assigning values to all of the components

          a[1][0] = (1.0)/(5.0);   // the A matrix of the Butcher tableau
      
          a[2][0] = (3.0)/(40.0);
          a[2][1] = (9.0)/(40.0);
		
          a[3][0] = (44.0)/(45.0);
          a[3][1] = (-56.0)/(15.0);
          a[3][2] = (32.0)/(9.0);
		
          a[4][0] = (19372.0)/(6561.0);
          a[4][1] = (-25360.0)/(2187.0);
          a[4][2] = (64448.0)/(6561.0);
          a[4][3] = (-212.0)/(729.0);
    
          a[5][0] = (9017.0)/(3168.0);
          a[5][1] = (-355.0)/(33.0);
          a[5][2] = (46732.0)/(5247.0);
          a[5][3] = (49.0)/(176.0);
          a[5][4] = (-5103.0)/(18656.0);
		
          a[6][0] = (35.0)/(384.0);
          a[6][1] = 0.0;
          a[6][2] = (500.0)/(1113.0);
          a[6][3] = (125.0)/(192.0);
          a[6][4] = (-2187.0)/(6784.0);
          a[6][5] = (11.0)/(84.0);
		
          b[0] = (35.0)/(384.0);   // b array
          b[1] = 0.0;
          b[2] = (500.0)/(1113.0);
          b[3] = (125.0)/(192.0);
          b[4] = (-2187.0)/(6784.0);
          b[5] = (11.0)/(84.0);
          b[6] = 0.0;

          bEmb[0] = (5179.0)/(57600.0);   // the embedded method
          bEmb[1] = 0.0;
          bEmb[2] = (7571.0)/(16695.0);
          bEmb[3] = (393.0)/(640.0);
          bEmb[4] = (-92097.0)/(339200.0);
          bEmb[5] = (187.0)/(2100.0);
          bEmb[6] = (1.0)/(40.0);

          btheta = new BthDopr(b); // use pre-defined btheta for Dormand-Prince
                              // interpolation (defined in interpolants folder)
		
          c[0] = 0.0;   // c array
          c[1] = (1.0)/(5.0);
          c[2] = (3.0)/(10.0);
          c[3] = (4.0)/(5.0);
          c[4] = (8.0)/(9.0);
          c[5] = 1.0;
          c[6] = 1.0;

          /*
             Note that all solvers take in a Butcher tableau (except
             for the DormandPrince solver, becuase it is hard coded
             into that solver) because all solvers need a scheme.  All
             solvers need a Butcher tableau with a and b defined;
             however bEmb, btheta, and c are not common to all.  bEmb
             is only for embedded schemes, btheta is only for schemes
             with an interpolant, and c is for schemes that are not in
             autonomous form.  

             Note the last parameter, the string.  This String can
             either be: "FSALenabled" or "FSALdisabled".  If the
             particular Butcher scheme has the "first same as last"
             property, you can have the solver take advantage of this
             property for efficiency.  However, if the scheme does not
             have the "first same as last" property, you must only use
             "FSALdisabled", otherwise an error occurs.  Note that
             only explicit solvers take advantage of "first same as
             last" Butcher schemes for efficiency.  There are also
             other parameters that are not dealt with here that can be
             found in other testers or in the Btableau module of the
             modules folder 
          */

          // initialize a Butcher tableau object with all of these components

          Btableau butcher = new Btableau(a, b, bEmb, btheta, c, "FSALenabled");

          // the tolerance arrays

          /*
             These are the tolerance arrays. These two tolerances
             determine how accurately the ODE will be solved.  These
             arrays are of dimension 28, as in this case the problem
             is of dimension 28.  They are all assigned the same value
             in this case, as a general tolerance is often good enough
             for the whole scheme. However, the arrays can be given all
             different values if you wish to have control over
             specific solution components.  Only solvers with stepsize
             control take in tolerances, and in some cases these
             tolerances are scalars, not arrays.  
          */

          double[] atol = new double[28];   // absolute tolerances all
          for(int i= 0; i< 28; i++)         // set to 1E-8
               atol[i] = 1.0E-8;

          double[] rtol = new double[28];   // relative tolerances all
          for(int i= 0; i< 28; i++)         // set to 1E-4
               rtol[i] = 1.0E-4;

          // run the ErkTriple scheme with the desired parameters

          /*
             Here we run an ErkTriple scheme with the parameters created
             above.  Here's an explanation of the parameters/arguments:

             1) new PlFtest() -> this is a PlFtest object (one that
             has to extend the ODE class). It is an ODE function that
             the routine takes in that it uses to solve the problem.
             All solvers have this in common. They all solve ODE
             problems, so they need a problem to solve (a function)

             2) new Span(0.0, 3.0) -> this is a Span object. It
             contains the lower and upper bounds of the solution.  All
             solvers take in a Span object, and those solvers with
             interpolation can also take in 2 types of enhanced Span
             objects.  The first type of enhanced Span objects is
             Span(double[] times), where the array of times in this
             Span object will be the times at which the solution will
             be output.  The second type of enhanced Span object is
             Span(double a, double b, double inc), where a and b are
             the first and last times as above, but inc is the
             increment after the first time that will be output.  For
             example, if you enter Span(0.0, 10.0, 1.0), the solution
             at the points 0, 1, 2 . . . 9, 10 would be
             output. Generally, the schemes should have
             interpolants. These enhanced Span objects can be used to
             initialize a scheme that does not have an interpolant,
             but the solution will not be output at the specified
             points.  Note that the only solvers that can have an
             interpolant are ErkTriple and DormandPrince

             3) x -> this is the array created above, the initial
             value to the IVP.  All solvers have this in common, as
             all solvers solve IVPs and, therefore, need an initial
             value

             4) 0.5 -> this is the initial stepsize that the code will
             attempt to take. If it is assigned a value less than or
             equal to 0, the initial stepsize is chosen via the
             initial stepsize selection routine.  Note that all
             solvers have this h argument in common, but for solvers
             with automatic stepsize control, the h is the initial
             stepsize, while for constant step solvers, h is the
             constant step (the stepsize that will be used throughout
             the integration)

             5) butcher -> this is the Butcher tableau created above,
             which defines the scheme.  Note again that all solvers
             have this argument in common but the Dormand-Prince
             scheme, where it is hard coded in

             6) 5.0 -> this is the highOrder argument, an argument
             that only this solver has.  This argument tells us that
             the higher order of the embedded scheme is
             5. (Dormand-Prince is a 5th-order/4th-order embedded
             scheme.) This is needed for error estimation and
             automatic stepsize control

             7) atol -> this is the absolute tolerance array created
             above.  Note that only solvers with automatic stepsize
             control take in this argument.  In some solvers, this
             argument is a scalar

             8) rtol -> this is the relative tolerance array created
             above.  Note that only solvers with automatic stepsize
             control take in this argument.  In some solvers, this
             argument is a scalar

             9) "" -> this is a placeholder for a filename for
             solution to be written to. In this case, the empty string
             indicates that no file is written to.  All solvers can
             take in a filename becuase all solvers can output their
             solutions to a file.  This string must be a filename or the
             empty string, else an error occurs

             10) "StiffDetect_Off" -> this string indicates that the
             stiffness detection feature of the solver is turned off.
             This string must either be: "StiffDetect_Halt" (solver
             halts if it detects stiffness in problem) or
             "StiffDetect_Off"; otherwise an error occurs.  Note that
             stiffness detection is only available for the
             Dormand-Prince scheme; only solvers implementing the
             Dormand-Prince Butcher tableau can do stiffness detection
             (e.g., ErkTriple and DormandPrince solvers)

             11) "EventLoc_Off" -> this string indicates that the
             event location feature of the solver is turned off.  This
             String must either be: "EventLoc_Halt" (solver halts if
             an event occurs in problem) or "EventLoc_Off"; otherwise
             an error occurs.  Note that only solvers with
             interpolants can do event location (e.g., ErkTriple and
             DormandPrince solvers)

             12) "Stats_On" -> this string means that statistics
             output is on verbose mode.  The other 2 options for this
             String are "Stats_Intermediate" (an intermediate amount
             of statisticss are output, depending on the solver) and
             "Stats_Off" (no statistics output, except after the
             solver completes the problem).  If this string is not one of
             the 3 mentioned above, an error is produced.  All the
             solvers can take in this String argument, in all its 3
             forms.  
          */

          ErkTriple.erk_triple(new PlFTest(), new Span(0.0, 3.0), x, 0.5, butcher, 5.0, atol, rtol, "", "StiffDetect_Off", "EventLoc_Off", "Stats_On");

          // run another ErkTriple scheme with similar parameters

          /*
             note the differences in the routine from the first call:
             
             1) The previous initial stepsize (0.5) has been replaced
             with -1.0.  We are calling the initial stepsize selection
             routine this time (by entering this parameter that is
             less than or equal to 0)

             2) Instead of "" we have put a filename to which the
             solution is written to.  Since the string is simply the
             filename and has no path, by default Java will write the
             file to the current directory.  You can, however, choose
             to put it in a different directory by specifying the
             entire path/filename.  Throughout these examples we will
             only be writing to the current directory.
             
             3) The stats option has been changed to
             "Stats_Intermediate", so that there will not be so many
             statistics output this time
          */

          ErkTriple.erk_triple(new PlFTest(), new Span(0.0, 3.0), x, -1.0, butcher, 5.0, atol, rtol, "erkTripleTestSoln.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_Intermediate");

          // run a third ErkTriple scheme with similar parameters

          /*
             This scheme is similar to the 2 above, but called
             a little more conveniently:

             1) Instead of using the PlFtest class we created with
             this project (which is the Pleiades problem of the CWI
             test set), we use the Pleiades class which is included in
             this package in the functions directory.  The functions
             directory includes several pre-defined problems that you
             may want to look at and test with the solvers.  However,
             non-stiff problems should be solved with non-stiff
             solvers, and stiff problems with stiff solvers.
             DormandPrince, ErkTriple, Erk, and ErkSD are non-stiff
             solvers, whereas Imex and ImexSD are stiff solvers.  These
             pre-defined functions save a lot of trouble and are good
             for testing these solvers (or any other for that matter)
             
             2) Note also that a pre-defined Butcher tableau
             (mentioned above) was used to define the scheme.  This
             predifined Butcher tableau: Btableau("dopr54") defines a
             Dormand-Prince 5th-order/4th-order scheme with an
             interpolant, and also allows for stiffness detection
             (this is the same Butcher tableau that was explicitly
             defined above).  Since the Dormand-Prince Butcher scheme
             is a "first same as last" scheme, this is automatically
             enabled as well, for efficiency

             3) Note also that the last string is now "Stats_Off" so
             that no stats will be output except for at the end of the
             integration interval (the final solution and such) 
          */

          ErkTriple.erk_triple(new Pleiades(), new Span(0.0, 3.0), x, -1.0, new Btableau("dopr54"), 5.0, atol, rtol, "", "StiffDetect_Off", "EventLoc_Off", "Stats_Off");

          // run a fourth ErkTriple scheme that does interpolation

          /*
             Note that the scheme is the same as the one above except
             that the output points will be different; this is achieved
             through interpolation of the numerical solution

             Before we actually run the routine of the scheme, we must
             initialize an array of times at which we wish to have
             solution values
          */

          double[] times = new double[31];

          for(int i= 0; i< 31; i++)   // the interval [0.0, 3.0] is a good interval, so we
          {   // will choose evenly spaced points on that interval
               times[i] = i * 0.1;
          }

          /*
             Here we are running an ErkTriple routine with continuous
             interpolation.  Before, we had only the points and
             solutions that the Erk Triple routine gave with its
             automatic stepsize control scheme; now we can get
             solution values with the same order of accuracy at any
             point we wish.  Note that a difference between it and the
             3rd routine is that we use an enhanced Span object
             (mentioned above).  This Span object is of the first
             enhanced type: Span(double[] times) and the solver
             interpolates the solution at the times in this double
             array.  One other small difference that this routine has
             from the 3rd routine is that we are writing the
             (interpolated) solution to a file 
           */

          ErkTriple.erk_triple(new Pleiades(), new Span(times), x, -1.0, new Btableau("dopr54"), 5.0, atol, rtol, "erkTripleInterpSoln.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_Off");

          // run another ErkTriple scheme that does interpolation

          /*
             Note that the only difference between this scheme and the
             one above is that we define the interpolant using the
             second enhanced Span object type: Span(double a, double
             b, double inc).  Since this increment is quite small the
             solution will look very smooth.  The solution is written
             to a different file as well 
          */

          ErkTriple.erk_triple(new Pleiades(), new Span(0.0, 3.0, 0.01), x, -1.0, new Btableau("dopr54"), 5.0, atol, rtol, "erkTripleInterpSoln2.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_Off");

          // run an ErkTriple scheme that solves with constant step
          
          /*
             The ErkTriple solver can also solve an IVP with a
             constant step, like the Erk solver does.  The only
             difference is that an interpolant can be defined for this
             constant step solution, whereas Erk does not do
             interpolation.  To define an ErkTriple scheme as a
             constant step scheme, simply set the highOrder argument
             to a value less than or equal to zero.  This essentially
             turns on the constant step feature of ErkTriple.  When
             highOrder is less than or equal to zero, the values of
             all other parameters that have nothing to do with
             constant step are ignored.  Here is an explanation of the
             parameters/arguments:
             
             1) new Pleiades() -> the ODE (still the same)
             
             2) new Span(0.0, 3.0) -> the time span (as before).  Note
             also that an enhanced Span object could also be passed to
             this solver, but because there is no interpolant defined
             for the ERK4 Butcher scheme, only the normal span object
             is passed to this solver
             
             3) x -> the initial value (as always)
             
             4) 3.0E-4 -> the constant stepsize.  This is one of the
             changes invoked the solver switching to constant step.
             Before, this argument would be the initial stepsize, and
             could be less than or equal to zero (thus invoking the
             initial stepsize selection routine).  Now this argument
             is the constant stepsize, the stepsize that will be used
             throughout the entire integration.  When ErkTriple is in
             constant step mode, this argument can not be less than or
             equal to zero or an error occurs
             
             5) new Btableau("erk4") -> this is the Butcher tableau as
             always, but note that a different Butcher scheme has been
             defined for solving this IVP, namely the classic
             4th-order explicit Runge-Kutta scheme
             
             6) -1.0 -> this is the highOrder argument.  Note that
             this argument, being less than or equal to zero, triggers
             the ErkTriple solver to go into constant step mode.  This
             argument alone is the only trigger
             
             7) atol -> this is the absolute tolerance as before,
             which must be an array (as the constructor needs this
             argument); however its size and values are ignored due to
             the fact that this solver will be in constant step mode
             (as tolerances are only needed for automatic stepsize
             control)

             8) rtol -> this is the relative tolerance as before,
             which must be an array (as the constructor needs this
             argument); however its size and values are ignored due to
             the fact that this solver will be in constant step mode
             (as tolerances are only needed for automatic stepsize
             control)
             
             9) "erkTripleConstStep.txt" -> this is the path/filename
             that the solution (solved with a constant step) will be
             written to
             
             10) "StiffDetect_Off" -> this string is the same as that
             for previous schemes, and the constructor needs this
             argument; however its value will be be ignored as
             stiffness detection is not done when the stepsize is
             constant

             11) "EventLoc_Off" -> this string is the same as that for
             previous schemes, and the constructor needs this
             argument; however its value will be be ignored, as event
             location is not done when the stepsize is contant
             
             12) "Stats_On" -> this determines the statistics mode as
             always, and no statistics will be output.  If, however,
             the statistics mode was on, statistics would be output in
             the context of a constant step scheme, as ErkTriple will
             be solving with a constant step */

          ErkTriple.erk_triple(new Pleiades(), new Span(0.0, 3.0), x, 3.0E-4, new Btableau("erk4"), -1.0, atol, rtol, "erkTripleConstStep.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_Off");

	  // run another scheme with constant step

	  /*
	    Note that the only difference between this scheme and the
	    previous one is that there is an extra parameter, 100,
	    after "Stats_Off".  This tells the solver to only write
	    ~100 points to the file; otherwise constant step routines
	    have no way of specifying how many points should be
	    written to file.  Note that all solvers that can do
	    constant step (namely ErkTriple, Erk, and Imex) have this
	    option (where the default is 1000 when the optional
	    parameter is not used).  If this optional parameter is
	    used when ErkTriple is in stepsize control mode, it does
	    not affect anything else.  
          */

          ErkTriple.erk_triple(new Pleiades(), new Span(0.0, 3.0), x, 3.0E-4, new Btableau("erk4"), -1.0, atol, rtol, "erkTripleConstStep2.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_Off", 100);

          /*
             Here we test out our solver on a problem whose exact
             solution is a 5th-order polynomial example by first
             entering in the initial value and the 2 tolerance arrays.
             Note that this scheme will be integrated exactly as we
             are using the Dormand-Prince scheme which is 5th-order
             accurate */

          double[] soln = new double[1];
          soln[0] = 0;

          double[] atol1 = new double[1];
          atol1[0] = 1.0E-8;

          double[] rtol1 = new double[1];
          rtol1[0] = 1.0E-4;

          // now we run the ErkTriple solver with the Dormand-Prince scheme

          /*
             Here we are running ErkTriple with:
             
             1) no interpolation (as span only takes in 2 doubles, not
             an array or 3 doubles)
             2) initial stepsize selection (as -1.0 has been passed as h)
             3) we are writing to a file             
             4) stats are on 
          */

          ErkTriple.erk_triple(new Poly(), new Span(0.0, 10.0), soln, -1.0, new Btableau("dopr54"), 5.0, atol1, rtol1, "poly.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_On");

          /*
             now we define an interval of 21 evenly spaced points to
             which the solution should be interpolated.
          */

          double[] times2 = new double[21];

          for(int i= 0; i< 21; i++)
               times2[i] = i * 0.5;

          // we run another ErkTriple scheme on the polynomial example

          /*
             we now initialize and run a scheme with interpolation,
             given the time interval defined above.  The differences
             between this scheme and the previous are:
             
             1) interpolation is being done, as the Span object is of the first
             enhanced type: Span(double[] times)
             2) we are writing to a different file
             3) stats are off 
          */

          ErkTriple.erk_triple(new Poly(), new Span(times2), soln, -1.0, new Btableau("dopr54"), 5.0, atol1, rtol1, "polyInterp.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_Off");

          /*
             We now test out stiffness detection, a feature of only
             this solver and the specialized Dormand-Prince solver.
             Note also that this solver will only allow stiffness
             detection if it takes in a Dormand-Prince Butcher tableau
             (otherwise an error occurs) 
          */

          // we run an ErkTriple scheme with stiffness detection on
          // the Pleiades problem

          /*
             Note that this scheme is not much different than scheme 3
             except for:
             
             1) the stiffness detection option is now set to
             "StiffDetect_Halt" so that the problem will halt if
             stiffness is detected.  This problem, however, will
             terminate regularly (stiffness will not be detected) as
             this problem is non-stiff
	     2) stats are also turned on, and because stiffness
             detection is also turned on, some extra stats will be
             output due to this feature
          */

          ErkTriple.erk_triple(new Pleiades(), new Span(0.0, 3.0), x, -1.0, new Btableau("dopr54"), 5.0, atol, rtol, "", "StiffDetect_Halt", "EventLoc_Off", "Stats_Off");

          /*
             we are going to now run ErkTriple with stiffness
             detection on the Pollution problem, a stiff problem of
             the CWI testset.  First we initialize the array of
             initial values (note the use "u" instead of "x") 
          */

          // initial values

          double[] u = new double[20];

          u[0] = 0.0;
          u[1] = 0.2;
          u[2] = 0.0;
          u[3] = 0.04;
          u[4] = 0.0;
          u[5] = 0.0;
          u[6] = 0.1;
          u[7] = 0.3;
          u[8] = 0.01;
          u[9] = 0.0;
          u[10] = 0.0;
          u[11] = 0.0;
          u[12] = 0.0;
          u[13] = 0.0;
          u[14] = 0.0;
          u[15] = 0.0;
          u[16] = 0.007;
          u[17] = 0.0;
          u[18] = 0.0;
          u[19] = 0.0;

          // tolerance vectors

          /*
             decrease the tolerances a bit because the problem is stiff
          */
          
          double atol2[] = new double[20];
          for(int i= 0; i< 20; i++)
               atol2[i] = 1.0E-12;
               
          double rtol2[] = new double[20];
          for(int i= 0; i< 20; i++)
               rtol2[i] = 1.0E-6;

          // we now run the ErkTriple solver with stiffness detection
          // on the Pollution problem

          /*
             This scheme does not look very different from the
             previous scheme, the only big differences being the
             function and its span.  However the output will be a bit
             different due to the fact that stiffness will be detected
             before the problem is actually solved (the solver will
             halt) 
          */

          ErkTriple.erk_triple(new Pollution(), new Span(0.0, 60.0), u, -1.0, new Btableau("dopr54"), 5.0, atol2, rtol2, "", "StiffDetect_Halt", "EventLoc_Off", "Stats_Off");

          /*
             Finally, we test event location, a feature of only
             solvers that have interpolants.  For more information on
             calling routines with event location, take a look inside
             the folder "discontinuous_problems" 
          */

          // initial value

          double[] x1 = new double[1];

          x1[0] = 1.0;

          // the tolerances

          double[] atol3 = new double[1];
          for(int i= 0; i< 1; i++)
               atol3[i] = 1.0E-6;   // absolute tolerance 1E-6

          double[] rtol3 = new double[1];
          for(int i= 0; i< 1; i++)
               rtol3[i] = 1.0E-4;   // relative tolerance 1E-4

          // integration interval

          double t0 = 0.0;
          double tf = 20.0;

          // we do one call to the Erk Triple routine to integrate the
          // first piece of the problem f4 from the DEtest test set

          /*
             This scheme does not look that different from scheme 3
             except for:
             
             1) The function takes a parameter "1". This is becuase
             the function is piecewise and "1" denotes that we are
             using the first piece of the function.  We are going to
             solve the first piece of the function until we hit the
             discontinuity in the function (where the event occurs)
             (even though we specified a solution interval of t0 to
             tf, we will never reach tf).  We call the time of this
             discontinuity t* (so we will be solving over the interval
             [t0, t*]), where t* may depend on the initial value, and
             therefore it may not be determined until located by the
             solver.  We will then solve the other part of the
             function from t* + 1.0E-14 to tf (so that we do not have
             a time step that spans the event: that would not be
             valid).  So the second solution will be over the interval
             [~t*, tf]

             2) And of course the "EventLoc_Halt" is used as we want
             the routine to halt upon location of an event (the
             discontinuity). We need to make a separate call to
             advance the solution past the discontinuity
             
             3) We also write the first part of the solution to a file

	     4) Note that we assign the value of the routine to a
	     double array.  This is because this static method of the
	     ErkTriple class actually returns a double array
	     containing some information about the solution.  This
	     information was never needed until now, so we did not
	     bother assigning the value of the routine to an array.
	     Note that this feature is only useful for event location 
          */

          double[] profile = ErkTriple.erk_triple(new F4(1), new Span(t0, tf), x1, -1.0, new Btableau("dopr54"), 5.0, atol3, rtol3, "f4_dopr_locOn.txt", "StiffDetect_Off", "EventLoc_Halt", "Stats_Off");

          // we then get some stats from this routine to use in the
          // second call to Erk Triple

          double hInit = profile[0]; // get h (average stepsize taken
                                     // in solving problem)
          double tStar = profile[1]; // get t*

          double[] x2 = new double[1]; // get y*
          x2[0] = profile[2];
          
          // we then do a small forward Euler step to step over the
          // discontiuity without adding an unacceptable amount of
          // error

          ODE function;   // function used for small forward Euler step
          double hfe = 1.0E-14;   // stepsize for small forward euler step

          function = new F4(2); // note that we do forward Euler with
                                // the function that
                                // will be used in the next ErkTriple call
               
          x[0] = x[0] + hfe * function.f(tStar, x)[0]; // update y*
                                                       // with y* + hf(t*, y*)
                         
          tStar = tStar + hfe;   // update t* with t* + h

          // we make a second ErkTriple solver call to integrate the
          // second piece of f4

          /*
             This scheme looks very much like the previous one except
             for:
             
             1) we call with the second piece of the function (after
             the discontinuity)

             2) we solve over the interval [~t*, tf] (having treated
             the discontinuity properly)

             3) we also use the average stepsize from the previous
             routine as the initial stepsize to this routine (which is
             better than going back to the initial stepsize selection
             routine to get initial stepsize)

             4) we also take the final value from the last solution
             (modified by a small forward Euler step to avoid the
             discontinuity) as the initial value

             5) we disable event location ("EventLoc_Off") because we
             have already passed the discontinuity, and proceed
             normally from here on
             
	     -> Note we are writing the solution to the same file as
	     we did the previous scheme, because we want a single
	     solution passing through all the events.  Because we are
	     doing this, there is this extra string parameter called
	     "Append".  This parameter makes the solver append the
	     solution to the file that it is writing to, instead of
	     overwriting it.  This is an optional parameter where the
	     solver defaults to overwrite when this parameter is not
	     specified.  Note that this optional parameter is only
	     useful for event location, and it is only an optional
	     parameter for those routines with event location (namely
	     ErkTriple and DormandPrince).  Note that in ErkTriple,
	     this optional parameter can also be used alongside the
	     other optional parameter, without any problem 
          */

          ErkTriple.erk_triple(new F4(2), new Span(tStar, tf), x2, hInit, new Btableau("dopr54"), 5.0, atol3, rtol3, "f4_dopr_locOn.txt", "StiffDetect_Off", "EventLoc_Off", "Stats_Off", "Append");

          /*
             Note that there are no dependencies between the special
             features of this solver (except for the fact that an
             interpolant has to exist for a certain scheme in order to
             have event location).  Other than that, for the special
             features interpolation, stiffness detection, event
             location, and stats output, as many or as few options can
             be turned on or off at once
             
             Also note that if a solver does not offer a certain
             option for a given feature, it does not have that
             feature 
           */

          // Start up the plotter to plot results from some schemes

          /*
             This is the plotting tool that can be used to plot the
             solutions to the ODEs after they have been solved with
             the solvers.  The plotter is a GUI that is relatively
             easy to use.  You choose the file and fill in the text
             fields to plot the solution you want, hit the update
             button, and a graph is plotted.  Here is a simple
             explanation of all the fields that create the plot.  It
             is recommended that you follow along with this when the
             GUI pops up when running this tester:

             1) file name -> this is the file containing the solution
             of the problem (with full path name as well).  For
             example, if you wish to plot a solution to the Pleiades
             problem above, you would enter that exact path/file that
             is the file that the Erk Triple routine wrote to.  A much
             easier way to enter this file is to click on the "Choose
             . . ." button below this field, where you can select the
             file by going through directories of files on your
             computer with a file chooser.

             2) Index for x (or t for time) -> this is the array index
             of the solution you wish to put on the x-axis (an integer
             between 0 and (dimension - 1) of the problem).  For
             example, the solution to the Pleiades problem is of
             dimension 28, so you could plot any of those solution
             dimensions by choosing 0-27.  You can also enter t,
             instead of a number, so that you can graph a solution
             dimension against time.

             3) Index for y (or t for time) -> this is the array index
             of the solution you wish to put on the y-axis (integer
             between 0 and (dimension - 1) of problem).  You can also
             put t on this axis as well

             4) title -> this is the name of the title you give your plot

             5) x label -> this is the label you wish to give to the x axis

             6) y label -> this is the label you with to give to the y axis

             7) x axis from -> this is the lower bound of graph
             display on the x axis.  (This does not need to filled in,
             as a default bound specific to the solution will be
             generated.  Note that this is the case for parameters 8,
             10 and 11 as well)

             8) (x axis) to -> upper bound of graph display on the x axis

             9) x tick -> this is distance between points (ticks) on
             the x axis.  For example, if x axis displayed from 0 to 3
             and tick was 1, there would be a tick at 0, 1, 2, and 3
             (i.e., occuring every 1).  There is also a default
             generated for this value (a small value that it defaults
             to if this field is left empty).  There is also a maximum
             value of ticks so that the plotter is not overloaded with
             drawing if the user enters in a very small value.
             parameter 12 has these properties as well
	     
             10) y axis from -> lower bound of graph display on the y axis

             11) (y axis) to -> upper bound of graph display on the y axis

             12) y tick -> distance between points (ticks) on the y axis

             We recommend trying the plotter out with the Pleiades
             problem, using the examples above as a guide.  There are
             also a few files of pre-solved solutions of other
             problems with other solvers in the plotFiles directory.
             Test with those and learn more about the plotter.
             Pictures of the plots for the solutions of ODEs can also
             be looked at in the CWI test set (in the related_files
             directory) for a good comparison 
          */

          Plotter.showPlotter(); // start up the plotter, when the
             // plotter window is closed, the plotter will close
     }
}

/*
   That is the end of the walkthrough for the major functions of this
   package. We encourage users to take a look at the other testers as
   well in order to learn even more.  There is a test project for each
   solver in this package.  After learning all about the solvers in
   this testers folder you should be able to go to the templates
   folder and start solving problems on your own!
*/
