      SUBROUTINE ALLS1(WHICH, JEX, FEX, NEQ, H, XIN, XOUT, 
     $                 WORK, IWORK, INFO)
      IMPLICIT         NONE
*
*     .. Scalar Arguments ..
      INTEGER          WHICH, NEQ, INFO
      DOUBLE PRECISION H
*
*     .. Array Arguments ..
      INTEGER          IWORK(NEQ)
      DOUBLE PRECISION XIN(NEQ), XOUT(NEQ), WORK(NEQ*(NEQ+1))
*
*     .. External Subroutines ..
      EXTERNAL         JEX, FEX
*
*-------------------------------------------------------------------------
*
*  Purpose
*  =======
*
*  ALLS1 is the driver routine for obtaining various palindromic schemes.
*  (See documentation for WHICH below.) It is written, together with
*  ONESTEP, for a general quadratic differential system:
*
*         y' = f(y),
*
*  where each component of f(y) is qudratic in y.
*
*                                       Ren-Cang Li, June 1, 1996
*                                       na.rcli@na-net.ornl.gov
*
*  Arguments:
*  ==========
*
*  WHICH     (input) INTEGER scalar
*            Identify WHICH palindromic schemes as follows:
*
*              -------------------------------------------
*             |                                           |
C             |    WHICH                  The METHOD      |
C             |                                           |
C             |      1                      s1odr2        |
C             |      2                      s3odr4        |
C             |      3                      s5odr4        |
C             |      4                      s5odr4a       |
C             |      5                      s7odr6        |
C             |      6                      s9odr6a       |
C             |      7                      s9odr6b       |
C             |      8                     s15odr8        |
C             |      9                     s17odr8a       |
C             |     10                     s17odr8b       |
C             |     11                    s31odr10a       |
C             |     12                    s31odr10b       |
C             |     13                    s33odr10a       |
C             |     14                    s33odr10b       |
C             |     15                    s33odr10c       |
*             |                                           |
*              -------------------------------------------
*
*  JEX       External Subroutine that calculates the Jacobian of f(y);
*
*  FEX       External Subroutine that calculates f(y).
*
*  NEQ       (input) INTEGER scalar
*            The dimension of the system.
*
*  H         (input) DOUBLE PRECISION scalar
*            Stepsize.
*
*  XIN       (input) DOUBLE PRECISION array, dimension (NEQ)
*            The value of y at T, the current time. (Since this is for
*            autonomous system, T does not appear as an argument.)
*
*  XOUT      (output) DOUBLE PRECISION array, dimension (NEQ)
*            The calculated y at T+H.
*
*  WORK      (working)  DOUBLE PRECISION array, dimension ( NEQ*NEQ+NEQ )
*
*  IWORK     (working) INTEGER array, dimension ( NEQ )
*
*  INFO      (output) INTEGER scalar
*            It may be used as a flag to indicate, e.g., the successfullness
*            of doing Gaussian elimination with partial pivoting.
C
C--------------------------------------------------------------------
C
C  local parameters that define various palindromic schemes.
C
      DOUBLE PRECISION DT2(2), DT3(3), DT4(3), DT5(4), DT6(5),
     $                 DT7(5), DT8(8), DT9(9), DT10(9), DT11(16),
     $                 DT12(16), DT13(17), DT14(17), DT15(17)
      DATA  DT2/ 1.351207191959657634D0,  -1.7024143839193152681D0/
      DATA  DT3/ 0.41449077179437573714D0, 0.41449077179437573714D0, 
     $          -0.65796308717750294857D0/
      DATA  DT4/ 0.78867513459481288226D0, 0.21132486540518711774D0,
     $          -1.0D0/
      DATA  DT5/ 0.78451361047755726382D0, 0.23557321335935813368D0,
     $          -1.1776799841788710069D0,  1.3151863206839112189D0/
      DATA  DT6/ 0.39216144400731413928D0, 0.33259913678935943860D0,
     $          -0.70624617255763935981D0, 0.082213596293550800230D0,
     $           0.79854399093482996340D0/
      DATA  DT7/ 0.39103020330868478817D0,  0.33403728961113601749D0,
     $          -0.70622728118756134346D0,  0.081877549648059445768D0,
     $           0.79856447723936218406D0/
      DATA  DT8/ 0.74167036435061295345D0, -0.40910082580003159400D0,
     $           0.19075471029623837995D0, -0.57386247111608226666D0,
     $           0.29906418130365592384D0,  0.33462491824529818378D0,
     $           0.31529309239676659663D0, -0.79688793935291635402D0/
      DATA  DT9/ 0.13020248308889008088D0,  0.56116298177510838456D0,
     $          -0.38947496264484728641D0,  0.15884190655515560090D0,
     $          -0.39590389413323757734D0,  0.18453964097831570709D0,
     $           0.25837438768632204729D0,  0.29501172360931029887D0,
     $          -0.60550853383003451170D0/
      DATA DT10/ 0.12713692773487857916D0,  0.56170253798880269972D0,
     $          -0.38253471994883018888D0,  0.16007605629464743119D0,
     $          -0.40181637432680696673D0,  0.18736671654227849724D0,
     $           0.26070870920779240570D0,  0.29039738812516162389D0,
     $          -0.60607448323584816258D0/
      DATA DT11/-0.48159895600253002870D0, 0.36303931544595926879D-2,
     $           0.50180317558723140279D0, 0.28298402624506254868D0,
     $           0.80702967895372223806D0, -0.26090580538592205447D-1,
     $          -0.87286590146318071547D0, -0.52373568062510581643D0,
     $           0.44521844299952789252D0, 0.18612289547097907887D0,
     $           0.23137327866438360633D0, -0.52191036590418628905D0,
     $           0.74866113714499296793D0, 0.66736511890604057532D-1,
     $          -0.80360324375670830316D0, 0.91249037635867994571D0/
      DATA DT12/ 0.27338476926228452782D0, 0.44587846502560283997D0,
     $           0.83219642847136307126D0, -0.83396868554957942879D0,
     $           0.27891843057015194293D0, 0.89032738045702532006D0,
     $           0.56681514845245709418D-1, -0.85737420814978887722D0,
     $          -0.46789492554836586111D0, -0.47919009182398264249D0,
     $           0.16724074680043708909D0, -0.87443151263376143307D0,
     $          -0.49873481853620165786D0, 0.58930536608974918851D0,
     $           0.83458937790882729775D0, 0.28614352562198582747D0/
      DATA DT13/ 0.70428877682658066880D-1, 0.87415651735353949041D0,
     $           0.55414604963802442707D-1, -0.66800477898797011598D-1,
     $          -0.62641308958799555593D0, 0.23682621087528762872D0,
     $          -0.42221063403170054210D0, 0.24222942201040859249D0,
     $           0.47374515478601436594D-1, 0.54386826052472423338D0,
     $          -0.93252230928447264311D0, 0.16960179883676464855D0,
     $           0.71608567578450563608D0, -0.80016730247310573512D0,
     $           0.23778185292256770747D0, -0.32330301550863943389D0,
     $           0.95529818470370207691D0/
      DATA DT14/ 0.12282427644721572094D0, 0.77644680890696440342D0,
     $           0.14881514553734297479D0, -0.17239125953506067249D0,
     $          -0.54745995781852463787D0, 0.14512932327306927479D0,
     $          -0.31564555153114460562D0, 0.12086865089833871979D0,
     $           0.17910277517866344258D0, 0.44263408813993245949D0,
     $          -0.81935337479593697464D0, 0.13445474141752884045D0,
     $           0.64444239169016646538D0, -0.71930149370201612557D0,
     $           0.21036902497348664610D0, -0.26908194941570516294D0,
     $           0.83629272067135846284D0/
      DATA DT15/ 0.12313526870982994083D0, 0.77644981696937310520D0,
     $           0.14905490079567045613D0, -0.17250761219393744420D0,
     $          -0.54871240818800177942D0, 0.14289765421841842100D0,
     $          -0.31419193263986861997D0, 0.12670943739561041022D0,
     $           0.17444734584181312998D0, 0.44318544665428572929D0,
     $          -0.81948900568299084419D0, 0.13382545738489583020D0,
     $           0.64509023524410605020D0, -0.71936337169922060719D0,
     $           0.20951381813463649682D0, -0.26828113140636051966D0,
     $           0.83647216092348048955D0/
C
C  Excutable statements
C
      IF (WHICH .EQ. 1) THEN
C
C        s1odr2
C
         CALL ONESTEP(JEX, FEX, NEQ, H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 2) THEN
C
C        s3odr4
C
         CALL ONESTEP(JEX, FEX, NEQ, DT2(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT2(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT2(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 3) THEN
C
C        s5odr4
C
         CALL ONESTEP(JEX, FEX, NEQ, DT3(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT3(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT3(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT3(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT3(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 4) THEN
C
C        s5odr4a
C
         CALL ONESTEP(JEX, FEX, NEQ, DT4(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT4(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT4(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT4(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT4(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 5) THEN
C
C        s7odr6
C
         CALL ONESTEP(JEX, FEX, NEQ, DT5(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT5(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT5(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT5(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT5(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT5(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT5(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 6) THEN
C
C        s9odr6a
C
         CALL ONESTEP(JEX, FEX, NEQ, DT6(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT6(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 7) THEN
C
C        s9odr6b
C
         CALL ONESTEP(JEX, FEX, NEQ, DT7(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT7(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 8) THEN
C
C        s15odr8
C
         CALL ONESTEP(JEX, FEX, NEQ, DT8(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT8(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 9) THEN
C
C        s17odr8a
C
         CALL ONESTEP(JEX, FEX, NEQ, DT9(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT9(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
      ELSE IF (WHICH .EQ. 10) THEN
C
C        s17odr8b
C
         CALL ONESTEP(JEX, FEX, NEQ, DT10(1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT10(1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
C
      ELSE IF (WHICH .EQ. 11) THEN
C
C        s31odr10a
C
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT11( 1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
C
      ELSE IF (WHICH .EQ. 12) THEN
C
C        s31odr10b
C
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT12( 1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
C
      ELSE IF (WHICH .EQ. 13) THEN
C
C        s33odr10a
C
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(17)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT13( 1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
C
      ELSE IF (WHICH .EQ. 14) THEN
C
C        s33odr10b
C
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(17)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT14( 1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
C
      ELSE IF (WHICH .EQ. 15) THEN
C
C        s33odr10c
C
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 1)*H,  XIN, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(17)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(16)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(15)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(14)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(13)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(12)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(11)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15(10)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 9)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 8)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 7)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 6)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 5)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 4)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 3)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 2)*H, XOUT, WORK, 
     $                WORK(NEQ+1), IWORK, INFO)
         CALL ONESTEP(JEX, FEX, NEQ, DT15( 1)*H, WORK, XOUT, 
     $                WORK(NEQ+1), IWORK, INFO)
C
      ELSE IF (WHICH .GT. 15) THEN
	   PRINT*, ' ERROR in ALLS1: Out of Bound'
      END IF
C
C     End of ALLS1
C
      RETURN
      END
